import sys
import gst
import gobject

from gobject import GObject
from youamp import Playlist, MAX_VOL

class Player(GObject):
    """
    A Playlist based player
    
    signals:
    song-changed: the song has changed
    song-played: the song was played
    tags-updated: the tags changed
    seek-chaned: the seek changed
    toggled: player was toggled
    """
    __gsignals__ = {"song-changed": (gobject.SIGNAL_RUN_LAST, None, (object,)),
                    "song-played": (gobject.SIGNAL_RUN_LAST, None, (object,)),
                    "tags-updated": (gobject.SIGNAL_RUN_LAST, None, (object,)),
                    "seek-changed": (gobject.SIGNAL_RUN_LAST, None, (float,)),
                    "toggled": (gobject.SIGNAL_RUN_LAST, None, (bool,))}
                                   
    def __init__(self, config):
        GObject.__init__(self)
        self.playing = False
        self.playlist = None

        self._current = None
        self._player = None
        self._seek_emit_id = None
        self._set_duration_id = None
        self._config = config
        self._vol = config["volume"] * MAX_VOL
        
        self._config.notify_add("volume", self._set_volume)
        self._config.notify_add("rg-preamp", lambda client, cxn_id, entry, data: self._apply_volume())
        self._config.notify_add("no-rg-preamp", lambda client, cxn_id, entry, data: self._apply_volume())
    
    def next(self, play=True, *args):
        if self.playlist.jump_to is not None:
            self.goto_pos(self.playlist.jump_index(), play)
            self.playlist.jump_to = None
        else:
            self.goto_pos(self.playlist.next_song(self._current), play)

    def tracks(self):
        return len(self.playlist)

    def goto_pos(self, pos, play=True):
        # emit song-played for previous track
        if self._current is not None:
            self._emit_played()
        
        pos %= len(self.playlist)
        self._current = self.playlist[pos]

        self.playlist.pos = pos
        
        if play: 
            self.start_playback()

    def previous(self):
        if self.get_time() > 3:
            self._emit_played()
            self.seek_to(0.0)
            return

        self.goto_pos(self.playlist.pos - 1)
    
    def get_time(self):
        """returns position in seconds"""
        try:
            s = self._player.query_position(gst.FORMAT_TIME)[0] / gst.SECOND
            return s
        except gst.QueryError, e:
            sys.stderr.write("get_time: %s\n" % e)

    def get_seek(self):
        """returns position as fraction"""
        try:
            length = float(self._player.query_duration(gst.FORMAT_TIME)[0])
        except gst.QueryError, e:
            sys.stderr.write("get_seek: %s\n" % e)
            return 0.0
        else:            
            return self._player.query_position(gst.FORMAT_TIME)[0] / length

    def seek_to(self, pos):
        try:
            length = float(self._player.query_duration(gst.FORMAT_TIME)[0])
        except gst.QueryError, e:
            sys.stderr.write("seek_to: %s\n" % e)
        else:        
            self._player.seek_simple(gst.FORMAT_TIME, gst.SEEK_FLAG_FLUSH, int(length * pos))
    
    def _emit_played(self):
        self.emit("song-played", self._current)
            
    def load_track(self):
        if self._current is None:
            self._current = self.playlist[self.playlist.pos]
        
        if self._player is not None:
            self._player.set_state(gst.STATE_NULL)
        
        # FIXME Maemo needs the playbin to be recreated every time
        # a new file is loaded
        self._player = gst.element_factory_make("playbin")
        
        bus = self._player.get_bus()
        bus.add_signal_watch()

        # set up signal handlers
        bus.connect("message::buffering", self._on_buffering)
        bus.connect("message::tag", self._on_tag)
        bus.connect("message::eos", self.next)
        bus.connect("message::error", self._on_error)
        bus.connect("message::state-changed", self._on_state_changed)

        self._player.set_property("uri", "file://"+self._current.uri)
        self._player.set_state(gst.STATE_PAUSED)

    def _on_state_changed(self, bus, message):
        # wait for playbin to move from ready to paused
        # up to there all tag events were fired
        if message.src.get_name().startswith("playbin"):
            state = message.parse_state_changed()
            if state[0:2] == [gst.STATE_READY, gst.STATE_PAUSED]:
                self._apply_volume()
                
                # FIXME duration should be set directly here
                if self._set_duration_id is not None:
                    gobject.source_remove(self._set_duration_id)
                self._set_duration_id = gobject.timeout_add(1000, self._set_duration)
                self.emit("song-changed", self._current)
    
    def _set_duration(self):
         self._current["duration"] = int(self._player.query_duration(gst.FORMAT_TIME)[0] / gst.SECOND)
         return False
   
    def start_playback(self):
        if not self.playing:
            self._emit_seek_id = gobject.timeout_add(1000, self._emit_seek)
            self.emit("toggled", True)

        self.playing = True
        self.load_track()
        self._player.set_state(gst.STATE_PLAYING)

    def _on_error(self, bus, message):
        sys.stderr.write("Error: %s\n" % message.parse_error()[0])
        
    def __del__(self):
        self._player.set_state(gst.STATE_NULL)

    def _on_tag(self, bus, message):
        self._update_song(message.parse_tag())
        self.emit("tags-updated", self._current)
        
    
    def mute(self):
        self._player.set_property("volume", 0.0)
    
    def _set_volume(self, client, cxn_id, entry, data):
        vol = min(1.0, max(0.0, entry.get_value().get_float()))

        self._vol = vol * MAX_VOL        
        
        self._apply_volume()
        
    def toggle(self):
        if self.playing:
            self._player.set_state(gst.STATE_PAUSED)
            self.playing = False
            gobject.source_remove(self._emit_seek_id)
        else:
            self._player.set_state(gst.STATE_PLAYING)
            self.playing = True
            
            self._emit_seek_id = gobject.timeout_add(1000, self._emit_seek)
        
        self.emit("toggled", self.playing)

    def _on_buffering(self, bus, message):
        print message.structure["buffer-percent"]
    
    def _update_song(self, new_tags):        
        for key in new_tags.keys():
            v = new_tags[key]

            if key in ("artist", "album", "title") and isinstance(v, list):
                v = ", ".join(v) # flatten
            
            self._current[key] = v

    def has_rg_info(self):
        cur = self._current
        
        return "replaygain-track-gain" in cur or "replaygain-album-gain" in cur
    
    def _get_rg_scale(self):
        pos = self._config["pos"]
        length = self.tracks()
        cur = self._current
        prev = self.playlist[(pos - 1) % length]
        next = self.playlist[(pos + 1) % length]
    
        if cur["album"] == "None":
            same_album = False
        else:
            same_album = cur["album"] in (prev["album"], next["album"])
        
        if self.has_rg_info():
            if same_album and "replaygain-album-gain" in cur:
                rg_value = float(cur["replaygain-album-gain"])
            else:
                rg_value = float(cur["replaygain-track-gain"])
                
            rg_value += self._config["rg-preamp"]
        else:
            rg_value = self._config["no-rg-preamp"]

        return pow(10, rg_value / 20.0) # rg_scale

    def _apply_volume(self):
        vol = self._vol

        vol = vol * self._get_rg_scale()  # scale global volume
        vol = min(10.0, max(vol, 0.0))    # cut to the gst.playbin range

        self._player.set_property("volume", vol)

    def _emit_seek(self):
        # do not send update if we are already paused
        try:
            self.emit("seek-changed", self.get_seek())
        except gst.QueryError:
            # dont hiccup
            pass
        
        return True # if True will be called repeatedly
