import gtk
import hildon
import gobject
import time
import urllib2
import sys
import dbus
import os.path

from youamp import VERSION, Playlist, PlaylistMux, IS_MAEMO
from youamp.ui.interface import UserInterface

from youamp.library import Library, check_db
from youamp.indexer import Indexer
from youamp.player import Player
from youamp.config import Config
import youamp.scrobbler as scrobbler

if IS_MAEMO:
    import osso
    
    osso_c = osso.Context("net.madman2k.youamp", VERSION, False)

class Controller:
    def __init__(self):
        check_db()
        self.config = Config("/apps/youamp/")
        check_config(self.config)
        self.player = Player(self.config)
        self.library = Library()
        self._indexer = Indexer()
        self.main_list = Playlist(title=_("Library"))
        self.main_list.pos = self.config["pos"]
        self.jump_to = None
        self.player.playlist = PlaylistMux(self.main_list)
        
        gobject.set_application_name("YouAmp")

        # last.fm submission
        self.scrobbler = scrobbler.Scrobbler()
        self.scrobbler.connect("error", lambda c, t, m: sys.stderr.write("scrobbler: "+m+"\n"))
        self.scrobbler.login(self.config["lastfm-user"], self.config["lastfm-pass"], ("you", VERSION))

        # RPC
        if IS_MAEMO:
            osso_rpc = osso.Rpc(osso_c)
            osso_rpc.set_rpc_callback("net.rojtberg.youamp", \
                                      "/net/rojtberg/youamp", \
                                      "net.rojtberg.youamp", self._handle_rpc)
        
        self.gui = UserInterface(self)

        # program
        program = hildon.Program()
        program.add_window(self.gui.window._w)
        program.set_property("can-hibernate", True)
    
        self._restore()
        
        self.player.connect("song-played", self._on_song_played)
        self.config.notify_add("music-folder", self._set_new_playlist)
        self.config.notify_add("is-browser", self._set_new_playlist)
        self.player.connect("song-changed", self._now_playing)
        self.player.connect("song-changed", self._update_pos)

        self._indexer.connect("update-complete", self._on_library_updated)

        # start updating library
        self._refresh_library()

    def _refresh_library(self, *args):
        self._banner = hildon.hildon_banner_show_animation(self.gui.window._w, "qgn_indi_pball_a", _("Updating database"))
        self._indexer.update(self.config["music-folder"])

    def _on_library_updated(self, caller, was_updated):
        gobject.idle_add(self._banner.destroy)
        
        if was_updated:
            gobject.idle_add(self._refresh_playlist)

    def _handle_rpc(self, interface, method, arguments, user_data):
        action = dict();
        action["toggle"] = self._player.toggle
        action["prev"] = self._player.previous
        action["next"] = self._player.next
        
        action[method]()

    def _update_pos(self, player, *args):
        if player.playlist is self.main_list:
            self.config["pos"] = self.main_list.pos

    def _restore(self):
        self._refresh_playlist()
 
        if self.config["shuffle"]:
            self.player.playlist.shuffle(True)

        # restore playlist pos
        try:
            self.player.load_track()
        except IndexError:
            # track might have been deleted
            pass
        
        self.gui.restore()
    
    def set_list_order(self, order):
        if order in ("album", "playcount"):
            self.main_list.order_by(order)
            return

        self.config["order-by"] = order
        self.config["shuffle"] = False
        self.config["pos"] = 0
        
        self._refresh_playlist()
    
    def _set_new_playlist(self, *args):        
        self._refresh_playlist()
        
        if self.config["shuffle"]:
            self.main_list.shuffle(True)
        
        self.config["pos"] = 0
    
    def _refresh_playlist(self):
        playlist = self.library.get_tracks(self.config)
            
        self.main_list.update(playlist)

    def _on_song_played(self, player, song):
        played = player.get_time()
        
        # workaround for bugged maemo
        if not "duration" in song:
            return
        
        if (played < song["duration"]/2 and played < 240) or song["duration"] < 30:
            return
        
        # call library from main thread
        gobject.idle_add(self.library.increment_played, song.uri)

        if self.scrobbler.is_connected():
            self.scrobbler.submit(song["artist"], song["title"], int(time.time()), album=song["album"], length=song["duration"])
        else:
            sys.stderr.write("scrobbler: song %s discarded\n" % (song["title"]))
                

    def _now_playing(self, player, song):        
        if not self.gui.window.visible():
            self.gui.show_notification(song)

        if self.scrobbler.is_connected():
            self.scrobbler.now_playing(song["artist"], song["title"], song["album"])
        else:
            sys.stderr.write("now playing: song %s discarded\n" % (song["title"]))

    def start(self):
        gtk.main()
        
    def quit(self, *args):
        # submit to last.fm
        if self.scrobbler.is_connected():
            try:
                self.scrobbler.flush()
            except urllib2.URLError, e:
                sys.stderr.write("lastfm.submit: %s\n" % e)
        
        gtk.main_quit()

def check_config(config):
    if not "volume" in config:
        # write default config
        config["volume"] = 0.5
        config["search-str"] = ""
        config["search-artist"] = ""
        config["search-album"] = ""
        config["pos"] = 0
        config["shuffle"] = False
        config["is-browser"] = False
        config["rg-preamp"] = 0     # preamp to adjust the default of 89db (value: db)
        config["no-rg-preamp"] = -10  # amp value to be used if no rg info is available (value: db)
    
    # order by was added in v0.3.5, therefore check for it seperately
    if not "order-by" in config:
        config["order-by"] = "album"

    # order by renamed in v0.5.8
    if config["order-by"] == "artist":
        config["order-by"] = "album"

    # music-folder was added in v0.3.8
    if not "music-folder" in config:
        config["music-folder"] = os.path.expanduser("~")
        
    # last.fm support was added in v0.4.0
    if not "lastfm-user" in config:
        config["lastfm-user"] = ""
        config["lastfm-pass"] = ""
