import sqlite3
import gst
import gobject
import os
import mutagen
import mutagen.easyid3

from gobject import GObject
from threading import Thread
from youamp import Song, music_dirs, known_exts, db_file, blacklist_dirs
from mutagen.id3 import ID3NoHeaderError

order = dict()
order["artist"] = "artist ASC, album ASC, tracknumber ASC"
order["date"] = "date DESC, "+order["artist"]
order["playcount"] = "playcount DESC, "+order["artist"]

class Library(GObject, Thread):
    __gsignals__ = {"update-complete": (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ())}

    def __init__(self):
        Thread.__init__(self)
        GObject.__init__(self)
        
        # set up connection for non thread communication
        self._conn = sqlite3.connect(db_file)
        self._cursor = self._conn.cursor()

    def __del__(self):
        self.join()
        self._conn.close()

    def get_artists(self):
        artists = self._cursor.execute("""SELECT DISTINCT artist FROM songs ORDER BY artist ASC""")
        
        return artists

    def get_albums(self, artist):
        where_clause = ""
        variables = ()
        
        if artist != "":
            where_clause += "WHERE artist = ?"
            variables += (artist,)

        albums = self._cursor.execute("""SELECT DISTINCT album
                                          FROM songs
                                          %s
                                          ORDER BY artist ASC, album ASC""" % where_clause, variables)
        
        return albums
    
    def get_tracks(self, config):        
        where_clause, variables = self._build_where_clause(config)

        playlist = self._cursor.execute("""SELECT uri, title, artist, album
                                          FROM songs 
                                          %s
                                          ORDER BY %s""" % (where_clause, order[config["order-by"]]), \
                                          variables).fetchall()

        return map(lambda e: Song(e), playlist)
    
    def _build_where_clause(self, config):
        where_clause = ""
        variables = ()
        
        if config["is-browser"]:
            artist = config["search-artist"]
            album = config["search-album"]
            
            if (artist != "" or album != ""):
                where_clause = "WHERE "
            
            # build query
            if artist != "":
                where_clause += "artist = ?"
                variables += (artist,)
            
            if artist != "" and album != "":
                where_clause += " AND "
            
            if album != "":
                where_clause += "album = ?"
                variables += (album,) 
        else:
            name = config["search-str"]
            
            if name != "":
                where_clause += "WHERE artist || album || title LIKE ?"
                name = "%"+"%".join(name.split())+"%"  # insert wildcard on spaces
                variables = (name,)
        
        return (where_clause, variables)

    def increment_played(self, song_uri):
        self._cursor.execute("""UPDATE songs SET playcount = playcount + 1 WHERE uri = ?""", (song_uri,))
        self._conn.commit()

    # start update thread
    def update(self):
        self.start()

    def _get_files(self):
        filelist = []
        for d in music_dirs:
            path = os.path.expanduser(d)

            for root, dirs, files in os.walk(path):
                if root.startswith(blacklist_dirs):
                    continue

                for f in files:
                    if f.lower().endswith(known_exts):
                        uri = "file://"+os.path.join(root, f)
                        uri = uri.replace("%", "%25") #escape the uri escape char
                        filelist.append(uri)

        return filelist
 
    def run(self):    
        # local connection for Thread
        conn = sqlite3.connect(db_file)
        cursor = conn.cursor()

        act_files = self._get_files()
        db_files = cursor.execute("SELECT uri FROM songs").fetchall()
    
        for uri, in db_files:
            try:
                act_files.remove(str(uri))   # convert unicode
            except ValueError:
                 cursor.execute("DELETE FROM songs WHERE uri = ?", (uri,))
                 print "Removed: %s" % uri

        # add new files
        for uri in act_files:
            song = {"title": "None", "artist": "None", "album": "None", "genre": "None", "tracknumber": 0}
            
            # only look at local files
            if not uri.startswith("file://"):
                continue
                
            path = uri[7:] # strip prefix
            path = path.replace("%25", "%") #unescape uri
            
            if path.lower().endswith("mp3"):
                try:
                    f = mutagen.easyid3.EasyID3(path)
                except ID3NoHeaderError:
                    sys.stderr.write("Bad ID3 Header, skipping: %s" % path)
                    continue
            else:
                f = mutagen.File(path)
                
            if f is None:
                sys.stderr.write("Not a media file, skipping: %s" % path)
                continue
            
            for k, v in f.items():
                # FIXME what if v is not a list
                try:
                    v = str(", ".join(v))   # flatten value
                except TypeError:
                    sys.stderr.write("wrong type for %s in %s" % (k, path))
                    continue
                
                if k == "tracknumber":                        
                    try:
                        slash = v.index("/")
                        v = v[0:slash]
                    except ValueError:
                        pass
                
                song[k] = v
            
            # no visible tags -> set title = filename
            if song["title"] == "None" and \
               song["artist"] == "None" and \
               song["album"] == "None":
                    fname = uri.rindex("/")
                    fnend = uri.rindex(".")
                    song["title"] = uri[fname+1:fnend]

            # store metadata in database (uri, title, artist, album, genre, tracknumber)
            cursor.execute("INSERT INTO songs VALUES (?, ?, ?, ?, ?, ?, 0, datetime('now'))", \
                (uri, song["title"], song["artist"], song["album"], song["genre"], song["tracknumber"]))

            print "Added: %s" % uri
                
        conn.commit()
        conn.close()
        self.emit("update-complete")

def check_db():
    conn = sqlite3.connect(db_file)
    c = conn.cursor()
    
    # the playcount column was introduced in v0.3.5
    try:
        c.execute("SELECT playcount FROM songs LIMIT 1")
    except sqlite3.OperationalError:
        c.execute("ALTER TABLE songs ADD COLUMN playcount INT")

def setup_db():
    conn = sqlite3.connect(db_file)
    c = conn.cursor()

    c.execute("CREATE TABLE songs (uri TEXT, title TEXT, artist TEXT, album TEXT, genre TEXT, tracknumber INT, playcount INT, date TEXT)")
    #c.execute("CREATE TABLE playlists (name TEXT)")

    conn.close()
