import gst
import gobject
import mutagen

from gobject import GObject
from youamp import Song, Playlist, MAX_VOL, IS_MAEMO

class Player(GObject):
    __gsignals__ = {"song-changed": (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,)),
                    "song-played": (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,)),
                    "tags-updated": (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,))}
                                   
    def __init__(self, config):
        GObject.__init__(self)
        self.playing = False
        self.playlist = Playlist()
        self._current = None
        self._player = None
        self._config = config
        self._vol = config["volume"] * MAX_VOL
        
        self._config.notify_add("volume", self._set_volume)
        self._config.notify_add("rg-preamp", lambda client, cxn_id, entry, data: self._apply_volume())
        self._config.notify_add("no-rg-preamp", lambda client, cxn_id, entry, data: self._apply_volume())

    def next(self, bus=None, message=None):
        if self.get_time() > 30:
            self.emit("song-played", self._current)
        
        self._config["pos"] = (self._config["pos"] + 1) % len(self.playlist)        
        self.start_playback()

    def tracks(self):
        return len(self.playlist)

    def goto_pos(self, pos):
        if self.get_time() > 30:
            self.emit("song-played", self._current)

        self._config["pos"] = pos % len(self.playlist)        
        self.start_playback()

    def previous(self):
        if self.get_time() > 30:
            self.emit("song-played", self._current)

        self._config["pos"] = (self._config["pos"] - 1) % len(self.playlist)        
        self.start_playback()
    
    def get_time(self):
        """returns position in seconds"""
        try:
            s = self._player.query_position(gst.FORMAT_TIME)[0]/ 1000000000 # ns*10^-9 = s
            return s
        except Exception:
            print "get_time: player not ready"

    def get_seek(self):
        """returns position as fraction"""
        try:
            length = float(self._player.query_duration(gst.FORMAT_TIME)[0])
        except Exception:
            print "get_seek: player not ready"
            return 0.0
        else:            
            return self._player.query_position(gst.FORMAT_TIME)[0]/length

    def seek_to(self, pos):
        try:
            length = float(self._player.query_duration(gst.FORMAT_TIME)[0])
        except Exception:
            print "seek_to: player not ready"
        else:        
            self._player.seek_simple(gst.FORMAT_TIME, gst.SEEK_FLAG_FLUSH, int(length*pos))

    def load_track(self):
        try:
            self._current = self.playlist[self._config["pos"]]
        except IndexError:
            print "Error: invalid playlist position"
            return
        
        if self._player is not None:
            self._player.set_state(gst.STATE_NULL)
        
        self._player = gst.element_factory_make("playbin")
        
        bus = self._player.get_bus()
        bus.add_signal_watch()

        # set up signal handlers
        bus.connect("message::buffering", self._on_buffering)
        bus.connect("message::tag", self._on_tag)
        bus.connect("message::eos", self.next)
        bus.connect("message::error", self._on_error)

        self._player.set_property("uri", self._current.uri)
        self._player.set_state(gst.STATE_PAUSED)
        # up to here core tag events were fired (rg info not there yet)
        
        # maemo has broken gstreamer, so use mutagen instead
        if not self._current.uri.endswith("mp3"):
            self._update_tags(self._current.uri[7:])
        
        self.emit("song-changed", self._current)

    def start_playback(self):
        self.playing = True
        self.load_track()
        self._player.set_state(gst.STATE_PLAYING)

    def _on_error(self, bus, message):
        print "Error: %s" % message.parse_error()[0]

    def __del__(self):
        self._player.set_state(gst.STATE_NULL)

    def _update_tags(self, path):        
        for k, v in mutagen.File(path).items():
            k = k.replace("_", "-") # use gstreamer formatting "-" istead of "_"
            v = str(", ".join(v))   # flatten value
            if k [0:10] == "replaygain":
                v = v[0:-3]
            
            self._current[k] = v
        
        self.emit("tags-updated", self._current)
        
        # FIXME immediate rg applying gives sound disortion (<=> volume > 1.0)
        gobject.timeout_add(3000, self._apply_volume)

    def _on_tag(self, bus, message):
        self._update_song(message.parse_tag())
        self._apply_volume()
        self.emit("tags-updated", self._current)
    
    def mute(self):
        self._player.set_property("volume", 0.0)
    
    def _set_volume(self, client, cxn_id, entry, data):
        vol = min(1.0, max(0.0, entry.get_value().get_float()))

        self._vol = vol * MAX_VOL        
        
        self._apply_volume()
		
    def toggle(self):
        if self.playing:
            self._player.set_state(gst.STATE_PAUSED)
            self.playing = False
        else:
            self._player.set_state(gst.STATE_PLAYING)
            self.playing = True

    def _on_buffering(self, bus, message):
        print message.structure["buffer-percent"]
	
    def _update_song(self, new_tags):        
        for key in new_tags.keys():
            v = new_tags[key]

            if key in ("artist", "album", "title") and isinstance(v, list):
                v = ", ".join(v) # flatten
            
            self._current[key] = v

    def has_rg_info(self):
        cur = self._current
        
        return cur.has_key("replaygain-track-gain") or cur.has_key("replaygain-album-gain")
    
    def _get_rg_scale(self):
        pos = self._config["pos"]
        length = self.tracks()
        cur = self._current
        prev = self.playlist[(pos - 1) % length]
        next = self.playlist[(pos + 1) % length]
    
        if cur["album"] == "None":
            same_album = False
        else:
            same_album = cur["album"] in (prev["album"], next["album"])
        
        if self.has_rg_info():
            if same_album and cur.has_key("replaygain-album-gain"):
                rg_value = float(cur["replaygain-album-gain"])
            else:
                rg_value = float(cur["replaygain-track-gain"])
        		
            rg_value += self._config["rg-preamp"]
        else:
            rg_value = self._config["no-rg-preamp"]

        return pow(10, rg_value/20.0) # rg_scale

    def _apply_volume(self):
        vol = self._vol

        vol = vol * self._get_rg_scale()  # scale global volume
        vol = min(10.0, max(vol, 0.0))    # cut to the gst.playbin range

        self._player.set_property("volume", vol)
