import gtk
import hildon
import gobject

from youamp.ui.window import Window
from youamp.ui.about import AboutDialog
from youamp.ui.preferences import Preferences

from youamp import Playlist, IS_MAEMO, VERSION

from youamp.library import Library, check_db
from youamp.player import Player
from youamp.config import Config

if IS_MAEMO:
    import osso
    
    osso_c = osso.Context("net.madman2k.youamp", VERSION, False)

class Controller:
    def __init__(self):
        check_db()
        self._config = Config("/apps/youamp/")
        self._check_config()
        self._player = Player(self._config)
        self._library = Library()
        self._window = Window(self._player, self._config, self._library)
        
        # programm
        program = hildon.Program()
        program.add_window(self._window)
        program.set_property("can-hibernate", True)
        
        # dialogs
        self._about_dialog = None
        self._preferences = None
        
        if IS_MAEMO:
            gtk.set_application_name("YouAmp")
            
        # menu
        menu = gtk.Menu()
        self._window.set_menu(menu)
        
        # view submenu
        view = gtk.MenuItem(_("View"))
        menu.append(view)
        
        view_sub = gtk.Menu()
        view.set_submenu(view_sub)
        
        search = gtk.RadioMenuItem(None, _("Search"))
        search.connect("activate", lambda caller: self._window.view.search_mode())
        view_sub.append(search)
        
        browse = gtk.RadioMenuItem(search, _("Browse"))
        browse.connect("activate", lambda caller: self._window.view.browse_mode())
        view_sub.append(browse)
    
        # order submenu
        order = gtk.MenuItem(_("Order"))
        menu.append(order)
        
        order_sub = gtk.Menu()
        order.set_submenu(order_sub)
        
        self._order_menu = dict()
        self._order_menu["artist"] = gtk.RadioMenuItem(None, _("Artist"))
        self._order_menu["artist"].connect("activate", lambda caller: self._set_list_order("artist"))
        self._order_menu["date"] = gtk.RadioMenuItem(self._order_menu["artist"], _("Date added"))
        self._order_menu["date"].connect("activate", lambda caller: self._set_list_order("date"))
        self._order_menu["playcount"] = gtk.RadioMenuItem(self._order_menu["artist"], _("Playcount"))
        self._order_menu["playcount"].connect("activate", lambda caller: self._set_list_order("playcount"))

        for k in ("artist", "date", "playcount"):
            order_sub.append(self._order_menu[k])

        prefs = gtk.ImageMenuItem(gtk.STOCK_PREFERENCES)
        prefs.connect("activate", self._show_preferences)
        menu.append(prefs)
        
        about = gtk.ImageMenuItem(gtk.STOCK_ABOUT)
        about.connect("activate", self._show_about_dialog)
        menu.append(about)
        menu.show_all()

        # restore config
        if self._player.tracks() > 0:
            self._player.load_track()
            
        if self._config["is-browser"]:
            browse.set_active(True)
        else:
            search.set_active(True)
        
        if not self._config["shuffle"]:
            self._order_menu[self._config["order-by"]].set_active(True)

        # signals
        self._window.connect("destroy", self.quit)
        self._window.connect("key-press-event", self._on_key_press)
        self._player.connect("song-played", self._on_song_played)
      
        # banner
        self._banner = hildon.hildon_banner_show_animation(self._window, "qgn_indi_pball_a", _("Updating database"))
        self._library.connect("update-complete", self._destroy_banner)

        # start updating thread
        self._library.update()

    def _check_config(self):
        if not self._config.has_key("volume"):
            # write default config
            self._config["volume"] = 0.5
            self._config["search-str"] = ""
            self._config["search-artist"] = ""
            self._config["search-album"] = ""
            self._config["pos"] = 0
            self._config["shuffle"] = False
            self._config["is-browser"] = False
            self._config["rg-preamp"] = 0     # preamp to adjust the default of 89db (value: db)
            self._config["no-rg-preamp"] = -10  # amp value to be used if no rg info is available (value: db)
        
        # order by was added in v0.3.5, therefore check for it seperately
        if not self._config.has_key("order-by"):
            self._config["order-by"] = "artist"

    def _set_list_order(self, order):
        self._config["order-by"] = order
        self._config["shuffle"] = False
        self._config["pos"] = 0

        playlist = self._library.get_tracks(self._config)
            
        self._player.playlist.update(playlist)

    def _on_song_played(self, caller, song):
        self._library.increment_played(song.uri)

    def _show_about_dialog(self, caller):
        if self._about_dialog is None:
            self._about_dialog = AboutDialog()
        
        self._about_dialog.show()
    
    def _show_preferences(self, caller):
        if self._preferences is None:
            self._preferences = Preferences(self._window, self._config)
            self._preferences.show_all()
    
        self._preferences.show()

    def _destroy_banner(self, caller):
        gobject.idle_add(self._banner.destroy)
    
    def _on_key_press(self, widget, event):
        key = gtk.gdk.keyval_name(event.keyval)
        
        if key == "Right":
            self._window._controls.on_next()
            return True # event handled
        elif key == "Left":
            self._window._controls.on_prev()
            return True # event handled
        if key == "F7":
            self._on_vol_up()
        elif key == "F8":
            self._on_vol_down()
        elif key == "F6":
            self._window.toggle_fullscreen()
    
    def _on_vol_down(self):
        vol = self._config["volume"]
        self._config["volume"] = max(0.0, vol - 0.1) # cut to range
    
    def _on_vol_up(self):
        vol = self._config["volume"]
        self._config["volume"] = min(1.0, vol + 0.1) # cut to range

    def start(self):
        gtk.main()
        
    def quit(self, caller=None):
        gtk.main_quit()