import gtk
import pango
import hildon

from youamp.ui.browser import Browser
from youamp.ui import NOTHING_SELECTED, ARTIST_SELECTED, ALBUM_SELECTED

class SearchView(gtk.VBox):
    def __init__(self, config, player, library):
        gtk.VBox.__init__(self)
        self.set_spacing(5)
        
        self._config = config
        self._library = library
        self._player = player
        self._is_browser = self._config["is-browser"]
        
        # Navi Controls
        navi = gtk.HBox()
        navi.set_spacing(5)

        # Search Label
        self._search_label = gtk.Label(_("Search"))
        self._search_label.set_size_request(100, -1)
        navi.pack_start(self._search_label, expand=False)
        
        # Browse Button
        self._browse_button = gtk.Button(_("Browse"))
        self._browse_button.connect("clicked", self._on_browse_click)
        self._browse_button.set_size_request(100, -1)
        self._browse_button.unset_flags(gtk.CAN_FOCUS)
        navi.pack_start(self._browse_button, expand=False)
        
        # Navi Artist Button
        self._artist = gtk.Button()
        self._artist.connect("clicked", self._on_artist_click)
        self._artist.set_size_request(200, -1)
        label = gtk.Label()
        label.set_ellipsize(pango.ELLIPSIZE_END)
        self._artist.add(label)
        self._artist.unset_flags(gtk.CAN_FOCUS)
        navi.pack_start(self._artist, expand=False)
        
        # Navi Album Button
        self._album = gtk.Button()
        self._album.connect("clicked", self._on_album_click)
        self._album.set_size_request(200, -1)
        label = gtk.Label()
        label.set_ellipsize(pango.ELLIPSIZE_END)
        self._album.unset_flags(gtk.CAN_FOCUS)
        self._album.add(label)
        navi.pack_start(self._album, expand=False)
        
        # Navi Search Entry
        self._search_entry = gtk.Entry()
        self._search_entry.connect("activate", self._search)
        self._search_entry.set_text(self._config["search-str"])
        navi.pack_start(self._search_entry, expand=True)
        
        # Shuffle Button
        self.shuffle = gtk.ToggleButton(_("shuffle"))
        self.shuffle.unset_flags(gtk.CAN_FOCUS)
        self.shuffle.set_active(self._config["shuffle"])
        self.shuffle.connect("toggled", self._on_shuffle_toggled)
        
        navi.pack_end(self.shuffle, expand=False)
        self.pack_start(navi, expand=False)
        
        # Scrolled View
        self._scroll = gtk.ScrolledWindow()
        self._scroll.set_policy(gtk.POLICY_NEVER, gtk.POLICY_ALWAYS)
        self._scroll.set_shadow_type(gtk.SHADOW_IN)
        hildon.hildon_helper_set_thumb_scrollbar(self._scroll, True)
        
        self._playlist = PlaylistView(player.playlist, self._config)
        self._playlist.connect("row-activated", self._on_row_activated)
        
        self._browser = Browser(self._config, self._library)
        
        self._scroll.add(self._playlist)

        self.pack_start(self._scroll)

        self.show_all()
 
        # Restore View
        if self._config["is-browser"]:
            self._artist.get_child().set_text(self._config["search-artist"])
            self._album.get_child().set_text(self._config["search-album"])
            self._browser._state = ARTIST_SELECTED | ALBUM_SELECTED
            self._search_label.hide()
            self._search_entry.hide()
        else:
            self._browse_button.hide()
            self._artist.hide()
            self._album.hide()
        
        self._restore_playlist()
            
        self._player.connect("song-changed", self._on_song_changed)
        
        # browser callbacks
        self._config.notify_add("search-artist", self._on_search_artist)
        self._config.notify_add("search-album", self._on_search_album)

    def _on_shuffle_toggled(self, caller):
        self._config["shuffle"] = not self._config["shuffle"]
        self._player.playlist.shuffle(self._config["shuffle"])
        
    def _restore_playlist(self):
        if self._config["is-browser"]:
            playlist = self._library.get_tracks(self._config)
            
            if self._config["search-artist"] == "":
                self._artist.get_child().set_text(_("All Artists"))
            
            if self._config["search-album"] == "":
                self._album.get_child().set_text(_("All Albums"))
        else:
            playlist = self._library.search(self._config)

        self._player.playlist.update(playlist)
        
        if self._config["shuffle"]:
            self._player.playlist.shuffle(True)

    def _browse_complete(self):
        playlist = self._library.get_tracks(self._config)
        self._set_playlist(playlist)
        
        self._search_entry.set_text("")
        self._config["is-browser"] = True
        
        self._show_playlist()
 
    def _search(self, caller):
        self._config["search-str"] = caller.get_text()
        self._config["shuffle"] = self.shuffle.get_active()
                
        playlist = self._library.search(self._config)
        
        self._set_playlist(playlist)
        
        self._config.notify("search-str")
        
        self._config["is-browser"] = False
    
    def _set_playlist(self, data):
        self._config["pos"] = 0
        self._playlist.set_cursor(0)

        self._player.playlist.update(data)
        
        if self._config["shuffle"]:
            self._player.playlist.shuffle(True)
    
    def browse_mode(self):
        self._search_label.hide()
        self._search_entry.hide()
        self._browse_button.show()
        
        if self._config["is-browser"]:
            self._artist.show()
            self._album.show()
        else:
            self._on_browse_click()
    
    def search_mode(self):
        self._browse_button.hide()
        self._artist.hide()
        self._album.hide()
        self._search_label.show()
        self._search_entry.show()
        
        if self._browser.get_parent() is not None:
            self._show_playlist()
    
    def _show_playlist(self):
        if self._playlist.get_parent() is None:
            self._scroll.remove(self._browser)
            self._scroll.add(self._playlist)

    def _show_browser(self):
        if self._browser.get_parent() is None:
            self._scroll.remove(self._playlist)
            self._scroll.add(self._browser)

    def _on_search_artist(self, client, cxn_id, entry, data):
        try:
            val = entry.get_value().get_string()
        except AttributeError:
            print "error"
        
        val = val if val != "" else _("All Artists")
        
        self._artist.get_child().set_text(val)
        self._artist.show()

        self._album.hide()
        print "search-artist changed"

    def _on_search_album(self, client, cxn_id, entry, data):
        try:
            val = entry.get_value().get_string()
        except AttributeError:
            print "error"
        
        val = val if val != "" else _("All Albums")
        
        self._album.get_child().set_text(val)
        self._album.show()
        
        self._browse_complete()
        print "search-album changed"
    
    def _on_album_click(self, caller):        
        if not (self._browser._state & ALBUM_SELECTED):
            self._browser._state |= ALBUM_SELECTED
            self._config.notify("search-album")
        
    def _on_artist_click(self, caller):
        if not (self._browser._state & ALBUM_SELECTED):
            self._config.notify("search-artist")

        self._show_browser()
        self._browser.show_albums()
        
        self._browser._state = ARTIST_SELECTED

    def _on_browse_click(self, caller=None):
        self._show_browser()
        self._browser.show_artists()
        self._browser._state = NOTHING_SELECTED

    def _on_song_changed(self, caller, newsong):
        self._playlist.set_cursor(self._config["pos"])

    def _on_row_activated(self, caller, path, column):
        self._player.goto_pos(path[0])

class PlaylistView(gtk.TreeView):
    def __init__(self, model, config):        
        gtk.TreeView.__init__(self, model)
    
        self.set_rules_hint(True)
        self.set_headers_visible(True)
        self.set_fixed_height_mode(True)
        keys = ("title", "artist", "album")
        cols = (_("Title"), _("Artist"), _("Album"))
        
        cell = gtk.CellRendererText()
        cell.set_property("ellipsize", pango.ELLIPSIZE_END)
            
        for i in xrange(len(keys)):
            col = gtk.TreeViewColumn(cols[i], cell)
            col.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
            col.set_fixed_width(200)
            col.set_cell_data_func(cell, self.data_func, keys[i])
            self.append_column(col)

        self.set_cursor(config["pos"])
    
    def data_func(self, col, cell, model, itr, key):
        cell.set_property("text", model[itr][key])

