/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2008  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Statistics.h"

#include <Lum/Base/L10N.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/WindowGroup.h>

#include "Configuration.h"
#include "Util.h"

Statistics::Statistics()
{
  SetExitAction(GetClosedAction());
}

Statistics::~Statistics()
{
  // no code
}

void Statistics::PreInit()
{
  Lum::Button      *button;
  Lum::ButtonRow   *buttonRow;
  Lum::Label       *label;
  Lum::Panel       *panel;
  Lum::WindowGroup *windowGroup;

  size_t           minWeight=0,maxWeight=0;
  Value            minValue=0, maxValue=0;
  Value            avgValue=0;
  size_t           avgValueDays=0;

  for (std::map<Lum::Base::Calendar,Day*>::const_iterator iter=data.data.begin();
       iter!=data.data.end();
       ++iter) {
    Value value=0;

    for (std::list<Entry>::const_iterator entry=iter->second->entries.begin();
         entry!=iter->second->entries.end();
         ++entry) {
      if (entry->item.empty()) {
        if (minWeight==0) {
          minWeight=entry->amount;
          maxWeight=entry->amount;
        }
        else {
          minWeight=std::min(minWeight,entry->amount);
          maxWeight=std::max(maxWeight,entry->amount);
        }
      }
      else {
        value+=entry->amount*entry->value;
      }
    }

    if (minValue==0 && maxValue==0) {
      minValue=value;
      maxValue=value;
    }
    else {
      minValue=std::min(minValue,value);
      maxValue=std::max(maxValue,value);
    }

    avgValue+=value;
    avgValueDays++;
  }

  minValue/=10;
  maxValue/=10;

  panel=new Lum::VPanel();

  label=new Lum::Label();
  label->SetFlex(true,true);

  std::wstring tmp;

  tmp=_(L"STAT_LABEL_WEIGHT_CHANGE_VALUE",L"Minimum $min, maximum $max, difference $delta ($unit)");
  Lum::Base::L10N::Substitute(tmp,L"$min",AmountToString(minWeight));
  Lum::Base::L10N::Substitute(tmp,L"$max",AmountToString(maxWeight));
  Lum::Base::L10N::Substitute(tmp,L"$delta",AmountToString(maxWeight-minWeight));
  Lum::Base::L10N::Substitute(tmp,L"$unit",weightUnit->Get());

  label->AddLabel(_(L"STAT_LABEL_WEIGHT_CHANGE",L"Weight change:"),tmp);
  label->AddLabel(_(L"STAT_LABEL_VALUE_MIN",L"Minimum value:"),ValueToString(minValue)+L" "+valueUnit->Get());
  if (avgValueDays>0) {
    label->AddLabel(_(L"STAT_LABEL_VALUE_AVG",L"Average value:"),ValueToString(avgValue/avgValueDays)+L" "+valueUnit->Get());
  }
  label->AddLabel(_(L"STAT_LABEL_VALUE_MAX",L"Maximum value:"),ValueToString(maxValue)+L" "+valueUnit->Get());

  panel->Add(label);

  panel->Add(new Lum::VSpace());

  buttonRow=new Lum::ButtonRow();
  buttonRow->SetFlex(true,false);

  button=new Lum::Button();
  button->SetFlex(true,true);
  button->SetModel(GetClosedAction());
  button->SetType(Lum::Button::typeDefault);
  button->SetText(_ld(dlgButtonClose));
  buttonRow->Add(button);

  panel->Add(buttonRow);

  windowGroup=new Lum::WindowGroup();
  windowGroup->SetMain(panel);

  SetTop(windowGroup);

  Dialog::PreInit();
}

void Statistics::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  Dialog::Resync(model,msg);
}
