#
# This is the main module for TrustMe
# Copyright (C) 2009, Henning Spruth
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA  02110-1301, USA.
#



import gtk
import os
#import signal
import gobject
from DataStore import DataStore, Record
import re

import Vault

try:
    import hildon
    isHildon=True
except:
    isHildon=False


version="0.5"

# The file header
HEADER="#TrustMe 1"

if not isHildon:
    class  hildon:
        def __init__(self):
	    pass
	    
	class Program:
	    def __init__(self):
		pass


# I think these are missing in the Diablo Python SDK
# http://maemo.org/api_refs/4.0/gtk/GtkIMContext.html#HildonGtkInputMode
HILDON_GTK_INPUT_MODE_ALPHA        = 1 << 0
HILDON_GTK_INPUT_MODE_NUMERIC      = 1 << 1
HILDON_GTK_INPUT_MODE_SPECIAL      = 1 << 2
HILDON_GTK_INPUT_MODE_HEXA         = 1 << 3
HILDON_GTK_INPUT_MODE_TELE         = 1 << 4
HILDON_GTK_INPUT_MODE_FULL         = (HILDON_GTK_INPUT_MODE_ALPHA | HILDON_GTK_INPUT_MODE_NUMERIC | HILDON_GTK_INPUT_MODE_SPECIAL)
HILDON_GTK_INPUT_MODE_MULTILINE    = 1 << 28
HILDON_GTK_INPUT_MODE_INVISIBLE    = 1 << 29
HILDON_GTK_INPUT_MODE_AUTOCAP      = 1 << 30
HILDON_GTK_INPUT_MODE_DICTIONARY   = 1 << 31

def errorPopup(message, toplevel=None):
    dialog=gtk.MessageDialog(toplevel,
                             gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT,
                             gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE,
                             message)
    dialog.set_title('Error')
    dialog.connect('response', lambda dialog, response: dialog.destroy())
    dialog.run()

def infoPopup(message, toplevel=None):
    dialog=gtk.MessageDialog(toplevel,
                             gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT,
                             gtk.MESSAGE_INFO, gtk.BUTTONS_OK,
                             message)
    dialog.set_title('Note')
    dialog.connect('response', lambda dialog, response: dialog.destroy())
    dialog.run()



def makePasswordEntry(returnCallback=None):
    entry=gtk.Entry()
    entry.set_visibility(False)
    #print entry.get_property('hildon-input-mode')
    if isHildon:
        entry.set_property('hildon-input-mode',  HILDON_GTK_INPUT_MODE_FULL|HILDON_GTK_INPUT_MODE_INVISIBLE)
        #entry.set_property('hildon-input-mode',  7 | 1<<29)
        #print entry.get_property('hildon-input-mode')
    # install a callback for return/enter keys, if specified. Usually the
    # returnCallback value something like 'my_button.pressed':
    if returnCallback:
        entry.connect('key-press-event', returnKeyHandler, returnCallback)
    return entry


def returnKeyHandler(widget, event, func):
    # Check for one of the Return keys
    val=event.keyval
    if val==gtk.keysyms.Return or val==gtk.keysyms.KP_Enter:
        func()
        return True
    else:
        return False

def defaultFileName():
    appDir=os.getenv('HOME')+"/.trustme"
    return appDir+"/default.db"


def showBanner(toplevel, msg):
    hildon.hildon_banner_show_information(toplevel,
                                          None,
                                          msg)


class Panel():

    def __init__(self, main):
        self.main=main
        self.vbox=gtk.VBox()

        self.toolbar=gtk.HBox()
        self.vbox.pack_end(self.toolbar, expand=False)
        self.toolbar.set_name('toolbar')

        self.lockButton=None
        # by default, disable "change password" menu entry
        self.canChangePassword=False

    def addButton(self, title, func, packAtEnd=False):
        b=gtk.Button(title)
        if packAtEnd:
            self.toolbar.pack_end(b, fill=False, expand=False, padding=4)
        else:
            self.toolbar.pack_start(b, fill=False, expand=False, padding=4)
        b.connect('clicked', func)
        return b

    def activate(self, arg=None):
        panes=self.main.panes
        child=panes.get_child()
        if child:
            panes.remove(child)
        panes.add(self.vbox)
        self.main.lockButton=self.lockButton

        # Some menu entries are only enabled for certain panels.
        # Initially, I compared self.__class__.__name__ against
        # the relevant classes but decided that was to kludgy.
        for item in self.main.changePasswordItems:
            item.set_sensitive(self.canChangePassword)
        
        self.prepare(arg)

    def prepare(self, arg):
        # virtual
        pass


class OpenPanel(Panel):

    def __init__(self, main):
        Panel.__init__(self, main)

        # Tool Bar
        hbox=gtk.HBox()
        self.vbox.pack_end(hbox, expand=False)
        self.openButton=self.addButton('Open', self.openCb)

        # Main Area
        vbox2=gtk.VBox()
        self.vbox.pack_start(vbox2, expand=True, fill=False)

        self.status=gtk.Label('')
        self.status.set_alignment(0,0)
        vbox2.pack_start(self.status)

        label=gtk.Label()
        label.set_markup('<big>Please enter the password:</big>')
        label.set_alignment(0,0)
        vbox2.pack_start(label)

        self.pwEntry=makePasswordEntry(self.openCb)
        vbox2.pack_start(self.pwEntry, gtk.FILL)
        self.vbox.show_all()


    def prepare(self, message=None):
        if message:
            self.setStatus(message)
        else:
            self.setStatus('')
        self.pwEntry.set_text('')
        self.pwEntry.grab_focus()
        self.main.clearAlarm()


    def setStatus(self, msg):
        self.status.set_text(msg)


    def openCb(self, dummy=None):

        password=self.pwEntry.get_text()
        if password=='':
            self.setStatus('Please specify a non-empyt password')
            return
        
        self.main.vault.setPassword(password)
        # paranoid
        self.pwEntry.set_text('')
        
        if not os.path.isfile(self.main.encryptedFileName):
            self.setStatus('Database does not exist?!?')
            return

        try:
            self.main.load()
        except Vault.EncryptedFileError, err:
            self.setStatus(err.value)
        else:
            # file was loaded successfully
            self.main.mainPanel.activate()
            

class MainPanel(Panel):

    def __init__(self, main):
        Panel.__init__(self, main)
        self.canChangePassword=True
        vbox=self.vbox

        # Tool Bar
        hbox=gtk.HBox()
        vbox.pack_end(hbox, expand=False)
        self.addButton('Add', self.addCb)
        self.addButton('Delete', self.deleteCb)
        self.addButton('Edit', self.editCb)

        # Thse are handled by the menu now
        #b=gtk.Button('ChangePw')
        #hbox.pack_start(b, fill=False, expand=False, padding=4)
        #b.connect('pressed', lambda self, main, dummy=None: main.changePasswordPanel.activate(), self.main)
        #self.addButton('Exit', self.main.destroyCb)

        self.lockButton=self.addButton('Lock', self.main.lockCb,
                                       packAtEnd=True)


        # Search Bar
        hbox=gtk.HBox()
        vbox.pack_start(hbox, expand=False, fill=True)
        label=gtk.Label('Search: ')
        hbox.pack_start(label, expand=False, fill=True) 
        self.searchMask=gtk.Entry()
        hbox.pack_start(self.searchMask, expand=True, fill=True)
        if isHildon:
            self.searchMask.set_property('hildon-input-mode',
                                         HILDON_GTK_INPUT_MODE_FULL)
        # 'x' button to clear the search mask
        button=gtk.Button('x')
        hbox.pack_start(button, expand=False, fill=True)
        button.connect("clicked",
                       lambda self, panel, dummy=None: panel.searchMask.set_text(''), self)

        #self.searchCategoryModel=gtk.ListStore(str)
        label=gtk.Label('Category: ')
        hbox.pack_start(label, expand=False, fill=True, padding=10) 
        self.searchCategory=gtk.combo_box_new_text()
        hbox.pack_start(self.searchCategory, expand=False, fill=True)
        self.searchMask.connect("changed", self.searchCb)
        self.searchCategory.connect("changed", self.searchCb)

        # Main Area
        scroll=gtk.ScrolledWindow()
        vbox.pack_start(scroll, expand=True, fill=True)
        scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scroll.set_border_width(6)
        if isHildon:
            hildon.hildon_helper_set_thumb_scrollbar(scroll, True)

        self.treeview=gtk.TreeView(self.main.store)
        scroll.add(self.treeview)
        self.treeview.set_rules_hint(True)
        #self.treeview.connect("cursor-changed", self.selectCb)
        self.treeview.set_grid_lines(gtk.TREE_VIEW_GRID_LINES_BOTH)
        self.treeview.set_headers_visible(False)
        #self.treeview.connect("row-activated", self.doubleCb, self)

        # create the TreeViewColumn to display the data
        col = gtk.TreeViewColumn('Name')
        self.treeview.append_column(col)
        cell = gtk.CellRendererText()
        col.pack_start(cell, True)
        col.add_attribute(cell, 'text', 1)

        self.treeview.connect("row-activated", self.editCb)
        self.vbox.show_all()

    def prepare(self, dummy):
        #print "in prepare"
        self.updateSearch()
        self.searchMask.set_text('')
        self.searchCategory.set_active(-1)
        # force update of the model
        self.searchCb()
        self.main.startAlarm()

    def searchCb(self, dummy=None):
        mask=self.searchMask.get_text()
        category=self.searchCategory.get_active_text()
        if category==None:
            category=''
        self.main.store.filter(mask, category)

    def updateSearch(self):
        currentCategory=self.searchCategory.get_active_text()
        self.searchCategory.get_model().clear()
        categories=self.main.store.getCategories()
        for cat in categories:
            self.searchCategory.append_text(cat)

    def addCb(self, dummy):
        store=self.main.store
        record=store.addRecord('', '', '', '', '')
        count=store.filter('', '')
        iter=store.get_iter_first()
        while iter!=None:
            if store[iter][0]==record.id:
                break
            iter=store.iter_next(iter)
        if iter:
            self.treeview.get_selection().select_iter(iter)
            self.editCb()


    def editCb(self, dummy1=None, dummy2=None, dummy3=None):
        store=self.main.store
        #print "editCb"
        self.main.startAlarm()
        selection=self.treeview.get_selection()
        (dummy, iter)=selection.get_selected()
        if iter!=None:
            recordId=store[iter][0]
            #print "Editing",store[iter][1],"index",recordId
            record=store.records[recordId]
            self.main.editPanel.activate(record)

    def deleteCb(self, widget):
        self.main.startAlarm()
        selection=self.treeview.get_selection()
        (dummy, iter)=selection.get_selected()
        if iter!=None:
            self.dialog=gtk.MessageDialog(parent=self.main.toplevel,
                                     flags=gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT,
                                     buttons=gtk.BUTTONS_YES_NO,
                                     message_format="Do you really want to delete this entry?"
                                     )
            result=self.dialog.run()
            self.dialog.destroy()
            self.dialog=None
            if result==gtk.RESPONSE_YES:
                self.main.store.deleteRecord(iter)
                self.main.save()


class EditPanel(Panel):

    def makeEntry(self, row, text, model=None, canCopy=0, autoCap=True):
        label=gtk.Label(text)
        label.set_alignment(0,0)
        self.grid.attach(label, 0, 1, row, row+1, gtk.FILL, gtk.FILL)

        if model:
            widget=gtk.ComboBoxEntry(model, 0)
            entry=widget.child
        else:
            widget=gtk.Entry()
            entry=widget

        if canCopy:
            copyButton=gtk.Button("copy")
            copyButton.connect("clicked",
                               lambda widget, entry, clipboard: clipboard.set_text(entry.get_text()), entry, self.main.clipboard)
            self.grid.attach(widget, 1, 2, row, row+1, gtk.FILL|gtk.EXPAND, gtk.FILL)
            self.grid.attach(copyButton, 2, 3, row, row+1, gtk.FILL, gtk.FILL)
        else:
            self.grid.attach(widget, 1, 3, row, row+1, gtk.FILL|gtk.EXPAND, gtk.FILL)

        if isHildon:
            if autoCap:
                entry.set_property('hildon-input-mode',
                                   HILDON_GTK_INPUT_MODE_FULL|HILDON_GTK_INPUT_MODE_AUTOCAP)
            else:
                entry.set_property('hildon-input-mode',
                                   HILDON_GTK_INPUT_MODE_FULL)
        
        return entry


    def makeEntryOld(self, row, text, model=None):
        label=gtk.Label(text)
        label.set_alignment(0,0)
        self.grid.attach(label, 0, 1, row, row+1, gtk.FILL, gtk.FILL)

        if model:
            widget=gtk.ComboBoxEntry(model, 0)
            entry=widget.child
        else:
            widget=gtk.Entry()
            entry=widget
        self.grid.attach(widget, 1, 2, row, row+1, gtk.FILL|gtk.EXPAND, gtk.FILL)
        return entry

    def makeText(self, row, text):
        #label=gtk.Label(text)
        #label.set_alignment(0,0)
        #grid.attach(label, 0, 1, row, row+1, gtk.FILL, gtk.FILL)

        #frame=gtk.Frame(" Info ")
        #self.grid.attach(frame, 0, 3, row, row+1, gtk.FILL|gtk.EXPAND, gtk.FILL|gtk.EXPAND)
        #frame.set_shadow_type(gtk.SHADOW_IN)
        #frame.set_border_width(3)
        
        #scroll=gtk.ScrolledWindow()
        #self.grid.attach(scroll, 0, 3, row, row+1, gtk.FILL|gtk.EXPAND, gtk.FILL|gtk.EXPAND)
        #frame.add(scroll)
        #scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        #scroll.set_policy(gtk.POLICY_ALWAYS, gtk.POLICY_ALWAYS)
        #scroll.set_border_width(6)
        #if isHildon:
        #    hildon.hildon_helper_set_thumb_scrollbar(scroll, True)
        text=gtk.TextView()
        self.grid.attach(text, 0, 3, row, row+1, gtk.FILL|gtk.EXPAND, gtk.FILL|gtk.EXPAND)
        #scroll.add(text)
        #text.set_size_request(0, 100)
        return text


    def __init__(self, main):
        Panel.__init__(self, main)

        # Tool Bar
        hbox=gtk.HBox()
        self.vbox.pack_end(hbox, expand=False)
        self.addButton('Commit', self.commitCb)
        self.addButton('Cancel', self.cancelCb)
        self.lockButton=self.addButton('Lock', self.main.lockCb, packAtEnd=True)


        self.grid=gtk.Table(rows=5, columns=2)
        self.vbox.pack_start(self.grid, fill=True)
        self.nameEntry=self.makeEntry(0, 'Name')
        self.userEntry=self.makeEntry(1, 'User', canCopy=True, autoCap=False)
        self.passwordEntry=self.makeEntry(2, 'Password', canCopy=True, autoCap=False)

        self.model=gtk.ListStore(str)
        self.categoryEntry=self.makeEntry(3, 'Category', self.model)
        self.infoText=self.makeText(4, 'Info')
        self.vbox.show_all()


    def prepare(self, record):
        # update the possible category choices
        self.model.clear()
        for cat in self.main.store.getCategories():
            self.model.append((cat,))

        self.record_id=record.id
        # set the data fields
        self.nameEntry.set_text(record.name)
        self.userEntry.set_text(record.user)
        self.passwordEntry.set_text(record.password)
        self.categoryEntry.set_text(record.category)
        self.infoText.get_buffer().set_text(record.info)
        self.nameEntry.grab_focus()
        self.main.startAlarm()

    def getRecord(self):
        buffer=self.infoText.get_buffer()
        result=Record(-1,
                     self.nameEntry.get_text(),
                     self.userEntry.get_text(),
                     self.passwordEntry.get_text(),
                     self.categoryEntry.get_text(),
                     buffer.get_text(buffer.get_start_iter(), buffer.get_end_iter()))
        return result

    def commitCb(self, dummy):
        # create a Record object from the entry fields
        newRecord=self.getRecord()
        # restore original integer identifier
        newRecord.id=self.record_id
        # replace the old record data with the new one
        store=self.main.store
        store.records[self.record_id]=newRecord
        iter=store.get_iter_first()
        while iter!=None:
            if store[iter][0]==self.record_id:
                break
            iter=store.iter_next(iter)
        if iter:
            store[iter][1]=newRecord.name
        else:
            print "Warning: could not find matching iter for record ID"
        # write things to disk
        self.main.save()
        # switch to main panel
        self.main.mainPanel.activate()


    def cancelCb(self, dummy):
        self.main.mainPanel.activate()
        



class ChangePasswordPanel(Panel):

    def __init__(self, main, initial=False):
        Panel.__init__(self, main)
        self.initial=initial

        # Tool Bar
        hbox=gtk.HBox()
        self.vbox.pack_end(hbox, expand=False)

        self.commitButton=self.addButton('Commit', self.commitCb)
        if not initial:
            self.addButton('Cancel', self.cancelCb)
            self.lockButton=self.addButton('Lock', self.main.lockCb, packAtEnd=True)

        # Main Area
        vbox2=gtk.VBox()
        self.vbox.pack_start(vbox2, expand=True, fill=False)

        self.status=gtk.Label('')
        self.status.set_alignment(0,0)
        vbox2.pack_start(self.status)

        label=gtk.Label('Please enter the new password:')
        label.set_alignment(0,0)
        vbox2.pack_start(label)

        self.pw1Entry=makePasswordEntry()
        vbox2.pack_start(self.pw1Entry, gtk.FILL)

        label=gtk.Label('Re-enter the password:')
        label.set_alignment(0,0)
        vbox2.pack_start(label)

        self.pw2Entry=makePasswordEntry(self.commitButton.clicked)
        vbox2.pack_start(self.pw2Entry, gtk.FILL)

        # couldn't pass the return handler in makePasswordEntry since
        # pw2Entry didn't exist at that time
        self.pw1Entry.connect('key-press-event', returnKeyHandler, self.pw2Entry.grab_focus)
        self.vbox.show_all()

    def prepare(self, message):
        if message:
            self.setStatus(message)
        else:
            self.setStatus('')
        self.pw1Entry.set_text('')
        self.pw1Entry.grab_focus()
        self.pw2Entry.set_text('')
        if not self.initial:
            self.main.startAlarm()
        
    def setStatus(self, msg):
        self.status.set_text(msg)


    def commitCb(self, dummy):
        pw1=self.pw1Entry.get_text()
        pw2=self.pw2Entry.get_text()

        if (pw1!=pw2):
            self.setStatus('The passwords do not match!')
            return

        self.main.vault.setPassword(pw1)
        pw1=None
        pw2=None
        self.main.save()
        self.main.mainPanel.activate()
        

    def cancelCb(self, dummy):
        self.main.mainPanel.activate()
        



class Main(hildon.Program):

    
    def queryFileName(self, title, openForRead):
        if openForRead:
            action=gtk.FILE_CHOOSER_ACTION_OPEN
        else:
            action=gtk.FILE_CHOOSER_ACTION_SAVE
            
        if isHildon:
            dialog=hildon.FileChooserDialog(self.toplevel, action)
        else:
            dialog=gtk.FileChooserDialog(
                title=title,
                action=action,
                buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK)
                )
        response=dialog.run()
        fn=dialog.get_filename()
        dialog.destroy()
        if response==gtk.RESPONSE_CANCEL:
            fn=None
        return fn


    def aboutCb(self, data):
        dialog=gtk.AboutDialog()
        dialog.set_name('TrustMe')
        dialog.set_version(version)
        dialog.set_copyright('(C) 2009, Henning Spruth')
        dialog.set_license('This program is free software and licensed under the GPL.\n\nSee /usr/share/common-licenses/GPL-2 for details.')
        dialog.set_wrap_license(True)
        dialog.set_website('http://trustme.garage.maemo.org')
        fn='/usr/share/icons/hicolor/scalable/apps/trustme.png'
        if os.path.isfile(fn):
            logo=gtk.gdk.pixbuf_new_from_file(fn)
            dialog.set_logo(logo)
        dialog.run()
        dialog.destroy()

    def exportCb(self, data):
        self.dialog=gtk.MessageDialog(parent=self.toplevel,
                                      flags=gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT,
                                      buttons=gtk.BUTTONS_YES_NO,
                                      message_format="Exporting is a security risk.\n\nDo you really want to export to an un-encrypted file?"
                                     )
        result=self.dialog.run()
        self.dialog.destroy()
        self.dialog=None
        if result==gtk.RESPONSE_YES:
            fn=self.queryFileName("Specify export file", False);
            if fn:
                self.clearAlarm()
                try:
                    content=self.store.toString()
                    fp=open(fn, 'w')
                    fp.write(HEADER+"\n")
                    fp.write(content)
                    fp.close()
                    if isHildon:
                        showBanner(self.toplevel, "Export Complete")
                except IOError:
                    errorPopup("Could not save file - I/O Error");


    #
    # Restore the database from a plain text file.
    #
    def importCb(self, data):
        self.dialog=gtk.MessageDialog(parent=self.toplevel,
                                      flags=gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT,
                                      buttons=gtk.BUTTONS_YES_NO,
                                      message_format="Importing will overwrite your complete database.\n\nAre you sure you want to do this?"
                                     )
        result=self.dialog.run()
        self.dialog.destroy()
        self.dialog=None
        if result==gtk.RESPONSE_YES:
            fn=self.queryFileName("Specify file to import", True);
            if fn:
                self.clearAlarm()
                try:
                    fp=open(fn, 'r')
                    header=fp.readline().strip()
                    content=fp.read()
                    fp.close()
                    if header != HEADER:
                        raise ValueError("Incorrect file header '"+header+"' detected")
                    self.store.fromString(content)
                    self.mainPanel.activate()
                    if isHildon:
                        showBanner(self.toplevel, "Import Complete")
                except ValueError, err:
                    errorPopup("Could not load file, reason: "+err.__str__());
                except IOError, err:
                    errorPopup("Could not load file: "+err.__str__());
                else:
                    # file was loaded successfully
                    # Update the encrypted database
                    self.vault.save(self.encryptedFileName, HEADER,content)
                    infoPopup("Database was restored. You may want to delete the un-encrypted import file.");




    def key_press_cb(self, widget, event, *args):
        # Fullscreen key
        if event.keyval==gtk.keysyms.F6:
            # "Full screen" hardware key
            if self.isFullScreen:
                self.toplevel.unfullscreen()
            else:
                self.toplevel.fullscreen()
            return True
        return False
                
    def window_state_cb(self, widget, event, *args):
        if event.new_window_state & gtk.gdk.WINDOW_STATE_FULLSCREEN:
            self.isFullScreen=True
        else:
            self.isFullScreen=False



    def lockCb(self, dummy):
        self.openPanel.activate('')


    def destroyCb(self, dummy, dummy2=None):
        gtk.main_quit()



    def userIsActive(self, dummy, dummy2):
        #print "boo"
        # iff the current panel has a lock button, start the countdown
        if self.lockButton:
            self.startAlarm()

    def startAlarm(self):
        #print "arm alarm"
        if self.lockButton:
            self.lockButton.set_label('Lock')
        self.timeout=300
        #signal.alarm(6)
        if self.alarm_source_id:
            gobject.source_remove(self.alarm_source_id)
        # The countdown increment in seconds
        delta=1
        self.alarm_source_id=gobject.timeout_add(1000*delta, self.alarmCb,delta)

    def clearAlarm(self):
        #print "alarm off"
        if self.lockButton:
            self.lockButton.set_label('Lock')
        #signal.alarm(0)
        if self.alarm_source_id:
            gobject.source_remove(self.alarm_source_id)
            self.alarm_source_id=None
            

    def alarmCb(self, delta):
        #print("Alarm!")
        self.timeout-=delta
        if self.timeout<0:
            if self.lockButton:
                self.lockButton.set_label('Lock')
            if self.dialog:
                self.dialog.destroy()
                self.dialog=None
            self.openPanel.activate("Timeout - please re-enter the password")
            return False;
        else:
            if self.lockButton:
                self.lockButton.set_label('Lock in '+str(self.timeout))
            return True

            
    def save(self):
        # turn alarm off - don't want to be interrupted while saving
        self.clearAlarm()
        content=self.store.toString()
        self.vault.save(self.encryptedFileName, HEADER,content)
        


    def load(self):
        # turn alarm off - don't want to be interrupted while loading
        self.clearAlarm()
        tuple=self.vault.load(self.encryptedFileName, "#TrustMe (\d+)$")
        schemaVersion=tuple[0]
        content=tuple[1]
        
        if schemaVersion!="1":
            raise Vault.EncryptedFileError("Unsupported schema version '"+str(schemaVersion)+"'")
        self.store.fromString(content)


    def createMenubar(self, parent):
	if isHildon:
            main_menu=gtk.Menu()
            self.toplevel.set_menu(main_menu)
            fileMenu=main_menu
            helpMenu=main_menu
        else:
            menu_bar=gtk.MenuBar()
            parent.pack_start(menu_bar, fill=True, expand=False)
            # File Menu
            item=gtk.MenuItem("File")
            menu_bar.append(item)
            fileMenu=gtk.Menu()
            item.set_submenu(fileMenu)
            # Help Menu
            item=gtk.MenuItem("Help")
            menu_bar.append(item)
            helpMenu=gtk.Menu()
            item.set_submenu(helpMenu)

        self.changePasswordItems=[];
        item=gtk.MenuItem("Change Password")
        item.connect_object("activate", lambda self, dummy=None: self.changePasswordPanel.activate(), self)
        fileMenu.append(item)
        # remember this item since it'll be disabled by some of the panels
        self.changePasswordItems.append(item);

        item=gtk.MenuItem("Export")
        item.connect_object("activate", self.exportCb, None)
        fileMenu.append(item)
        self.changePasswordItems.append(item);
        item=gtk.MenuItem("Import")
        item.connect_object("activate", self.importCb, None)
        fileMenu.append(item)
        self.changePasswordItems.append(item);

        item=gtk.MenuItem("About")
        item.connect_object("activate", self.aboutCb, None)
        helpMenu.append(item)
        
        item=gtk.MenuItem("Exit")
        item.connect_object("activate", self.destroyCb, "file.exit", None)
        fileMenu.append(item)
        
        #exit_item.show()
        if isHildon:
            main_menu.show_all()
        else:
            menu_bar.show_all()
        
#         if not isHildon:
#             file_item.set_submenu(main_menu)
#             menu_bar.append(file_item)

    def createGui(self):
        gtk.rc_parse_string('''
        style "big-button" {
             GtkButton::inner-border={10,10,10,10}
        }
        style "wide" {
             GtkScrollbar::slider-width=100
        }
        widget "top.*.toolbar.GtkButton" style "big-button"
        widget "top.*.GtkScrollbar" style:highest "wide"
       
        ''')

        if isHildon:
            self.toplevel=hildon.Window()
        else:
            self.toplevel = gtk.Window(gtk.WINDOW_TOPLEVEL)
        self.toplevel.set_title("TrustMe")
        if isHildon:
            gtk.set_application_name("TrustMe")
        self.toplevel.set_size_request(700, 400)
        self.toplevel.set_name('top')

        if not isHildon:
            vbox=gtk.VBox()
            self.toplevel.add(vbox)
            vbox.show()
            frame=gtk.Frame()
            vbox.pack_end(frame)
            self.panes=frame
            self.panes.show()
        else:
            self.panes=self.toplevel
            vbox=None

        self.createMenubar(vbox)
        self.clipboard=gtk.Clipboard()

        #self.toplevel.connect("delete_event", self.delete_event_cb, None)

        self.openPanel=OpenPanel(self)
        self.mainPanel=MainPanel(self)
        self.editPanel=EditPanel(self)
        self.changePasswordPanel=ChangePasswordPanel(self, False)
        self.initialChangePasswordPanel=ChangePasswordPanel(self, True)

        self.toplevel.connect("delete_event", self.destroyCb)

        self.toplevel.add_events(gtk.gdk.KEY_PRESS_MASK |gtk.gdk.BUTTON_PRESS_MASK)
        self.toplevel.connect("key-press-event", self.userIsActive)
        self.toplevel.connect("button-press-event", self.userIsActive)
        self.toplevel.connect("button_press_event", self.userIsActive)
        self.toplevel.connect("button-release-event", self.userIsActive)

        # manage full screen stuff
        self.toplevel.connect("key-press-event", self.key_press_cb)
        self.toplevel.connect("window-state-event", self.window_state_cb)
        self.isFullScreen=False

        self.toplevel.show()




    def run(self):
        if os.path.isfile(self.encryptedFileName):
            self.openPanel.activate()
        else:
            self.initialChangePasswordPanel.activate('You need to provide a password for the database')
        gtk.main()

        

    def __init__(self):
        hildon.Program.__init__(self)

        self.encryptedFileName=defaultFileName()

        self.store=DataStore()
        #self.prefs=Preferences()
        self.dialog=None
        self.lockButton=None
        self.alarm_source_id=None
        self.vault=Vault.Vault()

