#!/usr/bin/env python

#
#    RingClock for Maemo by CJ Manoj Kumar AKA shin@talk.maemo.org, based on a similar Screenlet clock by 
#       Paul Ashton & Craig Laparo
#
#    RingClock for Maemo is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    RingClock for Maemo is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with RingClock for Maemo.  If not, see <http://www.gnu.org/licenses/>.
#
    
# All the necessary imports
import sys, os
import pango
import pygtk
pygtk.require('2.0')
import gtk
from gtk import gdk
import gobject
import cairo
#import commands
#import traceback
from ConfigParser import *
import hildon
import hildondesktop    
import re
import math
import datetime
import osso
import time

alpha_channel = False

class RingClock(hildondesktop.HomeItem):
    """A ring that tells you the time on Maemo hildon desktop."""
    
    # internals
    p_layout = None

    hour = 0
    mins = 0
    secs = 0

    gui_update_interval = 1000 #ms
    ampm = True
    hourColor = [1,0,0,1]
    minsColor = [0,1,0,1]
    secsColor = [0,0,1,1]
    hourBGColor = [0.5,0.5,0.5,0.5]
    minsBGColor = [0.5,0.5,0.5,0.5]
    secsBGColor = [0.5,0.5,0.5,0.5]
    size = 100
    thickness = 15.0
    ringSpacing = 16.0
    blockSpacing = 0.7
    blockDivSpacing = 1.5
    hourMiddle = True
    allHours = True
    allMins = True
    allSecs = False
    zeroHour = False
    
    # Time Text related
    face_text_color    = (1.0, 1.0, 1.0, 1.0)
    face_text_font    = "Sans"
    text_alignment    = 'Center'


    # constructor
    def __init__(self):
    
        hildondesktop.HomeItem.__init__(self)
        #super(RingClock, self).__init__()

        self.set_resize_type(hildondesktop.HOME_ITEM_RESIZE_NONE)
        #self.set_resize_type(hildondesktop.HOME_ITEM_RESIZE_BOTH)
        self.set_size_request(200, 200)
        self.connect("settings", self._DoSettings)
        self.inbackground = False

        self.connect("expose-event", self.expose)
        self.connect("screen-changed", self.screen_changed)

        # Unmask events - To realize double click action on the widget and display a Calendar widget 
        self.add_events(gtk.gdk.BUTTON_PRESS_MASK|gtk.gdk.BUTTON_RELEASE_MASK|gtk.gdk.LEAVE_NOTIFY)
        self.connect("button-press-event", self.button_press)
        self.connect("button-release-event", self.button_release)
        self.connect("leave-notify-event", self.button_release)
        
        self.show_all()
        
        # set a timeout to change the images
        self.timer = gobject.timeout_add(1000, self.update)

        self.connect ("background", self.background)
        self.connect ("foreground", self.foreground)
        self.connect ("unrealize", self.unrealize)
        
    def _DoSettings(self,widget,data=None):
        menuitem = gtk.MenuItem("RingClock")
        menuitem.connect_object("activate", self._ShowSettings,None)
        return menuitem

    def _color_to_string(self,c):
        t = "#"

        if len(hex(c.red)[2:4])==1:
            t = t+'0'+hex(c.red)[2:4]
        else:
            t = t+hex(c.red)[2:4]
        if len(hex(c.green)[2:4])==1:
            t = t+'0'+hex(c.green)[2:4]
        else:
            t = t+hex(c.green)[2:4]
        if len(hex(c.blue)[2:4])==1:
            t = t+'0'+hex(c.blue)[2:4]
        else:
            t = t+hex(c.blue)[2:4]
        return t

    def _validate(self,widget,*data):
        self._prefSave('colhour',self._color_to_string(self._widgetTree.get_widget("color_hour").get_current_color()))
        self._prefSave('colmin',self._color_to_string(self._widgetTree.get_widget("color_min").get_current_color()))
        self._prefSave('colsec',self._color_to_string(self._widgetTree.get_widget("color_sec").get_current_color()))
        self._prefSave('colback',self._color_to_string(self._widgetTree.get_widget("color_back").get_current_color()))
        self._prefSave('colhour_alpha',str(self._widgetTree.get_widget("color_hour").get_current_alpha()))
        self._prefSave('colmin_alpha',str(self._widgetTree.get_widget("color_min").get_current_alpha()))
        self._prefSave('colsec_alpha',str(self._widgetTree.get_widget("color_sec").get_current_alpha()))
        self._prefSave('colback_alpha',str(self._widgetTree.get_widget("color_back").get_current_alpha()))
        self._prefSave('font',self._widgetTree.get_widget("fontselection").get_font_name())
        self._window.destroy()

    def _cancel(self,widget,*data):
        self._window.destroy()

    def _closebox(self,widget,*data):
        self._window.destroy()

    def _prefRead(self,key):
        default = {'colhour':'#ffffff','colhour_alpha':'65535','colmin':'#d8c81c','colmin_alpha':'65535','colsec':'#ffffff','colsec_alpha':'65535','colback':'#7a7c8f','colback_alpha':'33667','font':'Sans 16'}
        c = ConfigParser(default)
        c.read('/home/user/.ringclock')
        return c.get("DEFAULT", key)
    
    def _prefSave(self,key,value):

        default = {'colhour':'#ffffff','colhour_alpha':'65535','colmin':'#d8c81c','colmin_alpha':'65535','colsec':'#ffffff','colsec_alpha':'65535','colback':'#7a7c8f','colback_alpha':'33667','font':'Sans 16'}
        c = ConfigParser(default)
        c.read('/home/user/.ringclock')
        c.set('DEFAULT',key,value)
        fp = open('/home/user/.ringclock','w')
        c.write(fp)
        fp.close()

    def _ShowSettings(self,widget,data=None):
        import gtk.glade
        self._widgetTree = gtk.glade.XML('/usr/lib/ringclock/prefs.glade')
        self._window = self._widgetTree.get_widget("prefs")
        self._widgetTree.get_widget("color_hour").set_current_color(gtk.gdk.color_parse(self._prefRead('colhour')))
        self._widgetTree.get_widget("color_hour").set_has_opacity_control(True)
        self._widgetTree.get_widget("color_hour").set_current_alpha(int(self._prefRead('colhour_alpha')))
        self._widgetTree.get_widget("color_min").set_current_color(gtk.gdk.color_parse(self._prefRead('colmin')))
        self._widgetTree.get_widget("color_min").set_current_alpha(int(self._prefRead('colmin_alpha')))
        self._widgetTree.get_widget("color_min").set_has_opacity_control(True)
        self._widgetTree.get_widget("color_sec").set_current_color(gtk.gdk.color_parse(self._prefRead('colsec')))
        self._widgetTree.get_widget("color_sec").set_current_alpha(int(self._prefRead('colsec_alpha')))
        self._widgetTree.get_widget("color_sec").set_has_opacity_control(True)
        self._widgetTree.get_widget("color_back").set_current_color(gtk.gdk.color_parse(self._prefRead('colback')))
        self._widgetTree.get_widget("color_back").set_current_alpha(int(self._prefRead('colback_alpha')))
        self._widgetTree.get_widget("color_back").set_has_opacity_control(True)
        self._widgetTree.get_widget("fontselection").set_font_name(self._prefRead('font'))
    
        callbackMapping = {
            # Signal
          "on_bCancel_clicked": self._cancel,
          "on_bValidate_clicked": self._validate,
        }
        self._widgetTree.signal_autoconnect(callbackMapping)
        self._window.connect("delete-event", self._closebox)

        self._window.show()

        return True
        
    def expose(self, widget, event):

        global alpha_channel

        alloc = self.get_allocation()
        self.width, self.height = alloc.width, alloc.height

        #Get a cairo context
        ctx = event.window.cairo_create()

        ctx.set_source_rgba(1.0, 1.0, 1.0, 0.0) # Transparent

        # Draw the Context
        ctx.set_operator(cairo.OPERATOR_SOURCE)
        ctx.paint()
        
        # Draw the RingClock Background
        ctx.scale(1.0, 1.0)
        ctx.set_operator(cairo.OPERATOR_OVER)
        ctx.set_source_rgba(0,0,0,0.4)
        ctx.arc( 100, 100, self.size-(self.thickness/2.0), math.radians(0), math.radians(360) )
        ctx.fill()    
        
        self.DrawClockRings(ctx)
        self.DrawDate(ctx)

        return True

    def DrawClockRings(self, ctx):
    
        #Set the color parameters for the clock rings
        self.hourColor = self.get_pallette_info("colhour")
        self.minsColor = self.get_pallette_info("colmin")
        self.secsColor = self.get_pallette_info("colsec")
        self.hourBGColor = self.get_pallette_info("colback")
        self.minsBGColor = self.get_pallette_info("colback")
        self.secsBGColor = self.get_pallette_info("colback")
    
        # Update time
        self.updateClock()

        ctx.scale(1.0, 1.0)
        ctx.set_operator(cairo.OPERATOR_ADD)

        ctx.save()
        startrad = self.size-(self.thickness/2.0)
        
        ctx.set_line_width( self.thickness )

        #Hours
        for i in range(1, (25 - 12*self.ampm)):
            if (not self.zeroHour or self.hour != 24 - 12*self.ampm) and (i == self.hour or (i<=self.hour and self.allHours)):
                col = self.hourColor
            else:
                col = self.hourBGColor

            if self.hourMiddle: radius = startrad-(self.ringSpacing*2.0)
            else: radius = startrad

            if self.ampm: 
                pos = -120+(i*30)
                inc = 30
            else:
                pos = -105+(i*15)
                inc = 15

            gap = self.blockDivSpacing/2.0
                
            ctx.arc( 100, 100, radius, math.radians(pos+gap), math.radians(pos+inc-gap) )
            ctx.set_source_rgba(col[0], col[1], col[2], col[3])
  #          ctx.set_source_rgba (text_color.red / 65535.0, text_color.green/65535.0, text_color.blue/65535.0, float(self._prefRead('colsec_alpha'))/65535.0)        
            ctx.stroke()
        
        #Mins
        for i in range(1, 61):
            if i == self.mins or (i<=self.mins and self.allMins): col = self.minsColor
            else: col = self.minsBGColor

            if ((i - 1) % 5 == 0): leftGap = (self.blockDivSpacing/2.0)
            else: leftGap = (self.blockSpacing/2.0)

            if (i % 5 == 0): rightGap = (self.blockDivSpacing/2.0)
            else: rightGap = (self.blockSpacing/2.0)

            pos = -96+(i*6)
            
            ctx.arc( 100, 100, startrad-self.ringSpacing, math.radians(pos+leftGap), math.radians(pos+6-rightGap) )
            ctx.set_source_rgba(col[0], col[1], col[2], col[3])
            ctx.stroke()
        
        #Secs
        for i in range(1, 61):
            if i == self.secs or (i<=self.secs and self.allSecs): col = self.secsColor
            else: col = self.secsBGColor

            if self.hourMiddle: radius = startrad
            else: radius = startrad-(self.ringSpacing*2.0)

            if ((i - 1) % 5 == 0): leftGap = (self.blockDivSpacing/2.0)
            else: leftGap = (self.blockSpacing/2.0)

            if (i % 5 == 0): rightGap = (self.blockDivSpacing/2.0)
            else: rightGap = (self.blockSpacing/2.0)

            pos = -96+(i*6)
            
            ctx.arc( 100, 100, radius, math.radians(pos+leftGap), math.radians(pos+6-rightGap) )
            ctx.set_source_rgba(col[0], col[1], col[2], col[3])
            ctx.stroke()
        
        ctx.restore()
        
        return True

    def updateClock(self):
        self.gui_update_interval = 1000 - math.floor((time.time() - math.floor(time.time())) * 1000)
        tempTime = time.localtime()
        self.hour = tempTime[3]
        self.mins = tempTime[4]
        self.secs = tempTime[5]
        if self.hour == 0: self.hour = 24
        if self.mins == 0 and self.secs == 0: self.mins = 60
        if self.secs == 0: self.secs = 60
        if self.ampm:
            self.hour = self.hour % 12
            if self.hour == 0: self.hour = 12
        return True

    def DrawDate(self, ctx):

        # Get Color parameters for the text

        text_color1 = self.get_pallette_info("colsec")
        text_color2 = self.get_pallette_info("colmin")

        # Get date
        date = self.get_date_info() # [day, month, adate, year]
        #adate = self.get_date_ordinal(date[2])
        
        # Set size
        
        ctx.scale(1.0, 1.0)
        self.p_layout = ctx.create_layout()
        ctx.update_layout(self.p_layout) 
        
        p_fdesc = pango.FontDescription(self._prefRead('font'))
        ctx.save()

        # Draw day of the Week as in Monday, Tuesday  
        ctx.translate(80, 70)        
        p_fdesc.set_size(16 * pango.SCALE)
        self.p_layout.set_font_description(p_fdesc)  
        self.p_layout.set_width((self.width) * pango.SCALE)
        self.p_layout.set_alignment(pango.ALIGN_LEFT)
        self.p_layout.set_markup(date[0][:3]) 

        #ctx.set_source_rgba (text_color.red / 65535.0, text_color.green/65535.0, text_color.blue/65535.0, float(self._prefRead('colsec_alpha'))/65535.0)        
        ctx.set_source_rgba(text_color1[0], text_color1[1], text_color1[2], text_color1[3])
        ctx.show_layout(self.p_layout)
        
        ctx.restore()
        ctx.save()
        
        # Draw the date as in 1 ~ 30
        ctx.translate(60, 100)        
        p_fdesc.set_size(17 * pango.SCALE)
        self.p_layout.set_font_description(p_fdesc)  
        self.p_layout.set_width((self.width) * pango.SCALE)
        self.p_layout.set_alignment(pango.ALIGN_LEFT)
        #self.p_layout.set_markup(adate[0:2]  + '<sup>' + adate[2:4] + '</sup>') # For date ordinal superscripting
        self.p_layout.set_markup(date[2]) 

        #ctx.set_source_rgba (text_color.red / 65535.0, text_color.green/65535.0, text_color.blue/65535.0, float(self._prefRead('colsec_alpha'))/65535.0)        
        ctx.set_source_rgba(text_color2[0], text_color2[1], text_color2[2], text_color2[3])
        ctx.show_layout(self.p_layout)

        ctx.restore()
        ctx.save()

        # Draw the Month name as in Jan, Feb 
        ctx.translate(100, 100)        
        p_fdesc.set_size(17 * pango.SCALE)
        self.p_layout.set_font_description(p_fdesc)  
        self.p_layout.set_width((self.width) * pango.SCALE)
        self.p_layout.set_alignment(pango.ALIGN_LEFT)
        self.p_layout.set_markup(date[1]) 

        #ctx.set_source_rgba (text_color.red / 65535.0, text_color.green/65535.0, text_color.blue/65535.0, float(self._prefRead('colsec_alpha'))/65535.0)        
        ctx.set_source_rgba(text_color2[0], text_color2[1], text_color2[2], text_color2[3])
        ctx.show_layout(self.p_layout)

        ctx.fill()
        ctx.save() 
        ctx.restore()

        return True 

    def get_date_info(self):
        now = datetime.datetime.now()
        day = now.strftime("%A").upper()
        month = now.strftime("%b")
        adate = now.strftime("%d")
        year = now.strftime("%Y")
            
        return [day, month, adate, year]

    def get_date_ordinal(self, n):
        x = int(n)
        if 10 <= x % 100 < 20:
            return str(n) + 'th'
        else:
            return  str(n) + {1 : 'st', 2 : 'nd', 3 : 'rd'}.get(x % 10, "th")

    def get_pallette_info(self, key):

        #colname = "'" + key + "'"
        colrgb = gtk.gdk.color_parse(self._prefRead(key))
        colred = float(colrgb.red) /65535.0
        colblue = float(colrgb.blue) /65535.0
        colgreen = float(colrgb.green) /65535.0
        key = key + "_alpha"
        colalpha = float(self._prefRead(key))/65535.0
    
        return [ colred, colgreen, colblue, colalpha]

    # Event handlers for Button press and release actions
    def button_press(self, widget, event):
        # Differentiate between _2BUTTON_PRESS ( double click) and BUTTON_PRESS / _3BUTTON_PRESS
        # By "return False" it is indicated that the handling is being taken care
        if not event.type == gtk.gdk._2BUTTON_PRESS: 
            return False
        #self.update()
        osso_c = osso.Context("osso_test_sender", "0.0.1", False)
        self.alarm_popup(widget, osso_c)
        return True


    def button_release(self, widget, event):
        #Check if it was released or moved out.
        if event.type == gtk.gdk.BUTTON_RELEASE:
            pass
            #print "Flip Calendar - Button clicked!"
            # Calendar Popup from within button release creates lot of issues and moved to button press event
            #self.calendar_popup(widget)
            #return True

        else: 
            pass
            # This section is for a button click that "did not happen" or was bailed-out from, more like a drag or tap
            #print "Flip Calendar - Did you try clicking?"
            #self.update()
            #return True
        
    def alarm_popup(self, window, osso_c):
        osso_rpc = osso.Rpc(osso_c)
        osso_rpc.rpc_run("com.nokia.osso_worldclock",
            "/com/nokia/osso_worldclock",
            "com.nokia.osso_worldclock", "alarm")
        #print "RPC sent"

        return True
        
    def screen_changed(self, widget, old_screen=None):
        global alpha_channel

        screen = widget.get_screen()
        colormap = screen.get_rgba_colormap()
        if colormap == None:
            colormap = screen.get_rgb_colormap()
            alpha_channel = False
        else:
            alpha_channel = True

        widget.set_colormap(colormap)
        return False

    def update(self):

        self.queue_draw()
        return not self.inbackground

    def background(self, widget, data=None):

        self.inbackground = True
        if self.timer != None:
            gobject.source_remove(self.timer)
            self.timer = None

    def foreground(self, widget, data=None):

        #self.set_resize_type(hildondesktop.HOME_ITEM_RESIZE_BOTH)
        self.inbackground = False
        self.timer = gobject.timeout_add(1000, self.update)

    def unrealize(self, widget, date=None):
        # cancel timeout
        if self.timer:
            v = gobject.source_remove(self.timer)
            print "canceled ringclock timeout:", v
            self.timer = None

def hd_plugin_get_objects():
    plugin = RingClock()
    return [plugin]

if __name__ == '__main__':
    print RingClock()
    print RingClock().update()
    print RingClock()._ShowSettings('a')
    gtk.main()