/*
  PushIt - A simple Sokoban solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "GameArea.h"

#include <Lum/Base/Path.h>
#include <Lum/Base/Util.h>

#include <Lum/OS/Driver.h>

#include <iostream>
static GameArea::Prefs *prefs=new GameArea::Prefs();

GameArea::GameArea()
 : themeIndex(0),
   xOff(0),
   yOff(0),
   finished(new Lum::Model::Boolean),
   gameBackground(NULL),
   game(NULL),
   sokobanColor("red",Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   frameShineColor("grey95",Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   frameShadowColor("grey20",Lum::OS::display->GetColor(Lum::OS::Display::blackColor))
{
  SetPrefs(::prefs);

  SetCanFocus(true);

  finished->Set(false);
}

GameArea::~GameArea()
{
  delete gameBackground;
  delete game;
}

size_t GameArea::GetBoxSize() const
{
  return themes[themeIndex].boxSize;
}

Lum::Images::Image* GameArea::GetImage(Image image) const
{
  return themes[themeIndex].images[image];
}

size_t GameArea::GetSokobanXPos() const
{
  size_t sx;
  size_t sy;

  sokoban.GetSokoban(sx,sy);

  return sx;
}

size_t GameArea::GetSokobanYPos() const
{
  size_t sx;
  size_t sy;

  sokoban.GetSokoban(sx,sy);

  return sy;
}

bool GameArea::IsAreaCompletelyVisible() const
{
  return sokoban.GetWidth()<=width/GetBoxSize() && sokoban.GetHeight()<=height/GetBoxSize();
}

bool GameArea::CanScrollLeft() const
{
  return xOff>0;
}

bool GameArea::CanScrollUp() const
{
  return yOff>0;
}

bool GameArea::CanScrollRight() const
{
  return xOff+(width/GetBoxSize()-2)<sokoban.GetWidth();
}

bool GameArea::CanScrollDown() const
{
  return yOff+(height/GetBoxSize()-2)<sokoban.GetHeight();
}

void GameArea::MakeSokobanVisible()
{
  // If the area is completely visible, there is no need to move the
  // area offset to make the sokoban visible!
  if (IsAreaCompletelyVisible()) {
    return;
  }

  size_t sx;
  size_t sy;

  sokoban.GetSokoban(sx,sy);

  if (sx<xOff || sx>xOff+(width/GetBoxSize()-2)-1) {
    if (sx<(width/GetBoxSize()-2)/2) {
      xOff=0;
    }
    else if (sx>sokoban.GetWidth()-(width/GetBoxSize()-2)/2) {
      xOff=sokoban.GetWidth()-(width/GetBoxSize()-2);
    }
    else {
      xOff=sx-(width/GetBoxSize()-2)/2;
    }
  }

  if (sy<yOff || sy>yOff+(height/GetBoxSize()-2)-1) {
    if (sy<(height/GetBoxSize()-2)/2) {
      yOff=0;
    }
    else if (sy>sokoban.GetHeight()-(height/GetBoxSize()-2)/2) {
      yOff=sokoban.GetHeight()-(height/GetBoxSize()-2);
    }
    else {
      yOff=sy-(height/GetBoxSize()-2)/2;
    }
  }
}

void GameArea::ScrollLeft()
{
  assert(xOff>0);

  xOff--;
  Redraw();
}

void GameArea::ScrollUp()
{
  assert(yOff>0);

  yOff--;
  Redraw();
}

void GameArea::ScrollRight()
{
  //assert();

  xOff++;
  Redraw();
}

void GameArea::ScrollDown()
{
  //assert();

  yOff++;
  Redraw();
}

void GameArea::MoveLeft()
{
  sokoban.MoveLeft();
  RebuildGame();
  MakeSokobanVisible();
  Redraw();

  finished->Set(sokoban.IsFinished());
}

void GameArea::MoveUp()
{
  sokoban.MoveUp();
  RebuildGame();
  MakeSokobanVisible();
  Redraw();

  finished->Set(sokoban.IsFinished());
}

void GameArea::MoveRight()
{
  sokoban.MoveRight();
  RebuildGame();
  MakeSokobanVisible();
  Redraw();

  finished->Set(sokoban.IsFinished());
}

void GameArea::MoveDown()
{
  sokoban.MoveDown();
  RebuildGame();
  MakeSokobanVisible();
  Redraw();

  finished->Set(sokoban.IsFinished());
}

void GameArea::DrawSokobanInternal(Lum::OS::DrawInfo *draw, size_t x, size_t y, size_t size)
{
  draw->PushForeground(sokobanColor);
  draw->FillRectangle(x,y,size,size);
  draw->PopForeground();

  // Left
  draw->PushForeground(frameShineColor);
  draw->DrawLine(x,y,x,y+size-1);
  /// Top
  draw->DrawLine(x,y,x+size-1,y);
  draw->PopForeground();

  // Right
  draw->PushForeground(frameShadowColor);
  draw->DrawLine(x+size-1,y,x+size-1,y+size-1);
  // Bottom
  draw->DrawLine(x,y+size-1,x+size-1,y+size-1);
  draw->PopForeground();
}

void GameArea::DrawPackageInternal(Lum::OS::DrawInfo *draw, size_t x, size_t y, size_t size)
{
  draw->PushForeground(Lum::OS::display->fillColor);
  draw->FillRectangle(x,y,size,size);
  draw->PopForeground();

  // Left
  draw->PushForeground(frameShineColor);
  draw->DrawLine(x,y,x,y+size-1);
  /// Top
  draw->DrawLine(x,y,x+size-1,y);
  draw->PopForeground();

  // Right
  draw->PushForeground(frameShadowColor);
  draw->DrawLine(x+size-1,y,x+size-1,y+size-1);
  // Bottom
  draw->DrawLine(x,y+size-1,x+size-1,y+size-1);
  draw->PopForeground();
}

void GameArea::DrawGoalInternal(Lum::OS::DrawInfo *draw, size_t x, size_t y, size_t size)
{
  draw->PushForeground(frameShadowColor);
  draw->FillRectangle(x,y,size,size);
  draw->PopForeground();

}

void GameArea::DrawFloorInternal(Lum::OS::DrawInfo *draw, size_t x, size_t y, size_t size)
{
  draw->PushForeground(Lum::OS::Display::whiteColor);
  draw->FillRectangle(x,y,size,size);
  draw->PopForeground();
}

void GameArea::DrawWallInternal(Lum::OS::DrawInfo *draw,
                                size_t x, size_t y, size_t size,
                                size_t cx, size_t cy)
{
  Lum::OS::FillRef fill=Lum::OS::display->GetFill(Lum::OS::Display::backgroundFillIndex);

  fill->Draw(draw,x,y,size,size,x,y,size,size);

  // Left
  if (cx==0 || !sokoban.IsWall(cx-1,cy)) {
    draw->PushForeground(frameShineColor);
    draw->DrawLine(x,y,x,y+size-1);
    draw->PopForeground();
  }
  /// Top
  if (cy==0 || !sokoban.IsWall(cx,cy-1)) {
    draw->PushForeground(frameShineColor);
    draw->DrawLine(x,y,x+size-1,y);
    draw->PopForeground();
  }
  // Right
  if (cx==sokoban.GetWidth()-1 || !sokoban.IsWall(cx+1,cy)) {
    draw->PushForeground(frameShadowColor);
    draw->DrawLine(x+size-1,y,x+size-1,y+size-1);
    draw->PopForeground();
  }
  // Bottom
  if (cy==sokoban.GetHeight()-1 || !sokoban.IsWall(cx,cy+1)) {
    draw->PushForeground(frameShadowColor);
    draw->DrawLine(x,y+size-1,x+size-1,y+size-1);
    draw->PopForeground();
  }
}

/**
  Create bitmap containing the game area background build up from walls, floors,
  goals and backgrounds.
 */
void GameArea::RebuildGameBackground()
{
  delete gameBackground;
  delete game;

  gameBackground=Lum::OS::driver->CreateBitmap(sokoban.GetWidth()*GetBoxSize(),
                                               sokoban.GetHeight()*GetBoxSize());

  game=Lum::OS::driver->CreateBitmap(sokoban.GetWidth()*GetBoxSize(),
                                     sokoban.GetHeight()*GetBoxSize());

  for (size_t cx=0; cx<sokoban.GetWidth(); ++cx) {
    for (size_t cy=0; cy<sokoban.GetHeight(); ++cy) {
      unsigned long  type=sokoban.Get(cx,cy);

      if (type & Sokoban::typeWall) {
        if (GetImage(wallImage)!=NULL) {
          GetImage(wallImage)->Draw(gameBackground->GetDrawInfo(),
                                    cx*GetBoxSize(),
                                    cy*GetBoxSize());
        }
        else {
          DrawWallInternal(gameBackground->GetDrawInfo(),cx*GetBoxSize(),cy*GetBoxSize(),GetBoxSize(),cx,cy);
        }
      }
      else if (type & Sokoban::typeGoal) {
        if (GetImage(goalImage)!=NULL) {
          GetImage(goalImage)->Draw(gameBackground->GetDrawInfo(),
                                    cx*GetBoxSize(),
                                    cy*GetBoxSize());
        }
        else {
          DrawGoalInternal(gameBackground->GetDrawInfo(),cx*GetBoxSize(),cy*GetBoxSize(),GetBoxSize());
        }
      }
      else if (type & Sokoban::typeFloor) {
        if (GetImage(floorImage)!=NULL) {
          GetImage(floorImage)->Draw(gameBackground->GetDrawInfo(),
                                     cx*GetBoxSize(),
                                     cy*GetBoxSize());
        }
        else {
          DrawFloorInternal(gameBackground->GetDrawInfo(),cx*GetBoxSize(),cy*GetBoxSize(),GetBoxSize());
        }
      }
      else {
        Lum::OS::FillRef fill=Lum::OS::display->GetFill(Lum::OS::Display::backgroundFillIndex);

        fill->Draw(gameBackground->GetDrawInfo(),
                   cx*GetBoxSize(),cy*GetBoxSize(),GetBoxSize(),GetBoxSize(),
                   cx*GetBoxSize(),cy*GetBoxSize(),GetBoxSize(),GetBoxSize());
      }
    }
  }
}

void GameArea::RebuildGame()
{
  game->GetDrawInfo()->CopyFromBitmap(gameBackground,
                                      0,0,sokoban.GetWidth()*GetBoxSize(),sokoban.GetHeight()*GetBoxSize(),
                                      0,0);

  for (size_t cx=0; cx<sokoban.GetWidth(); ++cx) {
    for (size_t cy=0; cy<sokoban.GetHeight(); ++cy) {
      unsigned long  type=sokoban.Get(cx,cy);

      //
      // Draw foreground
      //

      if (type & Sokoban::typePackage) {
        if (GetImage(packageImage)!=NULL) {
          GetImage(packageImage)->Draw(game->GetDrawInfo(),
                                       cx*GetBoxSize(),
                                       cy*GetBoxSize());
        }
        else {
          DrawPackageInternal(game->GetDrawInfo(),cx*GetBoxSize(),cy*GetBoxSize(),GetBoxSize());
        }
      }

      if (type & Sokoban::typeSokoban) {
        if (GetImage(sokobanImage)!=NULL) {
          GetImage(sokobanImage)->Draw(game->GetDrawInfo(),
                                       cx*GetBoxSize(),
                                       cy*GetBoxSize());
        }
        else {
          DrawSokobanInternal(game->GetDrawInfo(),cx*GetBoxSize(),cy*GetBoxSize(),GetBoxSize());
        }
      }
    }
  }
}

void GameArea::EvaluateThemeName()
{
  for (size_t i=0; i<themes.size(); i++) {
    if (themes[i].name==themeName) {
      themeIndex=i;

      xOff=0;
      yOff=0;

      // Only rebuild game area if there is a valid Sokoban assigned!
      if (sokoban.GetWidth()>0) {
        RebuildGameBackground();
        RebuildGame();
        Redraw();
      }
      break;
    }
  }
}

void GameArea::SetSokoban(const Sokoban& sokoban)
{
  this->sokoban=sokoban;

  xOff=0;
  yOff=0;

  RebuildGameBackground();
  RebuildGame();
  Redraw();
  finished->Set(false);
}

void GameArea::CalcSize()
{
  if (Lum::OS::display->GetSize()<Lum::OS::Display::sizeNormal) {
    width=10*32;
    height=10*32;
  }
  else {
    width=20*32;
    height=20*32;
  }

  minWidth=10*32;
  minHeight=10*32;

  Object::CalcSize();
}

void GameArea::Draw(int x, int y, size_t w, size_t h)
{
  Object::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  DrawBackground(this->x,this->y,this->width,this->height);

  if (sokoban.GetWidth()==0 || sokoban.GetHeight()==0) {
    return;
  }

  Lum::OS::DrawInfo *draw=GetDrawInfo();

  if (IsAreaCompletelyVisible()) {
    draw->CopyFromBitmap(game,
                         0,0,sokoban.GetWidth()*GetBoxSize(),sokoban.GetHeight()*GetBoxSize(),
                         this->x,this->y);
  }
  else {
    draw->CopyFromBitmap(game,
                         xOff*GetBoxSize(),yOff*GetBoxSize(),
                         width-2*GetBoxSize(),
                         height-2*GetBoxSize(),
                         this->x+GetBoxSize(),this->y+GetBoxSize());

    // Up
    if (CanScrollUp()) {
      if (GetImage(upImage)!=NULL) {
        GetImage(upImage)->Draw(draw,
                                this->x+(width-GetBoxSize())/2,
                                this->y);
      }
      else {
        // TODO
      }
    }

    // Left
    if (CanScrollLeft()) {
      if (GetImage(leftImage)!=NULL) {
        GetImage(leftImage)->Draw(draw,
                                  this->x,
                                  this->y+(height-GetBoxSize())/2);
      }
      else {
        // TODO
      }
    }

    // Down
    if (CanScrollDown()) {
      if (GetImage(downImage)!=NULL) {
        GetImage(downImage)->Draw(draw,
                                  this->x+(width-GetBoxSize())/2,
                                  this->y+height-GetBoxSize());
      }
      else {
        // TODO
      }
    }

    // Right
    if (CanScrollRight()) {
      if (GetImage(rightImage)!=NULL) {
        GetImage(rightImage)->Draw(draw,
                                   this->x+width-GetBoxSize(),
                                   this->y+(height-GetBoxSize())/2);
      }
      else {
        // TODO
      }
    }
  }
}

bool GameArea::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (event.type==Lum::OS::MouseEvent::down && PointIsIn(event) && event.button==Lum::OS::MouseEvent::button1) {
    size_t x,y;
    size_t sx,sy;

    if (!IsAreaCompletelyVisible()) {
      if ((size_t)event.x<this->x+GetBoxSize() && CanScrollLeft()) {
        ScrollLeft();
        return true;
      }
      if ((size_t)event.x>=this->x+width-GetBoxSize() && CanScrollRight()) {
        ScrollRight();
        return true;
      }
      if ((size_t)event.y<this->y+GetBoxSize() && CanScrollUp()) {
        ScrollUp();
        return true;
      }
      if ((size_t)event.y>=this->y+height-GetBoxSize() && CanScrollDown()) {
        ScrollDown();
        return true;
      }

      x=(event.x-this->x)/GetBoxSize()-1+xOff;
      y=(event.y-this->y)/GetBoxSize()-1+yOff;
    }
    else {
      x=(event.x-this->x)/GetBoxSize();
      y=(event.y-this->y)/GetBoxSize();
    }

    sokoban.GetSokoban(sx,sy);

    if ((sx==x && sy==y)  || (sx!=x && sy!=y)) {
      // No move requested
      return true;
    }

    if (sx==x) {
      // Vertical movement
      if (y>sy) {
        while (y>GetSokobanYPos() && sokoban.CanMoveDown()) {
          MoveDown();
        }
      }
      else {
        while (y<GetSokobanYPos() && sokoban.CanMoveUp()) {
          MoveUp();
        }
      }
    }
    else if (sy==y) {
      // Horizontal movement
      if (x>sx) {
        while (x>GetSokobanXPos() && sokoban.CanMoveRight()) {
          MoveRight();
        }
      }
      else {
        while (x<GetSokobanXPos() && sokoban.CanMoveLeft()) {
          MoveLeft();
        }
      }
    }

    return true;
  }

  return false;
}

bool GameArea::HandleKeyEvent(const Lum::OS::KeyEvent& event)
{
  if (event.type==Lum::OS::KeyEvent::down) {
    switch (event.key) {
    case Lum::OS::keyLeft:
      if (!sokoban.CanMoveLeft()) {
        Lum::OS::display->Beep();
      }
      else {
        MoveLeft();
        finished->Set(sokoban.IsFinished());
      }
      return true;
    case Lum::OS::keyRight:
      if (!sokoban.CanMoveRight()) {
        Lum::OS::display->Beep();
      }
      else {
        MoveRight();
        finished->Set(sokoban.IsFinished());
      }
      return true;
    case Lum::OS::keyUp:
      if (!sokoban.CanMoveUp()) {
        Lum::OS::display->Beep();
      }
      else {
        MoveUp();
        finished->Set(sokoban.IsFinished());
      }
      return true;
    case Lum::OS::keyDown:
      if (!sokoban.CanMoveDown()) {
        Lum::OS::display->Beep();
      }
      else {
        MoveDown();
        finished->Set(sokoban.IsFinished());
      }
      return true;
    case Lum::OS::keyBackspace:
    case Lum::OS::keyDelete:
      Undo();
      return true;
    default:
      return false;
    }
  }

  return false;
}

Lum::Model::Boolean* GameArea::GetFinishedModel() const
{
  return finished.Get();
}

void GameArea::Undo()
{
  sokoban.Undo();
  RebuildGame();
  Redraw();
  finished->Set(sokoban.IsFinished());
}

