/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of pidgin-carman.
 *
 *  pidgin-carman is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  pidgin-carman is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with pidgin-carman.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* system headers */
#include <sys/types.h>
#include <string.h>
#include <regex.h>
#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef ENABLE_NLS
#include <glib/gi18n-lib.h>
#endif /* ENABLE_NLS */

/* libpurple headers */
#include <debug.h>
#include <sound.h>
#include <version.h>

/* local headers */
#include "pidgin-carman.h"
#include "pidgin-dbus.h"

static DBusConnection *connection = NULL;

static struct infosharing_event evt;

static PurpleConversationUiOps null_conv_uiops =
{
	NULL,			/* create_conversation  */
	NULL,			/* destroy_conversation */
	NULL,			/* write_chat           */
	NULL,			/* write_im             */
	NULL,			/* write_conv           */
	NULL,			/* chat_add_users       */
	NULL,			/* chat_rename_user     */
	NULL,			/* chat_remove_users    */
	NULL,			/* chat_update_user     */
	NULL,			/* present              */
	NULL,			/* has_focus            */
	NULL,			/* custom_smiley_add    */
	NULL,			/* custom_smiley_write  */
	NULL,			/* custom_smiley_close  */
	NULL,			/* send_confirm         */
	NULL,
	NULL,
	NULL,
	NULL
};

static gboolean match(const char *string, char *pattern)
{
	int status;
	regex_t re;
	if (regcomp(&re, pattern, REG_EXTENDED | REG_ICASE | REG_NOSUB) != 0)
		return FALSE;

	status = regexec(&re, string, 0, (regmatch_t *) NULL, 0);

	regfree(&re);

	return (status? FALSE : TRUE);
}

static PurpleBuddy *find_buddy(const char *name)
{
	GList *accounts = purple_accounts_get_all_active();
	GList *aux;
	PurpleAccount *account;
	PurpleBuddy *buddy = NULL;

	for (aux = accounts; aux != NULL; aux = aux->next) {
		account = aux->data;
		buddy = purple_find_buddy(account, name);
		if (buddy)
			break;
	}

	g_list_free(accounts);

	/* returns buddy if found, NULL otherwise */
	return buddy;
}

static int common_send(PurpleBuddy *buddy, char *message)
{
	PurpleConversation *conv = purple_conversation_new(PURPLE_CONV_TYPE_IM,
						buddy->account, buddy->name);

	debug_info("Sending message \"%s\" to %s\n", message, buddy->name);

	purple_conversation_set_ui_ops(conv, &null_conv_uiops);

	purple_conv_im_send(PURPLE_CONV_IM(conv), message);

	return 0;
}

static int send_request_authorization(const char *id)
{
	PurpleBuddy * buddy;

	buddy = find_buddy(id);
	if (!buddy) {
		debug_error("Invalid request authorization for buddy %s\n", id);
		return -1;
	}

	return common_send(buddy, PLUGIN_GREETING_MSG_FMT);
}

static int send_accept_conn(const char *id)
{
	PurpleBuddy * buddy;

	buddy = find_buddy(id);
	if (!buddy) {
		debug_error("Invalid reply authorization for buddy %s\n", id);
		return -1;
	}

	return common_send(buddy, PLUGIN_ACCEPTED_FMT);
}

static int send_reject_conn(const char *id)
{
	PurpleBuddy * buddy;

	buddy = find_buddy(id);
	if (!buddy) {
		debug_error("Invalid reply authorization for buddy %s\n", id);
		return -1;
	}

	return common_send(buddy, PLUGIN_REJECTED_FMT);
}

static int send_close_connection(const char *id)
{
	PurpleBuddy * buddy;

	buddy = find_buddy(id);
	if (!buddy) {
		debug_error("Invalid connection close for buddy %s\n", id);
		return -1;
	}

	return common_send(buddy, PLUGIN_CLOSE_CONN_FMT);
}

static int send_data_msg(const char *id, double lat, double lon,
		double alt, double track, double speed, double rpm)
{
	PurpleBuddy * buddy;
	char data[128];

	buddy = find_buddy(id);
	if (!buddy) {
		debug_error("Invalid GPS/OBD data send for buddy %s\n", id);
		return -1;
	}

	snprintf(data, 128, PLUGIN_DATA_FMT, lat, lon, alt, speed, track, rpm);
	return common_send(buddy, data);
}

static struct infosharing_command cmd = {
	send_request_authorization,
	send_close_connection,
	send_data_msg,
	send_accept_conn,
	send_reject_conn
};

static gboolean received_greeting_msg(char *id,
	const char *name)
{
	debug_info("Greeting message received from %s\n", id);

	if (evt.authorize(id, name) < 0)
		send_reject_conn(id);

	return TRUE;
}

static gboolean received_accepted_msg(char *id,
	const char *name)
{
	debug_info("Accepted connection message received from %s\n", id);

	evt.accepted(id, name);

	return TRUE;
}

static gboolean received_rejected_msg(char *sender)
{
	debug_info("Rejected connection message received from %s\n", sender);

	evt.rejected(sender);

	return TRUE;
}

static gboolean received_close_conn_msg(char *sender)
{
	debug_info("Close connection message received from %s\n", sender);

	evt.disconnected(sender);

	return TRUE;
}

static gboolean received_data_msg(
	char *sender,
	char *stripped)
{
	debug_info("GPS/OBD data message received from %s\n", sender);

	double lat, lon, alt, speed, track, rpm;

	sscanf(stripped, PLUGIN_DATA_FMT,
		&lat, &lon, &alt, &speed, &track, &rpm);

	evt.data(sender, lat, lon, alt, speed, track, rpm);

	return TRUE;
}

static void buddy_signed_off_cb(
	PurpleBuddy *buddy)
{
	debug_info("Sign off from buddy: %s\n", buddy->name);

	evt.disconnected(buddy->name);
}

static gboolean receiving_msg_cb(
	PurpleAccount * account,
	char **sender,
	char **message,
	PurpleConversation *conv,
	PurpleMessageFlags *flags)
{
	const char *name;

	PurpleBuddy *buddy = purple_find_buddy(account, *sender);

	name = buddy->server_alias? : (buddy->alias? : buddy->name);

	/* strips HTML tags from a string */
	char *stripped = purple_markup_strip_html(*message);

	if (match(stripped, PLUGIN_DATA_PATTERN))
		return received_data_msg(*sender, stripped);

	if (match(stripped, PLUGIN_GREETING_MSG_PATTERN))
		return received_greeting_msg(*sender, name);

	if (match(stripped, PLUGIN_ACCEPTED_MSG_PATTERN))
		return received_accepted_msg(*sender, name);

	if (match(stripped, PLUGIN_REJECTED_MSG_PATTERN))
		return received_rejected_msg(*sender);

	if (match(stripped, PLUGIN_CLOSE_CONN_MSG_PATTERN))
		return received_close_conn_msg(*sender);

	g_free(stripped);

	/* TRUE if the message should be canceled or FALSE otherwise */
	return FALSE;
}

static gboolean sound_event_cb(
	PurpleSoundEventID event,
	PurpleAccount *account)
{
	if (event && (PURPLE_SOUND_RECEIVE | /* msg received */
		PURPLE_SOUND_FIRST_RECEIVE | /* msg received starts a conv */
		PURPLE_SOUND_SEND)) {        /* msg sent */

		/* FIXME: Play sound if msg is *NOT* from carman plugin */
		return TRUE;
	}

	/* return TRUE if the sound should not be played or FALSE otherwise */
	return FALSE;
}

static gboolean plugin_load(PurplePlugin *plugin)
{
	void *blist_handle = purple_blist_get_handle();
	void *conv_handle = purple_conversations_get_handle();
	void *sound_handle = purple_sounds_get_handle();
	DBusError derr;

	dbus_error_init(&derr);

	/* Initialize D-Bus connection */
	connection = dbus_connection_open(CARMAND_BUS_PATH, &derr);
	if (dbus_error_is_set(&derr)) {
		gchar *title, *message;
		title = g_strdup_printf(_("Unable to Load %s Plugin"),
			plugin->info->name);
		message = g_strdup_printf(_("Can't open D-Bus connection: %s"),
			derr.message);
		dbus_error_free(&derr);
		purple_notify_error(NULL, title, message, NULL);
		debug_error(message);
		g_free(title);
		g_free(message);
		return FALSE;
	}
	dbus_connection_setup_with_g_main(connection, NULL);

	if (!dbus_init(connection, &cmd)) {
		gchar *title, *message;
		title = g_strdup_printf(_("Unable to Load %s Plugin"),
			plugin->info->name);
		message = g_strdup(_("Pidgin-carman's D-Bus server is not running"));
		purple_notify_error(NULL, title, message, NULL);
		debug_error(message);
		g_free(title);
		g_free(message);
		return FALSE;
	}

	dbus_init_callbacks(&evt);

	/* Libpurple callbacks */
	purple_signal_connect(blist_handle,
		"buddy-signed-off", plugin,
		PURPLE_CALLBACK(buddy_signed_off_cb), NULL);
	purple_signal_connect(sound_handle,
		"playing-sound-event", plugin,
		PURPLE_CALLBACK(sound_event_cb), NULL);
	purple_signal_connect(conv_handle,
		"receiving-im-msg", plugin,
		PURPLE_CALLBACK(receiving_msg_cb), NULL);

	debug_info("Plugin loaded\n");

	return TRUE;
}

static gboolean plugin_unload(PurplePlugin *plugin)
{
	void *blist_handle = purple_blist_get_handle();
	void *conv_handle = purple_conversations_get_handle();
	void *sound_handle = purple_sounds_get_handle();

	purple_signal_disconnect(blist_handle,
		"buddy-signed-off", plugin,
		PURPLE_CALLBACK(buddy_signed_off_cb));
	purple_signal_disconnect(sound_handle,
		"playing-sound-event", plugin,
		PURPLE_CALLBACK(sound_event_cb));
	purple_signal_disconnect(conv_handle,
		"receiving-im-msg", plugin,
		PURPLE_CALLBACK(receiving_msg_cb));

	dbus_exit();
	if (connection)
		dbus_connection_unref(connection);

	debug_info("Plugin unloaded\n");

	return TRUE;
}

static PurplePluginInfo info = {
	PURPLE_PLUGIN_MAGIC,
	PURPLE_MAJOR_VERSION,
	PURPLE_MINOR_VERSION,
	PURPLE_PLUGIN_STANDARD,
	NULL,
	0,
	NULL,
	PURPLE_PRIORITY_DEFAULT,

	PLUGIN_ID,
	NULL,
	PACKAGE_VERSION,

	NULL,
	NULL,

	"Bruno de Oliveira Abinader <bruno.abinader@openbossa.org>",
	"http://www.indt.org.br",

	plugin_load,
	plugin_unload,
	NULL,

	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static void init_plugin(PurplePlugin *plugin)
{
	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");

	info.name = _("Carman Information Share");
	info.summary = _("Carman Information Share plugin for Pidgin");
	info.description = _("Allows user to share OBD-II and GPS data to other users using Carman.");

	debug_info("Plugin initialized\n");
}

PURPLE_INIT_PLUGIN(carman, init_plugin, info)
