/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of pidgin-carman.
 *
 *  pidgin-carman is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  pidgin-carman is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with pidgin-carman.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <glib.h>

#ifdef ENABLE_NLS
#include <glib/gi18n-lib.h>
#endif /* ENABLE_NLS */

#include <dbus/dbus.h>

#include "pidgin-dbus.h"

#define BUDDY_ACCEPT	1
#define BUDDY_REJECT	0
#define BUDDY_AUTH_TIMEOUT	45000 /* 45 sec max time to receive buddy response */
#define AGENT_AUTH_TIMEOUT	40000 /* 40 sec max time to receive user(agent) response */
#define IDLE_TIMEOUT		30

#define OBD_PID_RPM	0x0C
#define OBD_PID_SPEED	0x0D

#define is_obd_rpm(pid)		pid == OBD_PID_RPM

#define is_obd_speed(pid)	pid == OBD_PID_SPEED

/* Buddy data reporting frequency */
#define DATA_REPORTING		5000

/* D-Bus definitions */
#define CARMANPLUGIN	"org.indt.carmanplugin"
#define MANAGER_PATH	"/"
#define MANAGER_IFACE	"org.indt.carmanplugin.Manager"

/* Buddy path is dynamic: based on buddy's address */
#define BUDDY_IFACE	"org.indt.carmanplugin.Buddy"
#define ERROR_IFACE	"org.indt.carmanplugin.Error"

#define MANAGER_INTROSPECT	DBUS_INTROSPECT_1_0_XML_DOCTYPE_DECL_NODE \
				"<node name=\""MANAGER_PATH"\">\n"	\
					"\t<interface name=\""MANAGER_IFACE"\">\n" \
					"\t\t<method name=\"RequestAuthorization\">\n" \
					"\t\t\t<arg type=\"s\" direction=\"in\"/>\n" \
					"\t\t</method>\n" \
					"\t\t<method name=\"RegisterAgent\">\n" \
					"\t\t\t<arg type=\"o\" direction=\"in\"/>\n" \
					"\t\t</method>\n" \
					"\t\t<method name=\"UnregisterAgent\">\n" \
					"\t\t\t<arg type=\"o\" direction=\"in\"/>\n" \
					"\t\t</method>\n" \
					"\t\t<method name=\"ListBuddies\">\n" \
					"\t\t\t<arg type=\"ao\" direction=\"out\"/>\n" \
					"\t\t</method>\n" \
					"\t\t<method name=\"RemoveBuddy\">\n" \
					"\t\t\t<arg type=\"o\" direction=\"in\"/>\n" \
					"\t\t</method>\n" \
					"\t\t<signal name=\"BuddyCreated\">\n" \
					"\t\t\t<arg type=\"o\"/>\n" \
					"\t\t</signal>\n" \
					"\t\t<signal name=\"BuddyRemoved\">\n" \
					"\t\t\t<arg type=\"o\"/>\n" \
					"\t\t</signal>\n" \
					"\t</interface>\n"

#define BUDDY_INTROSPECT	DBUS_INTROSPECT_1_0_XML_DOCTYPE_DECL_NODE \
				"<node name=\"%s\">\n"	\
					"\t<interface name=\""BUDDY_IFACE"\">\n" \
					"\t\t<method name=\"GetInfo\">\n" \
					"\t\t\t<arg type=\"a{sv}\" direction=\"out\"/>\n" \
					"\t\t</method>\n" \
					"\t\t<signal name=\"DataAvailable\">\n" \
					"\t\t\t<arg type=\"dddddd\"/>\n" \
					"\t\t</signal>\n" \
					"\t</interface>\n" \
				"</node>"


#define CARMAND_GPS_STATUS_MATCH_PATTERN	"sender='org.indt.carmand',type='signal'," \
						"interface='org.indt.carmand.GPS'," \
						"member='StatusChanged'"

#define CARMAND_GPS_DATA_MATCH_PATTERN	"sender='org.indt.carmand',type='signal'," \
					"interface='org.indt.carmand.GPS'," \
					"member='DataAvailable'"

#define CARMAND_OBD_MATCH_PATTERN	"sender='org.indt.carmand',type='signal'," \
					"interface='org.indt.carmand.OBD'," \
					"member='DataAvailable'"

struct buddy_info {
	gchar *id;
	gchar *name;
};

struct buddy {
	gchar *id;
	gchar *name;
	gchar *path;
	time_t last;
};

struct authorization {
	gchar		*id;
	gchar		*busid;
	DBusMessage	*msg;
	guint		watch;
};

struct agent {
	gchar *busid;
	gchar *path;
	gchar *id;		/* Pending id/email to be authorized */
	DBusPendingCall *call;  /* Pending authorization */
};

struct position_data {
	double latitude;
	double longitude;
	double altitude;
	double track;
	double speed;
	double rpm;
};

struct DBusMethodTable {
	const gchar *name;
	DBusHandlerResult (*handler)(DBusConnection *conn, DBusMessage *msg, void *user_data);
};

static DBusConnection *connection = NULL;
static struct agent *agent = NULL;
static struct authorization *pending = NULL;
static struct position_data pos;
static GSList *buddies = NULL;
static guint tmwatch = 0;
static guint buddy_counter = 0;
static const struct infosharing_command *cmd;
static gchar *gps_status = NULL;

static gboolean match(DBusConnection *conn, gboolean add, const gchar *pattern)
{
	DBusError derr;

	dbus_error_init(&derr);
	if (add)
		dbus_bus_add_match(conn, pattern, &derr);
	else
		dbus_bus_remove_match(conn, pattern, &derr);

	if (dbus_error_is_set(&derr))
		goto match_error;

	return TRUE;

match_error:
	debug_fprintf("%s match rule \"%s\" failed: %s",
			(add ? "Adding" : "Removing"),
			pattern, derr.message);
	dbus_error_free(&derr);

	return FALSE;
}

static gboolean name_match(DBusConnection *conn, gboolean add, const gchar *busid)
{
	gchar *pattern;
	gboolean ret;

	pattern = g_strconcat("interface=", DBUS_INTERFACE_DBUS,
			",member=NameOwnerChanged,arg0=", busid, NULL);
	ret = match(conn, add, pattern);
	g_free(pattern);
	return ret;
}

static void agent_free(struct agent *agent)
{
	if (!agent)
		return;

	name_match(connection, FALSE, agent->busid);

	g_free(agent->busid);
	g_free(agent->path);
	if (agent->id)
		g_free(agent->id);
	if (agent->call)
		dbus_pending_call_cancel(agent->call);
	g_free(agent);
}

static void authorization_free(struct authorization *auth)
{
	if (!auth)
		return;

	name_match(connection, FALSE, pending->busid);

	if (auth->watch)
		g_source_remove(auth->watch);
	dbus_message_unref(auth->msg);
	g_free(auth->id);
	g_free(auth->busid);
	g_free(auth);
}

static void buddy_info_free(struct buddy_info *buddy_info)
{
	if (!buddy_info)
		return;

	g_free(buddy_info->id);
	g_free(buddy_info->name);
	g_free(buddy_info);
}

static void buddy_free(struct buddy *buddy)
{
	if (!buddy)
		return;

	g_free(buddy->path);
	g_free(buddy->id);
	g_free(buddy->name);
	g_free(buddy);
}

static int buddy_id_cmp(const struct buddy *buddy, const gchar *id)
{
	return strncmp(buddy->id, id, sizeof(buddy->id));
}

static int buddy_path_cmp(const struct buddy *buddy, const gchar *path)
{
	return strcmp(buddy->path, path);
}

void dbus_message_iter_append_dict_entry(DBusMessageIter *dict,
					const char *key, int type, void *val)
{
	DBusMessageIter entry;
	DBusMessageIter value;
	char *sig;

	dbus_message_iter_open_container(dict, DBUS_TYPE_DICT_ENTRY, NULL, &entry);

	dbus_message_iter_append_basic(&entry, DBUS_TYPE_STRING, &key);

	switch (type) {
	case DBUS_TYPE_STRING:
		sig = DBUS_TYPE_STRING_AS_STRING;
		break;
	case DBUS_TYPE_BYTE:
		sig = DBUS_TYPE_BYTE_AS_STRING;
		break;
	case DBUS_TYPE_INT16:
		sig = DBUS_TYPE_INT16_AS_STRING;
		break;
	case DBUS_TYPE_UINT32:
		sig = DBUS_TYPE_UINT32_AS_STRING;
		break;
	case DBUS_TYPE_BOOLEAN:
		sig = DBUS_TYPE_BOOLEAN_AS_STRING;
		break;
	case DBUS_TYPE_VARIANT:
	default:
		sig = DBUS_TYPE_VARIANT_AS_STRING;
		break;
	}

	dbus_message_iter_open_container(&entry, DBUS_TYPE_VARIANT, sig, &value);

	dbus_message_iter_append_basic(&value, type, val);

	dbus_message_iter_close_container(&entry, &value);

	dbus_message_iter_close_container(dict, &entry);
}

static dbus_bool_t emit_signal_valist(DBusConnection *conn,
		const char *path,
		const char *interface,
		const char *name,
		int first,
		va_list var_args)
{
	DBusMessage *signal;
	dbus_bool_t ret;

	signal = dbus_message_new_signal(path, interface, name);
	if (!signal) {
		debug_fprintf("No memory %s.%s signal", interface,  name);
		return FALSE;
	}

	ret = dbus_message_append_args_valist(signal, first, var_args);
	if (!ret)
		goto fail;

	ret = dbus_connection_send(conn, signal, NULL);

fail:
	dbus_message_unref(signal);

	return ret;
}

static gboolean emit_dbus_signal(DBusConnection *conn,
		const char *path, const char *iface,
		const char *name, int type, ...)
{
	va_list args;
	gboolean ret;

	va_start(args, type);

	ret = emit_signal_valist(conn, path, iface,
			name, type, args);

	va_end(args);

	return ret;
}

static DBusHandlerResult send_dbus_msg_and_unref(DBusConnection *conn,
							DBusMessage *msg)
{
	if (!msg)
		return DBUS_HANDLER_RESULT_NEED_MEMORY;

	dbus_connection_send(conn, msg, NULL);
	dbus_message_unref(msg);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static gboolean authorization_expired(gpointer user_data)
{
	DBusMessage *reply;

	debug_fprintf("%s\n", __func__);

	if (!pending)
		return FALSE;

	reply = dbus_message_new_error(pending->msg,
					ERROR_IFACE ".Rejected",
					"Connection rejected");

	authorization_free(pending);
	pending = NULL;

	send_dbus_msg_and_unref(connection, reply);

	return FALSE;
}
static DBusHandlerResult request_authorization(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	DBusMessage *reply;
	const gchar *id;
	/*
	 * Carman python UI or any other client use this method to
	 * start the authorization: authorization shall be sent through
	 * the network using XMPP or any other transport protocol
	 */

	if (pending) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".InProgress",
				"Buddy authorization in progress");
		return send_dbus_msg_and_unref(conn, reply);
	}

	if (!dbus_message_get_args(msg, NULL,
				DBUS_TYPE_STRING, &id,
				DBUS_TYPE_INVALID)) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".InvalidArgs",
				"Invalid buddy address");
		return send_dbus_msg_and_unref(conn, reply);
	}

	if (buddies) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".AlreadyConnected",
				"Connected Buddies capacity achieved");
		return send_dbus_msg_and_unref(conn, reply);
	}

	/* FIXME: For now we're limiting buddies to 1 only */
#if 0
	if (g_slist_find_custom(buddies, id, (GCompareFunc) buddy_id_cmp)) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".AlreadyExists",
				"Buddy already exists");
		return send_dbus_msg_and_unref(conn, reply);
	}
#endif

	if (strcmp("Disconnected", gps_status) == 0) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".GPSDisconnected",
				"GPS disconnected");
		return send_dbus_msg_and_unref(conn, reply);
	}

	/* FIXME: call a function to send the authorization request.
	 * On success BuddyAdded shall be sent and a new buddy path
	 * shall be sent. On error we need to analyze if a synchronous
	 * response makes sense.
	 */
	if (cmd->connect(id) < 0) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".InvalidArgs",
				"Invalid buddy address");
		return send_dbus_msg_and_unref(conn, reply);
	}

	pending = g_new0(struct authorization, 1);
	pending->id = g_strdup(id);
	pending->msg = dbus_message_ref(msg);
	pending->busid = g_strdup(dbus_message_get_sender(msg));
	pending->watch = g_timeout_add(BUDDY_AUTH_TIMEOUT,
					authorization_expired, NULL);
	name_match(conn, TRUE, pending->busid);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult register_agent(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	DBusMessage *reply;
	const gchar *path, *busid;

	if (agent) {
		/* Agent already registered */
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".AlreadyExists",
				"Agent already exists");
		return send_dbus_msg_and_unref(conn, reply);
	}

	if (!dbus_message_get_args(msg, NULL,
				DBUS_TYPE_OBJECT_PATH, &path,
				DBUS_TYPE_INVALID)) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".InvalidArgs",
				"Invalid agent path");
		return send_dbus_msg_and_unref(conn, reply);
	}

	reply = dbus_message_new_method_return(msg);
	if (!reply)
		return DBUS_HANDLER_RESULT_NEED_MEMORY;

	busid = dbus_message_get_sender(msg);
	agent = g_new0(struct agent, 1);
	agent->busid = g_strdup(busid);
	agent->path = g_strdup(path);

	/* Track agent disconnection */
	name_match(conn, TRUE, busid);

	return send_dbus_msg_and_unref(conn, reply);
}

static DBusHandlerResult unregister_agent(DBusConnection *conn,
					DBusMessage *msg, void *user_data)
{
	DBusMessage *reply;
	const gchar *path;
#if 0
	*busid = dbus_message_get_sender(msg);
#endif
	gboolean ret;

	if (!agent) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".DoesNotExists",
				"Agent does not exists");

		return send_dbus_msg_and_unref(conn, reply);
	}

	ret = dbus_message_get_args(msg, NULL,
				DBUS_TYPE_OBJECT_PATH, &path,
				DBUS_TYPE_INVALID);

	if (ret == FALSE || strcmp(agent->path, path) != 0) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".InvalidArgs",
				"Invalid agent path");
		return send_dbus_msg_and_unref(conn, reply);
	}

#if 0
	/* FIXME: Check bus id and path */
	if (strcmp(busid, agent->busid) != 0)
		return /* not allowed */
#endif

	reply = dbus_message_new_method_return(msg);
	if (!reply)
		return DBUS_HANDLER_RESULT_NEED_MEMORY;

	agent_free(agent);
	agent = NULL;

	return send_dbus_msg_and_unref(conn, reply);
}

static DBusHandlerResult list_buddies(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	DBusMessage *reply;
	DBusMessageIter iter, array_iter;
	struct buddy *buddy;
	GSList *l;

	reply = dbus_message_new_method_return(msg);
	if (!reply)
		return DBUS_HANDLER_RESULT_NEED_MEMORY;

	dbus_message_iter_init_append(reply, &iter);
	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY,
				DBUS_TYPE_STRING_AS_STRING, &array_iter);

	for (l = buddies; l; l = l->next) {
		buddy = l->data;
		dbus_message_iter_append_basic(&array_iter,
				DBUS_TYPE_STRING, &buddy->id);
	}

	dbus_message_iter_close_container(&iter, &array_iter);

	return send_dbus_msg_and_unref(conn, reply);
}

static DBusHandlerResult remove_buddy(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	DBusMessage *reply;
	struct buddy *buddy;
	const gchar *path;
	GSList *l;

	if (!dbus_message_get_args(msg, NULL,
				DBUS_TYPE_OBJECT_PATH, &path,
				DBUS_TYPE_INVALID)) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".InvalidArgs",
				"Invalid path");
		return send_dbus_msg_and_unref(conn, reply);
	}

	l = g_slist_find_custom(buddies, path, (GCompareFunc) buddy_path_cmp);
	if (!l) {
		reply = dbus_message_new_error(msg,
				ERROR_IFACE ".DoesNotExists",
				"Buddy doesn't exists");
		return send_dbus_msg_and_unref(conn, reply);
	}

	reply = dbus_message_new_method_return(msg);
	if (!reply)
		return DBUS_HANDLER_RESULT_NEED_MEMORY;

	buddy = l->data;

	/*
	 * Ignore the return value: Remote device will
	 * automatically detect a problem after 30 seconds.
	 */
	cmd->disconnect(buddy->id);

	/* FIXME: is it necessary check busid? */
	dbus_connection_unregister_object_path(conn, path);

	emit_dbus_signal(conn, MANAGER_PATH, MANAGER_IFACE,
			"BuddyRemoved", DBUS_TYPE_OBJECT_PATH, &path,
			DBUS_TYPE_INVALID);

	return send_dbus_msg_and_unref(conn, reply);
}

struct DBusMethodTable manager_methods[] = {
	{ "RequestAuthorization",	request_authorization	},
	{ "RegisterAgent",		register_agent		},
	{ "UnregisterAgent",		unregister_agent	},
	{ "ListBuddies",		list_buddies		},
	{ "RemoveBuddy",		remove_buddy		},
	{}
};

static void manager_unregister(DBusConnection *connection, void *user_data)
{
	debug_fprintf("Unregistered manager path\n");
}

static DBusHandlerResult manager_introspect(DBusConnection *conn,
				DBusMessage *msg, const gchar *base_xml)
{
	DBusMessage *reply = dbus_message_new_method_return(msg);
	const gchar *path = dbus_message_get_path(msg);
	gchar **children;
	gchar *xml, *nodes = NULL, *node;
	int i;

	if (!reply)
		return DBUS_HANDLER_RESULT_NEED_MEMORY;

	if (!dbus_connection_list_registered(conn, path, &children))
		goto done;

	for (i = 0; children[i]; i++) {
		if (nodes) {
			node = g_strconcat(nodes, "\t<node name=\"",
						children[i], "\"/>\n", NULL);
			g_free(nodes);
			nodes = node;
		} else {
			nodes = g_strconcat("\t<node name=\"",
						children[i], "\"/>\n", NULL);
		}
	}

	dbus_free_string_array(children);

done:

	xml = g_strconcat(base_xml, (nodes ? : ""), "</node>", NULL);
	dbus_message_append_args(reply,
			DBUS_TYPE_STRING, &xml,
			DBUS_TYPE_INVALID);
	g_free(nodes);
	g_free(xml);

	return send_dbus_msg_and_unref(conn, reply);
}

static DBusHandlerResult manager_message(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	struct DBusMethodTable *method;
	const gchar *name = dbus_message_get_member(msg);

	if (dbus_message_is_method_call(msg,
				DBUS_INTERFACE_INTROSPECTABLE,
				"Introspect"))
		return manager_introspect(conn, msg, MANAGER_INTROSPECT);

	if (strcmp(MANAGER_IFACE, dbus_message_get_interface(msg)) != 0)
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	for (method = &manager_methods[0]; method; method++)
		if (strcmp(method->name, name) == 0)
			return method->handler(conn, msg, user_data);

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusObjectPathVTable manager_table = {
	.unregister_function    = manager_unregister,
	.message_function       = manager_message,
};

static gboolean register_manager(void)
{
	if (!dbus_connection_register_object_path(connection, MANAGER_PATH,
				&manager_table, NULL)) {
		return FALSE;
	}

	return TRUE;
}

static void unregister_manager(void)
{
	dbus_connection_unregister_object_path(connection, MANAGER_PATH);
}

static DBusHandlerResult connection_filter(DBusConnection *conn,
				DBusMessage *msg, void *user_data)
{
	const gchar *name, *old, *new;
	int mode;

	debug_fprintf("%s %s.%s\n", __func__, dbus_message_get_interface(msg),
						dbus_message_get_member(msg));

	if (dbus_message_is_signal(msg,
		DBUS_INTERFACE_DBUS, "NameOwnerChanged")) {

		if (!dbus_message_get_args(msg, NULL,
					DBUS_TYPE_STRING, &name,
					DBUS_TYPE_STRING, &old,
					DBUS_TYPE_STRING, &new,
					DBUS_TYPE_INVALID)) {
			return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
		}

		/* Ignore service creations */
		if (*new != '\0')
			return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

		if (agent && strcmp(agent->busid, name) == 0) {
			/* Agent exited */
			debug_fprintf("Agent exited: %s\n", name);
			agent_free(agent);
			agent = NULL;
		} else if (pending && strcmp(pending->busid, name) == 0) {
			/* RequestAuthorization client exitted */
			authorization_free(pending);
			pending = NULL;
		} else
			return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

		/* FIXME: Reply pending requests */

	} else if (dbus_message_is_signal(msg,
			"org.indt.carmand.OBD", "DataAvailable")) {
		double value1, value2, value3, value4;
		int pid;

		dbus_message_get_args(msg, NULL,
				DBUS_TYPE_INT32, &pid,
				DBUS_TYPE_DOUBLE, &value1,
				DBUS_TYPE_DOUBLE, &value2,
				DBUS_TYPE_DOUBLE, &value3,
				DBUS_TYPE_DOUBLE, &value4,
				DBUS_TYPE_INVALID);

		if (is_obd_rpm(pid))
			pos.rpm = value1;
		else if (is_obd_speed(pid))
			pos.speed = value1;
	} else if (dbus_message_has_interface(msg, "org.indt.carmand.GPS")) {
		const char *member;

		member = dbus_message_get_member(msg);
		if (strcmp("DataAvailable", member) == 0) {
			dbus_message_get_args(msg, NULL,
				DBUS_TYPE_INT32, &mode,
				DBUS_TYPE_DOUBLE, &pos.latitude,
				DBUS_TYPE_DOUBLE, &pos.longitude,
				DBUS_TYPE_DOUBLE, &pos.altitude,
				DBUS_TYPE_DOUBLE, &pos.speed,
				DBUS_TYPE_DOUBLE, &pos.track,
				DBUS_TYPE_INVALID);
		} else if (strcmp("StatusChanged", member) == 0) {
			const char *new_status;
			dbus_message_get_args(msg, NULL,
				DBUS_TYPE_STRING, &new_status,
				DBUS_TYPE_INVALID);
			if (gps_status)
				g_free(gps_status);
			gps_status = g_strdup(new_status);
		}
	}

	/* Return always not handled otherwise this signal will
	 * not arrive for other apps interested on this signal */

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusHandlerResult buddy_introspect(DBusConnection *conn,
	DBusMessage *msg)
{
	DBusMessage *reply = dbus_message_new_method_return(msg);
	const gchar *path = dbus_message_get_path(msg);
	gchar *xml;
	if (!reply)
		return DBUS_HANDLER_RESULT_NEED_MEMORY;

	debug_fprintf("Instrospecting buddy path:%s\n", path);

	xml = g_strdup_printf(BUDDY_INTROSPECT, path);
	dbus_message_append_args(reply,
			DBUS_TYPE_STRING, &xml,
			DBUS_TYPE_INVALID);

	g_free(xml);
	return send_dbus_msg_and_unref(connection, reply);
}

static DBusHandlerResult buddy_get_info(DBusConnection *conn,
	DBusMessage *msg, void *user_data)
{
	const gchar *property;
	struct buddy *buddy = user_data;
	DBusMessageIter iter, dict, entry, value;
	DBusMessage *reply = dbus_message_new_method_return(msg);
	if (!reply)
		return DBUS_HANDLER_RESULT_NEED_MEMORY;

        dbus_message_iter_init_append(reply, &iter);

	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY,
			DBUS_DICT_ENTRY_BEGIN_CHAR_AS_STRING
			DBUS_TYPE_STRING_AS_STRING DBUS_TYPE_VARIANT_AS_STRING
			DBUS_DICT_ENTRY_END_CHAR_AS_STRING, &dict);

	/* Name */
	property = "Name";

	dbus_message_iter_open_container(&dict, DBUS_TYPE_DICT_ENTRY,
			NULL, &entry);

	dbus_message_iter_append_basic(&entry, DBUS_TYPE_STRING, &property);

	dbus_message_iter_open_container(&entry, DBUS_TYPE_VARIANT,
			DBUS_TYPE_STRING_AS_STRING, &value);

	dbus_message_iter_append_basic(&value, DBUS_TYPE_STRING, &buddy->name);

	dbus_message_iter_close_container(&entry, &value);

	dbus_message_iter_close_container(&dict, &entry);

	/* Email */
	property = "Email";

	dbus_message_iter_open_container(&dict, DBUS_TYPE_DICT_ENTRY,
			NULL, &entry);

	dbus_message_iter_append_basic(&entry, DBUS_TYPE_STRING, &property);

	dbus_message_iter_open_container(&entry, DBUS_TYPE_VARIANT,
			DBUS_TYPE_STRING_AS_STRING, &value);

	dbus_message_iter_append_basic(&value, DBUS_TYPE_STRING, &buddy->id);

	dbus_message_iter_close_container(&entry, &value);

	dbus_message_iter_close_container(&dict, &entry);

	dbus_message_iter_close_container(&iter, &dict);

	return send_dbus_msg_and_unref(conn, reply);
}

static void buddy_unregister(DBusConnection *conn, void *user_data)
{
	struct buddy *buddy = user_data;

	buddies = g_slist_remove(buddies, buddy);
	if (buddies == NULL) {

		/* Remove local GPS and OBD signals filter */
		match(conn, FALSE, CARMAND_GPS_DATA_MATCH_PATTERN);
		match(conn, FALSE, CARMAND_OBD_MATCH_PATTERN);

		g_source_remove(tmwatch);
		tmwatch = 0;
	}

	buddy_free(buddy);

	debug_fprintf("Unregistered buddy path\n");
}

static DBusHandlerResult buddy_message(DBusConnection *conn,
		DBusMessage *msg, void *user_data)
{
	if (dbus_message_is_method_call(msg,
				DBUS_INTERFACE_INTROSPECTABLE,
				"Introspect"))
		return buddy_introspect(conn, msg);
	else if (dbus_message_is_method_call(msg,
				BUDDY_IFACE,
				"GetInfo"))
		return buddy_get_info(conn, msg, user_data);

	if (strcmp(BUDDY_IFACE, dbus_message_get_interface(msg)) != 0)
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusObjectPathVTable buddy_table = {
	.unregister_function    = buddy_unregister,
	.message_function       = buddy_message,
};

static gboolean data_reporting_watch(gpointer user_data)
{
	struct buddy *buddy;
	GSList *l, *lrm = NULL;
	time_t t, dif;

	/*
	 * Check if there is a buddy that has not
	 * sent GPS/OBD data in the last 30 seconds
	 */
	t = time(NULL);
	for (l = buddies; l; l = l->next) {
		buddy = l->data;
		dif = difftime(t, buddy->last);
		if (dif < IDLE_TIMEOUT)
			continue;

		debug_fprintf("Removing buddy(%s): No GPS data received\n",
								buddy->path);
		lrm = g_slist_append(lrm, g_strdup(buddy->path));
	}

	for (l = lrm; l; l = l->next) {
		const char *path = lrm->data;
		DBusMessage *msg;

		/* buddy is removed from the list: see buddy_unregister */
		dbus_connection_unregister_object_path(connection, path);

		msg = dbus_message_new_signal(MANAGER_PATH,
				MANAGER_IFACE, "BuddyRemoved");
		if (!msg)
			break;

		dbus_message_append_args(msg,
				DBUS_TYPE_OBJECT_PATH, &path,
				DBUS_TYPE_INVALID);

		send_dbus_msg_and_unref(connection, msg);
	}

	if (lrm) {
		g_slist_foreach(lrm, (GFunc) g_free, NULL);
		g_slist_free(lrm);
	}

	for (l = buddies; l; l = l->next) {
		buddy = l->data;
		fprintf(stderr, "Sending to: %s (%lf, %lf, %lf, %lf, %lf, %lf)\n",
				buddy->id, pos.latitude, pos.longitude,
				pos.altitude, pos.track, pos.speed, pos.rpm);

		cmd->write(buddy->id, pos.latitude, pos.longitude,
				pos.altitude, pos.track, pos.speed, pos.rpm);
	}

	/* Reset rpm to avoid send wrong values. For GPS data
	 * it doesn't matter, we always send the last value no
	 * matter if GPS is fixing or fixed.
	 */
	pos.rpm = -1;

	return TRUE;
}

static void data_reporting_destroyed(gpointer user_data)
{
	debug_fprintf("Stopping buddy data reporting\n");
}

static gboolean register_buddy(const char *email,
	const char *name)
{
	struct buddy *buddy;

	buddy = g_new0(struct buddy, 1);
	buddy->id = g_strdup(email);
	buddy->name = g_strdup(name);
	buddy->last = time(NULL);

	buddy->path = g_strdup_printf("/buddy%d", buddy_counter++);

	if (!dbus_connection_register_fallback(connection,
				buddy->path, &buddy_table, buddy)) {
		debug_fprintf("D-Bus path registration failed: %s\n", buddy->path);
		buddy_free(buddy);
		return FALSE;
	}

	emit_dbus_signal(connection, MANAGER_PATH, MANAGER_IFACE,
			"BuddyCreated", DBUS_TYPE_OBJECT_PATH, &buddy->path,
			DBUS_TYPE_INVALID);

	if (buddies == NULL) {

		/* Filter local GPS and OBD signals */
		match(connection, TRUE, CARMAND_GPS_DATA_MATCH_PATTERN);
		match(connection, TRUE, CARMAND_OBD_MATCH_PATTERN);

		/* First buddy: Start the Buddy data reporting */
		tmwatch = g_timeout_add_full(G_PRIORITY_DEFAULT,
				DATA_REPORTING,
				data_reporting_watch,
				NULL,
				data_reporting_destroyed);
	}

	buddies = g_slist_append(buddies, buddy);

	return TRUE;
}

static void agent_release(struct agent *agent)
{
	DBusMessage *msg = dbus_message_new_method_call(agent->busid, agent->path,
			"org.indt.carmanplugin.Agent", "Release");
	if (msg == NULL)
		return;

	dbus_message_set_no_reply(msg, TRUE);

	send_dbus_msg_and_unref(connection, msg);
}

static void agent_authorize_reply(DBusPendingCall *call, void *user_data)
{
	DBusMessage *reply =  dbus_pending_call_steal_reply(call);
	DBusError derr;
	struct buddy_info *buddy_info = user_data;

	dbus_error_init(&derr);
	if (dbus_set_error_from_message(&derr, reply)) {
		cmd->reject(agent->id);
		debug_fprintf("agent reply: %s\n", derr.message);
		dbus_error_free(&derr);
		goto done;
	}

	/* FIXME: For now, we're limiting buddies to 1 only */
	if (buddies) {
		cmd->reject(agent->id);
		goto done;
	}

	register_buddy(buddy_info->id, buddy_info->name);
	cmd->accept(agent->id);

done:
	dbus_message_unref(reply);
	dbus_pending_call_unref(agent->call);
	agent->call = NULL;
	g_free(agent->id);
	agent->id = NULL;
}

static int buddy_authorize(const char *id, const char *name)
{
	struct buddy_info *buddy_info;
	DBusMessage *msg;

	debug_fprintf("%s id:%d\n", __func__, id);

	if (strcmp(gps_status, "Disconnected") == 0) {
		debug_fprintf("GPS not connected\n");
		return -EPERM;
	}

	if (pending) {
		debug_fprintf("Waiting authorization! " \
				"Incomming connections blocked!\n");
		return -EPERM;
	}

	if (!agent) {
		debug_fprintf("No agent available!\n");
		return -EPERM;
	}

	if (agent->call) {
		debug_fprintf("Agent busy: Authorization pending\n");
		return -EALREADY;
	}

	/* FIXME: Maybe we can connect same buddy in the future */
	if (g_slist_find_custom(buddies, id, (GCompareFunc) buddy_id_cmp)) {
		debug_fprintf("Buddy already connected\n");
		return -EALREADY;
	}

	msg = dbus_message_new_method_call(agent->busid,
			agent->path, "org.indt.carmanplugin.Agent", "Authorize");
	if (msg == NULL)
		return -ENOMEM;

	dbus_message_append_args(msg,
			DBUS_TYPE_STRING, &id,
			DBUS_TYPE_STRING, &name,
			DBUS_TYPE_INVALID);

	dbus_connection_send_with_reply(connection, msg,
			&agent->call, AGENT_AUTH_TIMEOUT);

	buddy_info = g_new0(struct buddy_info, 1);
	buddy_info->id = g_strdup(id);
	buddy_info->name = g_strdup(name);

	dbus_pending_call_set_notify(agent->call,
			agent_authorize_reply, buddy_info,
			(DBusFreeFunction) buddy_info_free);

	agent->id = g_strdup(id);

	return 0;
}

static int buddy_data(const char *id, double lat, double lon,
			double alt, double speed, double track, double rpm)
{
	DBusMessage *msg;
	struct buddy *buddy;
	GSList *l;

	l = g_slist_find_custom(buddies, id, (GCompareFunc) buddy_id_cmp);
	if (!l) {
		debug_fprintf("Buddy(%s) doesn't exists\n", id);
		return -EPERM;
	}

	buddy = l->data;
	buddy->last = time(NULL);

	fprintf(stderr, "Received from: %s (%lf, %lf, %lf, %lf, %lf, %lf)\n",
			buddy->id, lat, lon, alt, speed, track, rpm);

	msg = dbus_message_new_signal(buddy->path,
			BUDDY_IFACE, "DataAvailable");
	if (!msg)
		return -ENOMEM;

	dbus_message_append_args(msg,
			DBUS_TYPE_DOUBLE, &lat,
			DBUS_TYPE_DOUBLE, &lon,
			DBUS_TYPE_DOUBLE, &alt,
			DBUS_TYPE_DOUBLE, &speed,
			DBUS_TYPE_DOUBLE, &track,
			DBUS_TYPE_DOUBLE, &rpm,
			DBUS_TYPE_INVALID);

	dbus_connection_send(connection, msg, NULL);
	dbus_message_unref(msg);

	return 0;
}

static void buddy_disconnected(const char *id)
{
	DBusMessage *msg;
	struct buddy *buddy;
	GSList *l;

	/* Verify if buddy belongs to connected buddies list */
	l = g_slist_find_custom(buddies, id, (GCompareFunc) buddy_id_cmp);
	if (!l) {
		debug_fprintf("Buddy(%s) is not connected to carman, doing nothing\n", id);
		return;
	}

	buddy = l->data;

	/* Remote initiated disconnection */
	msg = dbus_message_new_signal(MANAGER_PATH,
			MANAGER_IFACE, "BuddyRemoved");
	if (!msg)
		return;

	dbus_message_append_args(msg,
			DBUS_TYPE_OBJECT_PATH, &buddy->path,
			DBUS_TYPE_INVALID);

	send_dbus_msg_and_unref(connection, msg);

	dbus_connection_unregister_object_path(connection, buddy->path);
}

static void buddy_accepted(const char *id,
	const char *name)
{
#if 0
	GSList *l;
	struct buddy *buddy;
#endif
	DBusMessage *reply;

	if (!pending)
		return;

	/* FIXME: Fow now, we're limiting buddies to 1 only */
	if (buddies)
		goto free_pending;

#if 0
	/* Check if buddy is already in the list */
        for (l = buddies; l; l = l->next) {
		buddy = l->data;
		/* FIXME: Ignore Resource */
		if (strncmp(buddy->id, id, strlen(buddy->id)) == 0)
			goto free_pending;
	}
#endif

	register_buddy(id, name);

free_pending:
	reply = dbus_message_new_method_return(pending->msg);
	send_dbus_msg_and_unref(connection, reply);

	authorization_free(pending);
	pending = NULL;
}

static void buddy_rejected(const char *id)
{
	DBusMessage *msg;

	/* Authorization request denied */
	if (!pending) {
		/* Ignore it: UI has been already notified */
		debug_fprintf("No authorization pending: " \
				"client already exitted\n");
		return;
	}

	/* FIXME: */
	if (strncmp(pending->id, id, strlen(pending->id)) != 0) {
		debug_fprintf("Buddy doesn't match(%s, %s)\n",
						pending->id, id);
		return;
	}

	msg = dbus_message_new_error(pending->msg,
			ERROR_IFACE ".Rejected",
			"Connection Rejected");

	authorization_free(pending);
	pending = NULL;

	send_dbus_msg_and_unref(connection, msg);
}

static void gps_status_cb(DBusPendingCall *call, void *data)
{
	DBusMessage *reply = dbus_pending_call_steal_reply(call);
	DBusError derr;
	const gchar *status;

	dbus_error_init(&derr);

	dbus_message_get_args(reply, &derr, DBUS_TYPE_STRING,
				&status, DBUS_TYPE_INVALID);
	if (dbus_error_is_set(&derr)) {
                debug_fprintf("verify_gps_status_cb: %s\n", derr.message);
                dbus_error_free(&derr);
                return;
        }
	dbus_message_unref(reply);

	gps_status = g_strdup(status);
}

static void verify_gps_status(DBusConnection *conn)
{
	DBusMessage *msg;
	DBusPendingCall *pending;

	msg = dbus_message_new_method_call("org.indt.carmand",
		"/org/indt/carmand", "org.indt.carmand.GPS", "Status");
	if (msg == NULL) {
		dbus_message_unref(msg);
		return;
	}

	dbus_connection_send_with_reply(connection, msg, &pending, -1);
	dbus_message_unref(msg);

	dbus_pending_call_set_notify(pending, gps_status_cb, NULL, NULL);

	dbus_pending_call_unref(pending);
}

gboolean dbus_init(DBusConnection *conn, const struct infosharing_command *command)
{
	DBusError derr;
	gboolean ret = FALSE;

	dbus_error_init(&derr);
	connection = dbus_connection_ref(conn);

	dbus_bus_register(connection, &derr);
	if (dbus_error_is_set(&derr)) {
		debug_fprintf("bus register: %s\n", derr.message);
		dbus_error_free(&derr);
		goto done;
	}

	if (dbus_bus_request_name(connection, CARMANPLUGIN,
				DBUS_NAME_FLAG_DO_NOT_QUEUE, &derr) !=
			DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER ) {
		debug_fprintf("request name: %s\n", derr.message);
		dbus_error_free(&derr);
		goto done;
	}

	ret = register_manager();
	if (ret == FALSE) {
		debug_fprintf("Can't register Manager interface\n");
		goto done;
	}

	if (!dbus_connection_add_filter(conn, connection_filter, NULL, NULL)) {
		debug_fprintf("Can't add D-Bus connection filter function\n");
		unregister_manager();
		return FALSE;
	}

	/* Initialize gps_status */
	verify_gps_status(connection);

	match(connection, TRUE, CARMAND_GPS_STATUS_MATCH_PATTERN);

	cmd = command;

	memset(&pos, 0, sizeof(struct position_data));
done:
	if (ret == FALSE)
		dbus_connection_unref(connection);
	return ret;
}

int dbus_init_callbacks(struct infosharing_event *evt)
{
	if (!evt)
		return -EINVAL;

	evt->authorize = buddy_authorize;
	evt->disconnected = buddy_disconnected;
	evt->data = buddy_data;
	evt->accepted = buddy_accepted;
	evt->rejected = buddy_rejected;

	return 0;
}

void dbus_exit(void)
{
	if (tmwatch) {
		g_source_remove(tmwatch);
		tmwatch = 0;
	}

	if (agent) {
		agent_release(agent);
		agent_free(agent);
		agent = NULL;
	}

	if (pending) {
		authorization_free(pending);
		pending = NULL;
	}

	match(connection, FALSE, CARMAND_GPS_STATUS_MATCH_PATTERN);

	unregister_manager();

	if (connection) {
		dbus_bus_release_name(connection,
			CARMANPLUGIN,
			NULL);

		dbus_connection_unref(connection);
	}
}
