/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Kyller Gorgonio <kyllercg@gmail.com>
 *          Andre Rodrigues <andre.rodrigues@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <gtk/gtk.h>
#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <stdlib.h>

#include "nm_ifaces_utils.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

GList *
nm_ifaces_utils_read_file (const gchar *filename)
{
    gchar *line = NULL;
    GList *data = NULL;
    GError *error = NULL;
    GIOChannel *iface = g_io_channel_new_file (filename, NM_IFACES_R_MODE,
                                                &error);

    if (error != NULL) {
        g_warning ("Error opening file %s: %s\n", filename, error->message);
        return NULL;
    }

    while (g_io_channel_read_line (iface, &line, NULL, NULL,
        &error) != G_IO_STATUS_EOF) {

        if (error != NULL) {
            g_warning ("Error reading file %s: %s\n", filename, error->message);
            return NULL;
        }

        data = g_list_append (data, line);
    }

    g_io_channel_close (iface);
    g_free (iface);

    return data;
}

gboolean
nm_ifaces_utils_save_file (GList *data, const gchar *filename)
{
    guint i;
    gchar *line;
    GError *error = NULL;
    GIOChannel *iface = g_io_channel_new_file (filename, NM_IFACES_RW_MODE,
                                                &error);

    if (error != NULL) {
        g_warning ("Error opening file %s: %s\n", filename, error->message);
        return FALSE;
    }

    for (i = 0; i < g_list_length(data); i++) {

        line = g_list_nth_data (data, i);
        g_io_channel_write_chars (iface, line, g_utf8_strlen(line, G_MAXINT),
                                   NULL, &error);
        if (error != NULL) {
            g_warning ("Error opening file %s: %s\n", filename, error->message);
            return FALSE;
        }
    }

    g_io_channel_close (iface);
    g_free (iface);

    return TRUE;
}

gchar *
nm_ifaces_utils_find_line (GList *data, const gchar *name, const gchar *field)
{
    gchar *line = "";
    gboolean found = FALSE;
    guint i;

    for (i = 0; i < g_list_length(data); i++) {

        line = g_list_nth_data (data, i);
        if (nm_ifaces_utils_match(name, line))
            found = TRUE;

        if (found == TRUE && nm_ifaces_utils_match (field, line))
            break;
    }

    return line;
}

gboolean
nm_ifaces_utils_match (const gchar *pattern, const gchar *data)
{
    gchar *aux = g_strdup (data);
    g_strstrip(aux);
    gboolean res = g_str_has_prefix (aux, pattern);
    g_free (aux);
    return res;
}

guint
nm_ifaces_utils_find_line_number (GList *data, const gchar *name,
				   const gchar *field)
{
    gchar *line;
    gboolean found = FALSE;
    guint i;

    for (i = 0; i < g_list_length(data); i++) {

        line = g_list_nth_data (data, i);
        if (nm_ifaces_utils_match (name, line))
            found = TRUE;

        if (found == TRUE && nm_ifaces_utils_match (field, line))
            break;
    }

    return i;
}

void
nm_ifaces_utils_change (GList *data, const gchar *name, const gchar *field,
                         const gchar *new_ip)
{
    gchar *new_line, *line;
    guint line_number;

    line = nm_ifaces_utils_find_line (data, name, field);
    line_number = nm_ifaces_utils_find_line_number (data, name, field);

    new_line = g_strconcat ("\t", field, " ", new_ip, "\n", NULL);

    data = g_list_insert (data, new_line, line_number);
    data = g_list_remove (data, g_list_nth_data (data, line_number + 1));
}

gchar *
nm_ifaces_utils_get_ip (GList *iface, const gchar *name, const gchar *field)
{
    gchar *line, *l_aux;
    gchar **aux;

    line = nm_ifaces_utils_find_line (iface, name, field);
    
    l_aux = g_strdup (line);
    g_strstrip(l_aux);
    
    aux = g_strsplit_set (l_aux, NM_IFACES_WHITE, G_MAXINT);

    g_free (l_aux);
    
    return g_strdup (aux[1]);
}

gboolean
nm_ifaces_utils_save_config (const gchar *name, const gchar *address,
                              const gchar *netmask, const gchar *gateway)
{
	GList *data = nm_ifaces_utils_read_file (NM_IFACES_FILE);

    if (data == NULL)
        return FALSE;

    nm_ifaces_utils_change (data, name, NM_IFACES_ADDRESS_ST, address);
    nm_ifaces_utils_change (data, name, NM_IFACES_NETMASK_ST, netmask);
    //nm_ifaces_utils_change (data, name, NM_IFACES_GATEWAY_ST, gateway);

    return nm_ifaces_utils_save_file (data, NM_IFACES_FILE);
}

gboolean
nm_ifaces_utils_save_config_e (const gchar *name, const gchar *address,
                                const gchar *netmask, const gchar *gateway)
{
    gchar *cmd = g_strconcat ("sudo /usr/sbin/nm-ifaces ", name, " ",
                               address, " ", netmask, " ", gateway, NULL);
    gint status = system(cmd);
    return status == 0;
}
