/*
 * Copyright: (C) 2008 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */


#include <stdlib.h>
#include <string.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtk.h>
#include <glib.h>

#include "data_tab.h"
#include "trip_tab.h"
#include "common_tab.h"
#include "mileagefile.h"

#define DATA_DISPLAY_MILEAGE_STR "Display Mileage"
#define DATA_DISPLAY_ACTIVITY_STR "Display Activity"
#define DATA_DISPLAY_ALERTS_STR "Display Alerts"
#define DATA_DISPLAY_TRIPS_STR "Display Trips"

static GtkWidget *car_selector;
static GtkWidget *display_selector;
static GtkWidget *table1;
static int        car_inserted = FALSE;
static GtkListStore *store = NULL;
static GtkTreeViewColumn *date_column = NULL;
static GtkTreeViewColumn *odo_column = NULL;
static GtkTreeViewColumn *fuel_column = NULL;
static GtkTreeViewColumn *mpg_column = NULL;
static GtkTreeViewColumn *cost_column = NULL;
static GtkTreeViewColumn *extra1_column = NULL;
static GtkTreeViewColumn *extra2_column = NULL;
static GtkWidget    *tree = NULL;
static GtkWidget *sw;


static void car_selector_clicked_cb(GtkWidget* w, gpointer data);
static void display_selector_clicked_cb(GtkWidget* w, gpointer data);
static void data_delete_clicked_cb(GtkWidget* w, gpointer data);
static void data_sort_column_cb(GtkWidget* w, gpointer data);
static int data_tab_sort_string(GtkTreeModel *, GtkTreeIter *, GtkTreeIter *, gpointer);
static void data_tree_selection_changed_cb(GtkTreeSelection *, gpointer);

void data_tab_layout(GtkWidget *window)
{
   GtkCellRenderer *renderer;
   GtkWidget    *button;
   GtkTreeSelection  *select;


   table1 = gtk_table_new(6, 8, FALSE);
   gtk_container_add(GTK_CONTAINER(window), table1);

   car_selector = gtk_combo_box_new_text();
   gtk_table_attach(GTK_TABLE(table1), car_selector, 0, 6, 0, 1, GTK_SHRINK, GTK_SHRINK, 2, 2);
   gtk_signal_connect(GTK_OBJECT(car_selector),
                     "changed",
                     GTK_SIGNAL_FUNC(car_selector_clicked_cb),
                     car_selector);

   button = gtk_button_new_with_label("Delete Entry");
   gtk_signal_connect(GTK_OBJECT(button),
                     "clicked",
                     GTK_SIGNAL_FUNC(data_delete_clicked_cb),
                     NULL);
   gtk_table_attach(GTK_TABLE(table1), button, 0, 3, 1, 2, GTK_SHRINK, GTK_SHRINK, 2, 2);

   display_selector = gtk_combo_box_new_text();
   gtk_table_attach(GTK_TABLE(table1), display_selector, 3, 6, 1, 2, GTK_SHRINK, GTK_SHRINK, 2, 2);
   gtk_signal_connect(GTK_OBJECT(display_selector),
                     "changed",
                     GTK_SIGNAL_FUNC(display_selector_clicked_cb),
                     display_selector);
   gtk_combo_box_append_text(GTK_COMBO_BOX(display_selector), DATA_DISPLAY_MILEAGE_STR);
   gtk_combo_box_append_text(GTK_COMBO_BOX(display_selector), DATA_DISPLAY_ALERTS_STR);
   gtk_combo_box_append_text(GTK_COMBO_BOX(display_selector), DATA_DISPLAY_ACTIVITY_STR);
   gtk_combo_box_append_text(GTK_COMBO_BOX(display_selector), DATA_DISPLAY_TRIPS_STR);

   store = gtk_list_store_new (7, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
                                  G_TYPE_STRING, G_TYPE_STRING);

   tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
   select = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));
   g_signal_connect (G_OBJECT (select), "changed",
                  G_CALLBACK (data_tree_selection_changed_cb),
                  NULL);

   renderer = gtk_cell_renderer_text_new ();
   date_column = gtk_tree_view_column_new_with_attributes (" Date ",
                                                   renderer,
                                                   "text", DATA_DATE_ROW,
                                                   NULL);
   gtk_signal_connect(GTK_OBJECT(date_column),
                     "clicked",
                     GTK_SIGNAL_FUNC(data_sort_column_cb),
                     (gpointer)DATA_DATE_ROW);
   gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(store), DATA_DATE_ROW, data_tab_sort_string, (gpointer)DATA_DATE_ROW, NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW (tree), date_column);
   odo_column = gtk_tree_view_column_new_with_attributes (" Odometer ",
                                                   renderer,
                                                   "text", DATA_ODO_ROW,
                                                   NULL);
   gtk_signal_connect(GTK_OBJECT(odo_column),
                     "clicked",
                     GTK_SIGNAL_FUNC(data_sort_column_cb),
                     (gpointer)DATA_ODO_ROW);
   gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(store), DATA_ODO_ROW, data_tab_sort_string, (gpointer)DATA_ODO_ROW, NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW (tree), odo_column);
   fuel_column = gtk_tree_view_column_new_with_attributes (" Gallons ",
                                                   renderer,
                                                   "text", DATA_FUEL_ROW,
                                                   NULL);
   gtk_signal_connect(GTK_OBJECT(fuel_column),
                     "clicked",
                     GTK_SIGNAL_FUNC(data_sort_column_cb),
                     (gpointer)DATA_FUEL_ROW);
   gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(store), DATA_FUEL_ROW, data_tab_sort_string, (gpointer)DATA_FUEL_ROW, NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW (tree), fuel_column);
   mpg_column = gtk_tree_view_column_new_with_attributes (" MPG ",
                                                   renderer,
                                                   "text", DATA_MPG_ROW,
                                                   NULL);
   gtk_signal_connect(GTK_OBJECT(mpg_column),
                     "clicked",
                     GTK_SIGNAL_FUNC(data_sort_column_cb),
                     (gpointer)DATA_MPG_ROW);
   gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(store), DATA_MPG_ROW, data_tab_sort_string, (gpointer)DATA_MPG_ROW, NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW (tree), mpg_column);
   cost_column = gtk_tree_view_column_new_with_attributes (" Cost ",
                                                   renderer,
                                                   "text", DATA_COST_ROW,
                                                   NULL);
   gtk_signal_connect(GTK_OBJECT(cost_column),
                     "clicked",
                     GTK_SIGNAL_FUNC(data_sort_column_cb),
                     (gpointer)DATA_COST_ROW);
   gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(store), DATA_COST_ROW, data_tab_sort_string, (gpointer)DATA_COST_ROW, NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW (tree), cost_column);
   extra1_column = gtk_tree_view_column_new_with_attributes ("",
                                                   renderer,
                                                   "text", DATA_EXTRA1_ROW,
                                                   NULL);
   gtk_signal_connect(GTK_OBJECT(extra1_column),
                     "clicked",
                     GTK_SIGNAL_FUNC(data_sort_column_cb),
                     (gpointer)DATA_EXTRA1_ROW);
   gtk_tree_view_append_column (GTK_TREE_VIEW (tree), extra1_column);
   extra2_column = gtk_tree_view_column_new_with_attributes ("",
                                                   renderer,
                                                   "text", DATA_EXTRA2_ROW,
                                                   NULL);
   gtk_signal_connect(GTK_OBJECT(extra2_column),
                     "clicked",
                     GTK_SIGNAL_FUNC(data_sort_column_cb),
                     (gpointer)DATA_EXTRA2_ROW);
   gtk_tree_view_append_column (GTK_TREE_VIEW (tree), extra2_column);

   gtk_tree_view_set_headers_visible(GTK_TREE_VIEW (tree), TRUE);
   gtk_tree_view_set_headers_clickable(GTK_TREE_VIEW (tree), TRUE);

   sw = gtk_scrolled_window_new (NULL, NULL);
   gtk_container_set_border_width(GTK_CONTAINER(sw), 5);
   gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
        GTK_SHADOW_ETCHED_IN); 
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
        GTK_POLICY_AUTOMATIC,
        GTK_POLICY_AUTOMATIC);
   gtk_table_attach(GTK_TABLE(table1), sw, 0, 6, 2, 8, GTK_SHRINK, GTK_SHRINK, 2, 2); 

   gtk_widget_set_size_request(sw, 690, 300);
   gtk_container_add(GTK_CONTAINER(sw), (GtkWidget *)tree);

   gtk_combo_box_set_active(GTK_COMBO_BOX(display_selector), 0);
}

void
data_sort_column_cb(GtkWidget *w, gpointer data)
{
   int col_id;
   GtkSortType  order;

#ifdef DEBUG
      fprintf(stdout, "Sort column %d selected.\n", (int)data);
#endif
   gtk_tree_sortable_get_sort_column_id(GTK_TREE_SORTABLE(store), &col_id, &order);

   /* IF previous sort was for this column */
   if (col_id == (int)data)
   {
      /* Then switch the column sort order */
      if (order == GTK_SORT_ASCENDING)
         gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(store), (int)data, GTK_SORT_DESCENDING);
      else
         gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(store), (int)data, GTK_SORT_ASCENDING);
   }
   /* Else sort this column in ascending order */
   else
      gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(store), (int)data, GTK_SORT_ASCENDING);
}

void
data_tab_load_car(char * car)
{
   if (car == NULL)
   {
      gtk_widget_destroy(car_selector);
      car_selector = gtk_combo_box_new_text();
      gtk_table_attach(GTK_TABLE(table1), car_selector, 0, 6, 0, 1, GTK_SHRINK, GTK_SHRINK, 2, 2);
      gtk_signal_connect(GTK_OBJECT(car_selector),
                     "changed",
                     GTK_SIGNAL_FUNC(car_selector_clicked_cb),
                     car_selector);
      gtk_widget_show_all(car_selector);
      car_inserted = FALSE;
   }
   else
   {
      gtk_combo_box_append_text((GtkComboBox *)car_selector, car);
      car_inserted = TRUE;
   }
}

GtkWidget *
data_tab_get_car_selector()
{
   return car_selector;
}

int
data_tab_get_display_selector()
{
   int     ret;
   gchar  *disp;


   disp = gtk_combo_box_get_active_text(GTK_COMBO_BOX(display_selector));
   if (disp == NULL)
      ret = DATA_DISPLAY_NOITEM;
   else if (strcmp(disp, DATA_DISPLAY_MILEAGE_STR) == 0)
      ret = DATA_DISPLAY_MILEAGE;
   else if (strcmp(disp, DATA_DISPLAY_ACTIVITY_STR) == 0)
      ret = DATA_DISPLAY_ACTIVITY;
   else if (strcmp(disp, DATA_DISPLAY_ALERTS_STR) == 0)
      ret = DATA_DISPLAY_ALERTS;
   else if (strcmp(disp, DATA_DISPLAY_TRIPS_STR) == 0)
      ret = DATA_DISPLAY_TRIPS;
   else
      ret = DATA_DISPLAY_NOITEM;

   g_free(disp);

   return ret;
}

int
data_tab_get_car_inserted()
{
   return car_inserted;
}

void
car_selector_clicked_cb(GtkWidget* w, gpointer data)
{
   char *car;

   /* Update all combo boxes to the same value */
   common_tab_update_car_combo_box(gtk_combo_box_get_active((GtkComboBox *)car_selector));

   if (data)
   {
      car = gtk_combo_box_get_active_text(data);
      if (car[strlen(car) - 1] == '\n')
      {
         car[strlen(car) - 1] = '\0';
      }

#ifdef DEBUG
      fprintf(stdout, "Car Selector choosen %s.\n", car);
#endif
#if 0
      /* Only called once in car_tab */
      data_tab_load_data(car);
#endif

      if (car)
         free(car);
   }
}

void
data_tab_load_data(char *car)
{
   common_tab_updateDisplayTab(car);
}

void
data_tab_update_car_combo_box(int value)
{
   gtk_combo_box_set_active((GtkComboBox *)car_selector, value);
}

void
data_tab_update_columns(
   char *date,
   char *odo,
   char *fuel,
   char *mpg,
   char *cost,
   char *extra1,
   char *extra2)
{
#ifdef DEBUG
      fprintf(stdout, "Updating data tab date title to:%s.\n", date);
#endif
   if (date_column)
      gtk_tree_view_column_set_title(date_column, date);
   if (odo_column)
      gtk_tree_view_column_set_title(odo_column, odo);
   if (fuel_column)
      gtk_tree_view_column_set_title(fuel_column, fuel);
   if (mpg_column)
      gtk_tree_view_column_set_title(mpg_column, mpg);
   if (cost_column)
   {
      if (strcmp("", cost) == 0)
         gtk_tree_view_column_set_visible(cost_column, FALSE);
      else
      {
         gtk_tree_view_column_set_title(cost_column, cost);
         gtk_tree_view_column_set_visible(cost_column, TRUE);
      }
   }

   if (extra1_column)
   {
      if (strcmp("", extra1) == 0)
         gtk_tree_view_column_set_visible(extra1_column, FALSE);
      else
      {
         gtk_tree_view_column_set_title(extra1_column, extra1);
         gtk_tree_view_column_set_visible(extra1_column, TRUE);
      }
   }

   if (extra2_column)
   {
      if (strcmp("", extra2) == 0)
         gtk_tree_view_column_set_visible(extra2_column, FALSE);
      else
      {
         gtk_tree_view_column_set_title(extra2_column, extra2);
         gtk_tree_view_column_set_visible(extra2_column, TRUE);
      }
   }

   /* When changing the table column names readjust the column widths */
   if (tree)
      gtk_tree_view_columns_autosize(GTK_TREE_VIEW(tree));
}

GtkListStore *
data_tab_get_store()
{
   return store;
}

void
data_delete_clicked_cb(GtkWidget* w, gpointer data)
{
   GtkTreeIter  iter;
   GtkTreeSelection  *sel = NULL;
   gboolean     selected;
   GtkWidget    *dialog;
   gchar        *date;
   gchar        *odo;
   gchar        *activity;
   int           disp;
   int        result;
   char  *car = NULL;

#ifdef DEBUG
   fprintf(stdout, "Delete Entry called \n");
#endif

   sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
   selected = gtk_tree_selection_get_selected(sel, NULL, &iter); 

   if (selected == TRUE)
   {
      gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_DATE_ROW, &date, -1);
      gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_ODO_ROW, &odo, -1);
      gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_FUEL_ROW, &activity, -1);
#ifdef DEBUG
      fprintf(stdout, "Retrieved %s to delete\n", date);
#endif

      dialog = gtk_message_dialog_new (NULL,
                GTK_DIALOG_DESTROY_WITH_PARENT,
                GTK_MESSAGE_ERROR,
                GTK_BUTTONS_YES_NO,
                "Warning, Are you sure you want\nto delete this entry?\nYes, Delete  No, Do Not Delete");
      result = gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);

      switch (result)
      {
         case GTK_RESPONSE_YES:
            car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));
            if (car != NULL)
            {
               if (car[strlen(car) - 1] == '\n')
               {
                  car[strlen(car) - 1] = '\0';
               }
               disp = data_tab_get_display_selector();
               if (disp == DATA_DISPLAY_NOITEM)
               {
                  if (car)
                     g_free(car);
                  return;
               }

			   if (disp == DATA_DISPLAY_MILEAGE)
                  deleteEntry(car, odo);
               else if (disp == DATA_DISPLAY_ACTIVITY)
                  deleteActivityAlertEntry(car, date, odo, activity, 0);
               else if (disp == DATA_DISPLAY_ALERTS)
                  deleteActivityAlertEntry(car, date, odo, activity, 1);
               else if (disp == DATA_DISPLAY_TRIPS)
                  deleteTripEntry(car, odo);

               /* Update all entries after delete */
               common_tab_load_data(car);
               common_tab_updateDisplayTab(car);
               g_free(car);
            }
         break;

         default:
         break;
      }

      g_free(date);
      g_free(odo);
      g_free(activity);
   }
   else
   {
      dialog = gtk_message_dialog_new (NULL,
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_ERROR,
                                  GTK_BUTTONS_CLOSE,
                                  "Warning, Please select a row to delete.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
   }
}

void
display_selector_clicked_cb(GtkWidget* w, gpointer data)
{
   char  *car = NULL;

   car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));
   if (car != NULL)
   {
      if (car[strlen(car) - 1] == '\n')
      {
         car[strlen(car) - 1] = '\0';
      }
   }
   common_tab_updateDisplayTab(car);
   g_free(car);
}

int
data_tab_sort_string(
   GtkTreeModel  *model,
   GtkTreeIter   *iter1,
   GtkTreeIter   *iter2,
   gpointer       user_data)
{
   /* This can receive data for five columns: DATA_DATE_ROW DATA_ODO_ROW DATA_FUEL_ROW, DATA_MPG_ROW, DATA_COST_ROW */
   /* The are the following types for different displays */
   /* Display   DATA_DATE_ROW   DATA_ODO_ROW  DATA_FUEL_ROW    DATA_MPG_ROW     DATA_COST_ROW */
   /* Mileage   DATE            DOUBLE        DOUBLE           DOUBLE           DOUBLE   */
   /* Alerts    INT             DOUBLE        STRING           STRING           NONE     */
   /* Activity  DATE            DOUBLE        STRING           STRING           DOUBLE   */
   /* Trips     DATE            DOUBLE        DATE             INT              STRING   */

   int     disp;
   int     mode; /* 0 = DOUBLE, 1 = STRING, 2 = INT, 3 = DATE */
   int     ret;
   gchar  *item1_str;
   gchar  *item2_str;
   GDate  *gdate1;
   GDate  *gdate2;


   disp = data_tab_get_display_selector();
   if (disp == DATA_DISPLAY_NOITEM)
      return -1;

   if (disp == DATA_DISPLAY_MILEAGE)
   {
      if ((int)user_data == DATA_DATE_ROW)
         mode = 3;
      else
         mode = 0;
   }
   else if (disp == DATA_DISPLAY_ACTIVITY)
   {
      if ((int)user_data == DATA_DATE_ROW)
         mode = 3;
      else if ((int)user_data == DATA_ODO_ROW)
         mode = 0;
      else if ((int)user_data == DATA_COST_ROW)
         mode = 0;
      else
         mode = 1;
   }
   else if (disp == DATA_DISPLAY_ALERTS)
   {
      if ((int)user_data == DATA_DATE_ROW)
         mode = 2;
      else if ((int)user_data == DATA_ODO_ROW)
         mode = 0;
      else
         mode = 1;
   }
   else if (disp == DATA_DISPLAY_TRIPS)
   {
      if ((int)user_data == DATA_DATE_ROW)
         mode = 3;
      else if ((int)user_data == DATA_FUEL_ROW)
         mode = 3;
      else if ((int)user_data == DATA_MPG_ROW)
         mode = 2;
      else if ((int)user_data == DATA_ODO_ROW)
         mode = 0;
      else
         mode = 1;
   }

   gtk_tree_model_get (model, iter1, (int)user_data, &item1_str, -1);
   gtk_tree_model_get (model, iter2, (int)user_data, &item2_str, -1);

   if (mode == 0)
   {
      if (atof(item1_str) > atof(item2_str))
         ret = -1;
      else if (atof(item2_str) > atof(item1_str))
         ret = 1;
      else
         ret = 0;

   }
   else if (mode == 1)
   {
      ret = strcmp(item1_str, item2_str);
   }
   else if (mode == 2)
   {
      if (atoi(item1_str) > atoi(item2_str))
         ret = -1;
      else if (atoi(item2_str) > atoi(item1_str))
         ret = 1;
      else
         ret = 0;
   }
   else if (mode == 3)
   {
      gdate1 = g_date_new();
      gdate2 = g_date_new();
      g_date_set_parse(gdate1, item1_str);
      g_date_set_parse(gdate2, item2_str);
   
      ret = g_date_compare(gdate1, gdate2);

      g_date_free(gdate1);
      g_date_free(gdate2);
   }

   g_free(item1_str);
   g_free(item2_str);
   return (ret);
}

void
data_tree_selection_changed_cb(
   GtkTreeSelection *selection,
   gpointer data)
{
   int           disp;
   GtkTreeSelection  *sel = NULL;
   gboolean     selected;
   GtkTreeIter  iter;
   char         *start_date_str;
   char         *end_date_str;
   char         *end_odo_str;
   char         *purpose_str;
   char         *person_str;
   char         *dest_str;
   char         *start_odo_str;
#ifdef DEBUG
   fprintf(stdout, "Row selected\n");
#endif

   disp = data_tab_get_display_selector();
   if (disp == DATA_DISPLAY_TRIPS)
   {
      /* Clear all the fields */
      trip_set_fields("", "", "", "", "", "", "");

      sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
      selected = gtk_tree_selection_get_selected(sel, NULL, &iter);

      if (selected == TRUE)
      {
         gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_DATE_ROW, &start_date_str, -1);
         gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_ODO_ROW, &start_odo_str, -1);
         gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_FUEL_ROW, &end_date_str, -1);
         gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_MPG_ROW, &end_odo_str, -1);
         gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_COST_ROW, &person_str, -1);
         gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_EXTRA1_ROW, &dest_str, -1);
         gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DATA_EXTRA2_ROW, &purpose_str, -1);

         trip_set_fields(start_date_str, end_date_str, start_odo_str, end_odo_str, person_str, dest_str, purpose_str);
         if (start_date_str)
            g_free(start_date_str);
         if (start_odo_str)
            g_free(start_odo_str);
         if (end_date_str)
            g_free(end_date_str);
         if (end_odo_str)
            g_free(end_odo_str);
         if (person_str)
            g_free(person_str);
         if (dest_str)
            g_free(dest_str);
         if (purpose_str)
            g_free(purpose_str);
      }
   }
}

void
data_tab_fullscreen(int fullscreen)
{
   if (sw)
   {
      if (fullscreen == 0)
         gtk_widget_set_size_request(sw, 690, 300);
      else
         gtk_widget_set_size_request(sw, 810, 370);
   }
}

