/*
 * Copyright: (C) 2008 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <gtk/gtk.h>

#include "common_tab.h"
#include "graph_tab.h"
#include "mileagecfg.h"
#include "mileagefile.h"

static GtkWidget *car_selector = NULL;
static GtkWidget *table1 = NULL;
static int        car_inserted = FALSE;
static GSList    *data_list = NULL;
static GSList    *sort_list = NULL;
static int        first_year = 0;
static int        last_year = 0;
static GtkWidget *graph_selector1 = NULL;
static GtkWidget *graph_selector2 = NULL;
static int        max_display = 100; 
static int        max_distance = 0;
static int        max_mileage = 0;
static int        max_cost = 0;
static GtkWidget *w_area = NULL;
static GdkColor   color[8];
static gboolean   create_colors = FALSE;
static GdkColor   color_black;
static char   *dist_str;
static char   *mile_str;
static char   *cost_unit_str;

static void car_selector_clicked_cb(GtkWidget* w, gpointer data);
static void graph_selector1_clicked_cb(GtkWidget* w, gpointer data);
static void graph_selector2_clicked_cb(GtkWidget* w, gpointer data);
static gboolean graph_expose_event(GtkWidget*, GdkEventExpose*, gpointer);
static void graph_draw_legend(int year, GtkWidget *w);
static void graph_draw_grid(GtkWidget *w);
static void graph_draw_text(GtkWidget *w);
static void graph_draw_data(GtkWidget *w);
static void graph_it(int year, GtkWidget *w);
static void graph_mileage_year(int year, GtkWidget *w);
static void graph_distance_year(int year, GtkWidget *w);
static void graph_cost_year(int year, GtkWidget *w);
static int  graph_get_max_distance();
static int  graph_get_max_mileage();
static int  graph_get_max_cost();

#define X_OFFSET 70
#define X_DELTA  40
#define X_NUM    13
#define X_MAX_NUM X_NUM * X_DELTA + X_OFFSET

#define Y_OFFSET 20
#define Y_DELTA  23
#define Y_NUM    11
#define Y_MAX_NUM Y_NUM * Y_DELTA + Y_OFFSET

#define MILEAGE_STR  "Mileage"
#define DISTANCE_STR "Distance"
#define COST_STR     "Cost"
#define ALL_YEARS    "All Years"

#define GRAPH_HEIGHT   300
#define GRAPH_WIDTH    600


void graph_tab_layout(GtkWidget *window)
{
   table1 = gtk_table_new(9,9,FALSE);
   gtk_container_add(GTK_CONTAINER(window), (GtkWidget *)table1);

   car_selector = gtk_combo_box_new_text();
   gtk_table_attach(GTK_TABLE(table1), car_selector, 0, 4, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
   gtk_signal_connect(GTK_OBJECT(car_selector),
                     "changed",
                     GTK_SIGNAL_FUNC(car_selector_clicked_cb),
                     car_selector);

   graph_selector1 = gtk_combo_box_new_text();
   gtk_table_attach(GTK_TABLE(table1), graph_selector1, 0, 4, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
   gtk_signal_connect(GTK_OBJECT(graph_selector1),
                     "changed",
                     GTK_SIGNAL_FUNC(graph_selector1_clicked_cb),
                     graph_selector1);
   gtk_combo_box_append_text(GTK_COMBO_BOX(graph_selector1), MILEAGE_STR);
   gtk_combo_box_append_text(GTK_COMBO_BOX(graph_selector1), DISTANCE_STR);
   gtk_combo_box_append_text(GTK_COMBO_BOX(graph_selector1), COST_STR);
   gtk_combo_box_set_active(GTK_COMBO_BOX(graph_selector1), 0);

   graph_selector2 = gtk_combo_box_new_text();
   gtk_table_attach(GTK_TABLE(table1), graph_selector2, 5, 9, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
   gtk_signal_connect(GTK_OBJECT(graph_selector2),
                     "changed",
                     GTK_SIGNAL_FUNC(graph_selector2_clicked_cb),
                     graph_selector2);
   gtk_combo_box_append_text(GTK_COMBO_BOX(graph_selector2), "No Graph");
   gtk_combo_box_append_text(GTK_COMBO_BOX(graph_selector2), ALL_YEARS);
   gtk_combo_box_set_active(GTK_COMBO_BOX(graph_selector2), 0);

   w_area = gtk_drawing_area_new();
   gtk_table_attach(GTK_TABLE(table1), w_area, 0, 9, 2, 9, GTK_FILL, GTK_FILL, 2, 2);
   /* TBD - need to fill rest of space and get dimensions */
   gtk_widget_set_size_request(w_area, GRAPH_WIDTH, GRAPH_HEIGHT);
   g_signal_connect(G_OBJECT(w_area), "expose_event", G_CALLBACK(graph_expose_event), NULL);
}

void
graph_tab_load_car(char * car)
{  
   if (car == NULL)
   {
      gtk_widget_destroy(car_selector);
      car_selector = gtk_combo_box_new_text();
      gtk_table_attach(GTK_TABLE(table1), car_selector, 0, 4, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
      gtk_signal_connect(GTK_OBJECT(car_selector),
                     "changed",
                     GTK_SIGNAL_FUNC(car_selector_clicked_cb),
                     car_selector);
      gtk_widget_show_all(car_selector);
      car_inserted = FALSE;
   }
   else
   {
      gtk_combo_box_append_text(GTK_COMBO_BOX(car_selector), car);
      car_inserted = TRUE;
   }
}

GtkWidget *
graph_tab_get_car_selector()
{
   return car_selector;
}  

int
graph_tab_get_car_inserted()
{
   return car_inserted;
}

void
graph_tab_update_car_combo_box(int value)
{
   gtk_combo_box_set_active(GTK_COMBO_BOX(car_selector), value);
}

void
graph_tab_select_car(char *car)
{
   GtkTreeModel  *model;
   GtkTreeIter    iter;
   gboolean valid;
   gint row_count = 0;

   /* Select first car in list if NULL */
   if (car == NULL)
   {
      if (car_inserted == TRUE)
      {
         gtk_combo_box_set_active(GTK_COMBO_BOX(car_selector), 0);
      }
   }
   else
   {
      model = gtk_combo_box_get_model(GTK_COMBO_BOX(car_selector));

      /* Get the first iter in the list */
      valid = gtk_tree_model_get_iter_first (model, &iter);

      while (valid)
      {
         /* Walk through the list, reading each row */
         gchar *str_data;

         /* Make sure you terminate calls to gtk_tree_model_get()
          * with a '-1' value */
         gtk_tree_model_get (model, &iter, 0, &str_data, -1);

         /* Look for the specified car and make it active */
         if (strcmp(car, str_data) == 0)
         {
            gtk_combo_box_set_active(GTK_COMBO_BOX(car_selector), row_count);
         }
         g_free (str_data);

         row_count ++;
         valid = gtk_tree_model_iter_next (model, &iter);
      }
   }
}

void
car_selector_clicked_cb(GtkWidget* w, gpointer data)
{
   char *car;

   /* Update all combo boxes to the same value */
   common_tab_update_car_combo_box(gtk_combo_box_get_active((GtkComboBox *)car_selector));

   if (data)
   {
      car = gtk_combo_box_get_active_text(data);
      if (car[strlen(car) - 1] == '\n')
      {
         car[strlen(car) - 1] = '\0';
      }

#ifdef DEBUG
      fprintf(stdout, "Car Selector choosen %s.\n", car);
#endif
#if 0
      /* Only called once in car_tab.c */
      common_tab_load_data(car);
#endif

      if (car)
         free(car);
   }
   if (w_area != NULL)
      gtk_widget_queue_draw_area(w_area, 0, 0, GRAPH_WIDTH, GRAPH_HEIGHT);
}

void
graph_selector1_clicked_cb(GtkWidget* w, gpointer data)
{
#ifdef DEBUG
      fprintf(stdout, "Graph Selector 1 selected\n");
#endif

   if (w_area != NULL)
      gtk_widget_queue_draw_area(w_area, 0, 0, GRAPH_WIDTH, GRAPH_HEIGHT);
}

void
graph_selector2_clicked_cb(GtkWidget* w, gpointer data)
{
#ifdef DEBUG
      fprintf(stdout, "Graph Selector 2 selected\n");
#endif

   if (w_area != NULL)
      gtk_widget_queue_draw_area(w_area, 0, 0, GRAPH_WIDTH, GRAPH_HEIGHT);
}

gboolean
graph_expose_event(
   GtkWidget *w,
   GdkEventExpose *event,
   gpointer data)
{
   char  *graphtype;
   int    units = 0;
#ifdef DEBUG
      fprintf(stdout, "Graph expose event\n");
#endif

   if (create_colors == FALSE)
   {
      color_black.red = 0;
      color_black.blue = 0;
      color_black.green = 0;
      color[0].red = 65535;
      color[0].blue = 0;
      color[0].green = 0;
      color[1].red = 0;
      color[1].blue = 65535;
      color[1].green = 0;
      color[2].red = 0;
      color[2].blue = 0;
      color[2].green = 65535;
      color[3].red = 65535;
      color[3].blue = 65535;
      color[3].green = 0;
      color[4].red = 0;
      color[4].blue = 65535;
      color[4].green = 65535;
      color[5].red = 65535;
      color[5].blue = 0;
      color[5].green = 65535;
      color[6].red = 32000;
      color[6].blue = 32000;
      color[6].green = 32000;
      color[7].red = 32000;
      color[7].blue = 65535;
      color[7].green = 32000;
      create_colors = TRUE;
   }

   cost_unit_str = "Cost/Unit";
   units = returnUnits();
   switch (units)
   {
      /* MPG */
      case 0:
         dist_str = "M/Day";
         mile_str = "MPG";
      break;

      /* L/100km */
      case 1:
         dist_str = "Km/Day";
         mile_str = "L/100km";
      break;

      /* L/10km */
      case 2:
         dist_str = "Km/Day";
         mile_str = "L/10km";
      break;

      /* Miles / L */
      case 3:
         dist_str = "M/Day";
         mile_str = "M/L";
      break;

      /* km / L */
      case 4:
         dist_str = "Km/Day";
         mile_str = "Km/L";
      break;
   }

   /* Based upon graph selection find max scale value */
   graphtype = gtk_combo_box_get_active_text(GTK_COMBO_BOX(graph_selector1));
   if (strncmp(graphtype, MILEAGE_STR, strlen(MILEAGE_STR)) == 0)
      max_display = max_mileage / 50 * 50 + 50;
   else if (strncmp(graphtype, DISTANCE_STR, strlen(DISTANCE_STR)) == 0)
      max_display = max_distance / 50 * 50 + 50;
   else if (strncmp(graphtype, COST_STR, strlen(COST_STR)) == 0)
      max_display = max_cost;

   if (graphtype)
      free(graphtype);

   graph_draw_legend(-1, w);

   gdk_gc_set_rgb_fg_color(w->style->fg_gc[GTK_WIDGET_STATE(w)], &color_black);

   graph_draw_grid(w);

   graph_draw_text(w);

   graph_draw_data(w);

   gdk_gc_set_rgb_fg_color(w->style->fg_gc[GTK_WIDGET_STATE(w)], &color_black);

   return FALSE;
}

void
graph_draw_legend(
   int year,
   GtkWidget  *area)
{
   static int offset = 0;
   PangoFontDescription  *fd;
   PangoContext          *context;
   PangoLayout           *layout;
   char                  number[5];

   if (year == -1)
      offset = 0;
   else
   {
      fd = area->style->font_desc;
      context = gtk_widget_get_pango_context(area);
      layout  = pango_layout_new(context);
      g_object_unref(context); 

      pango_layout_set_font_description(layout, fd);

      sprintf(number, "%4d", year);
      pango_layout_set_text(layout, number, -1);
      gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   offset, 0, layout);
      offset += 55;
   }
}

void
graph_draw_grid(GtkWidget *area)
{
   int x, y;

   gdk_gc_set_line_attributes(area->style->fg_gc[GTK_WIDGET_STATE(area)],
      1, GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_MITER);

   for (y = 0; y < Y_NUM; y++)
   {
      gdk_draw_line(area->window, area->style->fg_gc[GTK_WIDGET_STATE(area)],
         X_OFFSET, Y_OFFSET + Y_DELTA * y,
         X_OFFSET + (X_NUM - 1) * X_DELTA, Y_OFFSET + Y_DELTA * y);
#if 0
      fprintf(stdout, "Draw Grid line %d %d %d %d\n", X_OFFSET, Y_OFFSET + Y_DELTA * y, X_OFFSET + (X_NUM - 1) * X_DELTA, Y_OFFSET + Y_DELTA * y);
#endif
   }

   for (x = 0; x < X_NUM; x++)
   {
      gdk_draw_line(area->window, area->style->fg_gc[GTK_WIDGET_STATE(area)],
         X_OFFSET + X_DELTA * x, Y_OFFSET,
         X_OFFSET + X_DELTA * x, Y_OFFSET + (Y_NUM - 1) * Y_DELTA);
   }
}

void
graph_draw_text(GtkWidget *area)
{
   PangoFontDescription  *fd;
   PangoContext          *context;
   PangoLayout           *layout;
   char                  one[100];
   char                  two[100];
   char                  three[100];
   char                  four[100];
   char                  five[100];
   char                 *graphtype;

   fd = area->style->font_desc;
   context = gtk_widget_get_pango_context(area);
   layout  = pango_layout_new(context);
   g_object_unref(context);

   pango_layout_set_font_description(layout, fd);

   pango_layout_set_text(layout, "Jan", -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   X_OFFSET + 0 * X_DELTA - (X_DELTA / 2),
                   (Y_NUM * Y_DELTA), layout);

   pango_layout_set_text(layout, "Apr", -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   X_OFFSET + 3 * X_DELTA - (X_DELTA / 2),
                   (Y_NUM * Y_DELTA), layout);

   pango_layout_set_text(layout, "Jul", -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   X_OFFSET + 6 * X_DELTA - (X_DELTA / 2),
                   (Y_NUM * Y_DELTA), layout);

   pango_layout_set_text(layout, "Oct", -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   X_OFFSET + 9 * X_DELTA - (X_DELTA / 2),
                   (Y_NUM * Y_DELTA), layout);

   pango_layout_set_text(layout, "Jan", -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   X_OFFSET + 12 * X_DELTA - (X_DELTA / 2),
                   (Y_NUM * Y_DELTA), layout);

   pango_layout_set_text(layout, "0", -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   1, ((Y_NUM-1) * Y_DELTA) + Y_DELTA/2, layout);

   sprintf(one, "%-7.2f", max_display / 10.0 * 2.0);
   pango_layout_set_text(layout, one, -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   1, ((Y_NUM-3) * Y_DELTA) + Y_DELTA/2, layout);

   sprintf(two, "%-7.2f", max_display / 10.0 * 4.0);
   pango_layout_set_text(layout, two, -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   1, ((Y_NUM-5) * Y_DELTA) + Y_DELTA/2, layout);

   sprintf(three, "%-7.2f", max_display / 10.0 * 6.0);
   pango_layout_set_text(layout, three, -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   1, ((Y_NUM-7) * Y_DELTA) + Y_DELTA/2, layout);

   sprintf(four, "%-7.2f", max_display / 10.0 * 8.0);
   pango_layout_set_text(layout, four, -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   1, ((Y_NUM-9) * Y_DELTA) + Y_DELTA/2, layout);

   sprintf(five, "%-7.2f", (double)max_display);
   pango_layout_set_text(layout, five, -1);
   gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                   1, ((Y_NUM-11) * Y_DELTA) + Y_DELTA/2, layout);

   graphtype = gtk_combo_box_get_active_text(GTK_COMBO_BOX(graph_selector1));
   if (strncmp(graphtype, MILEAGE_STR, strlen(MILEAGE_STR)) == 0)
   {
      pango_layout_set_text(layout, mile_str, -1);
      gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                      0, (Y_NUM * Y_DELTA) / 2, layout);
   }
   else if (strncmp(graphtype, DISTANCE_STR, strlen(DISTANCE_STR)) == 0)
   {
      pango_layout_set_text(layout, dist_str, -1);
      gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                      0, (Y_NUM * Y_DELTA) / 2, layout);
   }
   else if (strncmp(graphtype, COST_STR, strlen(COST_STR)) == 0)
   {
      pango_layout_set_text(layout, cost_unit_str, -1);
      gdk_draw_layout(area->window, area->style->fg_gc[area->state],
                      0, (Y_NUM * Y_DELTA) / 2, layout);
   }

   if (graphtype)
      free(graphtype);
}

void
graph_draw_data(GtkWidget *area)
{
   char  *year;
   int    i, x;
   char   buf[5];

   year = gtk_combo_box_get_active_text(GTK_COMBO_BOX(graph_selector2));

   if (strncmp(year, ALL_YEARS, strlen(ALL_YEARS)) == 0)
   {
      for (i = first_year, x = 0; i <= last_year && i != 0; i++, x++)
      {
         gdk_gc_set_rgb_fg_color(area->style->fg_gc[GTK_WIDGET_STATE(area)], &color[x%8]);
         graph_draw_legend(i, area);
         graph_it(i, area);
      }
   }
   else if (strstr(year, "Year") != NULL)
   {
      strncpy(buf, year, 4);
      buf[4] = 0;
      gdk_gc_set_rgb_fg_color(area->style->fg_gc[GTK_WIDGET_STATE(area)], &color[0]);
      graph_draw_legend(atoi(buf), area);
      graph_it(atoi(buf), area);
   }

   if (year)
      free(year);
}

void
graph_tab_load_data(char *car)
{
   char  *str;
   char  date_str[1024];
   GDate new_date;
   GSList *last_item;
   int     i;


   /* Delete the previous list */
   if (sort_list)
   {
      g_slist_foreach(sort_list, common_delete_data, NULL);
      g_slist_free(sort_list);
      sort_list = NULL;
      data_list = NULL;
   }

   first_year = 0;
   last_year = 0;

   data_list = returnMileEntries(car);

   if (data_list != NULL)
   {
      /* sort the list in date order */
      sort_list = g_slist_sort(data_list, common_sort_date);

      str = sort_list->data;
      if (str != NULL)
      {
         sscanf(str, "%s", date_str);
         g_date_set_parse(&new_date, date_str);
         if (g_date_valid(&new_date) == TRUE)
            first_year = g_date_get_year(&new_date);
      }

      last_item = g_slist_last(sort_list);
      str = last_item->data;
      if (str != NULL)
      {
         sscanf(str, "%s", date_str);
         g_date_set_parse(&new_date, date_str);
         if (g_date_valid(&new_date) == TRUE)
            last_year = g_date_get_year(&new_date);
      }
   }
#ifdef DEBUG
   fprintf(stdout, "First year is %d, last year is %d\n", first_year, last_year);
#endif

   /* Destroy the previous car year selector and build a new one */
   gtk_widget_destroy(graph_selector2);
   graph_selector2 = gtk_combo_box_new_text();
   gtk_table_attach(GTK_TABLE(table1), graph_selector2, 5, 9, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
   gtk_signal_connect(GTK_OBJECT(graph_selector2),
                  "changed",
                  GTK_SIGNAL_FUNC(graph_selector2_clicked_cb),
                  graph_selector2);
   gtk_widget_show_all(graph_selector2);

   gtk_combo_box_append_text(GTK_COMBO_BOX(graph_selector2), "No Graph");
   gtk_combo_box_append_text(GTK_COMBO_BOX(graph_selector2), ALL_YEARS);
   gtk_combo_box_set_active(GTK_COMBO_BOX(graph_selector2), 0);

   /* Fill the menu item with the years */
   if (first_year != 0)
   {
      for (i = first_year; i <= last_year; i++)
      {
         sprintf(date_str, "%d Year", i);
         gtk_combo_box_append_text(GTK_COMBO_BOX(graph_selector2), date_str);
      }
   }

   /* get max values for graph scale */
   max_distance = graph_get_max_distance();
   max_mileage  = graph_get_max_mileage();
   max_cost     = graph_get_max_cost();

   if (w_area != NULL)
      gtk_widget_queue_draw_area(w_area, 0, 0, GRAPH_WIDTH, GRAPH_HEIGHT);
}

void
graph_it(
   int year,
   GtkWidget *area)
{
   char  *graphtype;

   /* Make width of line bigger then grid */
   gdk_gc_set_line_attributes(area->style->fg_gc[GTK_WIDGET_STATE(area)],
      2, GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_MITER);
   
   graphtype = gtk_combo_box_get_active_text(GTK_COMBO_BOX(graph_selector1));
   if (strncmp(graphtype, MILEAGE_STR, strlen(MILEAGE_STR)) == 0)
      graph_mileage_year(year, area);
   else if (strncmp(graphtype, DISTANCE_STR, strlen(DISTANCE_STR)) == 0)
      graph_distance_year(year, area);
   else if (strncmp(graphtype, COST_STR, strlen(COST_STR)) == 0)
      graph_cost_year(year, area);

   if (graphtype)
      free(graphtype);
}

void
graph_mileage_year(
   int year,
   GtkWidget *area)
{
   GSList  *iter;
   GString *data_item;
   char    *date;
   int      odometer;
   double   fuel;
   double   cost;
   char    *p;
   GDate    gdate;
   int      prev_odo = -1;
   double   prev_fuel = 0.0;
   double   mpg;
   int      units;
   int      x_val, y_val;
   int      p_x = -1;
   int      p_y = -1;
   int      y_max = Y_OFFSET + (Y_NUM - 1) * Y_DELTA;
   int      days_in_month;
   char    *saveptr;
   char    *cost_str;


   data_item = g_string_new(NULL);
   units = returnUnits();

   iter = sort_list;
   while (iter != NULL)
   {
      g_string_erase(data_item, 0, -1);
      g_string_append(data_item, iter->data);

      date = (char *)strtok_r(data_item->str, " ", &saveptr);
      odometer = atoi((char *)strtok_r(NULL, " ", &saveptr));
      fuel = atof((char *)strtok_r(NULL, " ", &saveptr));
      cost_str = (char *)strtok_r(NULL, " ", &saveptr);
      p = (char *)strtok_r(NULL, " ", &saveptr);
      /* IF no cost data */
      if (p == NULL)
      {
         p = cost_str;
         cost = 0.0;
      }

      g_date_set_parse(&gdate, date);

      days_in_month = g_date_get_days_in_month(g_date_get_month(&gdate), g_date_get_year(&gdate));

      if (g_date_get_year(&gdate) == year)
      {
         /* IF no previous entry then just save off previous values */
         if (strcmp(p, "1") == 0)
         {
         }
         else if (prev_odo == -1)
         {
         }
         else
         {
            if (units == 0)
               mpg = (odometer - prev_odo) / fuel;
            else if (units == 1)
               mpg = 100.0 * fuel / (odometer - prev_odo);
            else if (units == 2)
               mpg = 10.0 * fuel / (odometer - prev_odo);
            else if (units == 3)
               mpg = (odometer - prev_odo) / fuel;
            else if (units == 4)
               mpg = (odometer - prev_odo) / fuel;
            else
               mpg = (odometer - prev_odo) / fuel;

            y_val = y_max - (int)(mpg * ((y_max - Y_OFFSET) / (double)max_display));
            x_val = ((int)g_date_get_month(&gdate) - 1) * X_DELTA + X_OFFSET;
            x_val = x_val + (X_DELTA * g_date_get_day(&gdate) / days_in_month);

            if (p_x != -1)
            {
               if (x_val > X_MAX_NUM)
                  x_val = X_MAX_NUM;
               if (x_val < 0)
                  x_val = 0;
               if (y_val > Y_MAX_NUM)
                  y_val = Y_MAX_NUM;
               if (y_val < 0)
                  y_val = 0;

               gdk_draw_line(area->window, area->style->fg_gc[GTK_WIDGET_STATE(area)],
                  p_x, p_y, x_val, y_val);
#if 0
      fprintf(stdout, "Draw Mileage line %d %d %d %d\n", p_x, p_y, x_val, y_val);
#endif
            }
            p_x = x_val;
            p_y = y_val;
         }

      }
      prev_odo = odometer;
      prev_fuel = fuel;

      iter = g_slist_next(iter);
   }

   g_string_free(data_item, TRUE);
}

void
graph_distance_year(
   int year,
   GtkWidget *area)
{
   GSList  *iter;
   GString *data_item;
   char    *date;
   int      odometer;
   double   fuel;
   double   cost;
   char    *p;
   GDate    gdate;
   GDate    prev_date;
   int      prev_odo = -1;
   int      x_val, y_val;
   int      p_x = -1;
   int      p_y = -1;
   int      miles;
   int      days;
   int      y_max = Y_OFFSET + (Y_NUM - 1) * Y_DELTA;
   int      days_in_month;
   char    *saveptr;
   char    *cost_str;


   data_item = g_string_new(NULL);

   iter = sort_list;
   while (iter != NULL)
   {
      g_string_erase(data_item, 0, -1);
      g_string_append(data_item, iter->data);

      date = (char *)strtok_r(data_item->str, " ", &saveptr);
      odometer = atoi((char *)strtok_r(NULL, " ", &saveptr));
      fuel = atof((char *)strtok_r(NULL, " ", &saveptr));
      cost_str = (char *)strtok_r(NULL, " ", &saveptr);
      p = (char *)strtok_r(NULL, " ", &saveptr);
      /* IF no cost data */
      if (p == NULL)
      {
         p = cost_str;
         cost = 0.0;
      }

      g_date_set_parse(&gdate, date);

      days_in_month = g_date_get_days_in_month(g_date_get_month(&gdate), g_date_get_year(&gdate));

      if (g_date_get_year(&gdate) == year)
      {
         /* IF no previous entry then just save off previous values */
         if (strcmp(p, "1") == 0)
         { 
         }
         else if (prev_odo == -1)
         {
         }
         else
         {
            miles = odometer - prev_odo;
            days = g_date_days_between(&prev_date, &gdate);
            /* IF fillup on same day then days to 1 */
            if (days == 0)
               days = 1;
            y_val = y_max - (int)((double)miles / (double)days * (y_max - Y_OFFSET) / (double)max_display);
            x_val = ((int)g_date_get_month(&gdate) - 1) * X_DELTA + X_OFFSET;
            x_val = x_val + (X_DELTA * g_date_get_day(&gdate) / days_in_month);

            if (p_x != -1)
            {
               if (x_val > X_MAX_NUM)
                  x_val = X_MAX_NUM;
               if (x_val < 0)
                  x_val = 0;
               if (y_val > Y_MAX_NUM)
                  y_val = Y_MAX_NUM;
               if (y_val < 0)
                  y_val = 0;

               gdk_draw_line(area->window, area->style->fg_gc[GTK_WIDGET_STATE(area)],
                  p_x, p_y, x_val, y_val);
#if 0
      fprintf(stdout, "Draw distance line %d %d %d %d\n", p_x, p_y, x_val, y_val);
#endif
            }
            p_x = x_val;
            p_y = y_val;
         } 

      } 
      prev_odo = odometer;
      prev_date = gdate;

      iter = g_slist_next(iter);
   }
   g_string_free(data_item, TRUE);
}

void
graph_cost_year(
   int year,
   GtkWidget *area)
{
   GSList  *iter;
   GString *data_item;
   char    *date;
   int      odometer;
   double   fuel;
   double   cost;
   char    *p;
   GDate    gdate;
   int      x_val, y_val;
   int      p_x = -1;
   int      p_y = -1;
   int      y_max = Y_OFFSET + (Y_NUM - 1) * Y_DELTA;
   int      days_in_month;
   char    *saveptr;
   char    *cost_str;
   int      cost_units;
   int      first_pass = TRUE;
   double   cost_total;


   data_item = g_string_new(NULL);
   cost_units = returnCostUnits();

   iter = sort_list;
   while (iter != NULL)
   {
      g_string_erase(data_item, 0, -1);
      g_string_append(data_item, iter->data);

      date = (char *)strtok_r(data_item->str, " ", &saveptr);
      odometer = atoi((char *)strtok_r(NULL, " ", &saveptr));
      fuel = atof((char *)strtok_r(NULL, " ", &saveptr));
      cost_str = (char *)strtok_r(NULL, " ", &saveptr);
      p = (char *)strtok_r(NULL, " ", &saveptr);
      /* IF no cost data */
      if (p == NULL)
      {
         p = cost_str;
         cost = 0.0;
      }
      else
      {
         cost = atof(cost_str);
      }

      g_date_set_parse(&gdate, date);

      days_in_month = g_date_get_days_in_month(g_date_get_month(&gdate), g_date_get_year(&gdate));
   
      if (g_date_get_year(&gdate) == year)
      {
         /* IF no previous entry then just save off previous values */
         if (strcmp(p, "1") == 0)
         {
         }
         else if (first_pass == TRUE)
         {
            first_pass = FALSE;
         }
         else
         {
            if (cost_units == 0)
            {
               if (fuel == 0)
                  cost_total = 0;
               else
                  cost_total = cost / fuel;
            }
            else
               cost_total = cost;

            /* Only if cost entry exists */
            if (cost_total > 0.000000001)
            {
               y_val = y_max - (int)(cost_total * (y_max - Y_OFFSET) / (double)max_display);
               x_val = ((int)g_date_get_month(&gdate) - 1) * X_DELTA + X_OFFSET;
               x_val = x_val + (X_DELTA * g_date_get_day(&gdate) / days_in_month);

               if (p_x != -1)
               {
                  if (x_val > X_MAX_NUM)
                     x_val = X_MAX_NUM;
                  if (x_val < 0)
                     x_val = 0;
                  if (y_val > Y_MAX_NUM)
                     y_val = Y_MAX_NUM;
                  if (y_val < 0)
                     y_val = 0;

                  gdk_draw_line(area->window, area->style->fg_gc[GTK_WIDGET_STATE(area)],
                     p_x, p_y, x_val, y_val);
#if 0
      fprintf(stdout, "Draw cost line %d %d %d %d\n", p_x, p_y, x_val, y_val);
#endif
               }
               p_x = x_val;
               p_y = y_val;
            }
         }
      }
      iter = g_slist_next(iter);
   }
   g_string_free(data_item, TRUE);
}

int
graph_get_max_distance()
{
   GSList  *iter;
   GString *data_item;
   char    *date;
   int      odometer;
   double   fuel;
   double   cost;
   char    *p;
   GDate    gdate;
   GDate    prev_date;
   int      prev_odo = -1;
   int      miles;
   int      days;
   char    *saveptr;
   char    *cost_str;
   int      dist_per_day = 0;
   int      temp;

   data_item = g_string_new(NULL);

   iter = sort_list;
   while (iter != NULL)
   {
      g_string_erase(data_item, 0, -1);
      g_string_append(data_item, iter->data);

      date = (char *)strtok_r(data_item->str, " ", &saveptr);
      odometer = atoi((char *)strtok_r(NULL, " ", &saveptr));
      fuel = atof((char *)strtok_r(NULL, " ", &saveptr));
      cost_str = (char *)strtok_r(NULL, " ", &saveptr);
      p = (char *)strtok_r(NULL, " ", &saveptr);
      /* IF no cost data */
      if (p == NULL)
      {
         p = cost_str;
         cost = 0.0;
      }

      g_date_set_parse(&gdate, date);

      /* IF no previous entry then just save off previous values */
      if (strcmp(p, "1") == 0)
      {
      }
      else if (prev_odo == -1)
      {
      }
      else
      {
         miles = odometer - prev_odo;
         days = g_date_days_between(&prev_date, &gdate);
         /* IF fillup on same day then days to 1 */
         if (days == 0)
            days = 1;
         temp = (int)((double)miles / (double)days);
         if (temp > dist_per_day)
            dist_per_day = temp;
      }  
      prev_odo = odometer;
      prev_date = gdate;

      iter = g_slist_next(iter);
   }
   g_string_free(data_item, TRUE);

#ifdef DEBUG
      fprintf(stdout, "Max distance is %d\n", dist_per_day);
#endif
   return (dist_per_day);
}

int
graph_get_max_mileage()
{
   GSList  *iter;
   GString *data_item;
   char    *date;
   int      odometer;
   double   fuel;
   double   cost;
   char    *p;
   GDate    gdate;
   int      prev_odo = -1;
   char    *saveptr;
   char    *cost_str;
   double   max_mpg = 0.0;
   double   mpg;
   double   prev_fuel;
   int      units;

   data_item = g_string_new(NULL);
   units = returnUnits();

   iter = sort_list;
   while (iter != NULL)
   {
      g_string_erase(data_item, 0, -1);
      g_string_append(data_item, iter->data);

      date = (char *)strtok_r(data_item->str, " ", &saveptr);
      odometer = atoi((char *)strtok_r(NULL, " ", &saveptr));
      fuel = atof((char *)strtok_r(NULL, " ", &saveptr));
      cost_str = (char *)strtok_r(NULL, " ", &saveptr);
      p = (char *)strtok_r(NULL, " ", &saveptr);
      /* IF no cost data */
      if (p == NULL)
      {
         p = cost_str;
         cost = 0.0;
      }

      g_date_set_parse(&gdate, date);

      /* IF no previous entry then just save off previous values */
      if (strcmp(p, "1") == 0)
      {
      }
      else if (prev_odo == -1)
      {
      }
      else
      {
         if (units == 0)
            mpg = (odometer - prev_odo) / fuel;
         else if (units == 1)
            mpg = 100.0 * fuel / (odometer - prev_odo);
         else if (units == 2)
            mpg = 10.0 * fuel / (odometer - prev_odo);
         else if (units == 3)
            mpg = (odometer - prev_odo) / fuel;
         else if (units == 4)
            mpg = (odometer - prev_odo) / fuel;
         else
            mpg = (odometer - prev_odo) / fuel;

         if (mpg > max_mpg)
            max_mpg = mpg;
      }
      prev_odo = odometer;
      prev_fuel = fuel;

      iter = g_slist_next(iter);
   }
   g_string_free(data_item, TRUE);

#ifdef DEBUG
      fprintf(stdout, "Max mpg is %f\n", max_mpg);
#endif
   return ((int)max_mpg);
}

int
graph_get_max_cost()
{
   GSList  *iter;
   GString *data_item;
   char    *date;
   int      odometer;
   double   fuel;
   double   cost;
   char    *p;
   GDate    gdate;
   int      days_in_month;
   char    *saveptr;
   char    *cost_str;
   int      cost_units;
   double   cost_total;
   int      local_max_cost = 0;


   data_item = g_string_new(NULL);
   cost_units = returnCostUnits();

   iter = sort_list;
   while (iter != NULL)
   {
      g_string_erase(data_item, 0, -1);
      g_string_append(data_item, iter->data);

      date = (char *)strtok_r(data_item->str, " ", &saveptr);
      odometer = atoi((char *)strtok_r(NULL, " ", &saveptr));
      fuel = atof((char *)strtok_r(NULL, " ", &saveptr));
      cost_str = (char *)strtok_r(NULL, " ", &saveptr);
      p = (char *)strtok_r(NULL, " ", &saveptr);
      /* IF no cost data */
      if (p == NULL)
      {
         p = cost_str;
         cost = 0.0;
      }
      else
      {
         cost = atof(cost_str);
      }

      g_date_set_parse(&gdate, date);

      days_in_month = g_date_get_days_in_month(g_date_get_month(&gdate), g_date_get_year(&gdate));

      if (cost_units == 0)
      {
         if (fuel == 0)
            cost_total = 0;
         else
            cost_total = (int)ceil(cost / fuel);
      }
      else
         cost_total = cost;

      if (cost_total > local_max_cost)
         local_max_cost = (int)ceil(cost_total);

      iter = g_slist_next(iter);
   }
   g_string_free(data_item, TRUE);

#ifdef DEBUG
      fprintf(stdout, "Max cost is %d\n", local_max_cost);
#endif
   return (local_max_cost);
}

