/*
 * Copyright: (C) 2008 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */

#include <stdlib.h>
#include <string.h>

#include <gtk/gtkmain.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtk.h>

#include "trip_tab.h"
#include "common_tab.h"
#include "mileagefile.h"
#include "mileagecfg.h"

static GtkWidget *car_selector = NULL;
static GtkWidget  *table1 = NULL;
static int        car_inserted = FALSE;
static GtkWidget  *start_date, *end_date;
static GtkWidget  *start_odo, *end_odo;
static GtkWidget  *purpose_text;
static GtkWidget  *person_selector;
static GtkWidget  *dest_selector;

static void car_selector_clicked_cb(GtkWidget* w, gpointer data);
static void trip_process_clicked_cb(GtkWidget* w, gpointer data);

void trip_tab_layout(GtkWidget *window)
{
   GtkWidget  *start_label, *end_label;
   GtkWidget  *start_odo_label, *end_odo_label;
   GtkWidget  *person_label;
   GtkWidget  *dest_label;
   GtkWidget  *purpose_label;
   GtkWidget  *button;


   table1 = gtk_table_new(9,9,FALSE);
   gtk_container_add(GTK_CONTAINER(window), (GtkWidget *)table1);

   car_selector = gtk_combo_box_new_text();
   gtk_table_attach(GTK_TABLE(table1), car_selector, 0, 5, 0, 1, GTK_FILL, GTK_SHRINK, 2, 2);
   gtk_signal_connect(GTK_OBJECT(car_selector),
                     "changed",
                     GTK_SIGNAL_FUNC(car_selector_clicked_cb),
                     car_selector);

   start_label = gtk_label_new("Start Date YYYY/MM/DD");
   gtk_table_attach(GTK_TABLE(table1), start_label, 0, 4, 1, 2, GTK_FILL, GTK_SHRINK, 2, 2);

   start_date = gtk_entry_new_with_max_length(15); 
   gtk_table_attach(GTK_TABLE(table1), start_date, 5, 9, 1, 2, GTK_FILL, GTK_SHRINK, 2, 2);

   start_odo_label = gtk_label_new("Start Odometer"); 
   gtk_table_attach(GTK_TABLE(table1), start_odo_label, 0, 4, 2, 3, GTK_FILL, GTK_SHRINK, 2, 2);

   start_odo = gtk_entry_new_with_max_length(15); 
   gtk_table_attach(GTK_TABLE(table1), start_odo, 5, 9, 2, 3, GTK_FILL, GTK_SHRINK, 2, 2);

   end_label = gtk_label_new("End Date YYYY/MM/DD");
   gtk_table_attach(GTK_TABLE(table1), end_label, 0, 4, 3, 4, GTK_FILL, GTK_SHRINK, 2, 2);

   end_date = gtk_entry_new_with_max_length(15);
   gtk_table_attach(GTK_TABLE(table1), end_date, 5, 9, 3, 4, GTK_FILL, GTK_SHRINK, 2, 2);

   end_odo_label = gtk_label_new("End Odometer");
   gtk_table_attach(GTK_TABLE(table1), end_odo_label, 0, 4, 4, 5, GTK_FILL, GTK_SHRINK, 2, 2);

   end_odo = gtk_entry_new_with_max_length(15);
   gtk_table_attach(GTK_TABLE(table1), end_odo, 5, 9, 4, 5, GTK_FILL, GTK_SHRINK, 2, 2);

   person_label = gtk_label_new("Person");
   gtk_table_attach(GTK_TABLE(table1), person_label, 0, 4, 5, 6, GTK_FILL, GTK_SHRINK, 2, 2);

   person_selector = gtk_combo_box_entry_new_text();
   gtk_table_attach(GTK_TABLE(table1), person_selector, 5, 9, 5, 6, GTK_FILL, GTK_SHRINK, 2, 2);

   dest_label = gtk_label_new("Destination");
   gtk_table_attach(GTK_TABLE(table1), dest_label, 0, 4, 6, 7, GTK_FILL, GTK_SHRINK, 2, 2);

   dest_selector = gtk_combo_box_entry_new_text();
   gtk_table_attach(GTK_TABLE(table1), dest_selector, 5, 9, 6, 7, GTK_FILL, GTK_SHRINK, 2, 2);

   purpose_label = gtk_label_new("Purpose");
   gtk_table_attach(GTK_TABLE(table1), purpose_label, 0, 4, 7, 8, GTK_FILL, GTK_SHRINK, 2, 2);

   purpose_text = gtk_entry_new_with_max_length(60); 
   gtk_table_attach(GTK_TABLE(table1), purpose_text, 5, 9, 7, 8, GTK_FILL, GTK_SHRINK, 2, 2);

   button = gtk_button_new_with_label("Submit");
   gtk_signal_connect(GTK_OBJECT(button),
                     "clicked",
                     GTK_SIGNAL_FUNC(trip_process_clicked_cb),
                     NULL);
   gtk_table_attach(GTK_TABLE(table1), button, 6, 9, 0, 1, GTK_FILL, GTK_SHRINK, 2, 2);

   trip_init();
}

void
trip_tab_load_car(char * car)
{
   if (car == NULL)
   {
      gtk_widget_destroy(car_selector);
      car_selector = gtk_combo_box_new_text();
      gtk_table_attach(GTK_TABLE(table1), car_selector, 0, 5, 0, 1, GTK_FILL, GTK_SHRINK, 2, 2);
      gtk_signal_connect(GTK_OBJECT(car_selector),
                     "changed",
                     GTK_SIGNAL_FUNC(car_selector_clicked_cb),
                     car_selector);
      gtk_widget_show_all(car_selector);
      car_inserted = FALSE;
   }
   else
   {
      gtk_combo_box_append_text(GTK_COMBO_BOX(car_selector), car);
      car_inserted = TRUE;
   }
}

GtkWidget *
trip_tab_get_car_selector()
{
   return car_selector;
}

int
trip_tab_get_car_inserted()
{
   return car_inserted;
}

void
trip_tab_update_car_combo_box(int value)
{
   gtk_combo_box_set_active(GTK_COMBO_BOX(car_selector), value);
}

void
car_selector_clicked_cb(GtkWidget* w, gpointer data)
{
   char *car;

   /* Update all combo boxes to the same value */
   common_tab_update_car_combo_box(gtk_combo_box_get_active((GtkComboBox *)car_selector));

   if (data)
   {
      car = gtk_combo_box_get_active_text(data);
      if (car[strlen(car) - 1] == '\n')
      {
         car[strlen(car) - 1] = '\0';
      }

#ifdef DEBUG
      fprintf(stdout, "Car Selector choosen %s.\n", car);
#endif
#if 0
      /* Only called once in car_tab.c */
      common_tab_load_data(car);
#endif

      if (car)
         free(car);
   }
}

void
trip_process_clicked_cb(GtkWidget* w, gpointer data)
{
   char *start_date_text;
   char *end_date_text;
   char *start_odo_text;
   char *end_odo_text;
   char *purpose_str;
   char *person_text;
   char *dest_text;
   char *car;
   GtkWidget *dialog;


   purpose_str = (char *)gtk_entry_get_text(GTK_ENTRY(purpose_text));
   person_text = gtk_combo_box_get_active_text(GTK_COMBO_BOX(person_selector));
   dest_text = gtk_combo_box_get_active_text(GTK_COMBO_BOX(dest_selector));

   /* Validate the start date */
   start_date_text = (char *)gtk_entry_get_text(GTK_ENTRY(start_date));
   if (common_valid_date(start_date_text) == FALSE)
   {
      dialog = gtk_message_dialog_new (NULL,
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_ERROR,
                                  GTK_BUTTONS_CLOSE,
                                  "Warning, Entry not Stored! Invalid start date.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      if (person_text)
         g_free(person_text);
      if (dest_text)
         g_free(dest_text);
      return;
   }

   /* Validate the end date */
   end_date_text = (char *)gtk_entry_get_text(GTK_ENTRY(end_date));
   if ((strlen(end_date_text) != 0) && (common_valid_date(end_date_text) == FALSE))
   {
      dialog = gtk_message_dialog_new (NULL,
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_ERROR,
                                  GTK_BUTTONS_CLOSE,
                                  "Warning, Entry not Stored! Invalid end date.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      if (person_text)
         g_free(person_text);
      if (dest_text)
         g_free(dest_text);
      return;
   }

   /* check for a start odo string */
   start_odo_text = (char *)gtk_entry_get_text(GTK_ENTRY(start_odo));
   if ((start_odo_text == NULL) || (atof(start_odo_text) <= 0.0))
   {
      dialog = gtk_message_dialog_new (NULL,
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_ERROR,
                                  GTK_BUTTONS_CLOSE,
                                  "Warning, Entry not Stored! Invalid start odometer.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      if (person_text)
         g_free(person_text);
      if (dest_text)
         g_free(dest_text);
      return;
   }

   /* check for a end odo string */
   end_odo_text = (char *)gtk_entry_get_text(GTK_ENTRY(end_odo));
   if ((strlen(end_odo_text) != 0) && (atof(end_odo_text) <= 0.0))
   {
      dialog = gtk_message_dialog_new (NULL,
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_ERROR,
                                  GTK_BUTTONS_CLOSE,
                                  "Warning, Entry not Stored! Invalid end odometer.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      if (person_text)
         g_free(person_text);
      if (dest_text)
         g_free(dest_text);
      return;
   }

   /* add mileagecfg entry for person and destination */
   setPerson(person_text);
   setDestinations(dest_text);

   car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));
   if (car != NULL)
   {
      if (car[strlen(car) - 1] == '\n')
      {
         car[strlen(car) - 1] = '\0';
      }
      addTripEntry(car, start_date_text, start_odo_text, end_date_text, 
         end_odo_text, person_text, dest_text, purpose_str);

      common_tab_updateDisplayTab(car);
      g_free(car);

      trip_set_fields("", "", "", "", "", "", "");
      trip_init();
   }
   else
   {
      dialog = gtk_message_dialog_new (NULL,
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_ERROR,
                                  GTK_BUTTONS_CLOSE,
                                  "Warning, Entry not Stored! Please create car first.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
   }

   if (person_text)
      g_free(person_text);
   if (dest_text)
      g_free(dest_text);
}

void
trip_set_fields(
   char *start_date_str,
   char *end_date_str,
   char *start_odo_str,
   char *end_odo_str,
   char *person_str,
   char *dest_str,
   char *purpose_text_str)
{
   GtkTreeModel *model;
   GtkTreeIter  iter;
   gboolean     valid;
   int          counter;
   int          found;

   gtk_entry_set_text(GTK_ENTRY(start_date), start_date_str);
   gtk_entry_set_text(GTK_ENTRY(end_date), end_date_str);
   gtk_entry_set_text(GTK_ENTRY(start_odo), start_odo_str);
   gtk_entry_set_text(GTK_ENTRY(end_odo), end_odo_str);
   gtk_entry_set_text(GTK_ENTRY(purpose_text), purpose_text_str);

   /* Find the string in the combo_box_entry and make it active */
   model = gtk_combo_box_get_model(GTK_COMBO_BOX(person_selector));
   valid = gtk_tree_model_get_iter_first(model, &iter);
   counter = 0;
   found = 0;
   while (valid)
   {
      gchar *str;
      gtk_tree_model_get(model, &iter, 0, &str, -1);
      if (strcmp(person_str, str) == 0)
      {
         found = 1;
         g_free(str);
         break;
      }
      g_free(str);
      valid = gtk_tree_model_iter_next(model, &iter);
      counter++;
   }
   if (found == 1)
      gtk_combo_box_set_active(GTK_COMBO_BOX(person_selector), counter);

   /* Find the string in the combo_box_entry and make it active */
   model = gtk_combo_box_get_model(GTK_COMBO_BOX(dest_selector));
   valid = gtk_tree_model_get_iter_first(model, &iter);
   counter = 0;
   found = 0;
   while (valid)
   {
      gchar *str;
      gtk_tree_model_get(model, &iter, 0, &str, -1);
      if (strcmp(dest_str, str) == 0)
      {
         found = 1;
         g_free(str);
         break;
      }
      g_free(str);
      valid = gtk_tree_model_iter_next(model, &iter);
      counter++;
   }
   if (found == 1)
      gtk_combo_box_set_active(GTK_COMBO_BOX(dest_selector), counter);
}

void
trip_init()
{
   gchar **person_list;
   gchar **dest_list;
   int i;

   person_list = returnPersons();
   dest_list = returnDestinations();

   /* Rebuild person_selector */
   gtk_widget_destroy(person_selector);
   person_selector = gtk_combo_box_entry_new_text();
   gtk_table_attach(GTK_TABLE(table1), person_selector, 5, 9, 5, 6, GTK_FILL, GTK_SHRINK, 2, 2);
   gtk_widget_show_all(person_selector);

   if (person_list != NULL)
   {
      for (i = 0; i < g_strv_length(person_list); i++)
      {
         gtk_combo_box_append_text(GTK_COMBO_BOX(person_selector), person_list[i]);
      }
   }

   /* Rebuild destination selector */
   gtk_widget_destroy(dest_selector);
   dest_selector = gtk_combo_box_entry_new_text();
   gtk_table_attach(GTK_TABLE(table1), dest_selector, 5, 9, 6, 7, GTK_FILL, GTK_SHRINK, 2, 2);
   gtk_widget_show_all(dest_selector);

   if (dest_list != NULL)
   {
      for (i = 0; i < g_strv_length(dest_list); i++)
      {
         gtk_combo_box_append_text(GTK_COMBO_BOX(dest_selector), dest_list[i]);
      }
   }
   g_strfreev(person_list);
   g_strfreev(dest_list);
}



