/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Copyright (c) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/
 
#include "config.h"

#ifdef OSSO_IC

#include <osso-ic.h>
#include <osso-ic-dbus.h>
#include <dbus/dbus.h>

#else

#include <conic.h>

#endif

#if HILDON == 1

#include <hildon/hildon.h>

#else

#include <hildon-widgets/hildon-note.h>

#endif

#include "mauku.h"

static void iap_connected(const gchar* iap_name);
static void iap_disconnected(const gchar* iap_name);
#ifdef OSSO_IC
static void iap_status_changed(struct iap_event_t* event, void* user_data);
static DBusHandlerResult iap_state_changed(DBusConnection* connection, DBusMessage* message, void* user_data);
#else
static void connection_event(ConIcConnection* connection, ConIcConnectionEvent* event, gpointer user_data);
#endif

gboolean connected = FALSE;
GtkWidget* connection_waiting_widget = NULL;
#ifndef OSSO_IC
ConIcConnection* connection = NULL;
#endif

void network_init() {
#ifdef OSSO_IC
	DBusError error;
	const gchar* filter = "interface=" ICD_DBUS_INTERFACE;

	dbus_error_init(&error);
	dbus_bus_add_match((DBusConnection*)osso_get_sys_dbus_connection(osso_context), filter, &error);
	if (dbus_error_is_set(&error)) {
		fprintf(stderr, "ERROR: dbus_bus_add_match failed: %s\n", error.message);
		return;

	}
	if (!dbus_connection_add_filter((DBusConnection*)osso_get_sys_dbus_connection(osso_context), iap_state_changed, NULL, NULL)) {
		fprintf(stderr, "ERROR: dbus_connection_add_filter failed: %s\n", error.message);
		return;
	}

	osso_iap_cb(iap_status_changed);
	osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, NULL);
#else	
	if ((connection = con_ic_connection_new())) {
		g_signal_connect(G_OBJECT(connection), "connection-event", G_CALLBACK(connection_event), NULL);
		g_object_set(G_OBJECT(connection), "automatic-connection-events", TRUE, NULL);
		con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
	}	
#endif
}

void network_cleanup() {
#ifdef OSSO_IC
	osso_iap_disconnect(OSSO_IAP_ANY, NULL);
#else	
	con_ic_connection_disconnect(connection);
#endif

}

gboolean network_is_connected() {

	return connected;
}

gboolean network_make_connected() {
	
	if (!connected) {
		connection_waiting_widget = hildon_note_new_information(NULL, "Waiting for a network connection to come up...");
		hildon_note_set_button_text(HILDON_NOTE(connection_waiting_widget), "Cancel");
		g_object_ref(connection_waiting_widget);
#ifdef OSSO_IC
		osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, NULL);
#else
		con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
#endif
		gtk_dialog_run(GTK_DIALOG(connection_waiting_widget));
		gtk_widget_destroy(connection_waiting_widget);
		g_object_unref(connection_waiting_widget);
		connection_waiting_widget = NULL;
	}
	
	return connected;
}

static void iap_connected(const gchar* iap_name) {
	gchar* key = NULL;
	
	if (!connected) {
		connected = TRUE;
		ui_start_update_overview();
	}
}

static void iap_disconnected(const gchar* iap_name) {
	connected = FALSE;
}

#ifdef OSSO_IC

static void iap_status_changed(struct iap_event_t* event, void* user_data) {
	gdk_threads_enter();

#ifdef i386
        connected = TRUE;
        ui_start_update_overview();
#else
        if (event->type == OSSO_IAP_DISCONNECTED || event->type == OSSO_IAP_ERROR) {
                iap_disconnected(event->iap_name);
        } else if (event->type == OSSO_IAP_CONNECTED) {
                iap_connected(event->iap_name);
                if (connection_waiting_widget) {
                        gtk_widget_destroy(GTK_WIDGET(connection_waiting_widget));
                }
        }
#endif

	gdk_threads_leave();
}

static DBusHandlerResult iap_state_changed(DBusConnection* connection, DBusMessage* message, void* user_data) {
	DBusHandlerResult ret = DBUS_HANDLER_RESULT_HANDLED;
	DBusError error;
        gchar* name;
        gchar* type;
        gchar* state;

	gdk_threads_enter();

        dbus_error_init (&error);
        if (!dbus_message_is_signal(message, ICD_DBUS_INTERFACE, ICD_STATUS_CHANGED_SIG ) ||
            !dbus_message_get_args (message, &error, DBUS_TYPE_STRING, &name, DBUS_TYPE_STRING, &type,
                                    DBUS_TYPE_STRING, &state, DBUS_TYPE_INVALID)) {
                ret = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        } else if (!strcmp(state, "DISCONNECTING")) {
                iap_disconnected(name);
        } else if (!strcmp(state, "CONNECTED")) {
                iap_connected(name);
        }

	gdk_threads_leave();

        return ret;
}

#else

static void connection_event(ConIcConnection* connection, ConIcConnectionEvent* event, gpointer user_data) {
	gdk_threads_enter();
	
#ifdef i386
	connected = TRUE;
//	ui_start_update_overview();
#else
	switch (con_ic_connection_event_get_status(event)) {
		case CON_IC_STATUS_CONNECTED:
			iap_connected(con_ic_event_get_iap_id(CON_IC_EVENT(event)));
			break;
		case CON_IC_STATUS_DISCONNECTED:
			iap_disconnected(con_ic_event_get_iap_id(CON_IC_EVENT(event)));
			break;
	}
#endif
	if (connection_waiting_widget) {
		gtk_widget_destroy(GTK_WIDGET(connection_waiting_widget));
		connection_waiting_widget = NULL;
	}
	
	gdk_threads_leave();
}

#endif
