/*****************************************************************************
 *** Mauku - Micro-blogging client for Maemo devices
 ***
 *** Copyright (c) 2009 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "microfeedmisc.h"

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
void* microfeed_memory_allocate_bytes(size_t size) {
	void* pointer;
	
	if (!(pointer = malloc(size))) {
		fprintf(stderr, "ERROR: Failed to allocate memory.\n");
		
		exit(126);
	}
	memset(pointer, 0, size);
		
	return pointer;
}


void microfeed_memory_free(void* pointer) {	
	free(pointer);
}

char* microfeed_util_string_concatenate(const char* s, ...) {
	char* result;
	size_t size;
	size_t length;
	va_list argp;

	size = strlen(s);
	length = size + 1;
	result = (char*)malloc(length);
	memcpy(result, s, length);
	va_start(argp, s);
	while ((s = va_arg(argp, const char*))) {
		length = strlen(s);
		result = realloc(result, size + length + 1);
		memcpy(result + size, s, length + 1);
		size += length;
	}
	va_end(argp);

	return result;	
}

char* microfeed_util_string_base64_encode(const char* s, size_t length) {
	static const char base64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
	char* result;
	size_t result_length;
	const unsigned char* from;
	const unsigned char* end;
	char* to;
		
	result_length = ((length + 2) / 3) * 4;
	result = (char*)malloc(result_length + 1);
	result[result_length] = 0;
	for (from = s, to = result, end = from + length; from + 2 < end; from += 3, to += 4) {
		to[0] = base64[from[0] >> 2];
		to[1] = base64[((from[0] & 0x3) << 4) | ((from[1] & 0xf0) >> 4)];
		to[2] = base64[((from[1] & 0xf) << 2) | ((from[2] & 0xc0) >> 6)];
		to[3] = base64[from[2] & 0x3f];
	}
	if (from + 1 < end) {
		to[0] = base64[from[0] >> 2];
		to[1] = base64[((from[0] & 0x3) << 4) | ((from[1] & 0xf0) >> 4)];
		to[2] = base64[((from[1] & 0xf) << 2)];
		to[3] = '=';
	} else if (from < end) {
		to[0] = base64[from[0] >> 2];
		to[1] = base64[(from[0] & 0x3) << 4];
		to[2] = '=';
		to[3] = '=';		
	}
	
	return result;	
}


size_t microfeed_util_string_starts_with(const char* s, const char* prefix) {
	size_t length = 0;
	const char* p;
	
	p = prefix;
	while (*s && *p && *s == *p) {
		s++;
		p++;
	}
	if (*p == 0) {
		length = p - prefix;
	}
	
	return length;
}

static int hexadecimal_char_to_int(char c) {
	int retvalue = -1;

	if (c >= '0' && c <= '9') {
		retvalue = c - '0';
	} else if (c >= 'a' && c <= 'f') {
		retvalue = c - 'a' + 10;
	} else if (c >= 'A' && c <= 'F') {
		retvalue = c - 'A' + 10;
	}
	
	return retvalue;
}

char* microfeed_util_string_percent_encoding_escape(const char* s) {
	static const char hex[] = "0123456789ABCDEF";
	char* result;
	size_t length;
	const char* from;
	char* to;
	size_t offset;
	
	length = strlen(s) + 1;
	result = (char*)malloc(length);
	
	for (to = result, from = s; *from; from++, to++) {
		if ((*from >= 'a' && *from <= 'z') || (*from >= 'A' && *from <= 'Z') || (*from >= '0' && *from <= '9') ||
		    *from == '-' || *from == '_' || *from == '.' || *from == '~') {
			*to = *from;
		} else {
			length += 2;
			offset = to - result;
			result = realloc(result, length);
			to = result + offset;
			*to++ = '%';
			*to++ = hex[((unsigned char)*from) >> 4];
			*to = hex[((unsigned char)*from) & 0xf];
		}
	}
	*to = 0;
	
	return result;
}

char* microfeed_util_string_percent_encoding_unescape(const char* s) {
	char* result;
	const char* from;
	char* to;
	int tempvalue;
	int value;

	result = (char*)malloc(strlen(s) + 1);
	for (to = result, from = s; *from; from++, to++) {
		if (*from == '%') {
			if ((tempvalue = hexadecimal_char_to_int(from[1])) == -1) {
				free(result);
				result = NULL;
				break;
			}
			if ((value = hexadecimal_char_to_int(from[2])) == -1) {
				free(result);
				result = NULL;
				break;
			}
			value |= tempvalue << 4;
			if ((value >= 'a' && value <= 'z') || (value >= 'A' && value <= 'Z') || (value >= '0' && value <= '9') ||
			    value == '-' || value == '_' || value == '.' || value == '~') {
				*to = value;
			} else {
				*to++ = *from++;
				*to++ = *from++;
				*to = *from;
			}
		} else {
			*to = *from;
		}
	}
	if (result) {
		*to++ = 0;
		result = (char*)realloc(result, result - to);
	}

	return result;
}
