/*****************************************************************************
 *** Mauku - The micro-blogging client for Maemo devices 
 ***
 *** Copyright (c) 2007-2008 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"
 
#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>

#include <dbus/dbus.h>
#include <curl/curl.h>
#include <json.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdlib.h>
#include <signal.h>

#if HILDON == 1

#include <gtk/gtkenums.h>
#include <hildon/hildon.h>

#else

#include <hildon-widgets/hildon-input-mode-hint.h>
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-banner.h>
#include <hildon-widgets/hildon-note.h>

#endif

#include "mauku.h"

static void add_icon_set_into_icon_factory(GtkIconFactory* icon_factory, const gchar* stock_id);
static void init_actions();
static void init_icon_factory();
static void init_menu();
static void init_toolbar();
static void insert_into_toolbar(GtkToolbar* toolbar, GtkAction* action, gboolean insert_separator);
static void menu_position_func(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer user_data);
static gboolean on_about_button_release_event(GtkWidget* widget, GdkEventButton* event, View* view);
static void on_add_comment_activate(GtkAction* action, gpointer user_data);
static void on_close_this_window_activate(GtkAction* action, gpointer user_data);
static void on_open_contacts_activate(GtkAction* action, gpointer user_data);
static void on_open_explore_activate(GtkAction* action, gpointer user_data);
static void on_open_marked_items_activate(GtkAction* action, gpointer user_data);
static void on_open_overview_activate(GtkAction* action, gpointer user_data);
static void on_open_your_items_activate(GtkAction* action, gpointer user_data);
static void on_open_received_direct_messages_activate(GtkAction* action, gpointer user_data);
static void on_open_sent_direct_messages_activate(GtkAction* action, gpointer user_data);
static void on_open_replies_activate(GtkAction* action, gpointer user_data);
static void on_show_toolbar_notify(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data);
static void on_send_activate(GtkAction* action, gpointer user_data);
static void on_toggle_action_toggled(GtkToggleAction* toggle_action, gpointer user_data);
static void on_update_activate(GtkAction* action, gpointer user_data);
static void on_write_something_activate(GtkAction* action, gpointer user_data);
static void show_about_dialog();
static void show_about_top_bottom_widget(View* view);
void set_service_sensitive_actions();
#ifdef MAUKU_BACKTRACE_UPLOADS_ENABLED
static void on_upload_backtrace_activate(GtkAction* action, gpointer user_data);
#endif
void update_contacts_view(View* view);
void update_explore_view(View* view);
void update_overview(View* view);
void update_your_items_view(View* view);

static View* overview = NULL;
static gboolean is_toolbar_hidden = FALSE;
static GtkAction* update_action;
static GtkAction* send_action;
static GtkAction* add_comment_action;
static GtkAction* write_something_action;
static GtkAction* open_overview_action;
static GtkAction* open_contacts_action;
static GtkAction* open_marked_items_action;
static GtkAction* open_explore_action;
static GtkAction* open_your_items_action;
static GtkAction* open_received_direct_messages_action;
static GtkAction* open_sent_direct_messages_action;
static GtkAction* open_replies_action;
static GtkAction* settings_action;
static GtkAction* about_action;
/* static GtkAction* change_location_action; */
static GtkToggleAction* show_toolbar_action;
static GtkToggleAction* show_scrollbar_action;
static GtkToggleAction* new_item_notification_action;
static GtkAction* close_this_window_action;
static GtkAction* close_all_windows_action;
/* static GtkWidget* location_label; */
#ifdef MAUKU_BACKTRACE_UPLOADS_ENABLED
static GtkAction* upload_backtrace_action;
#endif

void ui_init() {
	init_icon_factory();
	init_actions();
	init_menu();
	init_toolbar();

	gtk_action_activate(open_overview_action);
	show_about_top_bottom_widget(overview);	
	if (!gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL) &&
	    !gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, NULL) &&
	    !gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_EMAIL_ENABLED, NULL)) {
		settings_show_dialog();
	}

}

void ui_close_view(View* view) {
	GtkWidget* note;

	if (view_get_view_count() <= (view_get_visibility(overview) ? 1 : 2)) {
		note = hildon_note_new_confirmation(GTK_WINDOW(view_get_window(view)), "Do you really want to quit Mauku?");
		if (gtk_dialog_run(GTK_DIALOG(note)) == GTK_RESPONSE_OK) {
			gtk_main_quit();
		}
		gtk_widget_destroy(note);
	} else if (view == overview) {
		view_set_visibility(overview, FALSE);
	} else {
		view_destroy(view);
	}
}

void ui_popup_menu() {
	gtk_menu_popup(hildon_program_get_common_menu(hildon_program_get_instance()), NULL, NULL, menu_position_func, NULL, 0,
	               gtk_get_current_event_time());

}
/*
void ui_set_location(const gchar* location_text) {
	gtk_label_set_text(GTK_LABEL(location_label), location_text);
}
*/
void ui_set_sensitive_actions(UiSensitiveActionType sensitive_actions) {
	gtk_action_set_sensitive(write_something_action, sensitive_actions & UI_SENSITIVE_ACTION_WRITE_SOMETHING);
	gtk_action_set_sensitive(send_action, sensitive_actions & UI_SENSITIVE_ACTION_SEND);
	gtk_action_set_sensitive(add_comment_action, sensitive_actions & UI_SENSITIVE_ACTION_ADD_COMMENT);
	gtk_action_set_sensitive(open_overview_action, sensitive_actions & UI_SENSITIVE_ACTION_OPEN_OVERVIEW);
	gtk_action_set_sensitive(open_contacts_action, sensitive_actions & UI_SENSITIVE_ACTION_OPEN_CONTACTS);
	gtk_action_set_sensitive(open_explore_action, sensitive_actions & UI_SENSITIVE_ACTION_OPEN_EXPLORE);
	gtk_action_set_sensitive(open_your_items_action, sensitive_actions & UI_SENSITIVE_ACTION_OPEN_YOUR_ITEMS);
	gtk_action_set_sensitive(open_marked_items_action, sensitive_actions & UI_SENSITIVE_ACTION_OPEN_MARKED_ITEMS);
	gtk_action_set_sensitive(update_action, sensitive_actions & UI_SENSITIVE_ACTION_UPDATE);

	set_service_sensitive_actions();
}

void ui_set_toolbar_hidden(gboolean toolbar_hidden) {
	is_toolbar_hidden = toolbar_hidden;
	gconf_client_notify(gconf, MAUKU_GCONF_KEY_SHOW_TOOLBAR);
}

void ui_start_update_overview() {
	if (view_exists(overview)) {
		view_start_update(overview, FALSE);
	}
}

static void add_icon_set_into_icon_factory(GtkIconFactory* icon_factory, const gchar* stock_id) {
	gchar* sizes[] = { "scalable", "12x12", "16x16", "26x26", "32x32", "34x34", "40x40", "50x50", "64x64", "250x250", NULL };
	GtkIconSet* icon_set = NULL;
	GtkIconSource* icon_source;
	GdkPixbuf* pixbuf;
	gchar* s;
	gint i;
	
	for (i = 0; sizes[i]; i++) {
		s = g_strconcat(DATADIR "/icons/hicolor/", sizes[i], "/hildon/", stock_id, (i ? ".png" : ".svg"), NULL);
		if ((pixbuf = gdk_pixbuf_new_from_file(s, NULL))) {
			if (!icon_set) {
				icon_set = gtk_icon_set_new_from_pixbuf(pixbuf);
			} else {
				icon_source = gtk_icon_source_new();
				gtk_icon_source_set_pixbuf(icon_source, pixbuf);
				gtk_icon_set_add_source(icon_set, icon_source);
			}
		}
		g_free(s);
	}
	if (icon_set) {
		gtk_icon_factory_add(icon_factory, stock_id, icon_set);
	}
}

static void init_actions() {
	update_action = gtk_action_new("update", "Update the feed", NULL, MAUKU_STOCK_VIEW_REFRESH);
	g_signal_connect(update_action, "activate", G_CALLBACK(on_update_activate), NULL);
	
	send_action = gtk_action_new("send", "Send a new item...", NULL, MAUKU_STOCK_MAIL_MESSAGE_NEW);
	g_signal_connect(send_action, "activate", G_CALLBACK(on_send_activate), NULL);

	add_comment_action = gtk_action_new("add_comment", "Add a comment / reply...", NULL, MAUKU_STOCK_MAIL_MESSAGE_NEW);
	g_signal_connect(add_comment_action, "activate", G_CALLBACK(on_add_comment_activate), NULL);

	write_something_action = gtk_action_new("write_something", "", NULL, MAUKU_STOCK_MAIL_MESSAGE_NEW);
	g_signal_connect(write_something_action, "activate", G_CALLBACK(on_write_something_activate), NULL);

	open_overview_action = gtk_action_new("open_overview", "Open overview", NULL, MAUKU_STOCK_OVERVIEW);
	g_signal_connect(open_overview_action, "activate", G_CALLBACK(on_open_overview_activate), NULL);

	open_contacts_action = gtk_action_new("open_contacts", "Open contacts", NULL, MAUKU_STOCK_CONTACTS);
	g_signal_connect(open_contacts_action, "activate", G_CALLBACK(on_open_contacts_activate), NULL);

	open_marked_items_action = gtk_action_new("open_marked_items", "Open marked items", NULL, NULL);
	g_signal_connect(open_marked_items_action, "activate", G_CALLBACK(on_open_marked_items_activate), NULL);

	open_explore_action = gtk_action_new("open_explore", "Open latest public items", NULL, NULL);
	g_signal_connect(open_explore_action, "activate", G_CALLBACK(on_open_explore_activate), NULL);

	open_your_items_action = gtk_action_new("open_your_items", "Open your items", NULL, NULL);
	g_signal_connect(open_your_items_action, "activate", G_CALLBACK(on_open_your_items_activate), NULL);

	open_received_direct_messages_action = gtk_action_new("open_received_direct_messages", "Open received direct messages", NULL, NULL);
	g_signal_connect(open_received_direct_messages_action, "activate", G_CALLBACK(on_open_received_direct_messages_activate), NULL);

	open_sent_direct_messages_action = gtk_action_new("open_sent_direct_messages", "Open sent direct messages", NULL, NULL);
	g_signal_connect(open_sent_direct_messages_action, "activate", G_CALLBACK(on_open_sent_direct_messages_activate), NULL);

	open_replies_action = gtk_action_new("open_replies", "Open replies", NULL, NULL);
	g_signal_connect(open_replies_action, "activate", G_CALLBACK(on_open_replies_activate), NULL);

	settings_action = gtk_action_new("settings", "Settings...", NULL, NULL);
	g_signal_connect(settings_action, "activate", G_CALLBACK(settings_show_dialog), NULL);
	
	about_action = gtk_action_new("about", "About...", NULL, NULL);
	g_signal_connect(about_action, "activate", G_CALLBACK(show_about_dialog), NULL);

/*
	change_location_action = gtk_action_new("change_location", "Change location...", NULL, MAUKU_STOCK_LOCATION_CHANGE);
	g_signal_connect(change_location_action, "activate", G_CALLBACK(jaiku_change_location), NULL);
*/
	show_toolbar_action = gtk_toggle_action_new("show_toolbar", "Show toolbar", NULL, NULL);
	gtk_toggle_action_set_active(show_toolbar_action, gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_SHOW_TOOLBAR, NULL));
	g_signal_connect(show_toolbar_action, "toggled", G_CALLBACK(on_toggle_action_toggled), MAUKU_GCONF_KEY_SHOW_TOOLBAR);

	show_scrollbar_action = gtk_toggle_action_new("show_scrollbar", "Show scrollbar", NULL, NULL);
	gtk_toggle_action_set_active(show_scrollbar_action, gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_SHOW_SCROLLBAR, NULL));
	g_signal_connect(show_scrollbar_action, "toggled", G_CALLBACK(on_toggle_action_toggled), MAUKU_GCONF_KEY_SHOW_SCROLLBAR);

	new_item_notification_action = gtk_toggle_action_new("new_item_notification", "New item notification", NULL, NULL);
	gtk_toggle_action_set_active(new_item_notification_action, gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_NEW_ITEM_NOTIFICATION, NULL));
	g_signal_connect(new_item_notification_action, "toggled", G_CALLBACK(on_toggle_action_toggled), MAUKU_GCONF_KEY_NEW_ITEM_NOTIFICATION);

	close_this_window_action = gtk_action_new("close_this_window", "Close this window", NULL, NULL);
	g_signal_connect(close_this_window_action, "activate", G_CALLBACK(on_close_this_window_activate), NULL);
	
	close_all_windows_action = gtk_action_new("close_all_windows", "Close all windows", NULL, NULL);
	g_signal_connect(close_all_windows_action, "activate", G_CALLBACK(gtk_main_quit), NULL);

#ifdef MAUKU_BACKTRACE_UPLOADS_ENABLED
	upload_backtrace_action = gtk_action_new("upload_backtrace", "Upload backtrace (aborts!)", NULL, NULL);
	g_signal_connect(upload_backtrace_action, "activate", G_CALLBACK(on_upload_backtrace_activate), NULL);
#endif
}

static void init_icon_factory() {
	GtkIconFactory* icon_factory;
	
	icon_factory = gtk_icon_factory_new();
	add_icon_set_into_icon_factory(icon_factory, MAUKU_STOCK_CONTACTS);
/*	add_icon_set_into_icon_factory(icon_factory, MAUKU_STOCK_LOCATION_CHANGE); */
	add_icon_set_into_icon_factory(icon_factory, MAUKU_STOCK_MAIL_MESSAGE_NEW);
	add_icon_set_into_icon_factory(icon_factory, MAUKU_STOCK_OVERVIEW);
	add_icon_set_into_icon_factory(icon_factory, MAUKU_STOCK_VIEW_REFRESH);
	gtk_icon_factory_add_default(icon_factory);
}

static void init_menu() {
	PangoFontDescription* font;
	GtkWidget* menu;
	GtkWidget* item;
	GtkWidget* submenu;
	GtkWidget* subsubmenu;

	font = pango_font_description_new();
	pango_font_description_set_size(font, 20 * PANGO_SCALE);

	menu = gtk_menu_new();

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item = gtk_action_create_menu_item(update_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item = gtk_action_create_menu_item(send_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item = gtk_action_create_menu_item(add_comment_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);

	item = gtk_menu_item_new_with_label("Open");
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	submenu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(open_overview_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(open_contacts_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), gtk_separator_menu_item_new());
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(open_marked_items_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(open_explore_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(open_your_items_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), gtk_separator_menu_item_new());

	item = gtk_menu_item_new_with_label("Twitter");
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item);
	subsubmenu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), subsubmenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(subsubmenu), item = gtk_action_create_menu_item(open_received_direct_messages_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(subsubmenu), item = gtk_action_create_menu_item(open_sent_direct_messages_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(subsubmenu), item = gtk_action_create_menu_item(open_replies_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), gtk_separator_menu_item_new());
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(settings_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(about_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);

/*	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item = gtk_action_create_menu_item(change_location_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
*/	gtk_menu_shell_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());

	item = gtk_menu_item_new_with_label("Show");
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	submenu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(GTK_ACTION(show_toolbar_action)));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(GTK_ACTION(show_scrollbar_action)));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item = gtk_action_create_menu_item(GTK_ACTION(new_item_notification_action)));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());
	
	item = gtk_menu_item_new_with_label("Close");
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	submenu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(close_this_window_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item = gtk_action_create_menu_item(close_all_windows_action));
	gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item)), font);

#ifdef MAUKU_BACKTRACE_UPLOADS_ENABLED
//	gtk_menu_shell_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());
//	gtk_menu_shell_append(GTK_MENU_SHELL(menu), gtk_action_create_menu_item(upload_backtrace_action));
#endif
	
	hildon_program_set_common_menu(hildon_program_get_instance(), GTK_MENU(menu));
}

static void init_toolbar() {
	GtkToolbar* toolbar;
	GtkToolItem*item;
	PangoFontDescription* font;

	toolbar = GTK_TOOLBAR(gtk_toolbar_new());
	
	insert_into_toolbar(toolbar, update_action, TRUE);
	insert_into_toolbar(toolbar, write_something_action, TRUE);
	insert_into_toolbar(toolbar, open_overview_action, TRUE);
	insert_into_toolbar(toolbar, open_contacts_action, FALSE);

/*	item = GTK_TOOL_ITEM(gtk_action_create_tool_item(change_location_action));
	gtk_tool_item_set_expand(item, TRUE);
	gtk_toolbar_insert(toolbar, item, -1);

	item = gtk_tool_item_new();
	location_label = gtk_label_new("");
	gtk_widget_set_size_request(location_label, 256, -1);
	gtk_label_set_ellipsize(GTK_LABEL(location_label), PANGO_ELLIPSIZE_END);
	font = pango_font_description_new();
	pango_font_description_set_size(font, 12 * PANGO_SCALE);
	gtk_widget_modify_font(location_label, font);
	gtk_misc_set_alignment(GTK_MISC(location_label), 0.0, 0.5);
	gtk_container_add(GTK_CONTAINER(item), location_label);
	gtk_toolbar_insert(toolbar, item, -1);
*/
	hildon_program_set_common_toolbar(hildon_program_get_instance(), toolbar);

	gconf_client_notify_add(gconf, MAUKU_GCONF_KEY_SHOW_TOOLBAR, on_show_toolbar_notify, toolbar, NULL, NULL);
	gconf_client_notify(gconf, MAUKU_GCONF_KEY_SHOW_TOOLBAR);
}

static void insert_into_toolbar(GtkToolbar* toolbar, GtkAction* action, gboolean append_separator) {
	GtkToolItem* item;

	item = GTK_TOOL_ITEM(gtk_action_create_tool_item(action));
	gtk_tool_item_set_expand(item, TRUE);
	gtk_toolbar_insert(toolbar, item, -1);
	if (append_separator) {
		item = gtk_separator_tool_item_new();
		gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(item), FALSE);
		gtk_tool_item_set_expand(item, TRUE);
		gtk_toolbar_insert(toolbar, item, -1);
	}
}

static void menu_position_func(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer user_data) {
	*x = 90;
	*y = 47;
	*push_in = FALSE;
}

static gboolean on_about_button_release_event(GtkWidget* widget, GdkEventButton* event, View* view) {
	view_set_top_bottom_widget(view, NULL, FALSE);
	
	return FALSE;
}

static void on_add_comment_activate(GtkAction* action, gpointer user_data) {
	View* view;
	
	view = view_get_topmost_view();
	if (view_get_user_data(view, "twitter:nick")) {
		sending_show_send_top_bottom_widget(view);
	} else {
		sending_show_add_comment_top_bottom_widget(view);
	}
}

static void on_close_this_window_activate(GtkAction* action, gpointer user_data) {
	View* view;
	
	if ((view = view_get_topmost_view())) {
		ui_close_view(view);
	}
}

static void on_open_contacts_activate(GtkAction* action, gpointer user_data) {
	View* view;

	view = view_new("Contacts", "Contacts",
	 UI_SENSITIVE_ACTION_ALL & ~(UI_SENSITIVE_ACTION_OPEN_CONTACTS | UI_SENSITIVE_ACTION_WRITE_SOMETHING | UI_SENSITIVE_ACTION_ADD_COMMENT),
	 update_contacts_view, NULL);
	view_start_update(view, TRUE);
}

static void on_open_received_direct_messages_activate(GtkAction* action, gpointer user_data) {
	View* view;
	
	view = view_new("Received", "Received direct messages",
	 UI_SENSITIVE_ACTION_ALL & ~(UI_SENSITIVE_ACTION_OPEN_DIRECT_MESSAGES | UI_SENSITIVE_ACTION_ADD_COMMENT),
	 twitter_update_generic_view, NULL);
	view_set_user_data(view, "twitter:url", "http://twitter.com/direct_messages.json", NULL);
	view_set_user_data(view, "twitter:type", GINT_TO_POINTER(TWITTER_ITEM_TYPE_RECEIVED_DM), NULL);
	view_start_update(view, TRUE);
}

static void on_open_sent_direct_messages_activate(GtkAction* action, gpointer user_data) {
	View* view;
	
	view = view_new("Sent", "Sent direct messages",
	 UI_SENSITIVE_ACTION_ALL & ~(UI_SENSITIVE_ACTION_OPEN_DIRECT_MESSAGES | UI_SENSITIVE_ACTION_ADD_COMMENT),
	 twitter_update_generic_view, NULL);
	view_set_user_data(view, "twitter:url", "http://twitter.com/direct_messages/sent.json", NULL);
	view_set_user_data(view, "twitter:type", GINT_TO_POINTER(TWITTER_ITEM_TYPE_SENT_DM), NULL);
	view_start_update(view, TRUE);
}

static void on_open_explore_activate(GtkAction* action, gpointer user_data) {
	View* view;
	
	view = view_new("Explore", "Latest public items",
	 UI_SENSITIVE_ACTION_ALL & ~(UI_SENSITIVE_ACTION_OPEN_EXPLORE | UI_SENSITIVE_ACTION_WRITE_SOMETHING | UI_SENSITIVE_ACTION_ADD_COMMENT),
	 update_explore_view, NULL);
	view_start_update(view, TRUE);
}

static void on_open_marked_items_activate(GtkAction* action, gpointer user_data) {
	View* view;
	
	view = view_new("Marked", "Marked items",
	 UI_SENSITIVE_ACTION_ALL & ~(UI_SENSITIVE_ACTION_OPEN_MARKED_ITEMS | UI_SENSITIVE_ACTION_WRITE_SOMETHING | UI_SENSITIVE_ACTION_ADD_COMMENT),
	 backend_update_marked_items_view, NULL);
	view_start_update(view, TRUE);
}

static void on_open_overview_activate(GtkAction* action, gpointer user_data) {
	if (overview) {
		view_set_visibility(overview, TRUE);	
	} else {
		overview = view_new("Overview", "Overview",
		 UI_SENSITIVE_ACTION_ALL & ~(UI_SENSITIVE_ACTION_OPEN_OVERVIEW | UI_SENSITIVE_ACTION_ADD_COMMENT),
		 update_overview, NULL);
		view_set_user_data(overview, "twitter:url", "http://twitter.com/statuses/friends_timeline.json", NULL);
		if (network_is_connected()) {
			view_start_update(overview, TRUE);
		}
	}
}

static void on_open_replies_activate(GtkAction* action, gpointer user_data) {
	View* view;

	view = view_new("Replies", "Replies",
	 UI_SENSITIVE_ACTION_ALL & ~(UI_SENSITIVE_ACTION_OPEN_REPLIES | UI_SENSITIVE_ACTION_ADD_COMMENT),
	 twitter_update_generic_view, NULL);
	view_set_user_data(view, "twitter:url", "http://twitter.com/statuses/replies.json", NULL);
	view_start_update(view, TRUE);
}

static void on_open_your_items_activate(GtkAction* action, gpointer user_data) {
	View* view;
	
	view = view_new("Your", "Your items",
	 UI_SENSITIVE_ACTION_ALL & ~(UI_SENSITIVE_ACTION_OPEN_YOUR_ITEMS | UI_SENSITIVE_ACTION_ADD_COMMENT),
	 update_your_items_view, NULL);
	view_set_user_data(view, "twitter:url", "http://twitter.com/statuses/user_timeline.json", NULL);
	view_start_update(view, TRUE);
}


static void on_show_toolbar_notify(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data) {
	GtkWidget* toolbar;

	toolbar = GTK_WIDGET(user_data);
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_SHOW_TOOLBAR, NULL) && !is_toolbar_hidden) {
		gtk_widget_show_all(toolbar);
	} else {
		gtk_widget_hide(toolbar);
	}	
}

static void on_send_activate(GtkAction* action, gpointer user_data) {
	sending_show_send_top_bottom_widget(view_get_topmost_view());
}

static void on_toggle_action_toggled(GtkToggleAction* toggle_action, gpointer user_data) {
	gchar* key;
	
	key = (gchar*)user_data;
	gconf_client_set_bool(gconf, key, gtk_toggle_action_get_active(toggle_action), NULL);

	/* Hack: Caching is not working properly. */
	gconf_client_suggest_sync(gconf, NULL);
	gconf_client_clear_cache(gconf);
}

static void on_update_activate(GtkAction* action, gpointer user_data) {
	View* view;
	
	if ((view = view_get_topmost_view())) {
		view_start_update(view, TRUE);
	}
}

static void on_write_something_activate(GtkAction* action, gpointer user_data) {
	if (gtk_action_get_sensitive(add_comment_action) /* && !view_get_user_data(view_get_topmost_view()) */) {
		on_add_comment_activate(add_comment_action, user_data);
	} else {
		on_send_activate(send_action, user_data);
	}
}

static void show_about_dialog() {
	GdkPixbuf* pixbuf;
	
	pixbuf = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/64x64/hildon/mauku.png", NULL);
	gtk_show_about_dialog(NULL, "name", "Mauku", "logo", pixbuf, "version", VERSION, 
	                      "copyright", "Henrik Hedberg <hhedberg@innologies.fi>", 
			      "website", "http://mauku.henrikhedberg.com/", NULL);
}

static void show_about_top_bottom_widget(View* view) {
	GtkWidget* event_box;
	GtkWidget* vbox;
	GtkWidget* hbox;
	GtkWidget* image;
	GtkWidget* label;
	
	event_box = gtk_event_box_new();
	g_signal_connect(event_box, "button-release-event", G_CALLBACK(on_about_button_release_event), view);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(event_box), vbox);
	gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 0);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	image = gtk_image_new_from_file("/usr/share/icons/hicolor/64x64/hildon/mauku.png");
	gtk_widget_set_size_request(image, 64, 64);
	gtk_misc_set_alignment(GTK_MISC(image), 0.5, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 5);
	label = gtk_label_new(NULL);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_label_set_markup(GTK_LABEL(label), "<big>Mauku</big> version " VERSION "\n<small>By Henrik Hedberg &lt;hhedberg@innologies.fi&gt;</small>");
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 5);

	view_set_top_bottom_widget(view, event_box, FALSE);
	gtk_widget_show_all(event_box);
}

void set_service_sensitive_actions() {
	gboolean jaiku_enabled;
	gboolean twitter_enabled;
	
	jaiku_enabled = gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL);
	twitter_enabled = gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, NULL);
	
	gtk_action_set_sensitive(open_received_direct_messages_action, twitter_enabled);
	gtk_action_set_sensitive(open_sent_direct_messages_action, twitter_enabled);
	gtk_action_set_sensitive(open_replies_action, twitter_enabled);
}

#ifdef MAUKU_BACKTRACE_UPLOADS_ENABLED
static void on_upload_backtrace_activate(GtkAction* action, gpointer user_data) {
	void (*func)();
	
	func();
}
#endif


void update_contacts_view(View* view) {
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL)) {
		jaiku_update_contacts_view(view);
	}
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, NULL)) {
		twitter_update_contacts_view(view);
	}
}

void update_explore_view(View* view) {
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL)) {
		jaiku_update_explore_view(view);
	}
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, NULL)) {
		twitter_update_explore_view(view);
	}
}

void update_overview(View* view) {
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL)) {
		jaiku_update_overview(view);
	}
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, NULL)) {
		twitter_update_generic_view(view);
	}
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_EMAIL_ENABLED, NULL)) {
		email_update_view(view);
	}
}

void update_your_items_view(View* view) {
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL)) {
		jaiku_update_your_jaikus_view(view);
	}
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, NULL)) {
		twitter_update_generic_view(view);
	}
}
