/*****************************************************************************
 *** Mauku - Micro-blogging client for Maemo devices
 ***
 *** Copyright (c) 2009 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "microfeedstore.h"
#include "microfeedmisc.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define DELTA 10

struct _MicrofeedStore {
	void** data;
	unsigned int size;
	unsigned int reserved;
	MicrofeedStoreCompareKeysFunction compare_keys;
	MicrofeedStoreGetKeyFunction get_key;
	MicrofeedStoreIterator* iterators;
	int unsorted;
};

struct _MicrofeedStoreIterator {
	MicrofeedStoreIterator* previous;
	MicrofeedStoreIterator* next;
	
	MicrofeedStore* store;
	unsigned int index;
};

static int get_index(MicrofeedStore* store, const void* key, unsigned int* index);

MicrofeedStore* microfeed_store_new_sorted(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key) {
	MicrofeedStore* store;
	
	store = microfeed_memory_allocate(MicrofeedStore);
	store->compare_keys = compare_keys;
	store->get_key = get_key;
	
	return store;
}

MicrofeedStore* microfeed_store_new_unsorted(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key) {
	MicrofeedStore* store;
	
	store = microfeed_store_new_sorted(compare_keys, get_key);
	store->unsorted = 1;
	
	return store;
}

void microfeed_store_free(MicrofeedStore* store) {
	MicrofeedStoreIterator* iterator;
	
	for (iterator = store->iterators; iterator; iterator = iterator->next) {
		iterator->store = NULL;
	}	
	free(store->data);
	store->data = NULL;
	store->iterators = NULL;
	store->size = store->reserved = 0;
	free(store);
}

void microfeed_store_foreach(MicrofeedStore* store, MicrofeedStoreForeachFunction foreach, void* user_data) {
	int i;
	
	for (i = 0; i < store->size; i++) {
		foreach(store->data[i], user_data);
	}
}

void* microfeed_store_get_impl(MicrofeedStore* store, const void* key) {
	void* data = NULL;
	unsigned int index;

	if (get_index(store, key, &index)) {
		data = store->data[index];
	}

	return data;
}

void* microfeed_store_get_index_impl(MicrofeedStore* store, unsigned int index) {
	void* data = NULL;
	
	if (index < store->size) {
		data = store->data[index];
	}
	
	return data;	
}

unsigned int microfeed_store_get_size(MicrofeedStore* store) {
	return store->size;
}

void microfeed_store_insert(MicrofeedStore* store, void* data) {
	unsigned int index;

	if (!get_index(store, store->get_key(data), &index)) {
		if (store->size == store->reserved) {
			store->reserved += DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}
	
		if (index < store->size) {
			memmove(store->data + index + 1, store->data + index,
			        (store->size - index) * sizeof(void*));
		}
		store->size++;
		store->data[index] = data;
	}
}

void* microfeed_store_replace(MicrofeedStore* store, void* data) {
	unsigned int index;
	void* existing;
	
	if (get_index(store, store->get_key(data), &index)) {
		existing = store->data[index];
	} else {
		existing = NULL;
		if (store->size == store->reserved) {
			store->reserved += DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}

		if (index < store->size) {
			memmove(store->data + index + 1, store->data + index,
		        	(store->size - index) * sizeof(void*));
		}
		store->size++;
	}
	store->data[index] = data;
	
	return existing;
}

int microfeed_store_remove(MicrofeedStore* store, const void* data) {
	int retvalue = 0;
	unsigned int index;

	if (get_index(store, store->get_key(data), &index)) {
		if (index + 1 < store->size) {
			memmove(store->data + index, store->data + index + 1,
		        	(store->size - index - 1) * sizeof(void*));
		}

		store->size--;
		if (store->size + 2 * DELTA == store->reserved) {
			store->reserved -= DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}

		retvalue = 1;
	}

	return retvalue;
}

void* microfeed_store_remove_key_impl(MicrofeedStore* store, const void* key) {
	void* data = NULL;
	unsigned int index;

	if (get_index(store, key, &index)) {
		data = store->data[index];
		if (index + 1 < store->size) {
			memmove(store->data + index, store->data + index + 1,
		        	(store->size - index - 1) * sizeof(void*));
		}

		store->size--;
		if (store->size + 2 * DELTA == store->reserved) {
			store->reserved -= DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}
	}

	return data;
}

void* microfeed_store_remove_index_impl(MicrofeedStore* store, unsigned int index) {
	void* data = NULL;

	if (index < store->size) {
		data = store->data[index];
		if (index + 1 < store->size) {
			memmove(store->data + index, store->data + index + 1,
		        	(store->size - index - 1) * sizeof(void*));
		}

		store->size--;
		if (store->size + 2 * DELTA == store->reserved) {
			store->reserved -= DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}
	}

	return data;
}

MicrofeedStoreIterator* microfeed_store_iterate(MicrofeedStore* store, const void* start_key) {
	MicrofeedStoreIterator* iterator;
	unsigned int index;
	
	iterator = microfeed_memory_allocate(MicrofeedStoreIterator);
	iterator->store = store;
	if (start_key) {
		if (get_index(store, start_key, &index)) {
			iterator->index = index;
		} else {
			iterator->index = store->size;
		}
	} else {
		iterator->index = 0;
	}
	if (store->iterators) {
		store->iterators->previous = iterator;
		iterator->next = store->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	store->iterators = iterator;
	
	return iterator;
}

int microfeed_store_compare_keys_direct(const void* key1, const void* key2) {

	return (key1 == key2 ? 0 : (key1 < key2 ? -1 : 1));
}

const void* microfeed_store_get_key_direct(const void* data) {
	
	return data;
}


void microfeed_store_iterator_free(MicrofeedStoreIterator* iterator) {
	if (iterator->next) {
		iterator->next->previous = iterator->previous;
	}
	if (iterator->previous) {
		iterator->previous->next = iterator->next;
	} else if (iterator->store) {
		iterator->store->iterators = iterator->next;
	}

	iterator->store = NULL;
	microfeed_memory_free(iterator);
}

void* microfeed_store_iterator_get_impl(MicrofeedStoreIterator* iterator) {
	void* data = NULL;
	
	if (iterator->store && iterator->index < iterator->store->size) {
		data = iterator->store->data[iterator->index];
	}
	
	return data;
}

void microfeed_store_iterator_next(MicrofeedStoreIterator* iterator) {
	if (iterator->store && iterator->index < iterator->store->size) {
		iterator->index++;
	}
}

static int get_index(MicrofeedStore* store, const void* key, unsigned int* index) {
	int retval = 0;
	int result;
	unsigned int i, min, max;

	if (store->unsorted) {
		*index = store->size;
		for (i = 0; i < store->size; i++) {
			if (store->compare_keys(key, store->get_key(store->data[i])) == 0) {
				*index = i;
				retval = 1;
			}
		}
	} else {
		if (store->size == 0) {
			*index = 0;
		} else if ((result = store->compare_keys(key, store->get_key(store->data[0]))) == 0) {
			*index = 0;
			retval = 1;	
		} else if (result < 0) {
			*index = 0;
		} else if (store->size == 1) {
			*index = 1;
		} else if ((result = store->compare_keys(key, store->get_key(store->data[store->size - 1]))) == 0) {
			*index = store->size -1;
			retval = 1;
		} else if (result > 0) {
			*index = store->size;
		} else if (store->size == 2) {
			*index = store->size - 1;
		} else {
			min = i = 0;
			max = store->size - 1;
			
			while (min <= max) {
				i = (min + max) / 2;
				if ((result = store->compare_keys(key, store->get_key(store->data[i]))) == 0) {
					retval = 1;
					break;
				} else if (result < 0) {
					max = i - 1;
				} else {
					i++;
					min = i;
				}
			}
			
			*index = i;
		}
	}

	return retval;
}

/*

typedef struct {
	int i;
} Testi;

static int compare_ints(const void* item1, const void* item2) {
	return *((const int*)item1) - *((const int*)item2);
}

static const void* get_int(const void* item) {
	return &((Testi*)item)->i;
}

int main(void) {
	MicrofeedStore* store;
	int i;
	Testi* p;
	MicrofeedStoreIterator* iterator;
	int previous = 0;
		
	srand(time(NULL));
	
	store = microfeed_store_new(compare_ints, get_int);

	for (i = 0; i < 20000; i++) {
		p = (Testi*)malloc(sizeof(Testi));
		p->i = rand();
		microfeed_store_insert(store, p);
	}
	printf("test\n");
	for (iterator = microfeed_store_iterate(store, NULL); p = microfeed_store_iterator_get(iterator, Testi); microfeed_store_iterator_next(iterator)) {
		if (p->i < previous) {
			printf("ERROR: %d < %d\n", p->i, previous);
		}
		previous = p->i;
	}

	return 0;
}

*/
