/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Copyright (c) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"
 
#include <glib.h>
#include <gtk/gtk.h>
#include <libosso.h>
#include <errno.h>
#include <gconf/gconf-client.h>

#if HILDON == 0
#include <osso-ic.h>
#include <osso-ic-dbus.h>
#endif

#include <dbus/dbus.h>
#include <curl/curl.h>
#include <json.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>

#if HILDON == 1

#include <gtk/gtkenums.h>
#include <hildon/hildon.h>

#else

#include <hildon-widgets/hildon-input-mode-hint.h>
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-banner.h>

#endif

#include "mauku.h"

static void activate_link(GtkAboutDialog* about, const gchar* link, gpointer data);
static gboolean start_version_check(gpointer data);
static void* version_thread(void* args);

osso_context_t* osso_context;
ImageCache* image_cache;

int main(int argc, char** argv) {
	gchar buffer[1024];

	g_thread_init(NULL);
	gdk_threads_init();
	gdk_threads_enter();
	gtk_init(&argc, &argv);
	g_set_application_name("Mauku");

	if ((osso_context = osso_initialize(MAUKU_DBUS_NAME, VERSION, TRUE, NULL)) == NULL) {
		fprintf(stderr, "ERROR: osso_initialize failed.\n");
		return 1;
	}

	miaouw_init(osso_context);

#ifdef MAUKU_BACKTRACE_UPLOADS_ENABLED
	miaouw_backtrace_uploader_init(g_find_program_in_path(argv[0]), "Mauku", VERSION, "http://mauku.henrikhedberg.com/backtrace_upload", 1024);
#endif

	gtk_about_dialog_set_url_hook(activate_link, NULL, NULL);
	image_cache = image_cache_new();

	snprintf(buffer, 1024, "%s/.mauku", (getenv("HOME") ? getenv("HOME") : "/home/user"));
	mkdir(buffer, 0744);
	snprintf(buffer, 1024, "%s/.mauku/icons", (getenv("HOME") ? getenv("HOME") : "/home/user"));
	mkdir(buffer, 0744);

	g_timeout_add(1, start_version_check, NULL);
	
	microfeed_thread_init();
	network_init();
	microfeed_http_init(1);
	settings_init();
	ui_init();
	text_to_speech_service_init();
	geoclue_init();

	gtk_main();

	geoclue_cleanup();
	text_to_speech_service_cleanup();
	settings_cleanup();
	microfeed_http_cleanup();
	network_cleanup();
	
	gdk_threads_leave();

	return 0;
}

static void activate_link(GtkAboutDialog* about, const gchar* link, gpointer data) {
	osso_rpc_t retval;

	osso_rpc_run(osso_context, "com.nokia.osso_browser", "/com/nokia/osso_browser", "com.nokia.osso_browser",
	             "open_new_window", &retval,  DBUS_TYPE_STRING, link, DBUS_TYPE_INVALID);
}

static gboolean start_version_check(gpointer data) {
	if (g_thread_create(version_thread, NULL, FALSE, NULL) == NULL) {
			/* TODO: Failed... */
	}

	return FALSE;
}

static void* version_thread(void* args) {
	MicrofeedHttp* http;
	const gchar* data;
	gchar* s;
	size_t len;
	GtkWidget* note;
	gint response;
	osso_rpc_t retval;

	http = microfeed_http_new();
	if ((data = microfeed_http_get_data(http,  "http://mauku.henrikhedberg.com/version/" VERSION, &len))) {
		if (len > 8 && !strncmp(data, "UPDATE: ", 8) && (s = strchr(data + 8, ' '))) {
			*s = 0;
			gdk_threads_enter();
			note = hildon_note_new_confirmation(NULL, s + 1);
			hildon_note_set_button_texts(HILDON_NOTE(note), "Update Mauku", "Continue using this version");
			response = gtk_dialog_run(GTK_DIALOG(note));
			gtk_widget_destroy(note);
			if (response == GTK_RESPONSE_OK) {
				osso_rpc_run(osso_context, "com.nokia.osso_browser", "/com/nokia/osso_browser", "com.nokia.osso_browser",
				             "open_new_window", &retval,  DBUS_TYPE_STRING, data + 8, DBUS_TYPE_INVALID);
				gtk_main_quit();
			}
			gdk_threads_leave();
		} else if (len > 9 && !strncmp(data, "DISPLAY: ", 9)) {
			gdk_threads_enter();
			note = hildon_note_new_information(NULL, data + 9);
			response = gtk_dialog_run(GTK_DIALOG(note));
			gtk_widget_destroy(note);			
			gdk_threads_leave();
		}
	}
	
	microfeed_http_free(http);

	return NULL;
}

