#include "scrobblerd.h"
#include "../common.h"
#include "lastfm.h"
#include <conic/conicevent.h>
#include <conic/coniciap.h>
#include <conic/conicconnection.h>
#include <conic/conicconnectionevent.h>

#define MAXTRACKS 6

GConfClient *gc;
sqlite3 *db = NULL;
gboolean gotHUP = FALSE;
gboolean isonline = FALSE;
gboolean onlineknown = FALSE;
GMainLoop *loop = NULL;
osso_context_t *osso = NULL;

guint g_timeout_id = 0;
guint g_timeout_secs = 0;

guint gc_notify_id_submit = 0;
gboolean gc_submit = FALSE;

guint gc_notify_id_lastfmuser = 0;
gchar *gc_lastfmuser = NULL;

guint gc_notify_id_lastfmpass = 0;
gchar *gc_lastfmpass = NULL;

guint gc_notify_id_lasterror = 0;
gint gc_errno = 0; /* >=10 = stop errors, remove the timer until reset */

static void on_connection_event (ConIcConnection *cnx, ConIcConnectionEvent *event, gpointer user_data);
void signalHandler(int sig);
gint submit_update(void);
gboolean firstrun_callback(gpointer data);
gboolean mainloop(gpointer data);
gboolean scrobbleSubmit(GString *songs, gboolean notify);

void signalHandler(int sig)
{
switch(sig){
			case SIGHUP:
				gotHUP = TRUE;
				syslog(LOG_NOTICE, "got HUP");
				break;		
			case SIGTERM:
				g_main_loop_quit(loop);
				break;		
		}
}

void run_schedule_mainloop(int debugid)
{
if (g_timeout_id == 0)
	{
	mainloop(NULL); /* it will set the timeout */
	if (debugid>0) syslog(LOG_NOTICE, "added timeout for mainloop (%d)", debugid);
	}	
}

void remove_schedule_mainloop(int debugid)
{
if (g_timeout_id > 0)
	{
	g_source_remove(g_timeout_id);
	g_timeout_id = 0;
	g_timeout_secs = 0;
	if (debugid>0) syslog(LOG_NOTICE, "removed timeout for mainloop (%d)", debugid);
	}	
}

gint dbus_callback (const gchar *interface, const gchar *method, GArray *arguments, gpointer data, osso_rpc_t *retval)
{
/*
  syslog(LOG_NOTICE, "scrobblerd dbus: %s, %s *%s*\n", interface, method, arguments->data);
*/
 	if (g_ascii_strcasecmp(method, "quit") == 0)
 		{
		g_main_loop_quit(loop);
		}
 	else if (g_ascii_strcasecmp(method, "run") == 0)
 		{
		if (onlineknown==FALSE) g_timeout_add(5*1000, firstrun_callback, NULL); /*this is here to allow for the connection callback to come back*/
	 										 else firstrun_callback(NULL); 
		}
 	else if (g_ascii_strcasecmp(method, "notify") == 0)
		{
		do
			{
			if (onlineknown == FALSE || isonline == FALSE) break;
			if (arguments->len!=4) break;

			osso_rpc_t art=g_array_index(arguments, osso_rpc_t, 0);
			osso_rpc_t tra=g_array_index(arguments, osso_rpc_t, 1);
			osso_rpc_t alb=g_array_index(arguments, osso_rpc_t, 2);
			osso_rpc_t dur=g_array_index(arguments, osso_rpc_t, 3);

			if (art.type!=DBUS_TYPE_STRING || art.value.s==NULL) break;
			if (tra.type!=DBUS_TYPE_STRING || tra.value.s==NULL) break;
			if (alb.type!=DBUS_TYPE_STRING || alb.value.s==NULL) break;
			if (dur.type!=DBUS_TYPE_UINT32 && dur.type!=DBUS_TYPE_INT32) break;

			syslog(LOG_NOTICE, "NOTIFY call valid: %s/%s", art.value.s, tra.value.s);

			GString *songs=g_string_new(NULL);

			CURL *curl = curl_easy_init();

			char *art2=curl_easy_escape(curl, art.value.s, 0);
			char *tra2=curl_easy_escape(curl, tra.value.s, 0);
			char *alb2=curl_easy_escape(curl, alb.value.s, 0);

			g_string_append_printf(songs, "a=%s&t=%s&b=%s&n=&m=&", art2, tra2, alb2);
	
			if (dur.value.u>0)
				g_string_append_printf(songs, "l=%d&", dur.value.u);
			else
				g_string_append_printf(songs, "l=&");
			
			curl_free(art2);
			curl_free(tra2);
			curl_free(alb2);
			
			curl_easy_cleanup(curl);

			if (scrobbleSubmit(songs, true)==false)
				syslog(LOG_ERR, "nowplaying submission failed");

			g_string_free(songs, true);		
			}while(false);
		}

  retval->type = DBUS_TYPE_INVALID;
  return OSSO_OK;
}

gboolean firstrun_callback(gpointer data)
{
if (isonline==TRUE && g_timeout_id > 0) /* gc vars are already valid and running */ /* only call if we're online */
	{
	remove_schedule_mainloop(0);
	run_schedule_mainloop(0);
	}
return(false);
}

static void gch_submit(GConfClient *gcc, guint con, GConfEntry *entry, gpointer user_data)
{
	if (entry->value) gc_submit = gconf_value_get_bool(entry->value);
							 else gc_submit = TRUE;

	if (gc_submit == TRUE && gc_errno<10 && g_timeout_id == 0)
		{
		run_schedule_mainloop(0);
		}
	else if (gc_submit == FALSE && g_timeout_id > 0)
		{
		remove_schedule_mainloop(0);
		}
}

static void gch_lastfmuser(GConfClient *gcc, guint con, GConfEntry *entry, gpointer user_data)
{
	const char *tmp;
	if (entry->value) tmp = gconf_value_get_string(entry->value);
							 else tmp = "";
	if (gc_lastfmuser) g_free(gc_lastfmuser);
	gc_lastfmuser = g_strdup(tmp);
	lastfm_loggedin = 0;
/*
	syslog(LOG_NOTICE, "updated lastfmuser");
*/
}

static void gch_lastfmpass(GConfClient *gcc, guint con, GConfEntry *entry, gpointer user_data)
{
	const char *tmp;
	if (entry->value) tmp = gconf_value_get_string(entry->value);
							 else tmp = "";
	if (gc_lastfmpass) g_free(gc_lastfmpass);
	gc_lastfmpass = g_strdup(tmp);
	lastfm_loggedin = 0;
/*
	syslog(LOG_NOTICE, "updated lastfmpass");
*/
}

static void gch_lasterror(GConfClient *gcc, guint con, GConfEntry *entry, gpointer user_data)
{
	if (entry->value) gc_errno = gconf_value_get_int(entry->value);
							 else gc_errno = 0;
	
	if (gc_submit == TRUE && gc_errno<10 && g_timeout_id == 0)
		{
		run_schedule_mainloop(0);
		}
	else if (gc_errno>=10 && g_timeout_id > 0)
		{
		remove_schedule_mainloop(0);
		}
}

int main(int argc, char *argv[])
{
int lfp=0;

openlog("scrobblerd", LOG_PID | LOG_PERROR, LOG_DAEMON);

if (getuid()==0)
	{
	syslog(LOG_ERR, "do not run as root!");
	exit(1);
	}

gboolean dbuscalled=false;
if (argc>1 && strcmp(argv[1], "--dbus")==0)
	{
	dbuscalled=true;
	}

lfp=open("/var/run/scrobblerd.lock",O_RDWR|O_CREAT,0640);
if (lfp<0)
	{
	syslog(LOG_ERR, "could not open lock file for writing");
	if (dbuscalled==false) exit(1);
	}
else if (flock(lfp, LOCK_EX | LOCK_NB)<0)
	{
	syslog(LOG_ERR, "already running");
	exit(1);
	}

if (dbuscalled==false)
	{
	pid_t pid;
	if ((pid = fork()) < 0)
		{
	  return 1;
		}
	else if (pid != 0)
		{
	  exit(0);
		}
	}

setsid();
chdir("/");
umask(022); /* 755 */

if (lfp>0)
	{
	char str[16];
	sprintf(str,"%d\n",getpid());
	write(lfp,str,strlen(str));
	}

signal(SIGCHLD,SIG_IGN);
signal(SIGTSTP,SIG_IGN);
signal(SIGTTOU,SIG_IGN);
signal(SIGTTIN,SIG_IGN);
signal(SIGHUP, signalHandler);
signal(SIGTERM, signalHandler);

g_type_init();
gc = gconf_client_get_default();

osso = osso_initialize("scrobblerd", PACKAGE_VERSION, FALSE/*dbuscalled*/, NULL);

if (osso_rpc_set_cb_f(osso, "com.nokia.scrobblerd", "com/nokia/scrobblerd", "com.nokia.scrobblerd", dbus_callback, NULL) != OSSO_OK)
	{
	syslog(LOG_ERR, "error setting dbus callback");
	exit(1);
	}

gconf_client_add_dir(gc, GCONF_PATH, GCONF_CLIENT_PRELOAD_NONE, NULL);
gc_notify_id_submit = gconf_client_notify_add(gc, GCONF_PATH_SUBMIT, gch_submit, NULL, NULL, NULL);
gc_notify_id_lastfmuser = gconf_client_notify_add(gc, GCONF_PATH_USERNAME, gch_lastfmuser, NULL, NULL, NULL);
gc_notify_id_lastfmpass = gconf_client_notify_add(gc, GCONF_PATH_PASSMD5, gch_lastfmpass, NULL, NULL, NULL);
gc_notify_id_lasterror = gconf_client_notify_add(gc, GCONF_PATH_LASTERROR, gch_lasterror, NULL, NULL, NULL);

gconf_client_notify(gc, GCONF_PATH_SUBMIT);
gconf_client_notify(gc, GCONF_PATH_USERNAME);
gconf_client_notify(gc, GCONF_PATH_PASSMD5);
/*don't notify lasterror*/

loop=g_main_loop_new(NULL, TRUE);

ConIcConnection *con=con_ic_connection_new();
g_signal_connect(G_OBJECT(con), "connection-event", G_CALLBACK(on_connection_event), NULL);
g_object_set(con, "automatic-connection-events", true, NULL);
con_ic_connection_connect(con, CON_IC_CONNECT_FLAG_AUTOMATICALLY_TRIGGERED | CON_IC_CONNECT_FLAG_UNMANAGED);

g_main_loop_run(loop);

remove_schedule_mainloop(0);

if (db) sqlite3_close(db);

gconf_client_notify_remove(gc, gc_notify_id_submit);
gconf_client_notify_remove(gc, gc_notify_id_lastfmuser);
gconf_client_notify_remove(gc, gc_notify_id_lastfmpass);
gconf_client_notify_remove(gc, gc_notify_id_lasterror);

if (gc_lastfmuser) g_free(gc_lastfmuser);
if (gc_lastfmpass) g_free(gc_lastfmpass);

if (osso) osso_deinitialize(osso);

syslog(LOG_NOTICE, "exiting");
return(0);
}

gboolean mainloop(gpointer data)
{
guint secs;

if (isonline==TRUE || gotHUP==TRUE)
	{
/*
	syslog(LOG_NOTICE, "mainloop()!");
*/
	gotHUP=FALSE;
	submit_update();

	if (gc_errno>1 && gc_errno<10) secs=ERROR_TIME_SECS; /* error occured */
														else secs=CHECK_TIME_SECS;
	}
else
	{
	secs=CHECK_TIME_SECS;
	}	

/*
syslog(LOG_NOTICE, "newsecs = %d (errno=%u)", secs, gc_errno);
*/
if (g_timeout_id>0 && g_timeout_secs == secs)
	{
	return(true); /*keep the same loop*/
	}
else
	{
	g_timeout_id = g_timeout_add(secs*1000, mainloop, NULL);
	g_timeout_secs = secs;
	}

return(false);
}

static void on_connection_event (ConIcConnection *cnx, ConIcConnectionEvent *event, gpointer user_data)
{
onlineknown = TRUE;
isonline=FALSE;

if (con_ic_connection_event_get_error(event)!=CON_IC_CONNECTION_ERROR_NONE) return;
if (con_ic_connection_event_get_status(event)==CON_IC_STATUS_CONNECTED) isonline=TRUE;
}

/*ret: found tracks to submit*/
gint submit_update()
{
if (!db)
	{
	int rc;
	rc = sqlite3_open(SCROBBLERDB, &db);
	if (rc)
		{
		db=NULL;
		syslog(LOG_DEBUG, "could not open sqlite3 db: %d", rc);
		saveError(6);
		return(-1);
		}
	sqlite3_exec(db, "PRAGMA synchronous = OFF;", NULL, NULL, NULL);
	}

char tq[512];

sprintf(tq, "SELECT idx, artist, track, album, duration, playedon FROM tracks WHERE submitdate=0 ORDER BY playedon");
sqlite3_stmt *stmt = NULL;
const char *dum;
int rc = sqlite3_prepare(db, tq, strlen(tq), &stmt, &dum);

if (rc)
		{
		syslog(LOG_WARNING, "could not select sqlite3");
		sqlite3_close(db);
		db=NULL;
		return(-1);
		}

rc = SQLITE_BUSY;

int cnt=0, i;
unsigned long int ups[MAXTRACKS];

for(i=0;i<MAXTRACKS;i++) ups[i]=0;

GString *songs=g_string_new(NULL);

CURL *curl = curl_easy_init();

while(rc == SQLITE_BUSY || rc == SQLITE_ROW)
	{
	rc = sqlite3_step(stmt);
	if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE || rc == SQLITE_DONE) break;
	else if (rc == SQLITE_ROW)
		{
			int idx = sqlite3_column_int(stmt, 0);
			char *art = (char *)sqlite3_column_text(stmt, 1);
			char *tra = (char *)sqlite3_column_text(stmt, 2);
			char *alb = (char *)sqlite3_column_text(stmt, 3);
			int dur = sqlite3_column_int(stmt, 4);
			int pla = sqlite3_column_int(stmt, 5);
/*
			char ti[64];
			strftime(ti, sizeof(ti), "%Y-%m-%d %H:%M:%S", gmtime(&pla));
*/

/*			syslog(LOG_ERR, "ftime for %u is %s", pla, ti);*/
						
			char *art2=curl_easy_escape(curl, art, 0);
			char *tra2=curl_easy_escape(curl, tra, 0);
			char *alb2=curl_easy_escape(curl, alb, 0);
			
			g_string_append_printf(songs, "a[%d]=%s&t[%d]=%s&b[%d]=%s&m[%d]=&l[%d]=%d&i[%d]=%d&o[%d]=P&r[%d]=&n[%d]=&",
			cnt, art2, cnt, tra2, cnt, alb2, cnt, cnt, dur, cnt, pla, cnt, cnt, cnt);
			
			curl_free(art2);
			curl_free(tra2);
			curl_free(alb2);
			
			ups[cnt]=idx;
			cnt++;
	
			if (cnt>MAXTRACKS) break;
		}
	}

curl_easy_cleanup(curl);

if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE)
	{
	syslog(LOG_WARNING, "could not select sqlite3 (2)");
	sqlite3_close(db);
	db=NULL;
	return(-1);
	}

sqlite3_finalize(stmt);
/*
syslog(LOG_WARNING, "songcount is %d", cnt);
*/
if (cnt==0)
	{
	g_string_free(songs, true);		
	return(0);
	}

/*fprintf(stderr, "songs=%s\n", songs->str);*/
if (scrobbleSubmit(songs, false)==true)
	{
	syslog(LOG_NOTICE, "submission accepted, %d track(s)", cnt);
	int tm=(int)time(NULL);
	g_string_free(songs, true);
	for(i=0;i<MAXTRACKS;i++)
		{
		if (ups[i]==0) continue;
		
		sprintf(tq, "UPDATE tracks SET submitdate=%i WHERE idx=%lu;", tm, ups[i]);
		if (sqlite3_exec(db, tq, NULL, NULL, NULL) != 0)
			{
			syslog(LOG_WARNING, "ERROR updating track #%lu on db", ups[i]);
			}
			
		}

	}

return(cnt);
}

gboolean scrobbleSubmit(GString *songs, gboolean notify)
{
gboolean retval=false;

if (gc_lastfmuser && gc_lastfmpass)
	{
	gint errno = 99;
	gint tries = 0;
	
	while(errno>=10 && tries<3)
		{
		if (notify==false) retval=lastfm_submit(gc_lastfmuser, gc_lastfmpass, songs, &errno);
									else retval=lastfm_notify(gc_lastfmuser, gc_lastfmpass, songs, &errno);
		tries++;
		}

/*
	syslog(LOG_ERR, "returned errno=%u", errno);
*/
	saveError(errno);
	}

return(retval);
}

void saveError(gint errno)
{
if (errno!=gc_errno)
	{
	gconf_client_set_int(gc, GCONF_PATH_LASTERROR, errno, NULL);
	gc_errno = errno; /* needed to re-add the timeout without waiting one loop for the gc update */
	}	

gconf_client_set_int(gc, GCONF_PATH_LASTERRORTIME, time(NULL), NULL);
}
