/*
 *  Miaouw - The Miaouw Library for Maemo Development
 *  Copyright (C) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <errno.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <libosso.h>

#include <miaouw/miaouwprivate.h>
#include <miaouw/miaouwwindow.h>

/**
 * SECTION:miaouwwindow
 * @short_description: Top-level application window for Hildon/Maemo development
 * @see_also: #GtkWindow
 *
 * The #MiaouwWindow has built-in support for the hard keys: fullscreen, zoom in/out, and d-pad.
 * It can be configured to directly change the font size of text widgets.
 * In addition, the D-pad keys can be attached to any scroll adjustments. 
 **/

enum {
	SIGNAL_CHANGE_FONT_SIZE,
	SIGNAL_COUNT
};

struct _MiaouwWindowPrivate {
	GtkAdjustment* vadjustment;
	GtkAdjustment* hadjustment;
	
	gboolean is_fullscreen;
	gboolean is_visible;
	gboolean is_urgency_hint_active;
	guint font_size;
};

static guint signals[SIGNAL_COUNT];

#if HILDON == 1
static gboolean is_display_on = TRUE;
#endif

static gboolean is_led_blinking = FALSE;
static guint urgency_hint_window_count = 0;

static gchar* led_pattern_name = NULL;

static void class_init(gpointer klass, gpointer data);
static void object_init(GTypeInstance* instance, gpointer klass);

static gboolean key_press_event(GtkWidget* widget, GdkEventKey* event);
static void scroll_backwards(GtkAdjustment* adjustment);
static void scroll_forwards(GtkAdjustment* adjustment);

#if HILDON == 1
static void on_display_event(osso_display_state_t state, gpointer data);
#endif

static void realize(GtkWidget* widget);
static gboolean visibility_notify_event(GtkWidget* widget, GdkEventVisibility* event);
static gboolean window_state_event(GtkWidget* widget, GdkEventWindowState* event);

GType miaouw_window_get_type() {
	static GType type = 0;
	static const GTypeInfo info = {
		sizeof (MiaouwWindowClass),
		NULL,   /* base_init */
		NULL,   /* base_finalize */
		class_init,
		NULL,   /* class_finalize */
		NULL,   /* class_data */
		sizeof (MiaouwWindow),
		0,
		object_init,
		NULL
	};

	if (!type) {
		type = g_type_register_static(HILDON_TYPE_WINDOW, "MiaouwWindow", &info, 0);
	}

	return type;
}

GtkWidget* miaouw_window_new() {
	return gtk_widget_new(MIAOUW_TYPE_WINDOW, NULL);
}

GtkWidget* miaouw_window_new_with_title(const gchar* title) {
	return gtk_widget_new(MIAOUW_TYPE_WINDOW, "title", title, NULL);
}

void miaouw_window_activate_urgency_hint(MiaouwWindow* window) {
	osso_rpc_t retval;

	if (!window->priv->is_urgency_hint_active) {
		if (!window->priv->is_visible) {
			gtk_window_set_urgency_hint(GTK_WINDOW(window), TRUE);
			window->priv->is_urgency_hint_active = TRUE;
			urgency_hint_window_count++;
			if (led_pattern_name && miaouw_osso_context && !is_led_blinking) {
				osso_rpc_run_system(miaouw_osso_context, "com.nokia.mce", "/com/nokia/mce/request", "com.nokia.mce.request",
					     "req_led_pattern_activate", &retval,  DBUS_TYPE_STRING, led_pattern_name, DBUS_TYPE_INVALID);
				is_led_blinking = TRUE;
			}
#if HILDON == 1
		} else if (!is_display_on && led_pattern_name && miaouw_osso_context && !is_led_blinking) {
			osso_rpc_run_system(miaouw_osso_context, "com.nokia.mce", "/com/nokia/mce/request", "com.nokia.mce.request",
				     "req_led_pattern_activate", &retval,  DBUS_TYPE_STRING, led_pattern_name, DBUS_TYPE_INVALID);
			is_led_blinking = TRUE;
#endif
		}
	}
}

guint miaouw_window_get_font_size(const MiaouwWindow* window) {
	return window->priv->font_size;
}

gboolean miaouw_window_is_fullscreen(const MiaouwWindow* window) {
	return window->priv->is_fullscreen;
}

void miaouw_window_set_font_size(MiaouwWindow* window, const guint pixels) {
	PangoFontDescription* font;

	window->priv->font_size = pixels;
	font = pango_font_description_new();
	pango_font_description_set_size(font, window->priv->font_size * PANGO_SCALE);
	g_signal_emit(window, signals[SIGNAL_CHANGE_FONT_SIZE], 0, font);
	pango_font_description_free(font);
}

void miaouw_window_set_hadjustment(MiaouwWindow* window, GtkAdjustment* hadjustment) {
	window->priv->hadjustment = hadjustment;
}

void miaouw_window_set_led_pattern_name(gchar* name) {
	if (led_pattern_name) {
		g_free(led_pattern_name);
	}
	led_pattern_name = g_strdup(name);
}

void miaouw_window_set_vadjustment(MiaouwWindow* window, GtkAdjustment* vadjustment) {
	window->priv->vadjustment = vadjustment;
}

static void class_init(gpointer klass, gpointer data) {
	GTK_WIDGET_CLASS(klass)->window_state_event = window_state_event;
	GTK_WIDGET_CLASS(klass)->key_press_event = key_press_event;
	GTK_WIDGET_CLASS(klass)->visibility_notify_event = visibility_notify_event;
	GTK_WIDGET_CLASS(klass)->realize = realize;

	signals[SIGNAL_CHANGE_FONT_SIZE] = g_signal_new("change-font-size", MIAOUW_TYPE_WINDOW, 0, 0, NULL, NULL,
	                                        g_cclosure_marshal_VOID__POINTER, G_TYPE_NONE, 1, G_TYPE_POINTER);

#if HILDON == 1
	if (miaouw_osso_context) {
		osso_hw_set_display_event_cb(miaouw_osso_context, on_display_event, NULL);
	}
#endif
}

static void object_init(GTypeInstance* instance, gpointer klass) {
	MiaouwWindow* window = MIAOUW_WINDOW(instance);

	window->priv = g_new0(MiaouwWindowPrivate, 1);
	window->priv->font_size = 16;
}


static gboolean key_press_event(GtkWidget* widget, GdkEventKey* event) {
	MiaouwWindow* window;
	GtkWidgetClass* widget_class;
	
	window = MIAOUW_WINDOW(widget);
	switch (event->keyval) {
		case HILDON_HARDKEY_FULLSCREEN:
			if (window->priv->is_fullscreen) {
				gtk_window_unfullscreen(GTK_WINDOW(widget));
			} else {
				gtk_window_fullscreen(GTK_WINDOW(widget));
			}
			break;
		case HILDON_HARDKEY_INCREASE:
			if (window->priv->font_size < 24) {
				miaouw_window_set_font_size(window, window->priv->font_size + 4);
			}
			break;
		case HILDON_HARDKEY_DECREASE:
			if (window->priv->font_size > 8) {
				miaouw_window_set_font_size(window, window->priv->font_size - 4);
			}
			break;
		case HILDON_HARDKEY_UP:
			if (window->priv->vadjustment) {
				scroll_backwards(window->priv->vadjustment);
			}
			break;			
		case HILDON_HARDKEY_DOWN:
			if (window->priv->vadjustment) {
				scroll_forwards(window->priv->vadjustment);
			}				
			break;
		case HILDON_HARDKEY_LEFT:
			if (window->priv->hadjustment) {
				scroll_backwards(window->priv->hadjustment);
			}
			break;			
		case HILDON_HARDKEY_RIGHT:
			if (window->priv->hadjustment) {
				scroll_forwards(window->priv->hadjustment);
			}			
			break;
	}
	
	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_WINDOW)));
	if (widget_class->key_press_event) {

		return widget_class->key_press_event(widget, event);
	}
	
	return FALSE;
}

static void scroll_backwards(GtkAdjustment* adjustment) {
	if (gtk_adjustment_get_value(adjustment) - adjustment->page_increment > 0) {
		gtk_adjustment_set_value(adjustment, gtk_adjustment_get_value(adjustment) - adjustment->page_increment);
	} else {
		gtk_adjustment_set_value(adjustment, 0);
	}
}

static void scroll_forwards(GtkAdjustment* adjustment) {
	if (gtk_adjustment_get_value(adjustment) + adjustment->page_increment < adjustment->upper - adjustment->page_size) {
		gtk_adjustment_set_value(adjustment, gtk_adjustment_get_value(adjustment) + adjustment->page_increment);
	} else {
		gtk_adjustment_set_value(adjustment, adjustment->upper - adjustment->page_size);			
	}
}
	
#if HILDON == 1

static void on_display_event(osso_display_state_t state, gpointer data) {
	osso_rpc_t retval;

	if (state == OSSO_DISPLAY_ON) {
		if (led_pattern_name && !is_display_on && urgency_hint_window_count == 0 && is_led_blinking) {
			osso_rpc_run_system(miaouw_osso_context, "com.nokia.mce", "/com/nokia/mce/request", "com.nokia.mce.request",
			     "req_led_pattern_deactivate", &retval,  DBUS_TYPE_STRING, led_pattern_name, DBUS_TYPE_INVALID);
			is_led_blinking = FALSE;
		}
		is_display_on = TRUE;
	} else {
		is_display_on = FALSE;
	}
}

#endif

void realize(GtkWidget* widget) {
	GtkWidgetClass* widget_class;
	
	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_WINDOW)));
	if (widget_class->realize) {
		widget_class->realize(widget);
	}

	gdk_window_set_events(widget->window, gdk_window_get_events(widget->window) | GDK_VISIBILITY_NOTIFY_MASK);
}

static gboolean visibility_notify_event(GtkWidget* widget, GdkEventVisibility* event) {
	MiaouwWindow* window;
	osso_rpc_t retval;
	GtkWidgetClass* widget_class;
	
	window = MIAOUW_WINDOW(widget);
	if (event->state != GDK_VISIBILITY_FULLY_OBSCURED) {
		window->priv->is_visible = TRUE;
		if (window->priv->is_urgency_hint_active) {
			gtk_window_set_urgency_hint(GTK_WINDOW(widget), FALSE);
			window->priv->is_urgency_hint_active = FALSE;
			urgency_hint_window_count--;
			if (urgency_hint_window_count == 0 && led_pattern_name && is_led_blinking) {
				osso_rpc_run_system(miaouw_osso_context, "com.nokia.mce", "/com/nokia/mce/request", "com.nokia.mce.request",
				     "req_led_pattern_deactivate", &retval,  DBUS_TYPE_STRING, led_pattern_name, DBUS_TYPE_INVALID);
				is_led_blinking = FALSE;
			}
		}		
	} else {
		window->priv->is_visible = FALSE;
	}

	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_WINDOW)));
	if (widget_class->visibility_notify_event) {
		
		return widget_class->visibility_notify_event(widget, event);
	}
	
	return FALSE;
}

static gboolean window_state_event(GtkWidget* widget, GdkEventWindowState* event) {
	GtkWidgetClass* widget_class;

	if (event->changed_mask & GDK_WINDOW_STATE_FULLSCREEN) {
		MIAOUW_WINDOW(widget)->priv->is_fullscreen = event->new_window_state & GDK_WINDOW_STATE_FULLSCREEN;
	}

	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_WINDOW)));
	if (widget_class->window_state_event) {
		
		return widget_class->window_state_event(widget, event);
	}
	
	return FALSE;
}

