/*
 *  Miaouw - The Miaouw Library for Maemo Development
 *  Copyright (C) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <miaouw/miaouw.h>

#define DEFAULT_INTERVAL 50
#define DEFAULT_DECELERATION 0.7
#define DEFAULT_DRAGGING_STOPPED_DELAY 100

/**
 * SECTION:miaouwscrolledwindow
 * @short_description: Implements drag scrolling and kinetic scrolling
 * @see_also: #GtkScrolledWindow
 *
 * A scrolled window derived from #GtkScrolledWindow that implements drag scrolling and kinetic scrolling.
 * Can be used as a drop-in replacement for the existing #GtkScrolledWindow. 
 *
 * If a direct child of the #MiaouwScrolledWindow has its own window (InputOnly is enough for events),
 * it is automatically activated when added as a child. All motion events in that area will be used to
 * scroll.
 *
 * If some descendant widgets capture button press, button release and/or motion nofity events, an user
 * can not scroll the area by pressing those widgets (unless the widget is activated). #GtkButton is a typical
 * example of that. Usually that is the desired behaviour.
 *
 * Any widget can be registered to provide pointer events for the #MiaouwScrolledWindow by using
 * the #miaouw_scrolled_window_activate_scrolling function.
 * 
 **/
 
static GdkWindow* current_gdk_window;
static MiaouwScrolledWindow* current_scrolled_window;
static GtkWidget* current_widget;
static gboolean synthetized_crossing_event;

static GTree* activated_widgets;

struct _MiaouwScrolledWindowPrivate {
	/* Settings */
	guint interval;
	gdouble deceleration;
	gboolean drag_scrolling;
	gboolean kinetic_scrolling;
	guint32 dragging_stopped_delay;
	gboolean scrolling_hints;
	
	/* Temporary variables */
	gboolean dragged;
	gboolean press_received;
	GdkWindow* synthetic_crossing_event_window;

	/* Disabling twice happening scrolling adjustment */
	GtkAdjustment* hadjustment;
	GtkWidget* viewport;

	/* Motion scrolling */
	gint start_x;
	gint start_y;
	gint previous_x;
	gint previous_y;
	gint farest_x;
	gint farest_y;
	guint32 start_time;
	guint32 previous_time;
	guint32 farest_time;
	gboolean going_right;
	gboolean going_down;
	
	/* Kinetic scrolling */
	guint scrolling_timeout_id;
	gdouble horizontal_speed;
	gdouble vertical_speed;
	gdouble horizontal_deceleration;
	gdouble vertical_deceleration;
	
	/* Internal scrollbars */
	GdkWindow* vertical_scrollbar_window;
	GdkWindow* horizontal_scrollbar_window;
	gint vertical_scrollbar_size;
	gint horizontal_scrollbar_size;
	guint hide_scrollbars_timeout_id;
	GdkGC* hilight_gc;
	GdkGC* shadow_gc;
};

static gint compare_pointers(gconstpointer a, gconstpointer b, gpointer user_data) {
	return a - b;
}

static void disable_hadjustment(MiaouwScrolledWindow* scrolled_window) {
	GtkAdjustment* hadjustment;
	GtkWidget* viewport;

	if ((hadjustment = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolled_window))) &&
	    scrolled_window->priv->hadjustment != hadjustment) {
		scrolled_window->priv->hadjustment = hadjustment;
		scrolled_window->priv->viewport = NULL;
		viewport = GTK_WIDGET(scrolled_window);
		while (GTK_IS_BIN(viewport)) {
			viewport = gtk_bin_get_child(GTK_BIN(viewport));
			if (GTK_IS_VIEWPORT(viewport)) {
				scrolled_window->priv->viewport = viewport;
				break;
			}
		}
	}
	g_signal_handlers_block_matched(scrolled_window->priv->hadjustment, G_SIGNAL_MATCH_DATA, 0, 0, 0, 0, scrolled_window->priv->viewport);
}

static void enable_hadjustment(MiaouwScrolledWindow* scrolled_window) {
	g_signal_handlers_unblock_matched(scrolled_window->priv->hadjustment, G_SIGNAL_MATCH_DATA, 0, 0, 0, 0, scrolled_window->priv->viewport);
}

static gboolean on_expose_event(GtkWidget* widget, GdkEventExpose* event, MiaouwScrolledWindow* scrolled_window) {
	gboolean ret = FALSE;

	if (GTK_WIDGET_DRAWABLE(widget)) {
		if (event->window == scrolled_window->priv->horizontal_scrollbar_window) {
			if (scrolled_window->priv->horizontal_scrollbar_size) {
				gdk_draw_line(event->window, scrolled_window->priv->hilight_gc, 0, 0, scrolled_window->priv->horizontal_scrollbar_size - 1, 0);
				gdk_draw_line(event->window, scrolled_window->priv->hilight_gc, 0, 1, 0, 9);
				gdk_draw_line(event->window, scrolled_window->priv->shadow_gc, scrolled_window->priv->horizontal_scrollbar_size - 1, 1, scrolled_window->priv->horizontal_scrollbar_size - 1, 9);		
				gdk_draw_line(event->window, scrolled_window->priv->shadow_gc, 0, 9, scrolled_window->priv->horizontal_scrollbar_size - 1, 9);
			}
			
			ret = TRUE;
		} else if (event->window == scrolled_window->priv->vertical_scrollbar_window) {
			if (scrolled_window->priv->vertical_scrollbar_size) {
				gdk_draw_line(event->window, scrolled_window->priv->hilight_gc, 0, 0, 9, 0);
				gdk_draw_line(event->window, scrolled_window->priv->hilight_gc, 0, 1, 0, scrolled_window->priv->vertical_scrollbar_size - 1);
				gdk_draw_line(event->window, scrolled_window->priv->shadow_gc, 9, 1, 9, scrolled_window->priv->vertical_scrollbar_size - 1);		
				gdk_draw_line(event->window, scrolled_window->priv->shadow_gc, 0, scrolled_window->priv->vertical_scrollbar_size - 1, 9, scrolled_window->priv->vertical_scrollbar_size - 1);
			}

			ret = TRUE;		
		}
	}
  
	return ret;
}

static gboolean adjust_scrollbar(MiaouwScrolledWindow* scrolled_window, GdkWindow* scrollbar_window, GtkAdjustment* adjustment, gint* previous_size, gboolean horizontal) {
	GtkWidget* widget;
	gint x, y;
	gint size;
	double position;
	GtkWidget* window;

	if (adjustment->page_size >= adjustment->upper - adjustment->lower) {
		*previous_size = 0;
		
		return FALSE;
	}

	widget = GTK_WIDGET(scrolled_window);
	size = ((double)adjustment->page_size) / (adjustment->upper - adjustment->lower) * (horizontal ? widget->allocation.height : widget->allocation.width);
	if (size != *previous_size) {
		*previous_size = size;
		if (horizontal) {
			gdk_window_resize(scrollbar_window, 10, size);
			gdk_window_clear(scrollbar_window);
			gdk_draw_line(scrollbar_window, scrolled_window->priv->hilight_gc, 0, 0, 9, 0);
			gdk_draw_line(scrollbar_window, scrolled_window->priv->hilight_gc, 0, 1, 0, size - 1);
			gdk_draw_line(scrollbar_window, scrolled_window->priv->shadow_gc, 9, 1, 9, size - 1);		
			gdk_draw_line(scrollbar_window, scrolled_window->priv->shadow_gc, 0, size - 1, 9, size - 1);
		} else {
			gdk_window_resize(scrollbar_window, size, 10);
			gdk_window_clear(scrollbar_window);
			gdk_draw_line(scrollbar_window, scrolled_window->priv->hilight_gc, 0, 0, size - 1, 0);
			gdk_draw_line(scrollbar_window, scrolled_window->priv->hilight_gc, 0, 1, 0, 9);
			gdk_draw_line(scrollbar_window, scrolled_window->priv->shadow_gc, size - 1, 1, size - 1, 9);		
			gdk_draw_line(scrollbar_window, scrolled_window->priv->shadow_gc, 0, 9, size - 1, 9);
		}
	}

	position = (adjustment->value - adjustment->lower) / (adjustment->upper - adjustment->lower);
	window = gtk_widget_get_toplevel(widget);
	if (horizontal) {
		gtk_widget_translate_coordinates(widget, window, widget->allocation.width - 20, position * widget->allocation.height, &x, &y);
		gdk_window_move(scrollbar_window, x, y);	
	} else {
		gtk_widget_translate_coordinates(widget, window, position * widget->allocation.width, widget->allocation.height - 20, &x, &y);
		gdk_window_move(scrollbar_window, x, y);	
	}
	
	return TRUE;
}

static gboolean hide_scrollbars_timeout(gpointer data) {
	MiaouwScrolledWindow* scrolled_window = MIAOUW_SCROLLED_WINDOW(data);

	gdk_threads_enter();
	gdk_window_hide(scrolled_window->priv->vertical_scrollbar_window);	
	gdk_window_hide(scrolled_window->priv->horizontal_scrollbar_window);	

	scrolled_window->priv->hide_scrollbars_timeout_id = 0;
	gdk_threads_leave();

	return FALSE;
}

static gdouble calculate_timeout_scroll_values(gdouble old_value, gdouble upper_limit, gdouble* scrolling_speed_pointer, gdouble deceleration, gdouble* other_deceleration, gdouble normal_deceleration) {
	gdouble new_value = old_value;
	
	if (*scrolling_speed_pointer > deceleration ||
	    *scrolling_speed_pointer < -deceleration) {
		if (old_value + *scrolling_speed_pointer <= 0.0) {
			new_value = -1.0;
			*scrolling_speed_pointer = 0.0;
			*other_deceleration = normal_deceleration;
		} else if (old_value + *scrolling_speed_pointer >= upper_limit) {
			new_value = upper_limit;
			*scrolling_speed_pointer = 0.0;
			*other_deceleration = normal_deceleration;
		} else {
			new_value = old_value + *scrolling_speed_pointer;
		}
		if (*scrolling_speed_pointer > deceleration) {
			*scrolling_speed_pointer -= deceleration;
		} else if (*scrolling_speed_pointer < -deceleration) {
			*scrolling_speed_pointer += deceleration;
		}
	}
	
	return new_value;
}

static void do_timeout_scroll(MiaouwScrolledWindow* scrolled_window) {
	GtkAdjustment* hadjustment;
	GtkAdjustment* vadjustment;
	gdouble hvalue;
	gdouble vvalue;
	
	hadjustment = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolled_window));
	vadjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolled_window));
	hvalue = calculate_timeout_scroll_values(hadjustment->value, hadjustment->upper - hadjustment->page_size,
	                        		 &scrolled_window->priv->horizontal_speed,
						 scrolled_window->priv->horizontal_deceleration,
						 &scrolled_window->priv->vertical_deceleration,
						 scrolled_window->priv->deceleration);
	vvalue = calculate_timeout_scroll_values(vadjustment->value, vadjustment->upper - vadjustment->page_size,
	                			 &scrolled_window->priv->vertical_speed,
						 scrolled_window->priv->vertical_deceleration,
						 &scrolled_window->priv->horizontal_deceleration,
						 scrolled_window->priv->deceleration);
	if (vvalue != vadjustment->value) {
		if (hvalue != hadjustment->value) {
			disable_hadjustment(scrolled_window);
			gtk_adjustment_set_value(hadjustment, hvalue);
			enable_hadjustment(scrolled_window);
		}
		gtk_adjustment_set_value(vadjustment, vvalue);
	} else if (hvalue != hadjustment->value) {
		gtk_adjustment_set_value(hadjustment, hvalue);
	}

	adjust_scrollbar(scrolled_window, scrolled_window->priv->horizontal_scrollbar_window,
	                 gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolled_window)),
			 &scrolled_window->priv->horizontal_scrollbar_size, FALSE);
	adjust_scrollbar(scrolled_window, scrolled_window->priv->vertical_scrollbar_window,
	                 gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolled_window)),
			 &scrolled_window->priv->vertical_scrollbar_size, TRUE);
}

static gboolean timeout_scroll(gpointer data) {
	gboolean ret = TRUE;
	MiaouwScrolledWindow* scrolled_window = MIAOUW_SCROLLED_WINDOW(data);

	gdk_threads_enter();
	do_timeout_scroll(scrolled_window);

	if (scrolled_window->priv->vertical_speed < scrolled_window->priv->deceleration &&
	    scrolled_window->priv->vertical_speed > -scrolled_window->priv->deceleration &&
	    scrolled_window->priv->horizontal_speed < scrolled_window->priv->deceleration &&
	    scrolled_window->priv->horizontal_speed > -scrolled_window->priv->deceleration) {
		scrolled_window->priv->scrolling_timeout_id = 0;
		if (!scrolled_window->priv->hide_scrollbars_timeout_id) {
			scrolled_window->priv->hide_scrollbars_timeout_id = g_timeout_add(500, hide_scrollbars_timeout, scrolled_window);
		}

		ret = FALSE;
	}
	gdk_threads_leave();
	
	return ret;
}

static gdouble calculate_motion_scroll_values(gdouble old_value, gdouble upper_limit, gint current_coordinate, gint previous_coordinate) {
	gdouble new_value = old_value;
	gint movement;
	
	movement = current_coordinate - previous_coordinate;

	if (old_value - movement < upper_limit) {
		new_value = old_value - movement;
	} else {
		new_value = upper_limit;			
	}
	
	return new_value;
}

static void do_motion_scroll(MiaouwScrolledWindow* scrolled_window, GtkWidget* widget, gint x, gint y, guint32 timestamp) {
	GtkAdjustment* hadjustment;
	GtkAdjustment* vadjustment;
	gdouble hvalue;
	gdouble vvalue;
	
	if (scrolled_window->priv->dragged ||
	    gtk_drag_check_threshold(widget, scrolled_window->priv->start_x, scrolled_window->priv->start_y, x, y)) {
		if (timestamp - scrolled_window->priv->previous_time > scrolled_window->priv->dragging_stopped_delay ||
		    !scrolled_window->priv->dragged) {
			scrolled_window->priv->dragged = TRUE;
			scrolled_window->priv->going_right = scrolled_window->priv->start_x < x;
			scrolled_window->priv->going_down = scrolled_window->priv->start_y < y;
			scrolled_window->priv->start_x = scrolled_window->priv->farest_x = x;
			scrolled_window->priv->start_y = scrolled_window->priv->farest_y = y;
			scrolled_window->priv->start_time = scrolled_window->priv->farest_time = timestamp;			
		} else {
			if ((scrolled_window->priv->going_right && x > scrolled_window->priv->farest_x) ||
			    (!scrolled_window->priv->going_right && x < scrolled_window->priv->farest_x)) {
				scrolled_window->priv->farest_x = x;
				scrolled_window->priv->farest_time = timestamp;
			}
			if ((scrolled_window->priv->going_down && y > scrolled_window->priv->farest_y) ||
			    (!scrolled_window->priv->going_down && y < scrolled_window->priv->farest_y)) {
				scrolled_window->priv->farest_y = y;
				scrolled_window->priv->farest_time = timestamp;
			}
			if (gtk_drag_check_threshold(widget, scrolled_window->priv->farest_x, scrolled_window->priv->farest_y, x, y)) {
				scrolled_window->priv->start_x = scrolled_window->priv->farest_x;
				scrolled_window->priv->farest_x = x;
				scrolled_window->priv->start_y = scrolled_window->priv->farest_y;
				scrolled_window->priv->farest_y = y;
				scrolled_window->priv->start_time = scrolled_window->priv->farest_time;			
				scrolled_window->priv->farest_time = timestamp;			
				scrolled_window->priv->going_right = scrolled_window->priv->start_x < x;
				scrolled_window->priv->going_down = scrolled_window->priv->start_y < y;
			}
		}

		hadjustment = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolled_window));
		vadjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolled_window));
		hvalue = calculate_motion_scroll_values(hadjustment->value, hadjustment->upper - hadjustment->page_size, x, scrolled_window->priv->previous_x);
		vvalue = calculate_motion_scroll_values(vadjustment->value, vadjustment->upper - vadjustment->page_size, y, scrolled_window->priv->previous_y);
		if (vvalue != vadjustment->value) {
			if (hvalue != hadjustment->value) {
				disable_hadjustment(scrolled_window);
				gtk_adjustment_set_value(hadjustment, hvalue);
				enable_hadjustment(scrolled_window);
			}
			gtk_adjustment_set_value(vadjustment, vvalue);
		} else if (hvalue != hadjustment->value) {
			gtk_adjustment_set_value(hadjustment, hvalue);
		}	
	}

	scrolled_window->priv->previous_y = y;
	scrolled_window->priv->previous_x = x;
	scrolled_window->priv->previous_time = timestamp;

	adjust_scrollbar(scrolled_window, scrolled_window->priv->horizontal_scrollbar_window,
	                 gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolled_window)),
			 &scrolled_window->priv->horizontal_scrollbar_size, FALSE);
	adjust_scrollbar(scrolled_window, scrolled_window->priv->vertical_scrollbar_window,
	                 gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolled_window)),
			 &scrolled_window->priv->vertical_scrollbar_size, TRUE);
}

static gboolean button_press_event(GtkWidget* widget, GdkEventButton* event, MiaouwScrolledWindow* scrolled_window) {
	gint x;
	gint y;
	GdkModifierType mask;

	if (!scrolled_window->priv->drag_scrolling) {

		return FALSE;
	}

	scrolled_window->priv->press_received = TRUE;

	if (event->time - scrolled_window->priv->previous_time < scrolled_window->priv->dragging_stopped_delay &&
	    gtk_drag_check_threshold(widget, scrolled_window->priv->previous_x, scrolled_window->priv->previous_y, x, y)) {
		if (scrolled_window->priv->scrolling_timeout_id) {
			g_source_remove(scrolled_window->priv->scrolling_timeout_id);
			scrolled_window->priv->scrolling_timeout_id = 0;
		}
		gdk_window_get_pointer(GTK_WIDGET(scrolled_window)->window, &x, &y, &mask);
/*		do_motion_scroll(scrolled_window, widget, x, y, event->time); */
	} else {
		if (scrolled_window->priv->scrolling_timeout_id) {
			g_source_remove(scrolled_window->priv->scrolling_timeout_id);
			scrolled_window->priv->scrolling_timeout_id = 0;
			scrolled_window->priv->previous_time = 0;
		} else {
			scrolled_window->priv->dragged = FALSE;
			scrolled_window->priv->previous_time = event->time;
		}
		gdk_window_get_pointer(GTK_WIDGET(scrolled_window)->window, &x, &y, &mask);
		scrolled_window->priv->start_x = scrolled_window->priv->previous_x = scrolled_window->priv->farest_x = x;
		scrolled_window->priv->start_y = scrolled_window->priv->previous_y = scrolled_window->priv->farest_y = y;
		scrolled_window->priv->start_time  = event->time;
	}

	if (scrolled_window->priv->scrolling_hints && !GTK_SCROLLED_WINDOW(scrolled_window)->hscrollbar_visible &&
	    adjust_scrollbar(scrolled_window, scrolled_window->priv->horizontal_scrollbar_window,
		             gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolled_window)),
	                     &scrolled_window->priv->horizontal_scrollbar_size, FALSE)) {
		gdk_window_raise(scrolled_window->priv->horizontal_scrollbar_window);
		gdk_window_show(scrolled_window->priv->horizontal_scrollbar_window);
	}
	if (scrolled_window->priv->scrolling_hints && !GTK_SCROLLED_WINDOW(scrolled_window)->vscrollbar_visible &&
	    adjust_scrollbar(scrolled_window, scrolled_window->priv->vertical_scrollbar_window,
		             gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolled_window)),
	                     &scrolled_window->priv->vertical_scrollbar_size, TRUE)) {
	 	gdk_window_raise(scrolled_window->priv->vertical_scrollbar_window);
		gdk_window_show(scrolled_window->priv->vertical_scrollbar_window);				 
	}
	if (scrolled_window->priv->hide_scrollbars_timeout_id) {
		g_source_remove(scrolled_window->priv->hide_scrollbars_timeout_id);
		scrolled_window->priv->hide_scrollbars_timeout_id = 0;
	}

	return FALSE;
}

#define ABSOLUTE_VALUE(x) ((x) < 0 ? -(x) : (x))

static gboolean button_release_event(GtkWidget* widget, GdkEventButton* event, MiaouwScrolledWindow* scrolled_window) {
	gint x;
	gint y;
	GdkModifierType mask;

	gdk_window_get_pointer(GTK_WIDGET(scrolled_window)->window, &x, &y, &mask);
	if (scrolled_window->priv->press_received &&
	    gtk_drag_check_threshold(widget, scrolled_window->priv->start_x, scrolled_window->priv->start_y, x, y)) {
		scrolled_window->priv->dragged = TRUE;
	}
	
	if (scrolled_window->priv->press_received && scrolled_window->priv->kinetic_scrolling &&
	    event->time - scrolled_window->priv->previous_time < scrolled_window->priv->dragging_stopped_delay) {
		scrolled_window->priv->vertical_speed = (gdouble)(scrolled_window->priv->start_y - y) / (event->time - scrolled_window->priv->start_time) * scrolled_window->priv->interval;
		scrolled_window->priv->horizontal_speed = (gdouble)(scrolled_window->priv->start_x - x) / (event->time - scrolled_window->priv->start_time) * scrolled_window->priv->interval;
		if (ABSOLUTE_VALUE(scrolled_window->priv->vertical_speed) > ABSOLUTE_VALUE(scrolled_window->priv->horizontal_speed)) {
			scrolled_window->priv->vertical_deceleration = scrolled_window->priv->deceleration;
			scrolled_window->priv->horizontal_deceleration = scrolled_window->priv->deceleration * ABSOLUTE_VALUE(scrolled_window->priv->horizontal_speed / scrolled_window->priv->vertical_speed);
		} else {
			scrolled_window->priv->horizontal_deceleration = scrolled_window->priv->deceleration;
			scrolled_window->priv->vertical_deceleration = scrolled_window->priv->deceleration * ABSOLUTE_VALUE(scrolled_window->priv->vertical_speed / scrolled_window->priv->horizontal_speed);
		}
		scrolled_window->priv->scrolling_timeout_id = g_timeout_add(scrolled_window->priv->interval, timeout_scroll, scrolled_window);

		do_timeout_scroll(scrolled_window);
	} else if (!scrolled_window->priv->hide_scrollbars_timeout_id) {
		scrolled_window->priv->hide_scrollbars_timeout_id = g_timeout_add(500, hide_scrollbars_timeout, scrolled_window);
	}
	scrolled_window->priv->previous_x = x;
	scrolled_window->priv->previous_y = y;
	scrolled_window->priv->previous_time = event->time;

    	scrolled_window->priv->press_received = FALSE;

	return FALSE;
}

static gboolean motion_notify_event(GtkWidget* widget, GdkEventMotion* event, MiaouwScrolledWindow* scrolled_window) {
	gint x;
	gint y;
	GdkModifierType mask;

	if (scrolled_window->priv->press_received) {
		gdk_window_get_pointer(GTK_WIDGET(scrolled_window)->window, &x, &y, &mask);
		do_motion_scroll(scrolled_window, widget, x, y, event->time);
	}

	return FALSE;
}

static gboolean event_handler(GdkEvent* event, MiaouwEventHandlerState* state, gpointer user_data) {
	gboolean stop_propagating;
	GdkEventCrossing crossing;
	GdkWindow* window;
	
	stop_propagating = FALSE;

	if (event->type == GDK_BUTTON_PRESS) {

		gdk_window_get_user_data(event->button.window, (gpointer)&current_widget);

		if ((current_scrolled_window = g_tree_lookup(activated_widgets, current_widget))) {
			current_gdk_window = event->button.window;
	
		stop_propagating = button_press_event(current_widget, &event->button, current_scrolled_window);
		} else {
			current_gdk_window = NULL;
		}

	} else if (event->any.window == current_gdk_window) {

		if (event->type == GDK_MOTION_NOTIFY) {
			if (current_scrolled_window->priv->dragged) {
				stop_propagating = motion_notify_event(current_widget, &event->motion, current_scrolled_window);
			} else {
				stop_propagating = motion_notify_event(current_widget, &event->motion, current_scrolled_window);
				if (current_scrolled_window->priv->dragged) {
					crossing.type = GDK_LEAVE_NOTIFY;
					crossing.window = event->motion.window;
					crossing.send_event = event->motion.send_event;
					crossing.subwindow = GTK_WIDGET(current_scrolled_window)->window;
					crossing.time = event->motion.time;
					crossing.x = event->motion.x;
					crossing.y = event->motion.y;
					crossing.x_root = event->motion.x_root;
					crossing.y_root = event->motion.y_root;
					crossing.mode = GDK_CROSSING_GRAB;
					crossing.detail = GDK_NOTIFY_ANCESTOR;
					crossing.focus = TRUE;
					crossing.state = event->motion.state;

					gtk_main_do_event((GdkEvent*)&crossing);
					synthetized_crossing_event = TRUE;
				}
			}
		} else if ((event->type == GDK_ENTER_NOTIFY || event->type == GDK_LEAVE_NOTIFY) &&
	        	   synthetized_crossing_event) {

			stop_propagating = TRUE;
		} else if (event->type == GDK_BUTTON_RELEASE) {

			stop_propagating = button_release_event(current_widget, &event->button, current_scrolled_window);   
		}
	}

	
	if (!stop_propagating) {
		miaouw_event_handler_next(event, state);
	}

	if (event->type == GDK_BUTTON_RELEASE && event->button.window == current_gdk_window) {

		crossing.type = GDK_ENTER_NOTIFY;
		crossing.window = event->button.window;
		crossing.send_event = event->button.send_event;
		crossing.subwindow = GTK_WIDGET(current_scrolled_window)->window;
		crossing.time = event->button.time;
		crossing.x = event->button.x;
		crossing.y = event->button.y;
		crossing.x_root = event->button.x_root;
		crossing.y_root = event->button.y_root;
		crossing.mode = GDK_CROSSING_UNGRAB;
		crossing.detail = GDK_NOTIFY_ANCESTOR;
		crossing.focus = TRUE;
		crossing.state = event->button.state;

		gtk_main_do_event((GdkEvent*)&crossing);
		synthetized_crossing_event = FALSE;
		
		current_gdk_window;
	}
	
	return stop_propagating;
}

static void add(GtkContainer* container, GtkWidget* widget) {
	GtkContainerClass* container_class;
	
	miaouw_scrolled_window_activate_scrolling(MIAOUW_SCROLLED_WINDOW(container), widget);
	
	container_class = GTK_CONTAINER_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_SCROLLED_WINDOW)));
	if (container_class->add) {
		container_class->add(container, widget);
	}
	
}

static void realize(GtkWidget* widget) {
	GtkWidgetClass* widget_class;
	MiaouwScrolledWindow* scrolled_window;
	GdkWindowAttr attr;
	GdkColor color;

	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_SCROLLED_WINDOW)));
	if (widget_class->realize) {
		widget_class->realize(widget);
	}

	widget->window = gtk_widget_get_parent_window (widget);
	g_object_ref (widget->window);
	
	scrolled_window = MIAOUW_SCROLLED_WINDOW(widget);
	
	attr.height = attr.width = 10;
	attr.event_mask = GDK_EXPOSURE_MASK;
	attr.wclass = GDK_INPUT_OUTPUT;
	attr.window_type = GDK_WINDOW_CHILD;
	attr.override_redirect = TRUE;
	scrolled_window->priv->vertical_scrollbar_window = gdk_window_new(widget->window, &attr, 0);
	scrolled_window->priv->horizontal_scrollbar_window = gdk_window_new(widget->window, &attr, 0);
	
	gdk_window_set_user_data(scrolled_window->priv->vertical_scrollbar_window, widget);
	gdk_window_set_user_data(scrolled_window->priv->horizontal_scrollbar_window, widget);
	g_signal_connect(widget, "expose-event", G_CALLBACK(on_expose_event), scrolled_window);

	color.red = color.green = color.blue = 0x9999;
	gdk_rgb_find_color(gdk_colormap_get_system(), &color);
	gdk_window_set_background(scrolled_window->priv->vertical_scrollbar_window, &color);
	gdk_window_set_background(scrolled_window->priv->horizontal_scrollbar_window, &color);	
	
	scrolled_window->priv->hilight_gc = gdk_gc_new(widget->window);
	color.red = color.green = color.blue = 0xcccc;
	gdk_gc_set_rgb_fg_color(scrolled_window->priv->hilight_gc, &color);
	scrolled_window->priv->shadow_gc = gdk_gc_new(widget->window);
	color.red = color.green = color.blue = 0x6666;
	gdk_gc_set_rgb_fg_color(scrolled_window->priv->shadow_gc, &color);

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
}

static void unrealize(GtkWidget* widget) {
	MiaouwScrolledWindow* scrolled_window;
	GtkWidgetClass* widget_class;

	scrolled_window = MIAOUW_SCROLLED_WINDOW(widget);
	
	if (scrolled_window->priv->vertical_scrollbar_window) {
		gdk_window_set_user_data(scrolled_window->priv->vertical_scrollbar_window, NULL);
		gdk_window_destroy(scrolled_window->priv->vertical_scrollbar_window);
		scrolled_window->priv->vertical_scrollbar_window = NULL;
	}
	if (scrolled_window->priv->horizontal_scrollbar_window) {
		gdk_window_set_user_data(scrolled_window->priv->horizontal_scrollbar_window, NULL);
		gdk_window_destroy(scrolled_window->priv->horizontal_scrollbar_window);
		scrolled_window->priv->horizontal_scrollbar_window = NULL;
	}
	if (scrolled_window->priv->hilight_gc) {
		g_object_unref(scrolled_window->priv->hilight_gc);
		scrolled_window->priv->hilight_gc = NULL;
	}
	if (scrolled_window->priv->shadow_gc) {
		g_object_unref(scrolled_window->priv->shadow_gc);
		scrolled_window->priv->shadow_gc = NULL;
	}
	
	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_SCROLLED_WINDOW)));
	if (widget_class->unrealize) {
		widget_class->unrealize(widget);
	}

	GTK_WIDGET_UNSET_FLAGS (widget, GTK_REALIZED);
}

static void dispose(GObject* object) {
	MiaouwScrolledWindow* scrolled_window;
	MiaouwScrolledWindowClass* klass;  

	scrolled_window = MIAOUW_SCROLLED_WINDOW(object);
	if (scrolled_window->priv->scrolling_timeout_id) {
		g_source_remove(scrolled_window->priv->scrolling_timeout_id);
		scrolled_window->priv->scrolling_timeout_id = 0;
	}
	if (scrolled_window->priv->hide_scrollbars_timeout_id) {
		g_source_remove(scrolled_window->priv->hide_scrollbars_timeout_id);
		scrolled_window->priv->hide_scrollbars_timeout_id = 0;
	}

	klass = MIAOUW_SCROLLED_WINDOW_GET_CLASS(object);
	G_OBJECT_CLASS(g_type_class_peek_parent(klass))->dispose(object);
}

static void class_init(gpointer klass, gpointer data) {
	G_OBJECT_CLASS(klass)->dispose = dispose;
	GTK_WIDGET_CLASS(klass)->realize = realize;
	GTK_CONTAINER_CLASS(klass)->add = add;

	activated_widgets = g_tree_new((GCompareFunc)compare_pointers);
	current_gdk_window = NULL;

	miaouw_event_handler_append(event_handler, NULL);
}

static void object_init(GTypeInstance* instance, gpointer klass) {
	MiaouwScrolledWindow* scrolled_window = MIAOUW_SCROLLED_WINDOW(instance);

	scrolled_window->priv = g_new0(MiaouwScrolledWindowPrivate, 1);
	scrolled_window->priv->interval = DEFAULT_INTERVAL;
	scrolled_window->priv->deceleration = DEFAULT_DECELERATION;
	scrolled_window->priv->drag_scrolling = TRUE;
	scrolled_window->priv->kinetic_scrolling = TRUE;
	scrolled_window->priv->dragging_stopped_delay = DEFAULT_DRAGGING_STOPPED_DELAY;
}

GType miaouw_scrolled_window_get_type() {
	static GType type = 0;
	static const GTypeInfo info = {
		sizeof (MiaouwScrolledWindowClass),
		NULL,   /* base_init */
		NULL,   /* base_finalize */
		class_init,
		NULL,   /* class_finalize */
		NULL,   /* class_data */
		sizeof (MiaouwScrolledWindow),
		0,
		object_init,
		NULL
	};

	if (!type) {
		type = g_type_register_static(GTK_TYPE_SCROLLED_WINDOW, "MiaouwScrolledWindow", &info, 0);
	}

	return type;
}

/**
 * miaouw_scrolled_window_new:
 * @hadjustment: a horizontal #GtkAdjustment
 * @vadjustment: a vertical #GtkAdjustment
 *
 * Similar function than the #gtk_scrolled_window_new.
 **/

GtkWidget* miaouw_scrolled_window_new(GtkAdjustment* hadjustment, GtkAdjustment* vadjustment) {
	if (hadjustment) {
	
		g_return_val_if_fail (GTK_IS_ADJUSTMENT (hadjustment), NULL);
	}
	if (vadjustment) {
	
		g_return_val_if_fail (GTK_IS_ADJUSTMENT (vadjustment), NULL);
	}
	
	return gtk_widget_new(MIAOUW_TYPE_SCROLLED_WINDOW, "hadjustment", hadjustment, "vadjustment", vadjustment,
	                      "hscrollbar-policy", GTK_POLICY_NEVER, "vscrollbar-policy", GTK_POLICY_NEVER, NULL);
}

/**
 * miaouw_scrolled_window_activate_scrolling:
 * @scrolled_window: a #MiaouwScrolledWindow
 * @widget: a #GtkWidget of which area is made active event source for drag and kinetic scrolling.
 *
 * Activates the widget so that pointer motion events inside the widget are used to scroll the #MiaouwScrolledWindow.
 * The widget can be a child of the #MiaouwScrolledWindow or even a separate widget ("touchpad" style).
 *
 * The direct child of the #MiaouwScrolledWindow (typically #GtkViewport) is activated automatically when added.
 * This function has to be
 * used if indirect descendant widgets are stopping propagation of the button press and release as well as motion events
 * (for example GtkButton is doing so) but scrolling should be possible inside their area too.
 *
 * This function adds #GDK_BUTTON_PRESS_MASK, #GDK_BUTTON_RELEASE_MASK, #GDK_POINTER_MOTION_MASK, and
 * #GDK_MOTION_HINT_MAKS into the widgets event mask.
 */

void miaouw_scrolled_window_activate_scrolling(MiaouwScrolledWindow* scrolled_window, GtkWidget* widget) {
	gtk_widget_add_events(widget, GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK);
	g_tree_insert(activated_widgets, widget, scrolled_window);
}

void miaouw_scrolled_window_set_drag_scrolling(MiaouwScrolledWindow* scrolled_window, gboolean drag_scrolling) {
	g_return_if_fail(MIAOUW_IS_SCROLLED_WINDOW(scrolled_window));

	if (scrolled_window->priv->drag_scrolling && !drag_scrolling) {
		if (scrolled_window->priv->scrolling_timeout_id) {
			g_source_remove(scrolled_window->priv->scrolling_timeout_id);
			scrolled_window->priv->scrolling_timeout_id = 0;
			scrolled_window->priv->previous_time = 0;
		}

		gdk_window_hide(scrolled_window->priv->vertical_scrollbar_window);	
		gdk_window_hide(scrolled_window->priv->horizontal_scrollbar_window);
		if (scrolled_window->priv->hide_scrollbars_timeout_id) {
			g_source_remove(scrolled_window->priv->hide_scrollbars_timeout_id);
			scrolled_window->priv->hide_scrollbars_timeout_id = 0;
		}

		scrolled_window->priv->press_received = FALSE;
	}

	scrolled_window->priv->drag_scrolling = drag_scrolling;
}

gboolean miaouw_scrolled_window_is_dragged(MiaouwScrolledWindow* scrolled_window) {
	g_return_if_fail(MIAOUW_IS_SCROLLED_WINDOW(scrolled_window));

	return scrolled_window->priv->dragged;
}

void miaouw_scrolled_window_set_scrolling_hints(MiaouwScrolledWindow* scrolled_window, gboolean enabled) {
	scrolled_window->priv->scrolling_hints = enabled;
}

