/*
 *  Miaouw - The Miaouw Library for Maemo Development
 *  Copyright (C) 2008 Henrik Hedberg <hhedberg@innologies.fi>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
 
#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <errno.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include <miaouw/miaouwgestureeventbox.h>
#include "miaouwmarshalers.h"

/**
 * SECTION:miaouwgestureeventbox
 * @short_description: An event box that interprets gestures
 * @see_also: #GtkEventBox
 *
 * An event box that interprets simple (mouse / pointer / stylus / finger) gestures.
 **/
 
enum {
	SIGNAL_GESTURE_NOTIFY,
	SIGNAL_GESTURE_EVENT,
	SIGNAL_COUNT
};

static guint signals[SIGNAL_COUNT];

struct _MiaouwGestureEventBoxPrivate {
	GtkWidget* toplevel_window;
	gboolean press_received;
	gint start_x;
	gint start_y;
	MiaouwGestureType current_gesture;
	MiaouwGestureType previous_gesture;
	gint previous_x;
	gint previous_y;
};

static gboolean button_press_event(GtkWidget* widget, GdkEventButton* event) {
	MiaouwGestureEventBox* gesture_event_box;
	GdkModifierType mask;
	GtkWidgetClass* widget_class;

	gesture_event_box = MIAOUW_GESTURE_EVENT_BOX(widget);
	if (event->button == 1) {
		gdk_window_get_pointer(gesture_event_box->priv->toplevel_window->window,
	                	       &(gesture_event_box->priv->start_x), &(gesture_event_box->priv->start_y), &mask);
		gesture_event_box->priv->press_received = TRUE;
		gesture_event_box->priv->previous_x = gesture_event_box->priv->start_x;
		gesture_event_box->priv->previous_y = gesture_event_box->priv->start_y;
		gesture_event_box->priv->previous_gesture = gesture_event_box->priv->current_gesture = MIAOUW_GESTURE_NONE;
	}
	
	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_GESTURE_EVENT_BOX)));
	if (widget_class->button_press_event) {
		widget_class->button_press_event(widget, event);
	}

	return FALSE;
}

#define ABSOLUTE_VALUE(x) ((x) < 0 ? -(x) : (x))

static void send_gesture_notify(MiaouwGestureEventBox* gesture_event_box, MiaouwGestureType gesture, gint x, gint y) {
	MiaouwGestureEvent gesture_event;

	if (gesture_event_box->priv->previous_gesture != MIAOUW_GESTURE_NONE) {
			gesture = MIAOUW_GESTURE_FAILED;
	}
	
	gesture_event.previous_gesture = gesture_event_box->priv->previous_gesture = gesture_event_box->priv->current_gesture;
	gesture_event.gesture = gesture_event_box->priv->current_gesture = gesture;
	gesture_event.previous_start_x = gesture_event_box->priv->previous_x;
	gesture_event.previous_start_y = gesture_event_box->priv->previous_y;
	gesture_event.x = gesture_event_box->priv->previous_x = x;
	gesture_event.y = gesture_event_box->priv->previous_y = y;
	gesture_event.start_x = gesture_event_box->priv->start_x;
	gesture_event.start_y = gesture_event_box->priv->start_y;

	g_signal_emit(gesture_event_box, signals[SIGNAL_GESTURE_NOTIFY], 0, &gesture_event);
}

static void do_doing(MiaouwGestureEventBox* gesture_event_box, gint x, gint y) {
	gint threshold;
	MiaouwGestureEvent gesture_event;

	threshold = MIAOUW_GESTURE_EVENT_BOX_GET_CLASS(gesture_event_box)->threshold;

	if (((gesture_event_box->priv->current_gesture == MIAOUW_GESTURE_RIGHT &&
	    x > gesture_event_box->priv->previous_x) ||
	     ((gesture_event_box->priv->current_gesture == MIAOUW_GESTURE_LEFT &&
	    x < gesture_event_box->priv->previous_x))) &&
	    ABSOLUTE_VALUE(gesture_event_box->priv->previous_y - y) < threshold) {
		gesture_event_box->priv->previous_x = x;
	} else if (((gesture_event_box->priv->current_gesture == MIAOUW_GESTURE_DOWN &&
	    y > gesture_event_box->priv->previous_y) ||
	     ((gesture_event_box->priv->current_gesture == MIAOUW_GESTURE_UP &&
	    y < gesture_event_box->priv->previous_y))) &&
	    ABSOLUTE_VALUE(gesture_event_box->priv->previous_x - x) < threshold) {
		gesture_event_box->priv->previous_y = y;
	} else if (gesture_event_box->priv->previous_x - x > threshold &&
	           ABSOLUTE_VALUE(gesture_event_box->priv->previous_y - y) < threshold) {
		send_gesture_notify(gesture_event_box, MIAOUW_GESTURE_LEFT, x, y);
	} else if (gesture_event_box->priv->previous_x - x < -threshold &&
	           ABSOLUTE_VALUE(gesture_event_box->priv->previous_y - y) < threshold) {
		send_gesture_notify(gesture_event_box, MIAOUW_GESTURE_RIGHT, x, y);
	} else if (gesture_event_box->priv->previous_y - y > threshold &&
	           ABSOLUTE_VALUE(gesture_event_box->priv->previous_x - x) < threshold) {
		send_gesture_notify(gesture_event_box, MIAOUW_GESTURE_UP, x, y);
	} else if (gesture_event_box->priv->previous_y - y < -threshold &&
	           ABSOLUTE_VALUE(gesture_event_box->priv->previous_x - x) < threshold) {
		send_gesture_notify(gesture_event_box, MIAOUW_GESTURE_DOWN, x, y);
	}		
}

static gboolean motion_notify_event(GtkWidget* widget, GdkEventMotion* event) {
	MiaouwGestureEventBox* gesture_event_box;
	gint x, y;
	GdkModifierType mask;
	GtkWidgetClass* widget_class;
	
	gesture_event_box = MIAOUW_GESTURE_EVENT_BOX(widget);
	if (gesture_event_box->priv->press_received && gesture_event_box->priv->current_gesture != MIAOUW_GESTURE_FAILED) {
		gdk_window_get_pointer(gesture_event_box->priv->toplevel_window->window, &x, &y, &mask); 
		do_doing(gesture_event_box, x, y);
	}

	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_GESTURE_EVENT_BOX)));
	if (widget_class->motion_notify_event) {
		widget_class->motion_notify_event(widget, event);
	}
	
	return FALSE;
}

static gboolean button_release_event(GtkWidget* widget, GdkEventButton* event) {
	MiaouwGestureEventBox* gesture_event_box;
	gint x, y;
	GdkModifierType mask;
	MiaouwGestureEvent gesture_event;
	GtkWidgetClass* widget_class;

	gesture_event_box = MIAOUW_GESTURE_EVENT_BOX(widget);
	if (gesture_event_box->priv->press_received && gesture_event_box->priv->current_gesture != MIAOUW_GESTURE_FAILED) {
		gdk_window_get_pointer(gesture_event_box->priv->toplevel_window->window, &x, &y, &mask); 
		do_doing(gesture_event_box, x, y);

		if (gesture_event_box->priv->current_gesture != MIAOUW_GESTURE_FAILED &&
		    gesture_event_box->priv->current_gesture != MIAOUW_GESTURE_NONE) {
			gesture_event.gesture = gesture_event_box->priv->current_gesture;
			gesture_event.previous_gesture = gesture_event_box->priv->previous_gesture;
			gesture_event.x = x;
			gesture_event.y = y;
			gesture_event.start_x = gesture_event_box->priv->start_x;
			gesture_event.start_y = gesture_event_box->priv->start_y;
			gesture_event.previous_start_x = gesture_event_box->priv->previous_x;
			gesture_event.previous_start_y = gesture_event_box->priv->previous_y;

			g_signal_emit(gesture_event_box, signals[SIGNAL_GESTURE_EVENT], 0, &gesture_event);
		}
	}

	gesture_event_box->priv->press_received = FALSE;

	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_GESTURE_EVENT_BOX)));
	if (widget_class->button_release_event) {
		widget_class->button_release_event(widget, event);
	}

	return FALSE;
}

static void hierarchy_changed(GtkWidget* widget, GtkWidget* previous_toplevel) {
	GtkWidgetClass* widget_class;

	MIAOUW_GESTURE_EVENT_BOX(widget)->priv->toplevel_window = gtk_widget_get_toplevel(widget);

	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_GESTURE_EVENT_BOX)));
	if (widget_class->hierarchy_changed) {
		widget_class->hierarchy_changed(widget, previous_toplevel);
	}
}

static void miaouw_gesture_event_box_class_init(gpointer klass, gpointer data) {
	GtkWidgetClass *widget_class;
	GtkSettings* settings;
	gint threshold;
	
	widget_class = GTK_WIDGET_CLASS(klass);
	widget_class->button_press_event = button_press_event;
	widget_class->button_release_event = button_release_event;
	widget_class->motion_notify_event = motion_notify_event;
	widget_class->hierarchy_changed = hierarchy_changed;


/**
 * MiaouwGestureEventBox::gesture-notify:
 * @widget: the object which received the signal.
 * @event: a #MiaouwGestureEvent.
 *
 * The gesture-notify is fired when an user has performed a potential gesture.
 *
 * An application can use this signal to make gestures visible. However, this signal
 * does not imply that the user has finalized the gesture. It may still continue, change or fail.
 * This signal is fired during pointer motion.
 *
 * The user can do two subsequential gestures. If third gesture is received, the failed gesture notify
 * is fired and no further gestures are interpreted until the user releases the pointer.
 **/
	
	signals[SIGNAL_GESTURE_NOTIFY] = g_signal_new("gesture-notify", MIAOUW_TYPE_GESTURE_EVENT_BOX, 0, 0,
	                                              NULL, NULL,
	                                              g_cclosure_marshal_VOID__BOXED, G_TYPE_NONE, 1,
						      MIAOUW_TYPE_GESTURE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);

/**
 * MiaouwGestureEventBox::gesture-event:
 * @widget: the object which received the signal.
 * @event: a #MiaouwGestureEvent.
 *
 * The gesture-event is fired when an user has performed a successful gesture.
 *
 * An successful gesture may contain one or two gestures. This event is fired when
 * the user releases the pointer.
 **/

	signals[SIGNAL_GESTURE_EVENT] = g_signal_new("gesture-event", MIAOUW_TYPE_GESTURE_EVENT_BOX, 0, 0, 
	                                             NULL, NULL,
	                                             g_cclosure_marshal_VOID__BOXED, G_TYPE_NONE, 1,
						     MIAOUW_TYPE_GESTURE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);

	settings = gtk_settings_get_default();
	g_object_get(settings, "gtk-dnd-drag-threshold", &threshold, NULL);
	MIAOUW_GESTURE_EVENT_BOX_CLASS(klass)->threshold = threshold;

	/* TODO: unref settings? */
}

static void miaouw_gesture_event_box_init(GTypeInstance* instance, gpointer klass) {
	MiaouwGestureEventBox* gesture_event_box = MIAOUW_GESTURE_EVENT_BOX(instance);

	gesture_event_box->priv = g_new0(MiaouwGestureEventBoxPrivate, 1);

	gtk_widget_set_events(GTK_WIDGET(gesture_event_box),
	                      gtk_widget_get_events(GTK_WIDGET(gesture_event_box)) | GDK_BUTTON_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK);

}

GType miaouw_gesture_event_box_get_type() {
	static GType type = 0;
	static const GTypeInfo info = {
		sizeof (MiaouwGestureEventBoxClass),
		NULL,   /* base_init */
		NULL,   /* base_finalize */
		miaouw_gesture_event_box_class_init,
		NULL,   /* class_finalize */
		NULL,   /* class_data */
		sizeof (MiaouwGestureEventBox),
		0,
		miaouw_gesture_event_box_init,
		NULL
	};

	if (!type) {
		type = g_type_register_static(GTK_TYPE_EVENT_BOX, "MiaouwGestureEventBox", &info, 0);
	}

	return type;
}

GtkWidget* miaouw_gesture_event_box_new() {
	return gtk_widget_new(MIAOUW_TYPE_GESTURE_EVENT_BOX, NULL);
}

/* ******************** MiaouwGestureEvent ******************** */

MiaouwGestureEvent* miaouw_gesture_event_copy(const MiaouwGestureEvent* gesture_event) {
	MiaouwGestureEvent* new_gesture_event;

	g_return_val_if_fail(gesture_event, NULL);
	
	new_gesture_event = g_slice_new(MiaouwGestureEvent);
	*new_gesture_event = *gesture_event;
	
	return new_gesture_event;
}

void miaouw_gesture_event_free(MiaouwGestureEvent* gesture_event) {
	g_return_if_fail(gesture_event);

	g_slice_free(MiaouwGestureEvent, gesture_event);
}

GType miaouw_gesture_event_get_type(void) {
	static GType type = 0;
  
	if (!type) {
		type = g_boxed_type_register_static ("MiaouwGestureEvent",
		                                     (GBoxedCopyFunc)miaouw_gesture_event_copy,
					             (GBoxedFreeFunc)miaouw_gesture_event_free);
	}

	return type;
}
