/*
 *  Miaouw - The Miaouw Library for Maemo Development
 *  Copyright (C) 2008 Henrik Hedberg <hhedberg@innologies.fi>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gtk/gtk.h>
#include <libosso.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/utsname.h>
#include <curl/curl.h>

#include <miaouw/miaouw.h>

#define BLOCK_SIZE 1024

static gchar* construct_linking_info(gchar* path);
static void* do_upload_thread(void* data);
static size_t on_curl_read(void* ptr, size_t size, size_t nmemb, GString* string);
static size_t on_curl_write(void* ptr, size_t size, size_t nmemb, GString* string);
static void on_response(GtkDialog* dialog, gint response, void* not_used);
static gchar* read_fd(int fd);
static gchar* upload_data(gchar* url, gchar* data);

static gchar* global_url;
static gchar* global_data;

int main(int argc, char** argv) {
	osso_context_t* osso_context;
	gchar* backtrace;
	gchar* linking;
	struct utsname utsname;
	gchar* sysname;
	gchar* release;
	gchar* version;
	gchar* machine;
	GtkWidget* dialog;
	gchar* information_string;
	gchar* backtrace_string;
	gchar* question_string;
	GtkWidget* information_label;
	GtkWidget* backtrace_label;
	GtkWidget* question_label;
	PangoFontDescription* font;
	GtkWidget* scrolled_window;

	if (argc != 5) {
		fprintf(stderr, "Usage: %s path name version url\n", argv[0]);

		return 1;
	}

	global_url = argv[4];

	g_thread_init(NULL);
	gdk_threads_init();
	gdk_threads_enter();
	
	gtk_init(&argc, &argv);
	osso_context = osso_initialize(PACKAGE, VERSION, TRUE, NULL);
	g_set_application_name("Miaouw Backtrace Uploader");

	backtrace = read_fd(STDIN_FILENO);
	linking = construct_linking_info(argv[1]);
	if (uname(&utsname) == -1) {
		sysname = release = version = machine = NULL;
	} else {
		sysname = utsname.sysname;
		release = utsname.release;
		version = utsname.version;
		machine = utsname.machine;
	}
	
	dialog = gtk_dialog_new_with_buttons("Application Crashed", NULL, 0,
	                                     GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					     GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
					     NULL);

	information_string = g_strdup_printf(
	 "Application %s has a bug that caused it to quit. The following information was gathered:",
	 argv[2]);
	backtrace_string = g_strdup_printf("%s (%s) %s\n%s %s %s %s\n%s%s",
	 argv[2], argv[1], argv[3],
	 sysname, release, version, machine,
	 backtrace, linking);
	question_string = g_strdup_printf(
	 "To help fix the bug, the information (%d kB) is to be uploaded to %s.",
	 strlen(backtrace_string) / 1024, argv[4]);

	global_data = backtrace_string;

	information_label = gtk_label_new(information_string);
	backtrace_label = gtk_label_new(backtrace_string);
	question_label = gtk_label_new(question_string);

	font = pango_font_description_new();
	pango_font_description_set_size(font, 12 * PANGO_SCALE);
	gtk_widget_modify_font(backtrace_label, font);
	gtk_label_set_line_wrap(GTK_LABEL(information_label), TRUE);
	gtk_misc_set_alignment(GTK_MISC(information_label), 0.0, 0.0);
	gtk_misc_set_padding(GTK_MISC(information_label), 0, 4);
 	gtk_label_set_line_wrap(GTK_LABEL(question_label), TRUE);	
	gtk_misc_set_alignment(GTK_MISC(question_label), 0.0, 0.0);
	gtk_misc_set_padding(GTK_MISC(question_label), 0, 4);

	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), backtrace_label);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_ETCHED_IN);
	gtk_widget_set_size_request(scrolled_window, 633, 150);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), information_label);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), scrolled_window);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), question_label);

	g_signal_connect(G_OBJECT(dialog), "response", G_CALLBACK(on_response), backtrace_string);	

	gtk_widget_show_all(dialog);

	gtk_main();

	gdk_threads_leave();

	return 0;
}

static gchar* construct_linking_info(gchar* path) {
	int pipe_fds[2];
	gchar* argv[2];
	gchar* env[2];
	pid_t pid;
	gchar* ret;
	
	argv[0] = path;
	argv[1] = NULL;
	env[0] = "LD_TRACE_LOADED_OBJECTS=YES";
	env[1] = NULL;

	if (pipe(pipe_fds) == -1 || (pid = fork()) == -1) {
		
		return NULL;
	}
	if (!pid) {
		if (dup2(pipe_fds[1], STDOUT_FILENO) != -1) {
			close(pipe_fds[1]);
			close(pipe_fds[0]);
			execve(path, argv, env);
		}
		
		exit(1);
	}

	close(pipe_fds[1]);
	ret = read_fd(pipe_fds[0]);
	close(pipe_fds[0]);
	
	return ret;
}

static void* do_upload_thread(void* data) {
	GtkWidget* info_dialog;
	gchar* text;

	text = upload_data(global_url, global_data);
	
	gdk_threads_enter();

	gtk_widget_destroy(GTK_WIDGET(data));

	info_dialog = gtk_message_dialog_new(NULL, 0, GTK_MESSAGE_INFO, GTK_BUTTONS_CANCEL,
	 "%s", text);
	g_signal_connect(G_OBJECT(info_dialog), "response", G_CALLBACK(gtk_main_quit), NULL);	
	gtk_widget_show_all(info_dialog);

	gdk_threads_leave();

	return NULL;
}

static size_t on_curl_read(void* ptr, size_t size, size_t nmemb, GString* string) {
	size_t len;

	len = (size * nmemb <= string->len ? size * nmemb : string->len);
	memcpy(ptr, string->str, len);
	g_string_erase(string, 0, len);
	
	return len;
}

static size_t on_curl_write(void* ptr, size_t size, size_t nmemb, GString* string) {
	g_string_append_len(string, ptr, size * nmemb);
	
	return size * nmemb;
}

static void on_response(GtkDialog* dialog, gint response, void* not_used) {
	GtkWidget* info_dialog;
	
	gtk_widget_destroy(GTK_WIDGET(dialog));

	if (response != GTK_RESPONSE_REJECT) {
		info_dialog = gtk_message_dialog_new(NULL, 0, GTK_MESSAGE_INFO, GTK_BUTTONS_CANCEL,
		 "Uploading the backtrace information.\nPlease wait...");
		g_signal_connect(G_OBJECT(info_dialog), "response", G_CALLBACK(gtk_main_quit), NULL);	
		gtk_widget_show_all(info_dialog);

		if (!g_thread_create(do_upload_thread, info_dialog, FALSE, NULL)) {
			do_upload_thread(info_dialog);
		}
	} else {
		gtk_main_quit();
	}
}

static gchar* read_fd(int fd) {
	gchar* contents;
	gsize size;
	ssize_t ret;
	
	contents = NULL;
	size = 0;
	do {
		contents = g_renew(gchar, contents, size + BLOCK_SIZE);
		ret = read(fd, contents + size, BLOCK_SIZE);
		if (ret > 0) {
			size += ret;
		}
	} while (ret > 0);
	contents = g_renew(gchar, contents, size + 1);
	contents[size] = 0;
	
	return contents;	
}

static gchar* upload_data(gchar* url, gchar* data) {
	gchar* ret = NULL;
	struct curl_slist* header_slist = NULL;
	CURLcode code;
	CURL* curl = NULL;
	GString* post_string;
	GString* response_string;
	long status;

	post_string = g_string_new(data);
	response_string = g_string_new("");

	header_slist = curl_slist_append(header_slist, "Content-Type: text/plain");

	if ((code = curl_global_init(CURL_GLOBAL_ALL)) != 0) {
		ret = g_strdup_printf("Internal error: curl_global_init failed (%d).", code);
	} else if ((curl = curl_easy_init()) == NULL) {
		ret = g_strdup_printf("Internal error: curl_easy_init failed.");
	} else if ((code = curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, on_curl_write)) ||
		   (code = curl_easy_setopt(curl, CURLOPT_WRITEDATA, response_string)) ||
		   (code = curl_easy_setopt(curl, CURLOPT_READFUNCTION, on_curl_read)) ||
		   (code = curl_easy_setopt(curl, CURLOPT_READDATA, post_string)) ||
		   (code = curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1)) || 
		   (code = curl_easy_setopt(curl, CURLOPT_TIMEOUT, 300)) || 
		   (code = curl_easy_setopt(curl, CURLOPT_USERAGENT, "Miaouw/" VERSION)) ||
		   (code = curl_easy_setopt(curl, CURLOPT_POST, 1)) ||
		   (code = curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, post_string->len)) ||
		   (code = curl_easy_setopt(curl, CURLOPT_HTTPHEADER, header_slist)) ||
		   (code = curl_easy_setopt(curl, CURLOPT_URL, url))) {
		ret = g_strdup_printf("Internal error: %s.", curl_easy_strerror(code));
	} else if ((code = curl_easy_perform(curl))) {
		ret = g_strdup_printf("Uploading failed: %s.", curl_easy_strerror(code));
	} else if ((code = curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &status))) {
		ret = g_strdup_printf("Internal error: %s.", curl_easy_strerror(code));
	} else if (status != 200) {
		ret = g_strdup_printf("Uploading failed: server responded %ld.", status);
	} else if (response_string->len == 0) {
		ret = g_strdup_printf("Uploading failed (unknown reason).");
	} else {
		ret = g_strdup(response_string->str);
	}
	
	g_string_free(post_string, TRUE);
	g_string_free(response_string, TRUE);
	curl_slist_free_all(header_slist);
	if (curl) {
		curl_easy_cleanup(curl);
	}

	return ret;
}
