/* liqbase
 * Copyright (C) 2008 Gary Birkett
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 *
 * basic sketch
 *
 */




#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <ctype.h>

#include "liqapp.h"
#include "liqdraw.h"
#include "liqfont.h"
#include "liqcanvas.h"
#include "liqdoc.h"
//##################################################################

POINT *point_alloc()
{
	POINT *self = (POINT *)malloc(sizeof( POINT ));
	return self;
}

void point_new(POINT *self)
{
	memset((char *)self,0, sizeof( POINT ));	
}

void point_free(POINT *self)
{
	free(self);
}

inline void point_copy(POINT *self,POINT *s) 
{
	// inline memcpy would work for this, but it will do as is
	self->x=s->x;  self->y=s->y;  self->z=s->z; self->t=s->t;
}

POINT * point_clone(POINT *s) 
{
	POINT *self = point_alloc();
	point_copy(self,s);
	return self;
}

inline void point_getdiff(POINT *self,POINT *s,POINT *e) 
{
	self->x=e->x-s->x;  self->y=e->y-s->y;  self->z=e->z-s->z; self->t=e->t-s->t;
}
inline int point_issame(POINT *self,POINT *s) 
{
	return (self->x==s->x && self->y==s->y && self->z && self->z==s->z);
}

//##################################################################

void pointrange_start(POINTRANGE *self,POINT *p)
{
	self->xl = p->x;
	self->xr = p->x;
	self->yt = p->y;
	self->yb = p->y;
	self->zf = p->z;
	self->zb = p->z;
}
void pointrange_extendrubberband(POINTRANGE *self,POINT *p)
{
	if(self->xl > p->x)self->xl = p->x;
	if(self->xr < p->x)self->xr = p->x;
	if(self->yt > p->y)self->yt = p->y;
	if(self->yb < p->y)self->yb = p->y;
	if(self->zf < p->z)self->zf = p->z;
	if(self->zb > p->z)self->zb = p->z;
}


int pointrange_isconnected(POINTRANGE *self,POINTRANGE *b)
{
	
	if(self->xl > b->xr)return 0;
	if(self->xr < b->xl)return 0;
	if(self->yt > b->yb)return 0;
	if(self->yb < b->yt)return 0;
	return 1;
}



//##################################################################

STROKE *stroke_alloc()
{
	STROKE *self = (STROKE *)malloc(sizeof( STROKE ));
	stroke_new(self);
	return self;
}

void stroke_new(STROKE *self)
{
	memset((char *)self,0, sizeof( STROKE ));	
	self->pen_thick=1;
	//self->brushindex=-1;
}



STROKE *stroke_clone(STROKE *s) 
{
	STROKE *self = stroke_alloc();
	// first, a shallow copy..
	// memcpy isnt right here, we dont want to do a binary clone on all the links etc
	//memcpy((char *)self,(char *)s,sizeof( STROKE ));
	// only do the specific fields
	self->pen_y=s->pen_y;
	self->pen_u=s->pen_u;
	self->pen_v=s->pen_v;
	self->pen_thick=s->pen_thick;
	POINT *p=s->pointfirst;
	while(p)
	{
		stroke_appendpoint(self,point_clone(p));
		p=p->linknext;
	}
	
	return self;
}



void stroke_clear(STROKE *self)
{
	self->pointcount=0;
	// we throw away the points we added
	while(self->pointfirst)
	{
		POINT *p = self->pointfirst;
		self->pointfirst = p->linknext;
		point_free(p);
	}
	self->pointlast=NULL;
	if(self->quadchain)
	{
		free(self->quadchain);
		self->quadchain=NULL;
	}
}



int           stroke_totallength(STROKE *self)
{
	POINT *p = self->pointfirst;
	POINT *q = NULL;
	int m=0;
	while(p)
	{
		q=p->linknext;
		if(!q)break;
		
		int dx=q->x-p->x;
		int dy=q->y-p->y;
		int dm=sqrt(dx*dx+dy*dy);
		m+=dm;
		
		p=q;
	}
	return m;
}

void stroke_free(STROKE *self)
{
	stroke_clear(self);
	free(self);
}
void stroke_appendpoint(STROKE *self,POINT *p)
{
	//POINT *p = point_alloc();
	//p->x=s->x; p->y=s->y;  p->z=s->z;  p->t = s->t;
	if(self->pointcount==0)
	{
		p->linknext = NULL;
		p->linkprev = NULL;
		self->pointfirst = p;
		self->pointlast = p;
		self->pointcount = 1;
		pointrange_start( &self->boundingbox, p );		
	}
	else
	{
		p->linkprev = self->pointlast;	
		p->linknext = NULL;	
		self->pointlast->linknext = p;
		self->pointlast = p;
		self->pointcount++;
		pointrange_extendrubberband(&self->boundingbox, p);		
	}
}
void stroke_start(STROKE *self,int px,int py, int pz)
{
	POINT *p = point_alloc();	
	p->x=px; p->y=py;  p->z=pz;
	stroke_appendpoint(self,p);
}

//double atan2( double y, double x );

void stroke_extend(STROKE *self,int px,int py, int pz)
{
	switch(self->strokekind)
	{
		case 0:	// normal stroke
			{
				POINT *p = point_alloc();	
				p->x=px; p->y=py;  p->z=pz;	
				stroke_appendpoint(self,p);
			}
			break;

		case 1: // p2p line
			
			if(self->pointcount<2)
			{
				POINT *p = point_alloc();	
				p->x=px; p->y=py;  p->z=pz;	
				stroke_appendpoint(self,p);
			}
			else
			{
				POINT *p = self->pointlast;	
				p->x=px; p->y=py;  p->z=pz;
				pointrange_start( &self->boundingbox, self->pointfirst );
				pointrange_extendrubberband( &self->boundingbox, self->pointlast );
			}
			break;
		
		case 2: // box
			
			if(self->pointcount<2)
			{
				POINT *p = point_alloc();	
				p->x=px; p->y=py;  p->z=pz;	
				stroke_appendpoint(self,p);
			}
			else
			{
				POINT *p = self->pointlast;	
				p->x=px; p->y=py;  p->z=pz;
				pointrange_start( &self->boundingbox, self->pointfirst );
				pointrange_extendrubberband( &self->boundingbox, self->pointlast );
			}
			break;
		
		case 3: // filled
			if(self->pointcount<2)
			{
				POINT *p = point_alloc();	
				p->x=px; p->y=py;  p->z=pz;	
				stroke_appendpoint(self,p);
			}
			else
			{
				POINT *p = self->pointlast;	
				p->x=px; p->y=py;  p->z=pz;
				pointrange_start( &self->boundingbox, self->pointfirst );
				pointrange_extendrubberband( &self->boundingbox, self->pointlast );
			}
			break;

		case 4: // stamp
			if(self->pointcount<2)
			{
				POINT *p = point_alloc();	
				p->x=px; p->y=py;  p->z=pz;	
				stroke_appendpoint(self,p);
			}
			else
			{
				POINT *p = self->pointlast;	
				p->x=px; p->y=py;  p->z=pz;
				pointrange_start( &self->boundingbox, self->pointfirst );
				pointrange_extendrubberband( &self->boundingbox, self->pointlast );
			}
			break;
	}
}


int          stroke_hittest(STROKE *self,int px,int py)
{
	// depending upon what kind of stroke this is.
	// we react differently.
	// maybe using the stroke for this purpose is wrong
	// i should be using a primative which can use strokes as input.
	
	return 0;
}


void          stroke_ensurepositive(STROKE *self)
{
	if(self->strokekind>1 && self->pointcount>1)
	{
		POINT *p = self->pointfirst;	
		POINT *q = self->pointlast;
		if(p->x>q->x){int t=p->x; p->x=q->x; q->x=t; }
		if(p->y>q->y){int t=p->y; p->y=q->y; q->y=t; }
		
		
	}
}



int stroke_isconnected(STROKE *self,STROKE *o)
{
	//
	if( !pointrange_isconnected(&self->boundingbox,&o->boundingbox))
	{
		return 0;
	}
	// we MIGHT be connected
	
	POINT *p=self->pointfirst;
	while(p)
	{
		POINT *q=o->pointfirst;
		while(q)
		{
			int dx=p->x-q->x;
			int dx2=(dx*dx);


			int dy=p->y-q->y;
			int dy2=dy*dy;
			if(dx2+dy2<64)
			{
				if(sqrt(dx2+dy2)<8)
				{
					// ! we are connected
					return 1;
				}
			}

						
			q=q->linknext;
		}
		
		p=p->linknext;
	}
	// no, we arent
	return 0;
	
	
}


char *          stroke_quadchainbuild(STROKE *self)
{

	// depending upon what kind of stroke this is.
	// we react differently.
	// maybe using the stroke for this purpose is wrong
	// i should be using a primative which can use strokes as input.
	
	// i should be taking the entire stroke inside a unit SQUARE, not as I currently do and stretch a rectangle
	
	if(self->pointcount==0)
		return NULL;
	if(self->quadchain)
		return self->quadchain;
	
	int box = self->boundingbox.xl;
	int boy = self->boundingbox.yt;
	
	int bmx = self->boundingbox.xr-box;
	int bmy = self->boundingbox.yb-boy;

	int bcx = box+bmx/2;
	int bcy = boy+bmy/2;
	
	
	int bfac = (bmx > bmy) ? bmx : bmy;
	
	
	box = bcx-bfac/2;
	boy = bcy-bfac/2;
	
	bfac++;


	if(bmx==0 || bmy==0)
		return NULL;

	
	char quadchain[33];
	int quadused=0;
	int quadcurr=-1;
	
	


	
	POINT *p = self->pointfirst;
	while(p)
	{	
		// normalize to quadrant 0,1,2,3,4
		//int nx = 2 + ((p->x - bcx) * 5 / bfac);
		//int ny = 2 + ((p->y - bcy) * 5 / bfac);
		//int nq = ny*5+nx;
		
		// normalize to quadrant 0,1,2
		//int nx = 1 + ((p->x - bcx) * 3 / bfac);
		//int ny = 1 + ((p->y - bcy) * 3 / bfac);
		//int nq = ny*3+nx;		
	
		// normalize to quadrant 0,1,2
		int nx = ((p->x - box) * 3 / bfac);		// 3 steps
		int ny = ((p->y - boy) * 3 / bfac);		// 3 steps
		int nq = ny*3+nx;	
		
		if(quadcurr==-1)
		{
				//app_log("%i,%i,%i",nx,ny,nq);

			quadcurr = nq;
		}
		else
		{
			if(nq!=quadcurr)
			{
				//app_log("%i,%i,%i",nx,ny,nq);
				
				// stroke has changed to a different quadrant
				if(quadused<32)
					quadchain[quadused++] = 97 + quadcurr;
				else
					break;
				quadcurr = nq;
			}
		}		
		p=p->linknext;
	}
	if(quadused<32)
		quadchain[quadused++] = 97+quadcurr;
		
	quadchain[quadused++] = 0;
	self->quadchain = strdup(quadchain);
	return self->quadchain;
}


//##################################################################

PAGE *page_alloc()
{
	PAGE *self = (PAGE *)malloc(sizeof( PAGE ));
	page_new(self);
	return self;
}

void page_new(PAGE *self)
{
	memset((char *)self,0, sizeof( PAGE ));	
	//self->brushindex=-1;
	
	// when we know no better we assume baseline defaults
	// the implimentor is free to modify these to suit, or he can just scale himself between
	// 
	
	self->pixelwidth=800;//=canvas.pixelwidth;
	self->pixelheight=480;//canvas.pixelheight;

	self->dpix=225;// *canvas.scalew;
	self->dpiy=225;// *canvas.scaleh;

}


void page_coordchange_scr_to_page(PAGE *self,int scrx,int scry,int scrw,int scrh, int scrdpix,int scrdpiy,int *rx,int *ry)
{
//	*rx = scrx * (scrdpix * scrw) / (self->pixelwidth  * self->dpix);
//	*ry = scry * (scrdpiy * scrh) / (self->pixelheight * self->dpiy);

	*rx = scrx * (self->pixelwidth  * self->dpix) / (scrdpix * scrw);
	*ry = scry * (self->pixelheight * self->dpiy) / (scrdpiy * scrh);

}


void page_titlechange(PAGE *self,char *title)
{
	if(self->title){ free(self->title); self->title=NULL; }
	if(!title || *title==0) return;
	self->title = strdup(title);
}

// '''''''''''''''''''''''''''''''''''''''''''''''''''''''''

static char *filename_walkoverpath(char *filename)
{
	if(!filename || *filename==0)
	{
		return filename;
	}
	char *fnstart = filename;
	char *fnend  = filename;
	// walk quickly to the end
	while(*fnend)
	{
		//todo:make path handling safe
		if(*fnend=='/') fnstart=fnend+1;
		fnend++;
	}
	return fnstart;
}

char * pagefilename_rebuild(struct pagefilename *self,char *bufferresultfilename,int buffermax)
{
		if(self->filepath[0])
		{
			if(self->fileuser[0])
				snprintf(bufferresultfilename,buffermax, "%s/liq.%s.%s.page.%s",self->filepath,self->filedate,self->fileuser,self->filetitle);
			else
				snprintf(bufferresultfilename,buffermax, "%s/liq.%s.%s.page.%s",self->filepath,self->filedate,app.username,self->filetitle);
		}
		else
		{
			if(self->fileuser[0])
				snprintf(bufferresultfilename,buffermax,    "liq.%s.%s.page.%s",               self->filedate,self->fileuser,self->filetitle);
			else
				snprintf(bufferresultfilename,buffermax,    "liq.%s.%s.page.%s",                self->filedate,app.username,self->filetitle);
		}
		return bufferresultfilename;
}

int pagefilename_breakapart(struct pagefilename *self,char *filename)
{
	//todo:parse this properly
	
	// "liq.[date].[user].page.[keyword]"
	// "liq.[date].[user].page"
	// "liq.[user].[date].page.[keyword]"
	// "liq.[user].[date].page"
	// "liq.[date].page.[keyword]"
	// "liq.[date].page"
	
	
	// nullify input
	self->filepath[0]=0;
	//self->filename[0]=0;
	self->fileuser[0]=0;
	self->filedate[0]=0;
	self->fileclass[0]=0;
	self->filetitle[0]=0;
	
	

	
	
	
	
	
	
	


	if(!filename || *filename==0)
	{
		return -1;
	}
	char *pthstart = filename;
	char *fnstart = filename_walkoverpath(filename);
	char *fnend  = filename;
	while(*fnend)
	{
		fnend++;
	}

	int len=0;

	len = fnstart-pthstart;
	if(len>0)len--;
	if(len>255) len=255;

	strncpy(self->filepath,pthstart,len);
	self->filepath[len+1]=0;
	
	//app_log("cols :: trying :: %s",fnstart);
	
	//########################################### break into columns, removing the '.'s as we go
	char * inorig = strdup(fnstart);
		char *cols[80]={"X","X","X","X","X","X","X","X"};
		int colcount=0;
		char *indat=inorig;
		cols[0] = indat;
		colcount++;
		while(*indat && colcount<6)
		{
			if(*indat=='.')
			{
				*indat=0;
				indat++;
				cols[colcount++]=indat;
			}
			else
				indat++;
		}
		
		
		//app_log("X cols :: %i :: %s :: %s :: %s :: %s :: %s",colcount,cols[0],cols[1],cols[2],cols[3],cols[4]);
		
		if(colcount<1)
		{
			//app_log("no cols :: trying :: %s",fnstart);
			// invalid..
			return -1;
		}

		
		if(strcmp(cols[0],"liq")!=0)
		{
			return -1;
		}
		//if(strcmp(cols[2],"page")!=0) return -1;
		//if(strcmp(cols[3],"page")!=0) return -1;
		
		switch(colcount)
		{
			case 3:
					
				//app_log("cols :: %i :: %s :: %s :: %s",colcount,cols[0],cols[1],cols[2]);
					
				if(isdigit(*cols[1]) && (strcmp(cols[2],"page")==0))
				{
					// liq.[date].page  // ancient
					
					strncpy(self->filedate, cols[1],15); self->filedate[15]=  0;
					//strncpy(self->fileuser, cols[ ],15);self->fileuser[15]=0;
					strncpy(self->fileclass,cols[2],20);self->fileclass[20]=0;
					//strncpy(self->filetitle,cols[ ],20);self->filetitle[20]=0;
					free(inorig);
					return 0;
				}
				// should never occur.. heh, final last words
				free(inorig);
				return -1;

			case 4:
				
				
				//app_log("cols :: %i :: %s :: %s :: %s :: %s",colcount,cols[0],cols[1],cols[2],cols[3]);
				
				if(isdigit(*cols[1]) && (strcmp(cols[2],"page")==0))
				{
					// liq.[date].page.[keyword]  // ancient
					strncpy(self->filedate, cols[1],15); self->filedate[15]=  0;
					//strncpy(self->fileuser, cols[ ],15);self->fileuser[15]=0;
					strncpy(self->fileclass,cols[2],20);self->fileclass[20]=0;
					strncpy(self->filetitle,cols[3],20);self->filetitle[20]=0;
					free(inorig);
					return 0;
				}
				
				if(isdigit(*cols[1]) && (strcmp(cols[3],"page")==0))
				{
					// liq.[date].[user].page  // new
					strncpy(self->filedate, cols[1],15); self->filedate[15]=  0;
					strncpy(self->fileuser, cols[2],15);self->fileuser[15]=0;
					strncpy(self->fileclass,cols[3],20);self->fileclass[20]=0;
					//strncpy(self->filetitle,cols[ ],20);self->filetitle[20]=0;
					free(inorig);
					return 0;
				}

				if(isdigit(*cols[2]) && (strcmp(cols[3],"page")==0))
				{
					// liq.[user].[date].page  // ancient
					//app_log("cough");
					strncpy(self->filedate, cols[2],15); self->filedate[15]=  0;
					strncpy(self->fileuser, cols[1],15);self->fileuser[15]=0;
					strncpy(self->fileclass,cols[3],20);self->fileclass[20]=0;
					//strncpy(self->filetitle,cols[ ],20);self->filetitle[20]=0;
					free(inorig);
					return 0;
				}


				// should never occur.. heh, final last words
				free(inorig);
				return -1;
			
			case 5:
				
				//app_log("cols :: %i :: %s :: %s :: %s :: %s :: %s",colcount,cols[0],cols[1],cols[2],cols[3],cols[4]);
				
				if(isdigit(*cols[1]) && (strcmp(cols[3],"page")==0))
				{
					// liq.[date].[user].page.[keyword]    // new
					strncpy(self->filedate, cols[1],15); self->filedate[15]=  0;
					strncpy(self->fileuser, cols[2],15);self->fileuser[15]=0;
					strncpy(self->fileclass,cols[3],20);self->fileclass[20]=0;
					strncpy(self->filetitle,cols[4],20);self->filetitle[20]=0;
					free(inorig);
					return 0;
				}
				
				if(isdigit(*cols[2]) && (strcmp(cols[3],"page")==0))
				{
					// liq.[user].[date].page.[keyword]    // ancient
					strncpy(self->filedate, cols[2],15); self->filedate[15]=  0;
					strncpy(self->fileuser, cols[1],15);self->fileuser[15]=0;
					strncpy(self->fileclass,cols[3],20);self->fileclass[20]=0;
					strncpy(self->filetitle,cols[4],20);self->filetitle[20]=0;
					free(inorig);
					return 0;
				}

				// should never occur.. heh, final last words
				free(inorig);
				return -1;
		}
		
	
	free(inorig);
	
	
	
	
	
	
	
/*	
	
	

	//len = fnend-fnstart;
	//if(len>255) len=255;

	//strncpy(self->filename,fnstart,len);
	//self->filename[len+1]=0;

	char *p0=fnstart;

	if(strncmp(p0,"liq.",4)!=0) return -1;
	p0+=4;
	if(p0>=fnend) return -1;
	
	
	if(isdigit(*p0)==0)
	{
		// we have specified a name
		//app_log("sketch has username");
		char *pdot=instr(p0,".");
		if(!pdot)
		{
			// no dot, really odd
			return -1;
		}
		int diff = pdot-p0;
		if(diff>15)diff=15;
		strncpy(self->fileuser,p0,diff);
		p0=pdot+1;
	}
	else
	{
		//app_log("sketch has no username");
		// digits imply we have hit the date portion directly
		strncpy(self->fileuser,app.username,15);
	}
	

	strncpy(self->filedate,p0,15);
	self->filedate[15]=0;
	p0+=16;
	if(p0>=fnend) return -1;

	// we got the date portion
	// anything left is the terminator  (class &| Title)

	char *term = p0;

	char *pdot=instr(term,".");

	if(!pdot)
	{
		//app_log("{class}  term='%s'",term);
		// no dot, class only
		strcpy (self->filetitle,"");
		strncpy(self->fileclass,term,20);
		self->filetitle[0]=0;
		self->fileclass[20]=0;
		return 0;
	}

	char *fin = pdot+1;
	int tl1 = (unsigned int)(pdot-term);
	if(tl1>20)tl1=20;

	//app_log("{class}.{title}  term='%s'",term);
	strncpy(self->fileclass,term,tl1);
	strncpy(self->filetitle,fin,20);
	self->fileclass[tl1]=0;
	self->filetitle[20]=0;
	
 */
	
	return 0;

}

int pagefilename_test(char *filename)
{
	struct pagefilename self;

	if(	pagefilename_breakapart(&self,filename) == 0)
	{
		// got it ok, lets confirm...

		app_log("pagefilename: ok  path:'%s', dat:'%s', cls:'%s', tit:'%s'",self.filepath,self.filedate,self.fileclass,self.filetitle);
	}
	else
	{
		app_log("pagefilename: bad '%s'",filename);
		// invalid
	}
	return 0;
}


void          page_strokeremove(PAGE *self,STROKE *s)
{
	//
	if(!s)return;
	//if(! (s->linkpage==self) ) return;
	STROKE *l=s->linkprev;
	STROKE *r=s->linknext;
	
	if(self->strokefirst==s)
	{
		// remove the front most item
		self->strokefirst=r;
		s->linknext=NULL;

	}

	if(self->strokelast==s)
	{
		// remove the front most item
		self->strokelast=l;
		s->linkprev=NULL;
	}
	if(l)l->linknext=r;
	if(r)r->linkprev=l;
	
	s->linkprev=NULL;
	s->linknext=NULL;
	s->linkpage=NULL;
	
	stroke_free(s);
	
	self->strokecount--;
	
}



void page_strokeinsert(PAGE *self,STROKE *s)
{
	page_strokeupdate(self,s);

	//STROKE *s = stroke_alloc();
	s->linkprev = self->strokelast;
	if(!self->strokefirst) self->strokefirst = s;
	if( self->strokelast)  self->strokelast->linknext = s;
	self->strokelast = s;
	self->strokecount++;
	//return s;
}
void page_islandswap(PAGE *self,int islandnumberfrom,int islandnumberto)
{
		STROKE *o = self->strokefirst;
		while(o)
		{
			if(o->islandnumber==islandnumberfrom) o->islandnumber=islandnumberto;
			o=o->linknext;
		}
}

void page_islandclear(PAGE *self)
{
		STROKE *o = self->strokefirst;
		while(o)
		{
			o->islandnumber=0;
			o=o->linknext;
		}
		self->islandcount=0;
}

void page_islandcalcone(PAGE *self,STROKE *s)
{	
			// we need to check out the others
		STROKE *o = self->strokefirst;
		while(o)
		{
			if(o!=s)
			{
				// check
				if(stroke_isconnected(s,o))
				{
					//	app_log("conn");
					// yes, they are connected to us
					if(s->islandnumber==0)
					{
						// obtain the island ref from our neighbour
						if(o->islandnumber==0)
						{
							s->islandnumber = (self->islandcount++);
							o->islandnumber = s->islandnumber;
						}
						else
						{
							s->islandnumber=o->islandnumber;
						}
					}
					else
					{
						// islands merging, keep the lowest
						if(s->islandnumber < o->islandnumber)
							page_islandswap(self,o->islandnumber,s->islandnumber);
						else
							page_islandswap(self,s->islandnumber,o->islandnumber);

					}
				}
			}
			o=o->linknext;
		}
		if(s->islandnumber==0)
		{
			// make one now :)
			s->islandnumber = (self->islandcount++);
		}
		
}

void page_islandcalcall(PAGE *self)
{

	page_islandclear(self);
		// we need to check out the others
	STROKE *o = self->strokefirst;
	while(o)
	{
		page_islandcalcone(self,o);
		o=o->linknext;
	}
		
}


void page_strokeupdate(PAGE *self,STROKE *s)
{
POINT p1,p2;
	// the stroke has been extended
	// this may be handled by a callback from the stroke itself
	// not sure yet
	// however its kicked in we must update our bounding box
	p1.x = s->boundingbox.xl;
	p1.y = s->boundingbox.yt;
	p1.z = s->boundingbox.zf;
	p2.x = s->boundingbox.xr;
	p2.y = s->boundingbox.yb;
	p2.z = s->boundingbox.zb;
	if(self->strokecount==0)
	{
		pointrange_start(&self->boundingbox, &p1);
		pointrange_extendrubberband(&self->boundingbox, &p2);
	}
	else
	{
		pointrange_extendrubberband(&self->boundingbox, &p1);
		pointrange_extendrubberband(&self->boundingbox, &p2);
	}
	
	// if its island is 0 then we must scan for it :)
	//page_islandcalcone(self,s);
}


void          page_boundwholearea(PAGE *self)
{
	// make the page boundary be the full boundary
POINT p1,p2;
	p1.x = 0;
	p1.y = 0;
	p1.z = self->boundingbox.zf;
	p2.x = self->pixelwidth;
	p2.y = self->pixelheight;
	p2.z = self->boundingbox.zb;

	pointrange_extendrubberband(&self->boundingbox, &p1);
	pointrange_extendrubberband(&self->boundingbox, &p2);

}



void page_clear(PAGE *self)
{
	// we throw away the points we added
	self->strokecount=0;
	while(self->strokefirst)
	{
		STROKE *s = self->strokefirst;
		self->strokefirst = s->linknext;
		stroke_free(s);
	}

	if(self->title){ free(self->title); self->title=NULL; }
	if(self->filename) { free(self->filename); self->filename=NULL; }

	self->strokelast=NULL;
}

void page_free(PAGE *self)
{
	page_clear(self);
	free(self);
}

int page_filesave(PAGE *self,char *filename)
{
	app_log("filesave, saving to '%s'",filename);

	if(self->filename) { free(self->filename); self->filename=NULL; }
	self->filename = strdup(filename);

	FILE *fd;
	//int   ri;
	fd = fopen(filename, "w");
	if(fd==NULL){ app_log("filesave, cannot open '%s' for writing",filename); return -1; }
	// actual file data

	app_log("filesave, writing head");

	fprintf(fd,									"page:%i,%i,%i,%i\n",
																									self->pixelwidth,
																									self->pixelheight,
																									self->dpix,
																									self->dpiy
																									);
	app_log("filesave, writing strokes");
	STROKE *stroke=self->strokefirst;
	while(stroke)
	{
		
		fprintf(fd,								"\tstroke:%i,%i,%i,%i\n",
																									stroke->pen_y,
																									stroke->pen_u,
																									stroke->pen_v,
																									stroke->strokekind
																									);
		POINT *point;

		point = stroke->pointfirst;
		while(point)
		{
			fprintf(fd,							"\t\tpoint:%u,%i,%i,%i\n",
																									(unsigned int)point->t,
																									point->x,
																									point->y,
																									point->z
																									);
			point=point->linknext;
		}

		stroke=stroke->linknext;
	}

	app_log("filesave, closing");			
	fclose(fd);
	app_log("filesave, finished");
	//if(ri<0){ app_log("filesave, cannot write to '%s'",filename); return -2; }
	return 0;
}

int page_fileload(PAGE *self,char *filename)
{

	app_log("page_fileload '%s'",filename);
	struct doc doc;
	char *indat;
	int err=0;
	
	doc.renderfont=NULL;
	
	err=doc_initfromfilename(&doc,filename);
	//err=doc_initfromfilename(&doc,"/usr/bin/liq/moteineye.txt");
	//err=doc_initfromfilename(&doc,"/usr/share/liqbase/liqreader_readme.txt");
	if(err)
	{
    	{ return app_warnandcontinue(-1,"page_fileload couldnt open"); }						
	}

	page_clear(self);

	if(self->filename) { free(self->filename); self->filename=NULL; }
	self->filename = strdup(filename);

	STROKE *stroke=NULL;
	//POINT *point=NULL;

	int linenum=1;	
	struct docline *docline = doc.linefirst;
	if(!docline)
	{
		// empty file
		doc_close(&doc);
	   	{ return app_warnandcontinue(-1,"page_fileload file is empty"); }						
	}

	if(strncmp(docline->linedata,"page:",5) != 0)
	{
		// invalid header
		doc_close(&doc);
	   	{ return app_warnandcontinue(-1,"page_fileload invalid file header"); }						

	}

	while(docline)
	{

		indat = docline->linedata;
		
		//app_log("trying line: %5i %s",linenum,indat);

		int indentlevel=0;
		while(*indat==9)
		{
			indentlevel++;
			indat++;
		}
		
		{
			int  pagew=0;
			int  pageh=0;
			int  pagedpix=0;
			int  pagedpiy=0;
			int res = sscanf(indat,"page: %i, %i, %i, %i",&pagew,&pageh,&pagedpix,&pagedpiy);
			if(res==4)
			{
				//app_log("%4i ++page  ++ %i '%s' == %i,%i,%i,%i",linenum,res,indat,pagew,pageh,pagedpix,pagedpiy);
				self->pixelwidth =pagew;
				self->pixelheight=pageh;
				self->dpix=pagedpix;
				self->dpiy=pagedpiy;
			}
		}		

		{
			int  peny=0;
			int  penu=0;
			int  penv=0;
			int  strokekind=0;
			int  res = sscanf(indat,"stroke: %i, %i, %i, %i",&peny,&penu,&penv,&strokekind);
			if(res==3){ strokekind=0; res=4; }	// todo: make all this import cleaner, this is a fudge
			if(res==4)
			{
				//app_log("%4i ++stroke++ %i '%s' == %i,%i,%i kind=%i",linenum,res,indat,peny,penu,penv,strokekind);

				//if(stroke==NULL) 
				//	{ return app_warnandcontinue(-1,"page_fileload point without stroke"); }
				stroke = stroke_alloc();
				stroke->pen_y=peny;
				stroke->pen_u=penu;
				stroke->pen_v=penv;
				stroke->strokekind=strokekind;
				if(strokekind==4)
				{
					stroke->mediapage = self;
				}
			}
		}
/*		
		{
			int  peny=0;
			int  penu=0;
			int  penv=0;
			int  res = sscanf(indat,"stroke: %i, %i, %i",&peny,&penu,&penv);
			if(res==3)
			{
				//app_log("%4i ++stroke++ %i '%s' == %i,%i,%i",linenum,res,indat,peny,penu,penv);

				//if(stroke==NULL) 
				//	{ return app_warnandcontinue(-1,"page_fileload point without stroke"); }
				stroke = stroke_alloc();
				stroke->pen_y=peny;
				stroke->pen_u=penu;
				stroke->pen_v=penv;
			}
		}
*/		
		
		{
			int pt=0;
			int px=0;
			int py=0;
			int pz=0;
			int res = sscanf(indat,"point: %u, %i, %i, %i",&pt,&px,&py,&pz);
			if(res==4)
			{
				//app_log("%4i ++point ++ %i '%s' == %i,%i,%i,%i",linenum,res,indat,pt,px,py,pz);

				if(stroke==NULL) 
				{
					doc_close(&doc);
					{ return app_warnandcontinue(-1,"page_fileload point without stroke"); }
				}
				if(stroke->pointcount==0)
				{
					stroke_start(stroke,px,py,pz);
					page_strokeinsert(self,stroke);
				}
				else
				{
					stroke_extend(stroke,px,py,pz);
					page_strokeupdate(self,stroke);
				}

			}
			else
			{
				//app_log("%4i --point -- %i '%s' == %i,%i,%i,%i",linenum,res,indat,pt,px,py,pz);
			}

		}
		docline=docline->linknext;
		linenum++;
	}
	doc_close(&doc);
	
	

	//pagefilename_test(filename);

	return self->strokecount>0 ? 0 : -1;
}

void page_rendertocanvas(PAGE *self,int l,int t,int w,int h)
{

	if(l+w<0 || t+h<0) return;
	if(l>=canvas.pixelwidth) return;
	if(t>=canvas.pixelheight) return;
	// From
	int fox = self->boundingbox.xl;
	int foy = self->boundingbox.yt;
	int fmx = self->boundingbox.xr - self->boundingbox.xl;
	int fmy = self->boundingbox.yb - self->boundingbox.yt;
	int fmap2=fmx*fmx+fmy*fmy;
	if(fmx==0 || fmy==0) return;
	// To
	int tox = l;
	int toy = t;
	int tmx = w-1; if(tmx<0)tmx=0;
	int tmy = h-1; if(tmy<0)tmy=0;
	int tmap2 = tmx*tmx+tmy*tmy;
	if(tmy==0 || tmy==0) return;
	
	float adx = (float)self->dpix / (float)canvas.dpix;
	float ady = (float)self->dpiy / (float)canvas.dpiy;

	//================================ calc aspect ratio	
	float ax = adx * (float)tmx / (float)fmx;
	float ay = ady * (float)tmy / (float)fmy;
	float ar = (ax<=ay ? ax : ay);

	int rx = (ar * (float)fmx)/adx;
	int ry = (ar * (float)fmy)/ady;

//	int fx = ((float)tmx / rx);
//	int fy = ((float)tmy / ry);

/*	app_log("from %i,%i",fmx,fmy);
	app_log("to   %i,%i",tmx,tmy);
	app_log("ax   %f,%f,%f",ax,ay,ar);
	app_log("r    %f,%f",rx,ry);
	app_log("f    %i,%i",fx,fy);
*/	
	//================================ push altered aspect result into tmxy

	rx=(float)rx*0.9;
	ry=(float)ry*0.9;

	if(rx<tmx) tox+=(tmx-rx)/2;
	if(ry<tmy) toy+=(tmy-ry)/2;

	tmx = rx;
	tmy = ry;

	// automatic quality reduction skip factor
	int rpt=(fmap2/tmap2)/4;// /4;
	// For every point I am drawing from the Page
	// I must Subtract FO From Offset
	// I must then Divide by FM From Magnitude
	// Then Multiply by TM To Magnitude
	// Then add TO To Offset

	// optimize: combine t/f into a single variable up here.  reason against, it would need to be a float?
	// todo: ensure we render at correct aspect ratio, we can just adjust these parameters
	STROKE *stroke=self->strokefirst;
	while(stroke)
	{
		if(stroke->pointcount>=2)
		{
			unsigned char y=stroke->pen_y;
			unsigned char u=stroke->pen_u;
			unsigned char v=stroke->pen_v;

			// this is a good out from the dark function, but not for now..

			/*
			if(rpt>0)
			{
				y/=rpt;
				u=128+((u-128)/rpt);
				v=128+((v-128)/rpt);
			}
			*/
				//v=v-40;
				//if(v<0) v=255+v;

			POINT *p1;
			POINT *p2;
			p1 = stroke->pointfirst;
			
			int p1x=p1->x;	// rotate now..
			int p1y=p1->y;
			
			int p2x;
			int p2y;
			p2 = p1->linknext;
			while(p2)
			{
				
				p2x=p2->x;   // rotate now..
				p2y=p2->y;
				
				// the heavy math part is here...
				int lsx=tox+((p1x-fox)*tmx/fmx);
				int lsy=toy+((p1y-foy)*tmy/fmy);
				int lex=tox+((p2x-fox)*tmx/fmx);
				int ley=toy+((p2y-foy)*tmy/fmy);


				canvas_linecolor(lsx,lsy,lex,ley,    y,u,v);
				//canvas_line(lsx,lsy,lex,ley,    y);

				p1=p2;
				p2=p2->linknext;
				
				p1x=p2x;
				p1y=p2y;

				if(p2)
				{	// move us on, at this point, rendering half resolution is ok :)
					//todo: make this work properly, dropoff with scale is too steep
					int cnt=rpt;
					while(p2->linknext && cnt-->0)
						p2=p2->linknext;	
					// i could do this by a factor of anything and reduce resolution as difference increases
				}
			}
		}
		stroke=stroke->linknext;
	}

}

