#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys
import time
import os
from PyQt4 import QtCore
from PyQt4 import QtGui
from PyQt4.QtCore import Qt
from PyQt4.QtGui import QApplication
try:
    from PyQt4.QtMaemo5 import QMaemo5InformationBox
except:
    pass


greekSymbols = ["alpha", "beta", "chi", "delta", "Delta", "epsilon", "eta", \
"gamma", "Gamma", "iota", "kappa", "nu", "omega", "Omega", "omicron", "phi", \
"Phi", "pi", "Pi", "psi", "Psi", "rho", "sigma", "Sigma", "tau", "theta", \
"upsilon", "xi", "Xi"]

sympy_symbols_functions = ["acos()", "acosh()", "acot()", "acoth()", "asin()", \
"asinh()", "atan()", "atanh()", "cos()", "cosh()", "cot()", "coth()", \
"DiracDelta()", "E", "EulerGamma", "exp()", "factorial()", "fibonacci()", \
"gamma()", "GoldenRatio", "harmonic()", "hermite(n,x)", "I", "im()", \
"legendre(n,x)", "ln()", "ln(n,b)", "log()", "log(n,b)", "oo", "re()", "sin()", \
"sinh()", "sqrt()", "tan()", "tanh()", "zeta()"]

historyExpressionSamples = ["1/x", "(1+3/n)**n", "(7*x**2-4*x+2)/(5*x**2+9)", \
"exp(-x)", "log(x)", "sin(x)/x", "(sin(x+delta)-sin(x))/delta", \
"(x-3)**2/(x**2-9)"]

historyPointSamples = ["0", "oo", "-oo", "pi", "pi/2", "-pi", "-pi/2", "E", "I"]

directionType = {'bilateral':0, 'left':1, 'right':2}
simplifyType = {'none':0, 'expandterms':1, 'simplifyterms':2, 'expandall':3, \
                'simplifyall':4}
outputType = {'simple':0, 'bidimensional':1, 'typesetting':2, 'latex':3, \
              'mathml':4, 'c':5, 'fortran':6}
limitErrorMessage = u'Error: limit not calculated'

zoomFontSize = (10, 12, 14, 16, 20, 24, 28, 32)
zoomNumColumnsMaemo4 = (84, 68, 62, 52, 43, 36, 32, 27)
zoomNumColumnsMaemo5 = (97, 77, 70, 59, 49, 41, 36, 30)

maemo5InformationBoxTime = 3000


class LimitWindow(QtGui.QMainWindow):
    def __init__(self, parent=None):
        QtGui.QMainWindow.__init__(self, parent)
        self.setWindowTitle("Limit")

        self.window = QtGui.QWidget()
        self.setCentralWidget(self.window)

        if os.path.exists("/proc/component_version"):
            self.isMaemo = True
            try:
                self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
                self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
                self.zoomNumColumns = zoomNumColumnsMaemo5
                self.isMaemo5 = True
            except:
                self.isMaemo5 = False
                self.zoomNumColumns = zoomNumColumnsMaemo4
        else:
            self.zoomNumColumns = zoomNumColumnsMaemo5
            self.isMaemo = False
            self.isMaemo5 = False

        if not self.isMaemo:
            self.resize(800, 600)

        self.settings = QtCore.QSettings()
        self.loadSettings()
        # Setting the pretty printing of SymPy to use unicode characters and correct number of columns.
        init_printing(use_unicode=True, num_columns=self.zoomNumColumns[self.zoomLevel])

        self.historyInitExpression = QtCore.QStringList()
        for i in greekSymbols+sympy_symbols_functions+historyExpressionSamples:
            self.historyInitExpression.append(QtCore.QString(i))
        self.historyExpression=self.historyInitExpression+self.historyUserExpression

        self.historyVariable = QtCore.QStringList()
        for i in greekSymbols:
            self.historyVariable.append(QtCore.QString(i))

        self.historyPoint = QtCore.QStringList()
        for i in historyPointSamples+greekSymbols:
            self.historyPoint.append(QtCore.QString(i))

        self.directionLimit = 0
        self.nonCalculatedLimit = u""
        self.resultLimit = u""
        self.resultLimitSimp = u""
        self.timeLimit = 0.0
        self.createMenuAndActions()
        self.createItems()
        self.createLayouts()

    def createItems(self):
        self.expressionEditCompleter = QtGui.QCompleter()
        self.expressionEditCompleter.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.expressionEditModel = QtGui.QStringListModel()
        self.expressionEditCompleter.setModel(self.expressionEditModel)
        self.expressionEditModel.setStringList(self.historyExpression)

        self.expressionLabel = QtGui.QLabel(self.tr("Expression"))
        self.expressionEdit = QtGui.QLineEdit("")
        self.expressionEdit.setCompleter(self.expressionEditCompleter)

        self.variableEditCompleter = QtGui.QCompleter()
        self.variableEditCompleter.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.variableEditModel = QtGui.QStringListModel()
        self.variableEditCompleter.setModel(self.variableEditModel)
        self.variableEditModel.setStringList(self.historyVariable)

        self.variableLabel = QtGui.QLabel("Variable")
        self.variableEdit = QtGui.QLineEdit("x")
        self.variableEdit.setCompleter(self.variableEditCompleter)

        self.pointEditCompleter = QtGui.QCompleter()
        self.pointEditCompleter.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.pointEditModel = QtGui.QStringListModel()
        self.pointEditCompleter.setModel(self.pointEditModel)
        self.pointEditModel.setStringList(self.historyPoint)

        self.pointLabel = QtGui.QLabel(self.tr("Point"))
        self.pointEdit = QtGui.QLineEdit("0")
        self.pointEdit.setCompleter(self.pointEditCompleter)

        self.directionLabel = QtGui.QLabel(self.tr("Direction"))
        self.directionComboBox = QtGui.QComboBox()
        self.directionComboBox.addItem(self.tr("Bilateral"))
        self.directionComboBox.addItem(self.tr("Left"))
        self.directionComboBox.addItem(self.tr("Right"))
        self.directionComboBox.setCurrentIndex(self.directionLimit)

        self.font = QtGui.QFont()
        self.font.setFamily("DejaVu Sans Mono")
        if QtGui.QFontInfo(self.font).family() == "DejaVu Sans Mono":
            self.dejavuSansMono = True
        else:
            self.font.setFamily("Monospace")
            self.dejavuSansMono = False
        self.font.setPointSize(zoomFontSize[self.zoomLevel])
        self.resultEdit = QtGui.QTextBrowser()
        self.resultEdit.setReadOnly(False)
        self.resultEdit.setFont(self.font)

        # Connect buttons
        self.connect(self.expressionEdit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.variableEdit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.pointEdit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.directionComboBox,QtCore.SIGNAL("activated(int)"),self.changeDirectionLimit)

    def createLayouts(self):    
        self.h_box_expression = QtGui.QHBoxLayout()
        self.h_box_expression.addWidget(self.expressionLabel)
        self.h_box_expression.addWidget(self.expressionEdit)
        self.h_box_variable = QtGui.QHBoxLayout()
        self.h_box_variable.addWidget(self.variableLabel)
        self.h_box_variable.addWidget(self.variableEdit)
        self.h_box_variable.addWidget(self.pointLabel)
        self.h_box_variable.addWidget(self.pointEdit)
        self.h_box_variable.addWidget(self.directionLabel)
        self.h_box_variable.addWidget(self.directionComboBox)
        self.v_box = QtGui.QVBoxLayout()
        self.v_box.addLayout(self.h_box_expression)
        self.v_box.addLayout(self.h_box_variable)
        self.v_box.addWidget(self.resultEdit)
        self.window.setLayout(self.v_box)     

    def createMenuAndActions(self):
        if not self.isMaemo:
            fileMenu = self.menuBar().addMenu(self.tr("&File"))
#           saveAction = fileMenu.addAction(self.tr("&Salvar"))
#           shareAction = fileMenu.addAction(self.tr("&Share"))
            fileMenu.addSeparator()
            exitAction = QtGui.QAction(self.tr("E&xit"), self, shortcut="Ctrl+Q")
            fileMenu.addAction(exitAction)
      
        viewMenu = self.menuBar().addMenu(self.tr("&View"))
        zoomInAction = viewMenu.addAction(self.tr("Zoom in"))
        zoomOutAction = viewMenu.addAction(self.tr("Zoom out"))

        limitMenu = self.menuBar().addMenu(self.tr("&Limit"))
        configAction =  limitMenu.addAction(self.tr("Settings"))

        helpMenu = self.menuBar().addMenu(self.tr("&Help"))
        helpAction = helpMenu.addAction(self.tr("&Help"))
        helpMenu.addSeparator()
        aboutAction = helpMenu.addAction(self.tr("&About"))
        aboutQtAction = helpMenu.addAction(self.tr("About &Qt"))

        if self.isMaemo and (not self.isMaemo5):
            exitAction = QtGui.QAction(self.tr("E&xit"), self, shortcut="Ctrl+Q")
            self.menuBar().addAction(exitAction)

        self.connect(zoomInAction, QtCore.SIGNAL("triggered()"), self.zoomInMenu)
        self.connect(zoomOutAction, QtCore.SIGNAL("triggered()"), self.zoomOutMenu)
        if not self.isMaemo5:
            self.connect(exitAction, QtCore.SIGNAL("triggered()"), app, QtCore.SLOT("quit()"))
        self.connect(configAction, QtCore.SIGNAL("triggered()"), self.showConfigWindow)
        self.connect(helpAction, QtCore.SIGNAL("triggered()"), self.showHelpBox)
        self.connect(aboutAction, QtCore.SIGNAL("triggered()"), self.showAboutBox)
        self.connect(aboutQtAction, QtCore.SIGNAL("triggered()"), QtGui.qApp.aboutQt)

    def showHelpBox(self):
        QtGui.QMessageBox.about(self, self.tr("Help on Limit"),
            u'Mathematical operators : + - * / ** (power)\nExamples of functions : sqrt, exp, log, sin, acos\n' +\
            u'Examples of expressions : 1/x, sin(x)/x, exp(-x), (7*x**2-4*x+2)/(5*x**2+9)\n'+\
            u'Examples of variables : x, t, delta, Omega\n'+u'Examples of points : 0, 1, -oo, pi/2\n'+\
            u'Look at the SymPy site: http://sympy.org/')

    def showAboutBox(self):
        versionLimit = '1.0.0'
        siteLimit = 'http://www.RobertoColistete.net/Limit\n'
        infoLimit = self.tr("Calculation of limits using ")
        infoPython = "Python v"+python_version()
        infoSymPy = "Sympy v"+__version__+" (http://sympy.org/)"
        QtGui.QMessageBox.about(self, self.tr("About Limit"),
            'Limit v'+versionLimit+' - LGPLv3 (C) 2011 Roberto Colistete Jr.\n'+\
            siteLimit+infoLimit+infoPython+" and "+infoSymPy+u'\n\n* In loving memory of my wife Lorena *')

    def showConfigWindow(self):
        if self.isMaemo5:
            self.configWin = QtGui.QMainWindow(self)
            self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
            self.configWin.setAttribute(Qt.WA_Maemo5StackedWindow)
            self.configWin.window = QtGui.QWidget()
            self.configWin.setCentralWidget(self.configWin.window)
        else:
            self.configWin = QtGui.QDialog(self)

        self.configWin.setWindowTitle(self.tr("Limit (Settings)"))
        self.configWin.v_box = QtGui.QVBoxLayout()

        self.configWin.showLimitCheckBox = QtGui.QCheckBox(self.tr("Show not calculated limit before result"))
        if self.showLimit:
            self.configWin.showLimitCheckBox.toggle()

        self.configWin.showTimeCheckBox = QtGui.QCheckBox(self.tr("Show calculation time before limit result"))
        if self.showTime:
            self.configWin.showTimeCheckBox.toggle()

        self.configWin.numerApproxCheckBox = QtGui.QCheckBox(self.tr("Numerical approximation of the limit result"))
        if self.numerApprox:
            self.configWin.numerApproxCheckBox.toggle()

        self.configWin.h_box_numDig = QtGui.QHBoxLayout()
        self.configWin.numDigLabel = QtGui.QLabel(self.tr("Number of digits for numerical approximation"))
        self.configWin.numDigSpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding, QtGui.QSizePolicy.Minimum)
        self.configWin.numDigEdit = QtGui.QLineEdit(str(self.numDig))
        self.configWin.numDigEdit.setValidator(QtGui.QIntValidator(1,1000000,self.configWin.numDigEdit))
        self.configWin.numDigEdit.setMaximumSize(QtCore.QSize(100,100))
        self.configWin.h_box_numDig.addWidget(self.configWin.numDigLabel)
        self.configWin.h_box_numDig.addWidget(self.configWin.numDigEdit)
        self.configWin.h_box_numDig.addItem(self.configWin.numDigSpacer)

        self.configWin.h_box_simplify = QtGui.QHBoxLayout()
        self.configWin.simplifyLabel = QtGui.QLabel(self.tr("Simplification for non-numerical limit"))
        self.configWin.simplifySpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding,\
                                                          QtGui.QSizePolicy.Minimum)
        self.configWin.simplifyComboBox = QtGui.QComboBox()
        self.configWin.simplifyComboBox.addItem(self.tr("None"))
        self.configWin.simplifyComboBox.addItem(self.tr("Expand terms"))
        self.configWin.simplifyComboBox.addItem(self.tr("Simplify terms"))
        self.configWin.simplifyComboBox.addItem(self.tr("Expand all"))
        self.configWin.simplifyComboBox.addItem(self.tr("Simplify all"))
        self.configWin.simplifyComboBox.setCurrentIndex(self.simplifyResult)
        self.configWin.h_box_simplify.addWidget(self.configWin.simplifyLabel)
        self.configWin.h_box_simplify.addWidget(self.configWin.simplifyComboBox)
        self.configWin.h_box_simplify.addItem(self.configWin.simplifySpacer)

        self.configWin.h_box_output = QtGui.QHBoxLayout()
        self.configWin.outputLabel = QtGui.QLabel(self.tr("Output type for limit result"))
        self.configWin.outputSpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding, QtGui.QSizePolicy.Minimum)
        self.configWin.outputComboBox = QtGui.QComboBox()
        self.configWin.outputComboBox.addItem(self.tr("Simple"))
        self.configWin.outputComboBox.addItem(self.tr("Bidimensional"))
        self.configWin.outputComboBox.addItem(self.tr("Typesetting"))
        self.configWin.outputComboBox.addItem("LaTex")
        self.configWin.outputComboBox.addItem("MathML")
        self.configWin.outputComboBox.addItem("C")
        self.configWin.outputComboBox.addItem("Fortran")
        self.configWin.outputComboBox.setCurrentIndex(self.outputResult)
        self.configWin.h_box_output.addWidget(self.configWin.outputLabel)
        self.configWin.h_box_output.addWidget(self.configWin.outputComboBox)
        self.configWin.h_box_output.addItem(self.configWin.outputSpacer)

        if not self.isMaemo5:
            self.configWin.h_box_close = QtGui.QHBoxLayout()
            self.configWin.closeButton = QtGui.QPushButton("Close")
            self.configWin.h_box_close.addStretch(1)
            self.configWin.h_box_close.addWidget(self.configWin.closeButton)

        self.configWin.v_box.addWidget(self.configWin.showLimitCheckBox)
        self.configWin.v_box.addWidget(self.configWin.showTimeCheckBox)
        self.configWin.v_box.addWidget(self.configWin.numerApproxCheckBox)
        self.configWin.v_box.addLayout(self.configWin.h_box_numDig)
        self.configWin.v_box.addLayout(self.configWin.h_box_simplify)
        self.configWin.v_box.addLayout(self.configWin.h_box_output)
        if not self.isMaemo5:
            self.configWin.v_box.addStretch(1)
            self.configWin.v_box.addSpacing(12)
            self.configWin.v_box.addLayout(self.configWin.h_box_close)
            self.configWin.setLayout(self.configWin.v_box)
        else:
            self.configWin.window.setLayout(self.configWin.v_box)

        self.connect(self.configWin.showLimitCheckBox, QtCore.SIGNAL("stateChanged(int)"), self.toggleShowLimit)
        self.connect(self.configWin.showTimeCheckBox, QtCore.SIGNAL("stateChanged(int)"), self.toggleShowTime)
        self.connect(self.configWin.numerApproxCheckBox, QtCore.SIGNAL("stateChanged(int)"), self.toggleNumerApprox)
        self.connect(self.configWin.numDigEdit, QtCore.SIGNAL("textEdited (const QString&)"),self.changeNumDig)
        self.connect(self.configWin.simplifyComboBox, QtCore.SIGNAL("activated(int)"), self.changeSimplifyResult)
        self.connect(self.configWin.outputComboBox, QtCore.SIGNAL("activated(int)"), self.changeOutputResult)
        if not self.isMaemo5:
            self.configWin.closeButton.clicked.connect(self.configWin.close)
        self.configWin.show()

    def loadSettings(self):
        self.zoomLevel = self.settings.value('zoomLevel',3).toInt()[0]
        self.showLimit = self.settings.value('ShowLimit',True).toBool()
        self.showTime = self.settings.value('ShowTime',True).toBool()
        self.numerApprox = self.settings.value('NumerApprox',False).toBool()
        self.numDig = self.settings.value('NumDig',15).toInt()[0]
        self.simplifyResult = self.settings.value('SimplifyResult',simplifyType['simplifyterms']).toInt()[0]
        self.outputResult = self.settings.value('OutputResult',outputType['bidimensional']).toInt()[0]
        self.historyUserExpression = self.settings.value('historyUserExpression','').toStringList()

    def saveSettings(self):
        self.settings.setValue('zoomLevel',self.zoomLevel)
        self.settings.setValue('ShowLimit',self.showLimit) 	
        self.settings.setValue('ShowTime',self.showTime)
        self.settings.setValue('numerApprox',self.numerApprox) 	
        self.settings.setValue('NumDig',self.numDig)
        self.settings.setValue('SimplifyResult',self.simplifyResult)
        self.settings.setValue('OutputResult',self.outputResult)
        self.settings.setValue('historyUserExpression',self.historyUserExpression)

    def zoomInMenu(self):
        if self.zoomLevel < (len(zoomFontSize)-1):
            self.zoomLevel += 1
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nZoom in to font size "+str(zoomFontSize[self.zoomLevel])+"\n"), \
                                                  maemo5InformationBoxTime)
                QApplication.processEvents()
            self.font.setPointSize(zoomFontSize[self.zoomLevel])
            self.resultEdit.setFont(self.font)
            # Setting the pretty printing of SymPy to use unicode characters and correct number of columns.
            init_printing(use_unicode=True, num_columns=self.zoomNumColumns[self.zoomLevel])
            self.updateResult(flagCalculate=False)
            self.saveSettings()          

    def zoomOutMenu(self):
        if self.zoomLevel > 0:
            self.zoomLevel -= 1
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nZoom out to font size "+str(zoomFontSize[self.zoomLevel])+"\n"), \
                                                  maemo5InformationBoxTime)
                QApplication.processEvents()
            self.font.setPointSize(zoomFontSize[self.zoomLevel])
            self.resultEdit.setFont(self.font)
            # Setting the pretty printing of SymPy to use unicode characters and correct number of columns.
            init_printing(use_unicode=True, num_columns=self.zoomNumColumns[self.zoomLevel])
            self.updateResult(flagCalculate=False)
            self.saveSettings()          

    def toggleShowLimit(self, status):
        if status == Qt.Checked:
            self.showLimit = True
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nNot calculated limit is visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        else:
            self.showLimit = False
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nNot calculated limit is not visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        self.updateResult(flagCalculate=False)
        self.saveSettings()

    def toggleShowTime(self, status):
        if status == Qt.Checked:
            self.showTime = True
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nCalculation time is visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        else:
            self.showTime = False
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nCalculation time is not visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        self.updateResult(flagCalculate=False)
        self.saveSettings()

    def toggleNumerApprox(self, status):
        if status == Qt.Checked:
            self.numerApprox = True
        else:
            self.numerApprox= False
        self.saveSettings()

    def changeNumDig(self):
        if self.configWin.numDigEdit.text():
            self.numDig = int(self.configWin.numDigEdit.text())
        else:
            self.numDig = 0
        self.saveSettings()

    def changeSimplifyResult(self, option):
        self.simplifyResult = option
        self.saveSettings()

    def changeOutputResult(self, option):
        self.outputResult = option
        if self.isMaemo5:
            if self.outputResult == outputType['simple']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nLimit result converted to simple output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['bidimensional']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nLimit result converted to bidimensional output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['typesetting']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nLimit result converted to typesetting output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['latex']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nLimit result converted to LaTeX output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['mathml']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nLimit result converted to MathML output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['c']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nLimit result converted to C output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['fortran']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nLimit result converted to Fortran output\n"), maemo5InformationBoxTime)
            QApplication.processEvents()
        self.updateResult(flagCalculate=False)
        self.saveSettings()

    def changeDirectionLimit(self, option):
        self.directionLimit = option
        self.updateResult()

    def mapexpr(self,expr,func):
        if isinstance(expr,Add):
            return apply(Add,map(func,expr.args))
        else:
            return func(expr)

    def updateHistoriesAndResult(self):
        text = unicode(self.expressionEdit.text())
        if not text in self.historyExpression:
            self.historyUserExpression.prepend(text)
            self.historyUserExpression=self.historyUserExpression[:100]
            self.historyExpression=self.historyInitExpression+self.historyUserExpression
            self.expressionEditModel.setStringList(self.historyExpression)
        self.saveSettings()
        self.updateResult()

    def updateResult(self, flagCalculate=True):
        # On SymPy v0.7.0 and v0.7.1, 'sympify(limit(1/x,x,0,dir=\"-\")' does not work because of the embedded string.
        # So these 3 functions were created inside this functions and passed as locals() to "sympify". 
        def limitpos(func,var,varvalue):
            return limit(func,var,varvalue,dir='+')

        def limitneg(func,var,varvalue):
            return limit(func,var,varvalue,dir='-')

        def limitbi(func,var,varvalue):
            limpos = limit(func,var,varvalue,dir='+')
            limneg = limit(func,var,varvalue,dir='-')
            if limpos == limneg:
                return limpos
            else:
                return None

        if flagCalculate:
            self.resultEdit.clear()
            self.resultEdit.setTextColor(QtGui.QColor("darkGreen"))
            self.resultEdit.append(u"Calculating limit...")
            self.resultEdit.setTextColor(QtGui.QColor("black"))
            QApplication.processEvents()
            if self.isMaemo5:
                self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator,True)
                QApplication.processEvents()
    	    timet1=time.time()
            expressionLimit = unicode(self.expressionEdit.text())
            variableLimit = unicode(self.variableEdit.text()).strip()
            pointLimit = unicode(self.pointEdit.text()).strip()
            limitExpr = u'('+expressionLimit+u','+variableLimit+u','+pointLimit
            if self.directionLimit != directionType['bilateral']:
                limitExpr += u',dir=\"'
                if self.directionLimit == directionType['left']:
                    limitExpr += u'-\"'
                elif self.directionLimit == directionType['right']:
                    limitExpr += u'+\"'
            limitExpr += u')'
            if self.directionLimit == directionType['bilateral']:
                try:
                    self.nonCalculatedLimit = sympify(u'Limit'+limitExpr)
                except:
                    self.nonCalculatedLimit = u'Limit'+limitExpr
            else:
            # "Limit" has a bug not showing the "+" and "-" above the point value.
                self.nonCalculatedLimit = u'Limit'+limitExpr
            # On SymPy v0.7.0 and v0.7.1, 'sympify(limit(1/x,x,0,dir=\"-\")' does not work because of the embedded string.
            limitExpr = u'('+expressionLimit+u','+variableLimit+u','+pointLimit+u')'
            if self.directionLimit == directionType['bilateral']:
                try:
                    self.resultLimit = sympify(u'limitbi'+limitExpr,locals=locals())
                except:
                    self.resultLimit = limitErrorMessage
                if self.resultLimit is None:
                    self.resultLimit = u"Bilateral limit does not exist because the limits from the left and right are different."
            elif self.directionLimit == directionType['left']:
                try:
                    self.resultLimit = sympify(u'limitneg'+limitExpr,locals=locals())
                except:
                    self.resultLimit = limitErrorMessage
            elif self.directionLimit == directionType['right']:
                try:
                    self.resultLimit = sympify(u'limitpos'+limitExpr,locals=locals())
                except:
                    self.resultLimit = limitErrorMessage
            if (type(self.resultLimit) != unicode) and self.numerApprox:
                try:
                    self.resultLimit = sympify(u'N('+unicode(str(self.resultLimit))+u','+unicode(str(self.numDig))+u')')
                except:
                    self.resultLimit = limitErrorMessage
            if (self.resultLimit) and (type(self.resultLimit) != unicode) and (not self.numerApprox):
                if self.simplifyResult == simplifyType['none']:
                    self.resultLimitSimp = sympify(self.resultLimit)
                elif self.simplifyResult == simplifyType['expandterms']:
                    self.resultLimitSimp = sympify(self.mapexpr(self.resultLimit,expand))
                elif self.simplifyResult == simplifyType['simplifyterms']:
                    self.resultLimitSimp = sympify(self.mapexpr(self.resultLimit,simplify))
                elif self.simplifyResult == simplifyType['expandall']:
                    self.resultLimitSimp = sympify(expand(self.resultLimit))
                elif self.simplifyResult == simplifyType['simplifyall']:
                    self.resultLimitSimp = sympify(simplify(self.resultLimit))
            else:
                self.resultLimitSimp = self.resultLimit
    	    timet2=time.time()
            self.timeLimit=timet2-timet1
            if self.isMaemo5:
                self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator,False)
                QApplication.processEvents()
        self.nonCalculatedLimitOutput = self.nonCalculatedLimit
        self.resultOutput = self.resultLimitSimp
        if self.outputResult == outputType['bidimensional']:
            if self.directionLimit == directionType['bilateral']:
                self.nonCalculatedLimitOutput = self.fixUnicodeText(printing.pretty(self.nonCalculatedLimit))
            if (type(self.resultLimitSimp) != unicode):
                self.resultOutput = self.fixUnicodeText(printing.pretty(self.resultLimitSimp))
        elif self.outputResult == outputType['typesetting']:
            if self.directionLimit == directionType['bilateral']:
                self.nonCalculatedLimitOutput = self.fixUnicodeText(printing.pretty(self.nonCalculatedLimit))
            if (type(self.resultLimitSimp) != unicode):
                self.resultOutput = self.fixUnicodeText(printing.pretty(self.resultLimitSimp))
        elif self.outputResult == outputType['latex']:
            if self.directionLimit == directionType['bilateral']:
                self.nonCalculatedLimitOutput = latex(self.nonCalculatedLimit)
            if (type(self.resultLimitSimp) != unicode):
                self.resultOutput = latex(self.resultLimitSimp)
        elif self.outputResult == outputType['mathml']:
            if self.directionLimit == directionType['bilateral']:
                self.nonCalculatedLimitOutput = mathml(self.nonCalculatedLimit)
            if (type(self.resultLimitSimp) != unicode):
                self.resultOutput = mathml(self.resultLimitSimp)
        elif self.outputResult == outputType['c']:
            if (type(self.resultLimitSimp) != unicode):
                self.resultOutput = ccode(self.resultLimitSimp)
        elif self.outputResult == outputType['fortran']:
            if (type(self.resultLimitSimp) != unicode):
                self.resultOutput = fcode(self.resultLimitSimp)
        self.resultEdit.clear()
        if self.showTime:
            if self.timeLimit > 0.0:
                self.resultEdit.setTextColor(QtGui.QColor("darkGreen"))
                self.resultEdit.append(u"Calculated after %f seconds :" % self.timeLimit)
                self.resultEdit.setTextColor(QtGui.QColor("black"))
        if self.showLimit and self.nonCalculatedLimitOutput:
            self.resultEdit.setTextColor(QtGui.QColor("Blue"))
            self.resultEdit.append(u"%s\n=" % self.nonCalculatedLimitOutput)
            self.resultEdit.setTextColor(QtGui.QColor("black"))
        if (type(self.resultLimitSimp) != unicode):
            self.resultEdit.setTextColor(QtGui.QColor("black"))
        else:
            self.resultEdit.setTextColor(QtGui.QColor("Red"))
        self.resultEdit.append(u"%s" % unicode(self.resultOutput))

    def fixUnicodeText(self, str):
        if self.isMaemo:
            str = str.replace(u"⎽","_")
        if self.isMaemo5 and (not self.dejavuSansMono):
            str = str.replace(u"⎮",u'\u2223')
            str = str.replace(u"⎜",u'\u2223')
            str = str.replace(u"⎟",u'\u2223')
            str = str.replace(u"⎢",u'\u2223')
            str = str.replace(u"⎥",u'\u2223')
            str = str.replace(u"⎛",u"⌈")
            str = str.replace(u"⎡",u"⌈")
            str = str.replace(u"⎝",u"⌊")
            str = str.replace(u"⎣",u"⌊")
            str = str.replace(u"⎞",u"⌉")
            str = str.replace(u"⎤",u"⌉")
            str = str.replace(u"⎠",u"⌋")
            str = str.replace(u"⎦",u"⌋")
        str = str.replace(u"ℯ","e")
        str = str.replace(u"ⅈ","i")
        return str

if __name__ == "__main__":
    #This function means this was run directly, not called from another python file.
    app = QtGui.QApplication(sys.argv)
    app.setOrganizationDomain("RobertoColistete.net")
    app.setApplicationName("Limit")

    from platform import python_version
    from sympy import *
    from sympy import __version__
    from sympy.interactive.printing import init_printing
    from sympy.printing.mathml import mathml

    myapp = LimitWindow()
    myapp.show()
    sys.exit(app.exec_())
