/**
 * Copyright (C) 2006 by Koos Vriezen <koos ! vriezen ? xs4all ! nl>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License version 2 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
//#include <errno.h>
#include <string.h>

#include <gtk/gtk.h>

#include "kmplayer_lists.h"
#include "kmplayer.h"
#include "kmplayercontrol.h"
#include "actor.h"

using namespace KMPlayer;

KDE_NO_CDTOR_EXPORT FileDocument::FileDocument (short i, const String &s, KMPlayer::Control *src)
 : KMPlayer::SourceDocument (src, s), load_tree_version ((unsigned int)-1) {
    id = i;
}

KDE_NO_EXPORT KMPlayer::Node *FileDocument::childFromTag(const String &tag) {
    if (tag == nodeName ())
        return this;
    return NULL;
}

void FileDocument::readFromFile (const String & fn) {
    File file (fn);
    debugLog () << "readFromFile " << fn << endl;
    if (file.exists ()) {
        file.open (IO_ReadOnly);
        ByteArray data = file.readAll ();
        TextStream inxml (data);
        KMPlayer::readXML (this, inxml, String(), false);
        normalize ();
    }
    load_tree_version = m_tree_version;
}

void FileDocument::writeToFile (const String & fn) {
    File file (fn);
    debugLog () << "writeToFile " << fn << m_tree_version << " " << load_tree_version << endl;
    file.open (IO_WriteOnly);
    String xml = outerXML ();
    const char * utf = (const char *) xml;
    file.writeBlock (utf, strlen (utf));
}

KDE_NO_CDTOR_EXPORT Recents::Recents (Application *a)
    : FileDocument (id_node_recent_document, "recents://"),
      app(a) {
    title = "Most Recent";
}

KDE_NO_EXPORT void Recents::activate () {
    if (!resolved)
        defer ();
}

KDE_NO_EXPORT void Recents::defer () {
    if (!resolved) {
        resolved = true;
        readFromFile (path);
    }
}

KDE_NO_EXPORT KMPlayer::Node *Recents::childFromTag (const String & tag) {
    // kdDebug () << nodeName () << " childFromTag " << tag << endl;
    if (tag == "item")
        return new Recent (m_doc, app);
    else if (tag == "group")
        return new Group (m_doc, app);
    return FileDocument::childFromTag (tag);
}

KDE_NO_EXPORT void *Recents::message (KMPlayer::MessageType msg, void *data) {
    if (KMPlayer::MsgChildFinished) {
        finish ();
        return NULL;
    }
    return FileDocument::message (msg, data);
}

KDE_NO_CDTOR_EXPORT
Recent::Recent (KMPlayer::NodePtr & doc, Application * a, const String &url)
  : KMPlayer::Mrl (doc, id_node_recent_node), app (a) {
    src = url;
    setAttribute (KMPlayer::StringPool::attr_url, url);
}

KDE_NO_EXPORT void Recent::closed () {
    if (src.isEmpty ())
        src = getAttribute (KMPlayer::StringPool::attr_url);
    if (title.isEmpty ())
        title = getAttribute (KMPlayer::StringPool::attr_title);
}

KDE_NO_EXPORT void Recent::activate () {
    Document *doc = new SourceDocument (CONTROL (app), src);
    doc->setCaption (title);
    String pos = getAttribute ("clipBegin");
    if (!pos.isEmpty ())
        doc->clip_begin = parseTimeString (pos);
    CONTROL(app)->openDocument (doc, NULL);
}

KDE_NO_CDTOR_EXPORT
Group::Group (KMPlayer::NodePtr & doc, Application * a, const String & pn)
  : KMPlayer::Title (doc, KMPlayer::id_node_group_node), app (a) {
    title = pn;
    if (!pn.isEmpty ())
        setAttribute (KMPlayer::StringPool::attr_title, pn);
}

KDE_NO_EXPORT KMPlayer::Node *Group::childFromTag (const String & tag) {
    if (tag == "item")
        return new Recent (m_doc, app);
    else if (tag == "group")
        return new Group (m_doc, app);
    return NULL;
}

KDE_NO_EXPORT void Group::closed () {
    if (title.isEmpty ())
        title = getAttribute (KMPlayer::StringPool::attr_title);
}

// gconftool -s -t string /apps/maemo/kmplayer/playlist_file /home/koos/playlist.xml
KDE_NO_EXPORT void Playlist::defer () {
    if (playmode)
        KMPlayer::Document::defer ();
    else if (!resolved) {
        resolved = true;
        readFromFile (path);
    }
}

KDE_NO_EXPORT void Playlist::activate () {
    if (playmode)
        KMPlayer::Document::activate ();
    else if (!resolved)
        defer ();
}

KDE_NO_CDTOR_EXPORT Playlist::Playlist (Application *a, KMPlayer::Control *s, bool plmode)
    : FileDocument (KMPlayer::id_node_playlist_document, "Playlist://", s),
      app(a),
      playmode (plmode) {
    if (!plmode)
        title = "Favorites";
}

KDE_NO_EXPORT KMPlayer::Node *Playlist::childFromTag (const String & tag) {
    // kdDebug () << nodeName () << " childFromTag " << tag << endl;
    const char * name = (const char *) tag;
    if (!strcmp (name, "item"))
        return new PlaylistItem (m_doc, app, playmode);
    else if (!strcmp (name, "group"))
        return new PlaylistGroup (m_doc, app, playmode);
    else if (!strcmp (name, "object"))
        return new HtmlObject (m_doc, app, playmode);
    return FileDocument::childFromTag (tag);
}

KDE_NO_EXPORT void *Playlist::message (KMPlayer::MessageType msg, void *data) {
    if (KMPlayer::MsgChildFinished && !playmode) {
        finish ();
        return NULL;
    }
    return FileDocument::message (msg, data);
}

KDE_NO_CDTOR_EXPORT
PlaylistItemBase::PlaylistItemBase (KMPlayer::NodePtr &d, short i, Application *a, bool pm)
  : KMPlayer::Mrl (d, i), app (a), playmode (pm) {
}

KDE_NO_EXPORT void PlaylistItemBase::activate () {
    if (playmode) {
        Mrl::activate ();
    } else {
        NodePtr pl = new Playlist (app, CONTROL(app), true);
        String data;
        String pn;
        if (parentNode ()->id == KMPlayer::id_node_group_node &&
                (previousSibling() || nextSibling ())) {
            data = String ("<playlist>") +
                parentNode ()->innerXML () +
                String ("</playlist>");
            pn = parentNode ()->caption ();
        } else {
            data = outerXML ();
            pn = title.isEmpty () ? src : title;
        }
        pl->setCaption (pn);
        //debugLog () << "cloning to " << data << endl;
        TextStream inxml (data);
        KMPlayer::readXML (pl, inxml, String(), false);
        pl->normalize ();
        Node *cur = pl->firstChild ();
        pl->mrl ()->resolved = !!cur;
        if (parentNode ()->id == KMPlayer::id_node_group_node && cur) {
            KMPlayer::Node *sister = parentNode ()->firstChild ();
            while (sister && cur && sister != this) {
                sister = sister->nextSibling ();
                cur = cur->nextSibling ();
            }
        }
        CONTROL(app)->openDocument (pl, cur);
    }
}

KDE_NO_EXPORT void PlaylistItemBase::closed () {
    if (title.isEmpty ())
        title = getAttribute (KMPlayer::StringPool::attr_title);
}

KDE_NO_CDTOR_EXPORT
PlaylistItem::PlaylistItem (KMPlayer::NodePtr & doc, Application * a, bool pm, const String &url)
  : PlaylistItemBase (doc, KMPlayer::id_node_playlist_item, a, pm) {
    src = url;
    setAttribute (KMPlayer::StringPool::attr_url, url);
}

KDE_NO_EXPORT void PlaylistItem::closed () {
    if (src.isEmpty ())
        src = getAttribute (KMPlayer::StringPool::attr_url);
    PlaylistItemBase::closed ();
}

KDE_NO_EXPORT void PlaylistItem::begin () {
    if (playmode && firstChild ())
        firstChild ()->activate ();
    else
        Mrl::begin ();
}

KDE_NO_CDTOR_EXPORT
PlaylistGroup::PlaylistGroup (KMPlayer::NodePtr & doc, Application * a, const String & pn)
 : KMPlayer::Title (doc, KMPlayer::id_node_group_node),
   app (a),
   playmode (false) {
    title = pn;
    if (!pn.isEmpty ())
        setAttribute (KMPlayer::StringPool::attr_title, pn);
}

KDE_NO_CDTOR_EXPORT
PlaylistGroup::PlaylistGroup (KMPlayer::NodePtr & doc, Application * a, bool lm)
 : KMPlayer::Title (doc, KMPlayer::id_node_group_node),
   app (a),
   playmode (lm) {
}

KDE_NO_EXPORT KMPlayer::Node *PlaylistGroup::childFromTag (const String & tag) {
    const char * name = (const char *) tag;
    if (!strcmp (name, "item"))
        return new PlaylistItem (m_doc, app, playmode);
    else if (!strcmp (name, "group"))
        return new PlaylistGroup (m_doc, app, playmode);
    else if (!strcmp (name, "object"))
        return new HtmlObject (m_doc, app, playmode);
    return NULL;
}

KDE_NO_EXPORT void PlaylistGroup::closed () {
    if (title.isEmpty ())
        title = getAttribute (KMPlayer::StringPool::attr_title);
}

KDE_NO_CDTOR_EXPORT
HtmlObject::HtmlObject (KMPlayer::NodePtr &doc, Application *a, bool pm)
  : PlaylistItemBase (doc, KMPlayer::id_node_html_object, a, pm) {}

KDE_NO_EXPORT void HtmlObject::activate () {
    if (playmode)
        KMPlayer::Mrl::activate ();
    else
        PlaylistItemBase::activate ();
}

KDE_NO_EXPORT void HtmlObject::closed () {
    for (Node *n = firstChild (); n; n = n->nextSibling ()) {
        if (n->id == KMPlayer::id_node_param) {
            KMPlayer::Element *e = static_cast <KMPlayer::Element *> (n);
            String name = e->getAttribute (KMPlayer::StringPool::attr_name);
            if (name == "type")
                mimetype = e->getAttribute (KMPlayer::StringPool::attr_value);
            else if (name == "movie")
                src = e->getAttribute (KMPlayer::StringPool::attr_value);
        } else if (n->id == KMPlayer::id_node_html_embed) {
            KMPlayer::Element *e = static_cast <KMPlayer::Element *> (n);
            String type = e->getAttribute (KMPlayer::StringPool::attr_type);
            if (!type.isEmpty ())
                mimetype = type;
            String asrc = e->getAttribute (KMPlayer::StringPool::attr_src);
            if (!asrc.isEmpty ())
                src = asrc;
        }
    }
    PlaylistItemBase::closed ();
}

KDE_NO_EXPORT KMPlayer::Node *HtmlObject::childFromTag (const String & tag) {
    const char * name = (const char *) tag;
    if (!strcasecmp (name, "param"))
        return new DarkNode (m_doc, name, KMPlayer::id_node_param);
    else if (!strcasecmp (name, "embed"))
        return new DarkNode (m_doc, name, id_node_html_embed);
    return NULL;
}

Node *Generator::childFromTag (const String &tag) {
    const char *ctag = (const char *) tag;
    if (!strcmp (ctag, "generator"))
        return new GeneratorElement (m_doc, tag, id_node_gen_generator);
    return NULL;
}

String Generator::genReadAsk (Control *ctr, Node *n) {
    String title;
    String desc;
    String type = static_cast<Element*>(n)->getAttribute(StringPool::attr_type);
    String key = static_cast<Element*>(n)->getAttribute ("key");
    String def = static_cast<Element*>(n)->getAttribute ("default");
    String input;
    if (!key.isEmpty ())
        def = m_control->m_config->readEntry (
                String (GCONF_KEY_KEYS) + key, def);
    if (type == "file") {
        input = app_file_chooser (m_control->m_app, def,
                GTK_FILE_CHOOSER_ACTION_OPEN);
    } else if (type == "dir") {
        input = app_file_chooser (m_control->m_app, def,
                GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
        if (!input.isEmpty ())
            input += Char ('/');
    } else {
        for (Node *c = n->firstChild (); c; c = c->nextSibling ())
            switch (c->id) {
                case id_node_gen_title:
                    title = c->innerText ().stripWhiteSpace ();
                    break;
                case id_node_gen_description:
                    desc = c->innerText ().stripWhiteSpace ();
                    break;
            }
        input = getUserInput (ctr->m_app, title, desc, (const char *) def);
    }
    if (input.isNull ())
        canceled = true;
    else if (!key.isEmpty ())
        m_control->m_config->writeEntry (
                String (GCONF_KEY_KEYS) + key, input);
    return input;
}

String Generator::genReadUriGet (Control *ctr, Node *n) {
    String str;
    bool first = true;
    for (Node *c = n->firstChild (); c && !canceled; c = c->nextSibling ()) {
        String key;
        String val;
        switch (c->id) {
        case id_node_gen_http_key_value: {
            Node *q = c->firstChild ();
            if (q) {
                key = genReadString (ctr, q);
                q = q->nextSibling ();
                if (q && !canceled)
                    val = genReadString (ctr, q);
            }
            break;
        }
        default:
            key = genReadString (ctr, c);
            break;
        }
        if (!key.isEmpty ()) {
            if (first) {
                str += Char ('?');
                first = false;
            } else {
                str += Char ('&');
            }
            str += URL::encode_string (key);
            if (!val.isEmpty ())
                str += Char ('=') + URL::encode_string (val);
        }
    }
    return str;
}

String Generator::genReadString (Control *ctr, Node *n) {
    String str;
    bool need_quote = quote;
    quote = false;
    for (Node *c = n->firstChild (); c && !canceled; c = c->nextSibling ())
        switch (c->id) {
        case id_node_gen_uri:
        case id_node_gen_sequence:
            str += genReadString (ctr, c);
            break;
        case id_node_gen_literal:
            str += c->innerText ().stripWhiteSpace ();
            break;
        case id_node_gen_predefined: {
            String val = static_cast <Element *>(c)->getAttribute ("key");
            if (val == "data")
                str += ctr->data_dir;
            else if (val == "sysdata")
                str += "/usr/share/applications/kmplayer";
            break;
        }
        case id_node_gen_http_get:
            str += genReadUriGet (ctr, c);
            break;
        case id_node_gen_ask:
            str += genReadAsk (ctr, c);
            break;
        case id_node_text:
             str += c->nodeValue ().stripWhiteSpace ();
        }
    if (!static_cast <Element *>(n)->getAttribute ("encoding").isEmpty ())
        str = URL::encode_string (str);
    if (need_quote) {
        char *quoted = g_shell_quote ((const char *) str);
        str = quoted;
        g_free (quoted);
        quote = true;
    }
    return str;
}

String Generator::genReadInput (Control *ctrl, Node *n) {
    quote = false;
    return genReadString (ctrl, n);
}

String Generator::genReadProcess (Control *ctrl, Node *n) {
    String process;
    String str;
    quote = true;
    for (Node *c = n->firstChild (); c && !canceled; c = c->nextSibling ())
        switch (c->id) {
        case id_node_gen_program:
            process = String (genReadString (ctrl, c));
            break;
        case id_node_gen_argument:
            process += Char (' ') + genReadString (ctrl, c);
            break;
        }
    return process;
}

void Generator::activate () {
    String input;
    canceled = false;
    Node *n = firstChild ();
    if (n && n->id == id_node_gen_generator) {
        title = static_cast<Element *>(n)->getAttribute (StringPool::attr_name);
        for (Node *c = n->firstChild (); c && !canceled; c = c->nextSibling ())
            switch (c->id) {
            case id_node_gen_input:
                input = genReadInput (m_control, c);
                break;
            case id_node_gen_process:
                process = genReadProcess (m_control, c);
            }
    }
    if (canceled)
        return;
    if (!input.isEmpty () && process.isEmpty ()) {
        message (MsgInfoString, &input);
        openFile (m_control->m_app, input);
    } else if (!process.isEmpty ()) {
        data = new TextStream;
        if (input.isEmpty ()) {
            message (MsgInfoString, &process);
            begin ();
        } else {
            String cmdline (input + " | " + process);
            message (MsgInfoString, &cmdline);
            if (!media_info)
                media_info = new MediaInfo (this, ActorAgent::Data);
            state = state_activated;
            media_info->wget (input);
        }
    }
}

void Generator::begin () {
    if (!unix_process)
        unix_process = new UnixProcess (
                this, UnixProcess::StdIn | UnixProcess::StdOut);
    String info;
    if (media_info)
        info = String ("Input data ") +
            String::number (media_info->rawData ().size () / 1024.0) + "kb | ";
    info += process;
    message (MsgInfoString, &info);
    if (unix_process->start (m_control->data_dir, process)) {
        state = state_began;
        write_pos = 0;
        if (media_info && media_info->rawData ().size ())
            unix_process->pollStdIn (true);
    } else {
        String err ("Couldn't start process");
        message (MsgInfoString, &err);
        deactivate ();
    }
}

void Generator::deactivate () {
    if (unix_process)
        unix_process->stop ();
    delete unix_process;
    unix_process = NULL;
    delete data;
    data = NULL;
    FileDocument::deactivate ();
}

void *Generator::message (KMPlayer::MessageType msg, void *content) {
    if (MsgMediaReady == msg) {
        if (!media_info->rawData ().size ()) {
            String err ("No input data received");
            message (MsgInfoString, &err);
            deactivate ();
        } else {
            begin ();
        }
    } else {
        return FileDocument::message (msg, content);
    }
    return NULL;
}

void Generator::readEvent (UnixProcess *) {
    char buf[1024];
    int nr;
    State cur_state = state;
    state = state_deferred;
    do {
        nr = unix_process->readStdOut (buf, sizeof (buf) - 1);
        buf[nr] = 0;
        if (nr)
            *data << buf;
    } while (nr > 0);
    if (unix_process->eof) {
        unix_process->closeStdOut ();
        String s = data->release ();
        if (!s.isEmpty ()) {
            TextStream inxml (s);
            Playlist *pl = new Playlist (m_control->m_app, m_control, true);
            NodePtr n = pl;
            KMPlayer::readXML (n, inxml, String (), false);
            pl->normalize ();
            pl->title = title;
            message (MsgInfoString, NULL);
            m_control->openDocument (n, NULL);
        } else {
            String err ("No data received");
            message (MsgInfoString, &err);
        }
        deactivate ();
    } else {
        state = cur_state;
    }
}

void Generator::writeEvent (UnixProcess *) {
    if (media_info) {
        ByteArray &ba = media_info->rawData ();
        int nr = unix_process->writeStdIn (
                ba.data() + write_pos, ba.size () - write_pos);
        write_pos += nr;
        if (write_pos < ba.size ())
            return;
    }
    unix_process->closeStdIn ();
    message (MsgInfoString, &process);
}

void Generator::processExited (UnixProcess *) {
    debugLog() << "Generator::processExited " << endl;
    if (active () && state_deferred != state)
        readEvent (NULL);
}

struct GeneratorTag {
    const char *tag;
    short id;
} gen_tags[] = {
    { "input", id_node_gen_input },
    { "process", id_node_gen_process },
    { "uri", id_node_gen_uri },
    { "literal", id_node_gen_literal },
    { "ask", id_node_gen_ask },
    { "title", id_node_gen_title },
    { "description", id_node_gen_description },
    { "process", id_node_gen_process },
    { "program", id_node_gen_program },
    { "argument", id_node_gen_argument },
    { "predefined", id_node_gen_predefined },
    { "http-get", id_node_gen_http_get },
    { "key-value", id_node_gen_http_key_value },
    { "key", id_node_gen_sequence },
    { "value", id_node_gen_sequence },
    { "sequence", id_node_gen_sequence },
    { NULL, -1 }
};

Node *GeneratorElement::childFromTag (const String &tag) {
    const char *ctag = (const char *) tag;
    for ( GeneratorTag *t = gen_tags; t->tag; ++t)
        if (!strcmp (ctag, t->tag))
            return new GeneratorElement (m_doc, tag, t->id);
    return NULL;
}

