/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2005 Koos Vriezen <koos.vriezen@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * until boost gets common, a more or less compatable one ..
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <gtk/gtkmain.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-async-ops.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-module-callback.h>
#include <libgnomevfs/gnome-vfs-standard-callbacks.h>

#include "kmplayertypes.h"

using namespace KMPlayer;

KMPLAYER_NO_CDTOR_EXPORT ByteArrayImpl::ByteArrayImpl (const ByteArrayImpl & b)
 : m_data (b.m_size ? (char *) malloc (b.m_size) : 0L),
   m_size (b.m_size),
   m_capacity (b.m_size),
   data_owner (true) {
    if (m_size)
        memcpy (m_data, b.m_data, m_size);
}

KMPLAYER_NO_EXPORT void ByteArrayImpl::resize (unsigned ns) {
    if (ns > m_capacity) {
        char * tmp = (char *) malloc (ns);
        if (m_size > 0)
            memcpy (tmp, m_data, m_size);
        m_capacity = m_size = ns;
        if (m_data && data_owner)
            free (m_data);
        m_data = tmp;
        data_owner = true;
    } else
        m_size = ns;
}

KMPLAYER_NO_EXPORT
ByteArrayImpl & ByteArrayImpl::operator = (const ByteArrayImpl & b) {
    if (m_data && data_owner)
        free (m_data);
    m_data = (b.m_size ? (char*) malloc (b.m_size) : 0L);
    m_size = b.m_size;
    m_capacity = b.m_size;
    if (m_size)
        memcpy (m_data, b.m_data, m_size);
}

KMPLAYER_NO_EXPORT ByteArrayImpl &ByteArrayImpl::operator = (const char *b) {
    int len = strlen (b);
    resize (len);
    memcpy (m_data, b, len);
    return *this;
}

//-----------------------------------------------------------------------------

template <>
void Shared<SharedChars>::attach (const Shared<SharedChars> &other) {
    if (data != other.data) {
        if (other.data)
            other.data->ref ();
        if (data)
            data->unref ();
        data = other.data;
    }
}

GLibString::GLibString (const Char & ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    takeUtf8 (g_strdup (buf));
}

void GLibString::takeUtf8 (char * p) {
    if (!data || data->ptr != p) {
        SharedChars *tmp = p ? new SharedChars (p) : 0L;
        if (data)
            data->unref ();
        data = tmp;
    }
}

int GLibString::compare (const char *str) const {
    if (!str && !data)
        return 0;
    if (str && data)
        return strcmp (str, data->ptr);
    return data ? 1 : -1;
}

int GLibString::compare (const GLibString & str) const {
    if (str.data == data)
        return 0;
    if (str.data && data)
        return strcmp (str.data->ptr, data->ptr);
    return data ? 1 : -1;
}

void GLibString::take (char * s) {
    const gchar *end;
    if (g_utf8_validate (s, -1, &end)) {
        takeUtf8 (s);
    } else {
        gsize bytes_read, bytes_written;
        takeUtf8 (g_locale_to_utf8 (s, -1, &bytes_read, &bytes_written, 0L));
    }
}

GLibString & GLibString::operator = (const char * s) {
    if (!s)
        clear ();
    else
        take (g_strdup (s));
    return *this;
}

GLibString & GLibString::operator = (const Char & ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    takeUtf8 (g_strdup (buf));
    return *this;
}

GLibString & GLibString::operator += (const GLibString & str) {
    if (!data) {
        if (str.data)
            takeUtf8 (g_strdup (str.data->ptr));
    } else if (str.data)
        takeUtf8 (g_strconcat (data->ptr, str.data->ptr, 0L));
    return *this;
}

KMPLAYER_NO_EXPORT GLibString & GLibString::operator += (const char *str) {
    if (!data)
        takeUtf8 (g_strdup (str));
    else
        takeUtf8 (g_strconcat (data->ptr, str, 0L));
    return *this;
}

GLibString GLibString::fill (Char ch, int length) {
    GLibString str;
    str.take (g_strnfill (length, (const char) ch)); //FIME
    return str;
}

GLibString GLibString::stripWhiteSpace () const {
    String str;
    if (data && data->ptr [0]) {
        String ws;
        gchar * iter = data->ptr;
        Char ch (g_utf8_get_char (iter));
        iter = g_utf8_next_char (iter);
        while (ch.isSpace () && iter && *iter) { // skip leading ws
            ch = Char (g_utf8_get_char (iter));
            iter = g_utf8_next_char (iter);
        }
        if (ch.isSpace ())
            return str;
        str += ch;
        do {
            if (!iter || !*iter)
                break;
            ch = Char (g_utf8_get_char (iter));
            iter = g_utf8_next_char (iter);
            while (ch.isSpace () && iter && *iter) {
                ws += ch;
                ch = Char (g_utf8_get_char (iter));
                iter = g_utf8_next_char (iter);
            }
            if (ch.isSpace ())
                break;  // skip ending ws
            if (!ws.isEmpty ()) {
                str += ws;
                ws.truncate (0);
            }
            str += ch;
        } while (iter && *iter);
    }
    return str;
}

bool GLibString::startsWith (const GLibString & prefix) const {
    if (!prefix.data && !data)
        return true;
    if (prefix.data && data)
        return g_str_has_prefix (data->ptr, prefix.data->ptr);
    return !prefix.data;
}

bool GLibString::endsWith (const GLibString & suffix) const {
    if (!suffix.data && !data)
        return true;
    if (suffix.data && data)
        return g_str_has_suffix (data->ptr, suffix.data->ptr);
    return !suffix.data;
}

int GLibString::indexOf (const GLibString & needle) const {
    if (!data)
        return needle.data ? -1 : 0;
    if (!needle.data)
        return 0;
    gchar * str = strstr (data->ptr, needle.data->ptr);
    if (!str)
        return -1;
    return int (str - data->ptr);
}

int GLibString::lastIndexOf (const GLibString & needle) const {
    if (!data)
        return needle.data ? -1 : 0;
    if (!needle.data)
        return length ();
    gchar * str = g_strrstr (data->ptr, needle.data->ptr);
    if (!str)
        return -1;
    return int (str - data->ptr);
}

GLibString GLibString::replace (Char c1, Char c2) const {
    GLibString str;
    if (data) {
        gchar cstr1 [8];
        int nr1 = g_unichar_to_utf8 (c1.unicode (), cstr1);
        cstr1 [nr1] = 0;
        gchar cstr2 [8];
        int nr2 = g_unichar_to_utf8 (c2.unicode (), cstr2);
        cstr2 [nr2] = 0;
        gchar *s = data->ptr;
        gchar *p = strstr (s, cstr1);
        while (p) {
            if (p != s)
                str += GLibString (s, p - s);
            str += GLibString (cstr2);
            s = p + nr1;
            p = strstr (s, cstr1);
        }
        if (s[0])
            str += GLibString (s);
    }
    return str;
}

int GLibString::toInt (bool * valid, int base) const {
    int i = 0;
    if (data) {
        char *endptr;
        i = (int) strtol (data->ptr, &endptr, base);
        if (valid)
            *valid = (data->ptr != endptr);
    } else if (valid)
        *valid = false;
    return i;
}

KMPLAYER_NO_EXPORT double GLibString::toDouble (bool * valid) const {
    if (data) {
        gchar * endptr;
        double d = g_ascii_strtod (data->ptr, &endptr);
        if (valid)
            *valid = (endptr != data->ptr);
        return d;
    } else {
        if (valid)
            *valid = false;
        return 0.0;
    }
}

GLibString GLibString::mid (const int index, unsigned len) const {
    if (!data)
        return GLibString ();
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    gchar * eos = off;
    if (!off)
        return GLibString ();
    while (eos && *eos && len--)
        eos = g_utf8_next_char (eos);
    GLibString str;
    if (!eos || !*eos)
        str.takeUtf8 (g_strdup (off));
    else
        str.takeUtf8 (g_strndup (off, eos-off));
    return str;
}

GLibString GLibString::left (int index) const {
    if (!data)
        return GLibString ();
    if (index < 0)
        return GLibString (*this);
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    if (!off)
        return GLibString ();
    GLibString str;
    str.takeUtf8 (g_strndup (data->ptr, off - data->ptr));
    return str;
}

void GLibString::truncate (int pos) {
    if (data) {
        if (pos == 0) {  // speed up clearing string
            clear ();
        } else {
            gchar * epos = g_utf8_offset_to_pointer (data->ptr, pos);
            if (epos)
                takeUtf8 (g_strndup (data->ptr, epos - data->ptr));
        }
    }
}

//-----------------------------------------------------------------------------

namespace KMPlayer {

class GnomeVFSFilePrivate {
public:
    GnomeVFSFilePrivate () : m_vfshandle (0L) {}
    ~GnomeVFSFilePrivate () {}
    GnomeVFSHandle * m_vfshandle;
    String m_path;
};

}

KMPLAYER_NO_CDTOR_EXPORT
GnomeVFSFile::GnomeVFSFile (const String & p) : d (new GnomeVFSFilePrivate) {
    d->m_path = p;
}

KMPLAYER_NO_CDTOR_EXPORT GnomeVFSFile::~GnomeVFSFile () {
    close ();
    delete d;
}

KMPLAYER_NO_EXPORT bool GnomeVFSFile::open (FileMode mode) {
    switch (mode) {
        case IO_ReadOnly:
            if (gnome_vfs_open (&d->m_vfshandle, (const char *) d->m_path, GNOME_VFS_OPEN_READ) == GNOME_VFS_OK)
                return true;
            d->m_vfshandle = 0L;
            break;
        case IO_WriteOnly: // fixme truncate
            if (gnome_vfs_open (&d->m_vfshandle, (const char *) d->m_path, GNOME_VFS_OPEN_WRITE) == GNOME_VFS_OK)
                return true;
            if (gnome_vfs_create (&d->m_vfshandle,(const char *) d->m_path, GNOME_VFS_OPEN_WRITE, TRUE, GNOME_VFS_PERM_USER_ALL) == GNOME_VFS_OK)
                return true;
            d->m_vfshandle = 0L;
            break;
        default:
            warningLog() << "open " << int (mode) << " not implemented" << endl;
    }
    return false;
}

KMPLAYER_NO_EXPORT void GnomeVFSFile::close () {
    if (d->m_vfshandle) {
        gnome_vfs_close (d->m_vfshandle);
        d->m_vfshandle = 0L;
    }
}

KMPLAYER_NO_EXPORT ByteArray GnomeVFSFile::readAll () {
    ByteArray str;
    char buf [2048];
    GnomeVFSFileSize nr;
    if (!d->m_vfshandle && !open (IO_ReadOnly))
        return str;
    gnome_vfs_seek (d->m_vfshandle, GNOME_VFS_SEEK_START, 0);
    do {
        if (gnome_vfs_read (d->m_vfshandle, buf, sizeof (buf) -1, &nr) != GNOME_VFS_OK)
            break;
        off_t size = str.size ();
        str.resize (size + nr);
        memcpy (str.data () + size, buf, nr);
    } while (nr > 0);
    return str;
}

KMPLAYER_NO_EXPORT off_t GnomeVFSFile::readBlock (char * buf, off_t max) {
    if (!d->m_vfshandle && !open (IO_ReadOnly))
        return 0;
    GnomeVFSFileSize nr;
    if (gnome_vfs_read (d->m_vfshandle, buf, max, &nr) != GNOME_VFS_OK)
        return 0;
    return nr;
}

KMPLAYER_NO_EXPORT bool GnomeVFSFile::exists () {
    GnomeVFSFileInfo info;
    return (gnome_vfs_get_file_info ((const char *) d->m_path, &info, GNOME_VFS_FILE_INFO_DEFAULT) == GNOME_VFS_OK);
}

KMPLAYER_NO_EXPORT off_t GnomeVFSFile::size () {
    GnomeVFSFileInfo info;
    if (gnome_vfs_get_file_info ((const char *) d->m_path, &info, GNOME_VFS_FILE_INFO_DEFAULT) != GNOME_VFS_OK)
        return 0;
    return info.size;
}

KMPLAYER_NO_EXPORT off_t GnomeVFSFile::writeBlock (const char * data, off_t length) {
    debugLog () << "writeBlock " << int (length) << endl;
    off_t len = length;
    while (len > 0) {
        GnomeVFSFileSize bytes_written;
        if (gnome_vfs_write (d->m_vfshandle, data + (length - len), len, &bytes_written) != GNOME_VFS_OK)
            break;
        len -= bytes_written;
    }
    return length - len;
}

//-----------------------------------------------------------------------------

GnomeVFSUrl::GnomeVFSUrl (const GnomeVFSUrl & base, const String & rel) {
    m_vfsurl= (base.m_vfsurl ?
            gnome_vfs_uri_resolve_relative (base.m_vfsurl, rel) :
            gnome_vfs_uri_new ((const char *) rel));
}

KMPLAYER_NO_CDTOR_EXPORT GnomeVFSUrl::GnomeVFSUrl (const String & url) {
    m_vfsurl = (url.isEmpty () ? 0L : gnome_vfs_uri_new ((const char *) url));
}

KMPLAYER_NO_CDTOR_EXPORT GnomeVFSUrl::GnomeVFSUrl (const GnomeVFSUrl & url) {
    m_vfsurl = (url.m_vfsurl ? gnome_vfs_uri_dup (url.m_vfsurl) : 0L);
}

KMPLAYER_NO_CDTOR_EXPORT GnomeVFSUrl::~GnomeVFSUrl () {
    if (m_vfsurl)
        gnome_vfs_uri_unref (m_vfsurl);
}

KMPLAYER_NO_EXPORT String GnomeVFSUrl::url () const {
    if (m_vfsurl) {
        GLibString s;
        s.take (gnome_vfs_uri_to_string (m_vfsurl, GNOME_VFS_URI_HIDE_NONE));
        return String (s);
    }
    return String ();
}

GnomeVFSUrl & GnomeVFSUrl::operator = (const String & url) {
    if (m_vfsurl)
        gnome_vfs_uri_unref (m_vfsurl);
    m_vfsurl = gnome_vfs_uri_new ((const char *) url);
    return *this;
}

GnomeVFSUrl & GnomeVFSUrl::operator = (const GnomeVFSUrl & url) {
    if (m_vfsurl)
        gnome_vfs_uri_unref (m_vfsurl);
    m_vfsurl = url.m_vfsurl ? gnome_vfs_uri_dup (url.m_vfsurl) : 0L;
    return *this;
}

KMPLAYER_NO_EXPORT String GnomeVFSUrl::path () const {
    if (m_vfsurl)
        return String (gnome_vfs_uri_get_path (m_vfsurl));
    return String ();
}

KMPLAYER_NO_EXPORT String GnomeVFSUrl::protocol () const {
    if (m_vfsurl)
        return String (gnome_vfs_uri_get_scheme (m_vfsurl));
    return String ();
}

KMPLAYER_NO_EXPORT String GnomeVFSUrl::decode_string (const String & s) {
    if (s.isEmpty ())
        return String ();
    String str;
    GLibString &gs = str;
    gs.take(gnome_vfs_unescape_string ((const char *) s, ""));
    return str;
}

KMPLAYER_NO_EXPORT String GnomeVFSUrl::encode_string (const String & s) {
    if (s.isEmpty ())
        return String ();
    String str;
    GLibString &gs = str;
    gs.take(gnome_vfs_escape_string ((const char *) s));
    return str;
}

KMPLAYER_NO_EXPORT bool GnomeVFSUrl::isLocalFile () const {
    if (m_vfsurl)
        return gnome_vfs_uri_is_local (m_vfsurl) ||
            protocol () == String ("file");
    return false;
}

//-----------------------------------------------------------------------------

String GMimeType::findByContent (const char * data, int size) {
    if (size <= 0)
        return String();
    const gchar *end;
    if (g_utf8_validate ((gchar *) data, size, &end))
        return "text/plain";
    if (g_utf8_get_char_validated (end, size - (end - data)) == (gunichar)-2)
        return "text/plain";
    return String (gnome_vfs_get_mime_type_for_data (data, size));
}

//-----------------------------------------------------------------------------

GLibTextStream & GLibTextStream::operator >> (Char & ch) {
    if (m_gstring) {
        m_utf8_str = g_string_free (m_gstring, false);
        m_gstring = 0L;
        m_iter = m_utf8_str;
    }
    if (m_iter) {
        ch = Char (g_utf8_get_char (m_iter));
        m_iter = g_utf8_next_char (m_iter);
    }
}

KMPLAYER_NO_EXPORT
GLibTextStream & GLibTextStream::operator = (const GLibTextStream & s) {
    if (m_gstring)
        g_string_free (m_gstring, true);
    else if (m_utf8_str)
        g_free (m_utf8_str);
    m_utf8_str = 0L;
    m_iter = 0L;
    m_gstring = g_string_new (s.m_gstring ? s.m_gstring->str :
            (s.m_utf8_str ? s.m_utf8_str : ""));
    return *this;
}

String GLibTextStream::readLine () {
    String str;
    Char ch;
    while (!atEnd ()) {
        *this >> ch;
        if (ch == '\r' || ch == '\n')
            break;
        str += ch;
    }
    gchar * iter = m_iter;
    while (!atEnd ()) { // FIXME: current is skipping emty lines as well
        *this >> ch;
        if (ch != '\r' && ch != '\n') {
            m_iter = iter;
            break;
        }
        iter = m_iter;
    }
    return str;
}

//-----------------------------------------------------------------------------
#if 0
namespace KMPlayer {

    class GnomeVFSGetJob : public IOJob {
    public:
        GnomeVFSGetJob (IOJobListener * rec, const String & u);
        ~GnomeVFSGetJob ();
        void kill (bool quiet = true);
        void finished ();
        bool error ();

        GnomeVFSAsyncHandle * network_handle;
        IOJobListener * m_receiver;
        bool job_closed;
        bool job_error;
        bool keep_quiet;
        bool continuing;
        char read_buf [2048];
    };

    IOJob * asyncGet (IOJobListener * receiver, const String & url) {
        return new GnomeVFSGetJob (receiver, url);
    }
}

//static GTree *uri_info;

//void getRedirection (gpointer data, gpointer d) {
//    debugLog() << " getRedirection: " << (char *)data << endl;
//}

//static void globalVFSAsyncModuleCallback (gconstpointer in,
//        gsize in_size,
//        gpointer out,
//        gsize out_size,
//        gpointer callback_data/*,
//        GnomeVFSModuleCallbackResponse response,
//        gpointer response_data*/) {
//    GnomeVFSModuleCallbackReceivedHeadersIn *in_args =
//        (GnomeVFSModuleCallbackReceivedHeadersIn *)in;
//    debugLog() << "globalVFSAsyncModuleCallback " << 
//        in_args->uri->text << endl;
//    g_list_foreach (in_args->headers, getRedirection, NULL);
//    //response (response_data);
//}


static void cb_async_get_close (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer p) {
    //debugLog () << "cb_async_get_close " << endl;
    if (!p)
        return;
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    get->finished ();
}

static void cb_async_get_read (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer buffer, GnomeVFSFileSize bytes_requested, GnomeVFSFileSize bytes_read, gpointer p) {
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    if (get->job_closed)
        return;
    get->continuing = true;
    if (bytes_read > 0) {
    debugLog () << "cb_async_get_read " << handle << endl;
        ByteArray data (bytes_read);
        memcpy (data.data (), (const char *) buffer, bytes_read);
        get->m_receiver->jobData (get, data);
    }
    //debugLog () << "cb_async_get_read " << int (bytes_read) << " canceled:" << get->job_closed << " not ok:" << (result != GNOME_VFS_OK) << " less:" << (bytes_read < bytes_requested) << " killed:" << !get->continuing << endl;
    if (result != GNOME_VFS_OK || bytes_read <= 0 || !get->continuing) {
        // || bytes_read < bytes_requested 
        get->job_closed = true;
        get->job_error = result != GNOME_VFS_ERROR_EOF;
        gnome_vfs_async_cancel (handle);
        gnome_vfs_async_close (handle, cb_async_get_close, p);
    } else {
        gnome_vfs_async_read (handle, get->read_buf, sizeof (get->read_buf) -1, cb_async_get_read, p);
    }
    get->continuing = false;
}

static void cb_async_get_open (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer p) {
    debugLog () << "cb_async_get_open " << handle << endl;
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    if (get->job_closed)
        return;
    if (result != GNOME_VFS_OK)
        get->finished ();  // kills get
    else
        gnome_vfs_async_read (handle, get->read_buf, sizeof (get->read_buf) -1, cb_async_get_read, p);
}

GnomeVFSGetJob::GnomeVFSGetJob (IOJobListener * rec, const String &url)
 : m_receiver (rec), network_handle (0L),
   job_closed (false), job_error (false),
   keep_quiet (false), continuing (false) {

    //if (!uri_info) {
    //    uri_info = g_tree_new ((GCompareFunc)::strcmp);
    //    gnome_vfs_module_callback_set_default
    //        (GNOME_VFS_MODULE_CALLBACK_HTTP_RECEIVED_HEADERS,
    //         globalVFSAsyncModuleCallback,
    //         uri_info,
    //         NULL);
    //}
    gnome_vfs_async_open (&network_handle, (const char *) url, GNOME_VFS_OPEN_READ, 0, cb_async_get_open , static_cast <void *> (this));
    debugLog() << "GnomeVFSGetJob " << network_handle << " " << url << endl;
}

GnomeVFSGetJob::~GnomeVFSGetJob () {
    if (network_handle && !job_closed) {
        gnome_vfs_async_cancel (network_handle);
        gnome_vfs_async_close (network_handle, cb_async_get_close, 0L);
    }
}

void GnomeVFSGetJob::kill (bool quiet) {
    keep_quiet = quiet;
    if (network_handle && !job_closed) {
        if (!continuing) {
            job_closed = true;
            gnome_vfs_async_cancel (network_handle);
            gnome_vfs_async_close (network_handle, cb_async_get_close, static_cast <void *> (this));//
        } else
            continuing = false;  // killed from jobData receiver callback
    } else if (!quiet)
        m_receiver->jobResult (this);
}

void GnomeVFSGetJob::finished () {
    network_handle = 0L;
    if (!keep_quiet)
        m_receiver->jobResult (this);
    delete this;
}

bool GnomeVFSGetJob::error () {
    return job_error;
}
#endif
//-----------------------------------------------------------------------------

#include <curl/curl.h>

static gint easyCompare (gconstpointer a, gconstpointer b) {
    return (long)a - (long)b;
}

namespace KMPlayer {
    class CurlGetJob {
    public:
        CurlGetJob (IOJobListener * rec, const String & u);
        ~CurlGetJob ();

        void init (IOJob *job);

        static size_t headerLine( void *p, size_t sz, size_t n, void *strm);
        static size_t dataReceived (void *p, size_t sz, size_t n, void *strm);
        static gboolean findKilledJob (gpointer key, gpointer val, gpointer d);

        IOJobListener * m_receiver;

        static bool global_init;
        static CURLM *multi;

        String url;
        String content_type;
        CURL *easy;
        bool quiet;
        bool killed;
        bool redirected;
        int error_code;
        unsigned long content_length;
    };
}

namespace {

    struct FdWatch {
        int fd;
        int chan_read_watch;
    };

}

namespace KMPlayer {

    IOJob *asyncGet (IOJobListener *receiver, const String &url) {
        return new IOJob (new CurlGetJob (receiver, url));
    }
}

inline gboolean
CurlGetJob::findKilledJob (gpointer key, gpointer value, gpointer data) {
    IOJob *job = (IOJob *)value;
    CurlGetJob *self = job->d;

    if (self->killed) {
        *(IOJob **)data = job;
        return true;
    }
    return false;
}

static gboolean curlChannelRead (GIOChannel *src, GIOCondition c, gpointer d);
bool CurlGetJob::global_init = false;
CURLM *CurlGetJob::multi = NULL;
static GTree *easy_list = NULL;
static FdWatch *fd_watch_array;
static int fd_watch_array_len;
static gint curl_timeout_timer;
static bool in_perform_multi;
static bool curl_job_killed;

static void curlChannelDestroy (gpointer) {
    //debugLog() << "channel destoyed" << endl;
}

static void updateReadWatches() {
    fd_set fd_read, fd_write, fd_excep;
    FD_ZERO (&fd_read);
    FD_ZERO (&fd_write);
    FD_ZERO (&fd_excep);
    int maxfd = 0;
    curl_multi_fdset (CurlGetJob::multi, &fd_read, &fd_write, &fd_excep, &maxfd);
    if (maxfd > 0 && !fd_watch_array) {
        fd_watch_array = new FdWatch[maxfd];
        memset (fd_watch_array, 0, maxfd * sizeof (FdWatch));
        fd_watch_array_len = maxfd;
    } else if (maxfd > fd_watch_array_len) {
        FdWatch *tmp = fd_watch_array;
        fd_watch_array = new FdWatch[maxfd];
        memset (fd_watch_array, 0, maxfd * sizeof (FdWatch));
        memcpy (fd_watch_array, tmp, fd_watch_array_len * sizeof (FdWatch));
        delete[] tmp;
        fd_watch_array_len = maxfd;
    }
    for (int i = 3; i < fd_watch_array_len; i++) {
        if (FD_ISSET (i, &fd_read)) {
            if (!fd_watch_array[i].fd) {
                fd_watch_array[i].fd = i;
                GIOChannel *channel = g_io_channel_unix_new (i);
                g_io_channel_set_encoding (channel, NULL, NULL);
                fd_watch_array[i].chan_read_watch = g_io_add_watch_full (channel, G_PRIORITY_DEFAULT, G_IO_IN, curlChannelRead, NULL, curlChannelDestroy);
                g_io_channel_unref (channel);
            }
        } else if (fd_watch_array[i].fd) {
            g_source_remove (fd_watch_array[i].chan_read_watch);
            fd_watch_array[i].fd = fd_watch_array[i].chan_read_watch = 0;
        }
    }
}

static void curlPerformMulti () {

    in_perform_multi = true;

    int running_handles = 0;
    CURLMcode curlCode;
    do {
        curl_job_killed = false;
        curlCode = curl_multi_perform (CurlGetJob::multi, &running_handles);
	if (curl_job_killed)
            do {
                IOJob *job = NULL;
                g_tree_traverse (easy_list, CurlGetJob::findKilledJob, G_IN_ORDER, &job);
                if (job)
                    delete job;
                else
                    break;
            } while (true);
    } while (CURLM_CALL_MULTI_PERFORM == curlCode && running_handles > 0);
    int messages;
    CURLMsg* msg = curl_multi_info_read (CurlGetJob::multi, &messages);
    while (msg) {
        if (CURLMSG_DONE == msg->msg) {
            IOJob *job = (IOJob *)g_tree_lookup (easy_list, msg->easy_handle);
            if (job)
                delete job;
        }
        msg = curl_multi_info_read (CurlGetJob::multi, &messages);
    }
    updateReadWatches();

    in_perform_multi = false;
}

static gboolean curlTimeOut (void *) {
    curlPerformMulti ();
    if (g_tree_height (easy_list))
        return 1; // continue
    curl_timeout_timer = 0;
    debugLog() << "clearing polling" << endl;
    return 0; // stop
}

static gboolean curlChannelRead (GIOChannel *src, GIOCondition c, gpointer d) {
    curlPerformMulti ();
}

size_t CurlGetJob::headerLine( void *ptr, size_t size, size_t nmemb, void *stream) {
    IOJob *job = (IOJob *) stream;
    CurlGetJob *self = job->d;
    //char buf[256];
    //snprintf (buf, sizeof (buf), "%s", (char *)ptr);
    //debugLog() << "header: " << buf << endl;
    // Last-Modified:
    if (!self->killed) {
        if (!strncmp ((char *)ptr, "HTTP/", 5)) {
            char *p = strchr ((char *)ptr + 5, ' ');
            if (p) {
                int rc = strtol (p +1, NULL, 10);
                if (rc >= 400)
                    self->error_code = rc;
                else if (rc >= 300)
                    self->redirected = true;
            }
        } else if (self->redirected && !strncasecmp ((char *)ptr, "location:", 9)) {
            String nuri ((char *) ptr + 9, size * nmemb - 9);
            self->redirected = false;
            self->m_receiver->redirected (job, nuri.stripWhiteSpace ());
        } else if (!strncasecmp ((char *)ptr, "content-length:", 15)) {
            self->content_length = strtol ((char *)ptr + 15, NULL, 10);
        } else if (!strncasecmp ((char *)ptr, "content-type:", 13)) {
            self->content_type = String (
                    (char *)ptr + 13, size * nmemb - 13).stripWhiteSpace ();
        }
    }
    return size * nmemb;
}

size_t CurlGetJob::dataReceived (void *ptr, size_t size, size_t nmemb, void *stream) {
    IOJob *job = (IOJob *) stream;
    CurlGetJob *self = job->d;
    ByteArray data ((char *)ptr, size * nmemb);
    //debugLog() << "dataReceived " << (size * nmemb) << endl;
    if (!self->killed)
        self->m_receiver->jobData (job, data);
    return size * nmemb;
}
static int job_count;

KMPLAYER_NO_CDTOR_EXPORT
CurlGetJob::CurlGetJob (IOJobListener * rec, const String & u)
 : m_receiver (rec), url (u),
   quiet (false), killed (false), redirected (false),
   error_code (0), content_length (0)
{}

void CurlGetJob::init (IOJob *job) {
    if (!global_init) {
        global_init = !curl_global_init (CURL_GLOBAL_ALL);
        easy_list = g_tree_new (easyCompare);
        debugLog() << "global curl init: " << global_init << endl;
    }
    if (!multi)
        multi = curl_multi_init ();
    easy = curl_easy_init ();
    debugLog() << "CurlGetJob::CurlGetJob " << easy << " " << ++job_count << endl;
    g_tree_insert (easy_list, easy, job);
    //curl_easy_setopt (easy, CURLOPT_VERBOSE, true);
    curl_easy_setopt (easy, CURLOPT_NOPROGRESS, true);
    curl_easy_setopt (easy, CURLOPT_WRITEFUNCTION, CurlGetJob::dataReceived);
    curl_easy_setopt (easy, CURLOPT_WRITEDATA, job);
    curl_easy_setopt (easy, CURLOPT_HEADERFUNCTION, CurlGetJob::headerLine);
    curl_easy_setopt (easy, CURLOPT_WRITEHEADER, job);
    curl_easy_setopt (easy, CURLOPT_FOLLOWLOCATION, 1);
    curl_easy_setopt (easy, CURLOPT_MAXREDIRS, 10);
    //CURLSH *curl_share_init( );
    //curl_easy_setopt (easy, CURLOPT_SHARE, m_curlShareHandle);
    //curl_share_setopt(share_handle,CURL_LOCK_DATA_DNS);
    //curl_share_cleanup(CURLSH * share_handle );
    curl_easy_setopt (easy, CURLOPT_DNS_CACHE_TIMEOUT, 60 * 5);
    curl_easy_setopt (easy, CURLOPT_ENCODING, "");
    curl_easy_setopt (easy, CURLOPT_USERAGENT, "Mozilla/4.0 (compatible; MSIE 6.0; X11; Linux armv6l; U) Opera 8.5 [en_US] Tablet browser 0.0.14 RX-34_2007SE_4.2007.26-8");
    curl_easy_setopt (easy, CURLOPT_SSL_VERIFYPEER, 0);
    curl_easy_setopt (easy, CURLOPT_URL, (const char *)url);
    //if (!strncmp (u, "http", 4))
    //    curl_easy_setopt (easy, CURLOPT_HTTPGET, true);
    CURLMcode type = curl_multi_add_handle (multi, easy);
    if (type && type != CURLM_CALL_MULTI_PERFORM)
        debugLog() << "CurlGetJob add to multi failure" << endl;

    if (!curl_timeout_timer)
        curl_timeout_timer = g_timeout_add (50, curlTimeOut, NULL);
}

KMPLAYER_NO_CDTOR_EXPORT CurlGetJob::~CurlGetJob () {
    //debugLog() << "CurlGetJob::~CurlGetJob " << easy << " " << --job_count << endl;
    if (easy) {
        curl_multi_remove_handle (multi, easy);
        curl_easy_cleanup (easy);
        g_tree_remove (easy_list, easy);
    }
    if (0) {
        curl_multi_cleanup (multi);
        multi = NULL;
    }
}

KMPLAYER_NO_EXPORT IOJob::IOJob (IOJobPrivate *priv) : d (priv) {
    priv->init (this);
}

KMPLAYER_NO_EXPORT IOJob::~IOJob () {
    IOJobListener *listener = d->easy && !d->quiet ? d->m_receiver : NULL;

    if (listener)
        listener->jobResult (this);

    delete d;
}

KMPLAYER_NO_EXPORT void IOJob::kill (bool q) {
    d->quiet = q;
    d->killed = true;
    curl_job_killed = true;
    if (!in_perform_multi) {
        delete this;
	curlPerformMulti ();
    }
}

KMPLAYER_NO_EXPORT bool IOJob::error () {
    return d->error_code;
}

unsigned long IOJob::contentLength() const {
    return d->content_length;
}

String IOJob::contentType () const {
    return d->content_type;
}

//-----------------------------------------------------------------------------

static bool gconf_initialized = false;

KMPLAYER_NO_CDTOR_EXPORT GConf::GConf () {
    if (!gconf_initialized) {
        gconf_initialized = true;
        g_type_init ();
    }
    client = gconf_client_get_default ();
}

KMPLAYER_NO_CDTOR_EXPORT GConf::~GConf () {
    gconf_client_clear_cache (client);
    g_object_unref (client);
}

KMPLAYER_NO_EXPORT int GConf::readNumEntry (const String & key, int def) {
    GConfValue *val = gconf_client_get (client, (const char *) key, 0L);
    if (val) {
        int i = gconf_value_get_int (val);
        gconf_value_free (val);
        return i;
    }
    return def;
}

KMPLAYER_NO_EXPORT double GConf::readNumEntry (const String & key, double def) {
    GError * err;
    double val = gconf_client_get_float (client, (const char *) key, &err);
    bool has_err = !!err;
    g_error_free (err);
    return has_err ? def : val;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT Log & Log::operator << (const char * s) {
    fprintf (stderr, "%s", s);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (int i) {
    fprintf (stderr, "%d", i);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (unsigned int i) {
    fprintf (stderr, "%u", i);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (double d) {
    fprintf (stderr, "%f", d);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (const Single &s) {
    fprintf (stderr, "%.3f", (double)s);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (const SRect &r) {
    fprintf (stderr, "SRect(%.1f, %.1f, %.1f, %.1f)",
           (double)r.x(), (double)r.y(), (double)r.width(), (double)r.height());
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (const IRect &r) {
    fprintf (stderr, "SRect(%d, %d, %d, %d)",
            r.x (), r.y (), r.width (), r.height ());
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (void * d) {
    fprintf (stderr, "0x%x", d);
    return *this;
}

//-----------------------------------------------------------------------------

#include <glib/gprintf.h>

static GdkColor * colorFromRGB (int rgb) {
    GdkColormap * cmap = gdk_colormap_get_system ();
    GdkColor *color = g_new (GdkColor, 1);
    GdkColor *rcolor = NULL;
    char buf [16];
    g_sprintf (buf, "#%02X%02X%02X", (rgb & 0x00FF0000) >> 16, (rgb & 0x0000FF00) >> 8, (rgb & 0x000000FF));
    if (gdk_color_parse (buf, color) &&
                (gdk_colormap_alloc_color (cmap, color, 0, 0) ||
                 gdk_colormap_alloc_color (cmap, color, 0, 1)))
        rcolor = gdk_color_copy (color);
    g_free (color);
    return rcolor;
}

KMPLAYER_NO_CDTOR_EXPORT GColor::GColor (const String & desc) : m_color (NULL) {
    GdkColormap * cmap = gdk_colormap_get_system ();
    GdkColor *color = g_new (GdkColor, 1);
    if (!(gdk_color_parse ((const char *) desc, color) &&
                gdk_colormap_alloc_color (cmap, color, 0, 1))) {
        if (desc.lower () == "silver")
            m_color = colorFromRGB (0xbebebe);
        // TODO add more unsupported real live colors
        if (!m_color)
            errorLog () << "Could not parse " << desc << " color" << endl;
    } else {
        m_color = gdk_color_copy (color);
    }
    g_free (color);
    //debugLog () << desc << " -> " << int (m_color ? m_color->pixel : 0) << endl;
}

KMPLAYER_NO_CDTOR_EXPORT GColor::GColor (int rgb)
    : m_color (colorFromRGB(rgb)) {
    //debugLog () << "GColor::GColor (int " << rgb << ") -> " << int (m_color ? m_color->pixel : -1) << " str:" << buf << endl;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (int w, int h) {
    GdkVisual * visual = gdk_visual_get_system ();//gdk_rgb_get_visual
    m_pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, visual->depth, w, h);
}

KMPLAYER_NO_CDTOR_EXPORT
GPixbuf::GPixbuf (const ByteArray & data) : m_pixbuf (0L) {
    GdkPixbufLoader * pix_loader = gdk_pixbuf_loader_new ();
    GError *gerror = 0L;
    if (gdk_pixbuf_loader_write (pix_loader, (const guchar *) data.data (), data.size (), &gerror) && gdk_pixbuf_loader_close (pix_loader, 0L)) {
        m_pixbuf = gdk_pixbuf_loader_get_pixbuf (pix_loader);
        g_object_ref (m_pixbuf);
    } else
        g_free (gerror);
    g_object_unref (pix_loader);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (GdkPixbuf * pb, bool skip_addref)
  : m_pixbuf (pb) {
    if (!skip_addref && m_pixbuf)
        g_object_ref (m_pixbuf);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (const GPixbuf & other)
 : m_pixbuf (other.m_pixbuf) {
    if (m_pixbuf)
        g_object_ref (m_pixbuf);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::~GPixbuf () {
    reset ();
}

KMPLAYER_NO_EXPORT GPixbuf & GPixbuf::operator = (const GPixbuf & other) {
    if (&other != this) {
        reset ();
        m_pixbuf = other.m_pixbuf;
        if (m_pixbuf)
            g_object_ref (m_pixbuf);
    }
    return *this;
}

KMPLAYER_NO_EXPORT void GPixbuf::reset () {
    if (m_pixbuf) {
        g_object_unref (m_pixbuf);
        m_pixbuf = 0L;
    }
}

KMPLAYER_NO_EXPORT void GPixbuf::fill (const Color & c) {
    if (m_pixbuf)
        gdk_pixbuf_fill (m_pixbuf, c.rgb ());
}

KMPLAYER_NO_EXPORT GPixbuf GPixbuf::scale (int w, int h) {
    if (m_pixbuf)
        return GPixbuf (gdk_pixbuf_scale_simple (m_pixbuf, w, h, w > width () ? GDK_INTERP_NEAREST : GDK_INTERP_BILINEAR), true);
}


/*
void        gdk_pixbuf_xlib_init            (Display *display,
                                             int screen_num);


GdkPixbuf*  gdk_pixbuf_new_from_file        (const char *filename,
                                             GError **error);
// gdk_pixbuf_new_from_data gdk_pixbuf_new_from_xpm_data
void        gdk_pixbuf_copy_area            (const GdkPixbuf *src_pixbuf,
                                             int src_x,
                                             int src_y,
                                             int width,
                                             int height,
                                             GdkPixbuf *dest_pixbuf,
                                             int dest_x,
                                             int dest_y);

void        gdk_pixbuf_scale                (const GdkPixbuf *src,
                                             GdkPixbuf *dest,
                                             int dest_x,
                                             int dest_y,
                                             int dest_width,
                                             int dest_height,
                                             double offset_x,
                                             double offset_y,
                                             double scale_x,
                                             double scale_y,
                                             GdkInterpType interp_type);

g_object_unref() 

// http://developer.gnome.org/doc/API/2.0/gdk-pixbuf/gdk-pixbuf-animation.html

GdkPixbuf*  gdk_pixbuf_add_alpha            (const GdkPixbuf *pixbuf,
                                             gboolean substitute_color,
                                             guchar r,
                                             guchar g,
                                             guchar b);


typedef struct {
  gint x;
  gint y;
  gint width;
  gint height;
} GdkRectangle;

void        gdk_rectangle_union             (GdkRectangle *src1,
                                             GdkRectangle *src2,
                                             GdkRectangle *dest);

*/
