/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2008 Koos Vriezen <koos.vriezen@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#include <cairo.h>

#include "surface.h"
#include "viewarea.h"

using namespace KMPlayer;

Matrix::Matrix () : a (1.0), b (0.0), c (0.0), d (1.0), tx (0), ty (0) {}

Matrix::Matrix (const Matrix & m)
 : a (m.a), b (m.b), c (m.c), d (m.d), tx (m.tx), ty (m.ty) {}

Matrix::Matrix (Single xoff, Single yoff, float xscale, float yscale)
 : a (xscale), b (0.0), c (0.0), d (yscale), tx (xoff), ty (yoff) {}

void Matrix::getXY (Single & x, Single & y) const {
    x = Single (x * a) + tx;
    y = Single (y * d) + ty;
}

void Matrix::getWH (Single &w, Single &h) const {
    w *= a;
    h *= d;
}

void Matrix::getXYWH (Single & x, Single & y, Single & w, Single & h) const {
    getXY (x, y);
    w *= a;
    h *= d;
}

void Matrix::invXYWH (Single & x, Single & y, Single & w, Single & h) const {
    if (a > 0.00001 && d > 0.00001) {
        w /= a;
        h /= d;
        x = Single ((x - tx) / a);
        y = Single ((y - ty) / d);
    } else {
        warningLog () << "Not invering " << a << ", " << d << " scale" << endl;
    }
}

void Matrix::transform (const Matrix & matrix) {
    // TODO: rotate
    a *= matrix.a;
    d *= matrix.d;
    tx = Single (tx * matrix.a) + matrix.tx;
    ty = Single (ty * matrix.d) + matrix.ty;
}

void Matrix::scale (float sx, float sy) {
    a *= sx;
    d *= sy;
    tx *= sx;
    ty *= sy;
}

void Matrix::translate (Single x, Single y) {
    tx += x;
    ty += y;
}

KDE_NO_CDTOR_EXPORT Surface::Surface (ViewArea *widget)
  : bounds (SRect (0, 0, 342, 260)), //widget->width(),widget->height()))
    xscale (1.0),
    yscale (1.0),
    background_color (0),
    surface (0L),
    dirty (false),
    scroll (false),
    view_widget (widget) {
}

Surface::~Surface() {
    if (surface)
        cairo_surface_destroy (surface);
}

void Surface::clear () {
    m_first_child = 0L;
    background_color = 0;
}

void Surface::remove () {
    Surface *sp = parentNode ();
    if (sp) {
        sp->markDirty ();
        sp->removeChild (this);
    }
}

void Surface::resize (const SRect &rect, bool parent_resized) {
    SRect old_bounds = bounds;
    bounds = rect;
    if (parent_resized || old_bounds != rect) {

        if (parent_resized || old_bounds.size != rect.size) {
            virtual_size = SSize (); //FIXME try to preserve scroll on resize
            markDirty ();
            if (surface) {
                cairo_surface_destroy (surface);
                surface = NULL;
            }
            updateChildren (true);
        } else if (parentNode ()) {
            parentNode ()->markDirty ();
        }
        if (parentNode ())
            parentNode ()->repaint (old_bounds.unite (rect));
        else
            repaint ();
    }
}

void Surface::markDirty () {
    for (Surface *s = this; s && !s->dirty; s = s->parentNode ())
        s->dirty = true;
}

void Surface::updateChildren (bool parent_resized) {
    for (Surface *c = firstChild (); c; c = c->nextSibling ())
        if (c->node)
            c->node->message (MsgSurfaceBoundsUpdate, (void *) parent_resized);
        else
            errorLog () << "Surface without node" << endl;
}

KDE_NO_EXPORT
Surface *Surface::createSurface (NodePtr owner, const SRect & rect) {
    Surface *surface = new Surface (view_widget);
    surface->node = owner;
    surface->bounds = rect;
    appendChild (surface);
    return surface;
}

KDE_NO_EXPORT IRect Surface::toScreen (Single x, Single y, Single w, Single h) {
    //FIXME: handle scroll
    Matrix matrix (0, 0, xscale, yscale);
    matrix.translate (bounds.x (), bounds.y ());
    for (Surface *s = parentNode(); s; s = s->parentNode()) {
        matrix.transform(Matrix (0, 0, s->xscale, s->yscale));
        matrix.translate (s->bounds.x (), s->bounds.y ());
    }
    matrix.getXYWH (x, y, w, h);
    return IRect (x, y, w, h);
}

static void clipToScreen (Surface *s, Matrix &m, IRect &clip) {
    Surface *ps = s->parentNode ();
    if (!ps) {
        clip = IRect (s->bounds);
        m = Matrix (s->bounds.x (), s->bounds.y (), s->xscale, s->yscale);
    } else {
        clipToScreen (ps, m, clip);
        SRect r = s->bounds;
        Single x = r.x (), y = r.y (), w = r.width (), h = r.height ();
        m.getXYWH (x, y, w, h);
        clip = clip.intersect (IRect (x, y, w, h));
        Matrix m1 = m;
        m = Matrix (s->bounds.x (), s->bounds.y (), s->xscale, s->yscale);
        m.transform (m1);
        if (!s->virtual_size.isEmpty ())
            m.translate (-s->x_scroll, -s->y_scroll);
    }
}


KDE_NO_EXPORT void Surface::repaint (const SRect &r) {
    Matrix matrix;
    IRect clip;
    clipToScreen (this, matrix, clip);
    Single x = r.x (), y = r.y (), w = r.width (), h = r.height ();
    matrix.getXYWH (x, y, w, h);
    clip = clip.intersect (IRect (x, y, w, h));
    if (!clip.isEmpty ())
        view_widget->scheduleRepaint (clip);
}

KDE_NO_EXPORT void Surface::repaint () {
    Surface *ps = parentNode ();
    if (ps)
        ps->repaint (bounds);
    else
        view_widget->scheduleRepaint (IRect (bounds));
}

