/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <signal.h>

#include <glib.h>

#include <cerrno>
#include <cmath>

#if defined (HAVE_LIB_GPS)
  #include <gps.h>
#endif

#if defined(HAVE_LIB_LOCATION)
extern "C" {
  #include <location/location-gps-device.h>
  #include <location/location-gpsd-control.h>
}
#endif

#ifndef SIMPLE_CODE

#include <iomanip>
#include <sstream>

#include <Lum/OS/Probe.h>
#include <Lum/OS/Thread.h>

#include <Lum/Base/DateTime.h>
#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Boolean.h>
#include <Lum/Model/DataStream.h>
#include <Lum/Model/Header.h>
#include <Lum/Model/String.h>

#include <Lum/OS/Main.h>

#include <Lum/AppInfo.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/LED.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/PercentBar.h>
#include <Lum/Tab.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/View.h>

#include "Compass.h"
#include "TimeLine.h"
#endif
#include <iostream>

#ifdef SIMPLE_CODE
GMainContext *context;
GMainLoop    *loop = NULL;
#else

static long obsoleteDataAge = 30; // maximum ge of data in seconds until value is invalid
static long reconnectTime   = 10; // Period of time after which we check for disconnection and
                                  // trigger reconnection

class GPSData
{
public:
  struct Satellite
  {
    int  id;
    bool used;
    int  snr;
  };

private:
  bool                   connected;
  Lum::Base::SystemTime  connectedLastData;

  bool                   online;
  Lum::Base::SystemTime  onlineLastData;

  bool                   fix;
  Lum::Base::SystemTime  fixLastData;

  double                 latitude;
  Lum::Base::SystemTime  latitudeLastData;

  double                 longitude;
  Lum::Base::SystemTime  longitudeLastData;

  double                 altitude;
  Lum::Base::SystemTime  altitudeLastData;

  double                 speed;
  Lum::Base::SystemTime  speedLastData;

  double                 track;
  Lum::Base::SystemTime  trackLastData;

  double                 climb;
  Lum::Base::SystemTime  climbLastData;

  Lum::Base::SystemTime  time;
  Lum::Base::SystemTime  timeLastData;

  std::vector<Satellite> satellites;
  Lum::Base::SystemTime  satellitesLastData;

public:
  GPSData()
   : connectedLastData(0),
     onlineLastData(0),
     fixLastData(0),
     latitudeLastData(0),
     longitudeLastData(0),
     altitudeLastData(0),
     speedLastData(0),
     trackLastData(0),
     climbLastData(0),
     timeLastData(0),
     satellitesLastData(0)
  {
    // no code
  }

  //
  // Connected
  //
  void SetConnected(bool connected)
  {
    connectedLastData.SetToNow();
    this->connected=connected;
  }

  bool HasConnected() const
  {
    Lum::Base::SystemTime now;

    return now-connectedLastData<obsoleteDataAge;
  }

  bool GetConnected() const
  {
    assert(HasConnected());

    return connected;
  }

  //
  // Online
  //
  void SetOnline(bool online)
  {
    onlineLastData.SetToNow();
    this->online=online;
  }

  bool HasOnline() const
  {
    Lum::Base::SystemTime now;

    return now-onlineLastData<obsoleteDataAge;
  }

  bool GetOnline() const
  {
    assert(HasOnline());

    return online;
  }

  //
  // Fix
  //
  void SetFix(bool fix)
  {
    fixLastData.SetToNow();
    this->fix=fix;
  }

  bool HasFix() const
  {
    Lum::Base::SystemTime now;

    return now-fixLastData<obsoleteDataAge;
  }

  bool GetFix() const
  {
    assert(HasFix());

    return fix;
  }

  //
  // Latitude
  //
  void SetLatitude(double latitude)
  {
    latitudeLastData.SetToNow();
    this->latitude=latitude;
  }

  bool HasLatitude() const
  {
    Lum::Base::SystemTime now;

    return now-latitudeLastData<obsoleteDataAge;
  }

  double GetLatitude() const
  {
    assert(HasLatitude());

    return latitude;
  }

  //
  // Longitude
  //
  void SetLongitude(double longitude)
  {
    longitudeLastData.SetToNow();
    this->longitude=longitude;
  }

  bool HasLongitude() const
  {
    Lum::Base::SystemTime now;

    return now-longitudeLastData<obsoleteDataAge;
  }

  double GetLongitude() const
  {
    assert(HasLongitude());

    return longitude;
  }

  //
  // Altitude
  //
  void SetAltitude(double altitude)
  {
    altitudeLastData.SetToNow();
    this->altitude=altitude;
  }

  bool HasAltitude() const
  {
    Lum::Base::SystemTime now;

    return now-altitudeLastData<obsoleteDataAge;
  }

  double GetAltitude() const
  {
    assert(HasAltitude());

    return altitude;
  }

  //
  // Speed
  //
  void SetSpeed(double speed)
  {
    speedLastData.SetToNow();
    this->speed=speed;
  }

  bool HasSpeed() const
  {
    Lum::Base::SystemTime now;

    return now-speedLastData<obsoleteDataAge;
  }

  double GetSpeed() const
  {
    assert(HasSpeed());

    return speed;
  }

  //
  // Track
  //
  void SetTrack(double track)
  {
    trackLastData.SetToNow();
    this->track=track;
  }

  bool HasTrack() const
  {
    Lum::Base::SystemTime now;

    return now-trackLastData<obsoleteDataAge;
  }

  double GetTrack() const
  {
    assert(HasTrack());

    return track;
  }

  //
  // Climb
  //
  void SetClimb(double climb)
  {
    climbLastData.SetToNow();
    this->climb=climb;
  }

  bool HasClimb() const
  {
    Lum::Base::SystemTime now;

    return now-climbLastData<obsoleteDataAge;
  }

  double GetClimb() const
  {
    assert(HasClimb());

    return climb;
  }

  //
  // Time
  //
  void SetTime(Lum::Base::SystemTime& time)
  {
    timeLastData.SetToNow();
    this->time=time;
  }

  bool HasTime() const
  {
    Lum::Base::SystemTime now;

    return now-timeLastData<obsoleteDataAge;
  }

  Lum::Base::SystemTime GetTime() const
  {
    assert(HasTime());

    return time;
  }

  //
  // Satellites
  //
  void SetSatellites(const std::vector<Satellite>& satellites)
  {
    satellitesLastData.SetToNow();
    this->satellites=satellites;
  }

  bool HasSatellites() const
  {
    Lum::Base::SystemTime now;

    return now-satellitesLastData<obsoleteDataAge;
  }

  const std::vector<Satellite>& GetSatellites() const
  {
    assert(HasSatellites());

    return satellites;
  }
};

typedef Lum::Model::StdTable<GPSData::Satellite> SatellitesModel;
typedef Lum::Base::Reference<SatellitesModel>    SatellitesModelRef;

double GetRelativeQualityFromDbm(double quality)
{
  if (quality<0) {
    return 0;
  }
  if (quality>=0 && quality<40) {
    return 5*quality/2;
  }
  else {
    return 100;
  }
}

class SatellitesDataProvider : public SatellitesModel::DataProvider
{
private:
  Lum::PercentBar                 *bar;
  Lum::Model::DoubleDataStreamRef signal;

public:
  SatellitesDataProvider()
  : bar(new Lum::PercentBar()),
    signal(new Lum::Model::DoubleDataStream())
  {
    bar->SetFlex(true,true);
    bar->SetModel(signal);

    signal->SetNotificationMode(Lum::Model::DoubleDataStream::notifyExplicit);
    signal->SetChannels(1);
  }

  ~SatellitesDataProvider()
  {
    delete bar;
  }

  std::wstring GetString(const SatellitesModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      return Lum::Base::NumberToWString(iter->id);
    case 2:
      return iter->used ? L"X" : L"";
    default:
      return L"";
    }
  }

  Lum::Object* GetObject(const SatellitesModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 3:
      signal->Set(0,GetRelativeQualityFromDbm(iter->snr)/100.0);
      return bar;
    default:
      return NULL;
    }
  }
};

class MyWindow;
static MyWindow* window=NULL;
static Lum::AppInfo info;
#endif

#if defined(HAVE_LIB_GPS)
static void cb( struct gps_data_t * data,
               char * buf,
               size_t len,
               int level );
#endif

#ifndef SIMPLE_CODE

std::wstring DoubleToString(double value)
{
  if (!isnan(value)) {
    std::stringstream buffer;

    buffer.imbue(std::locale(""));
    buffer << std::fixed;
    buffer << std::showpoint;
    buffer.precision(2);

    buffer << value;

    return Lum::Base::StringToWString(buffer.str());
  }
  else {
    return L"";
  }
}

class MyWindow : public Lum::Dialog
{
private:
  Lum::Model::BooleanRef   connected;
  Lum::Model::BooleanRef   online;
  Lum::Model::BooleanRef   fix;
  Lum::Model::StringRef    time;
  Lum::Model::StringRef    latitude;
  Lum::Model::StringRef    longitude;
  Lum::Model::StringRef    altitude;
  Lum::Model::StringRef    speed;
  Lum::Model::StringRef    track;
  Lum::Model::DoubleRef    direction;
  Lum::Model::StringRef    climb;
  SatellitesModelRef       satellites;
  Lum::Model::DoubleDataStreamRef speedStream;
  Lum::Model::DoubleDataStreamRef altitudeStream;

  Lum::Model::ActionRef    aboutAction;

  Lum::Model::ActionRef    dataChangedAction;
  Lum::Model::ActionRef    refreshTimer;
  Lum::Model::ActionRef    reconnectCheckTimer;

#if defined(HAVE_LIB_LOCATION)
  LocationGPSDControl      *control;
#endif
#if defined(HAVE_LIB_GPS)
  struct gps_data_t        *gpsData;
  pthread_t                gpsThread;
#endif

public:
  Lum::OS::Mutex           dataMutex;
  GPSData                  data;

private:
  Lum::TextValue* CreateTextValue(Lum::Model::StringRef& model, size_t size)
  {
    Lum::TextValue *textValue;

    textValue=Lum::TextValue::Create(model,Lum::TextValue::left,true,false);
    textValue->SetWidth(Lum::Base::Size::stdCharWidth,size);

    return textValue;
  }

  void UpdateConnectionStatus()
  {
    Lum::OS::Guard<Lum::OS::Mutex> guard(dataMutex);

    data.SetConnected(gpsData!=NULL);
  }

  void InitializeGPS()
  {
    std::cout << "Initialize GPS..." << std::endl;
#if defined(HAVE_LIB_LOCATION)
    control=location_gpsd_control_get_default();

    if(control->can_control) {
      location_gpsd_control_start(control);
    }
#endif
#if defined(HAVE_LIB_GPS)
    // This fixed a bug in libgps, using locale aware functions to parse non-locale-aware float values
    setlocale(LC_NUMERIC,"C");
    gpsData=::gps_open("127.0.0.1","2947");

    if (gpsData!=NULL) {
      gps_query(gpsData,"j=0w+x\n");
      gps_set_callback(gpsData,cb,&gpsThread);
      std::cout << "GPS initialized!" << std::endl;
    }
    else {
      std::cout << "GPS NOT initialized: " << errno << std::endl;
    }

    UpdateConnectionStatus();
#endif
  }

  void DeinitializeGPS()
  {
    std::cout << "Deinitialize GPS..." << std::endl;

#if defined(HAVE_LIB_GPS)
    if (gpsData!=NULL) {
      ::gps_close(gpsData);
      gpsData=NULL;
    }
#endif

#if defined(HAVE_LIB_LOCATION)
    if(control->can_control) {
      location_gpsd_control_stop(control);
    }

    /*Device control cleanup */
    g_object_unref(control);
    control=NULL;
#endif

    UpdateConnectionStatus();

    std::cout << "GPS deinitialized!" << std::endl;
  }

  void ReconnectIfDisconnected()
  {
    if (data.HasConnected() && data.GetConnected()) {
      return;
    }

#if defined(HAVE_LIB_GPS)
    if (gpsData!=NULL) {
      ::gps_close(gpsData);
      gpsData=NULL;
    }

    gpsData=::gps_open("127.0.0.1","2947");

    if (gpsData!=NULL) {
      gps_query(gpsData,"j=0w+x\n");
      gps_set_callback(gpsData,cb,&gpsThread);
      data.SetConnected(true);
      std::cout << "GPS initialized!" << std::endl;
    }
    else {
      data.SetConnected(false);
      std::cout << "GPS NOT initialized: " << errno << std::endl;
    }

    UpdateConnectionStatus();
#endif
  }

public:
  MyWindow()
  : connected(new Lum::Model::Boolean(false)),
    online(new Lum::Model::Boolean(false)),
    fix(new Lum::Model::Boolean(false)),
    time(new Lum::Model::String(L"")),
    latitude(new Lum::Model::String(L"")),
    longitude(new Lum::Model::String(L"")),
    altitude(new Lum::Model::String(L"")),
    speed(new Lum::Model::String(L"")),
    track(new Lum::Model::String(L"")),
    direction(new Lum::Model::Double()),
    climb(new Lum::Model::String(L"")),
    satellites(new SatellitesModel(new SatellitesDataProvider())),
    speedStream(new Lum::Model::DoubleDataStream()),
    altitudeStream(new Lum::Model::DoubleDataStream()),
    aboutAction(new Lum::Model::Action()),
    dataChangedAction(new Lum::Model::Action()),
    refreshTimer(new Lum::Model::Action()),
    reconnectCheckTimer(new Lum::Model::Action())
#if defined(HAVE_LIB_LOCATION)
    ,control(NULL)
#endif
#if defined(HAVE_LIB_GPS)
    ,gpsData(NULL)
#endif
  {
    Observe(GetOpenedAction());
    Observe(aboutAction);
    Observe(dataChangedAction);
    Observe(refreshTimer);
    Observe(reconnectCheckTimer);

    ::window=this;
  }

  void PreInit()
  {
    Lum::Model::HeaderRef headerModel;
    Lum::Label            *label;
    Lum::Panel            *hPanel;
    Lum::Tab              *tab;
    Lum::Table            *table;
    Compass               *compass;

    tab=Lum::Tab::Create(true,true);

    hPanel=Lum::HPanel::Create(true,true);

    label=Lum::Label::Create(true,true);

    label->AddLabel(L"Connected:",new Lum::LED(connected));
    label->AddLabel(L"Online:",new Lum::LED(online));
    label->AddLabel(L"Fix:",new Lum::LED(fix));
    label->AddLabel(L"Time:",CreateTextValue(time,24)); // TODO: Measure by example
    label->AddLabel(L"Latitude:",CreateTextValue(latitude,10));
    label->AddLabel(L"Longitude:",CreateTextValue(longitude,10));
    label->AddLabel(L"Altitude:",CreateTextValue(altitude,10));
    label->AddLabel(L"Speed:",CreateTextValue(speed,10));
    label->AddLabel(L"Direction:",CreateTextValue(track,10));
    label->AddLabel(L"Climb:",CreateTextValue(climb,10));
    hPanel->Add(label);

    tab->Add(L"General",hPanel);

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Id",Lum::Base::Size::stdCharWidth,3);
    headerModel->AddColumn(L"Used",Lum::Base::Size::stdCharWidth,1);
    headerModel->AddColumn(L"Signal",Lum::Base::Size::stdCharWidth,25,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(true);
    table->SetHeaderModel(headerModel);
    table->SetModel(satellites);
    table->GetTableView()->SetAutoFitColumns(true);

    tab->Add(L"Satellites",table);

    compass=new Compass();
    compass->SetFlex(true,true);
    compass->SetModel(direction);

    tab->Add(L"Compass",compass);

    tab->Add(L"History",
             Lum::VPanel::Create(true,true)
             ->Add(new Lum::Text(L"Speed (m/s)"))
             ->AddSpace()
             ->Add(Lum::View::Create(TimeLine::Create(speedStream,true,true),true,true))
             ->AddSpace()
             ->Add(new Lum::Text(L"Altitude (m)"))
             ->AddSpace()
             ->Add(Lum::View::Create(TimeLine::Create(altitudeStream,true,true),true,true)));

    SetMain(tab);

    Lum::MenuDesc *menu=new Lum::MenuDesc();

    menu
      ->AddMenuItemSub(_ld(menuProject))
        ->AddMenuItemAction(_ld(menuProjectQuit),Lum::OS::qualifierControl,L"q",GetClosedAction());
    menu
      ->AddMenuItemSub(_ld(menuHelp))
        ->AddMenuItemAction(_ld(menuHelpHelp),NULL)
        ->AddMenuItemAction(_ld(menuHelpAbout),aboutAction);

    SetMenu(menu);

    Dialog::PreInit();
  }

  void GetGPSDataCopy(GPSData& data)
  {
    Lum::OS::Guard<Lum::OS::Mutex> guard(dataMutex);

    data=this->data;
  }

  void ShowChange(const GPSData& data)
  {
    if (data.HasConnected()) {
      connected->Set(data.GetConnected());
    }
    else {
      connected->Set(false);
    }

    if (data.HasOnline()) {
      online->Set(data.GetOnline());
    }
    else {
      online->Set(false);
    }

    if (data.HasFix()) {
      fix->Set(data.GetFix());
    }
    else {
      fix->Set(false);
    }

    if (data.HasTime()) {
      time->Set(data.GetTime().GetLocalLocaleDateTime());
    }
    else {
      time->SetNull();
    }

    if (data.HasLatitude()) {
      latitude->Set(DoubleToString(data.GetLatitude())+L"\u00b0");
    }
    else {
      latitude->SetNull();
    }

    if (data.HasLongitude()) {
      longitude->Set(DoubleToString(data.GetLongitude())+L"\u00b0");
    }
    else {
      longitude->SetNull();
    }

    if (data.HasAltitude()) {
      altitude->Set(DoubleToString(data.GetAltitude())+L"m");
    }
    else {
      altitude->SetNull();
    }

    if (data.HasSpeed()) {
      if (data.GetSpeed()>=1/3.6) {
        speed->Set(DoubleToString(data.GetSpeed()*3.6)+L"km/h");
      }
      else {
        speed->Set(DoubleToString(data.GetSpeed())+L"m/s");
      }
    }
    else {
      speed->SetNull();
    }

    if (data.HasTrack()) {
      track->Set(DoubleToString(data.GetTrack())+L"\u00b0");
      direction->Set(data.GetTrack());
    }
    else {
      track->SetNull();
      direction->SetNull();
    }

    if (data.HasClimb()) {
      climb->Set(DoubleToString(data.GetClimb())+L"m");
    }
    else {
      climb->SetNull();
    }

    if (data.HasSatellites()) {
      // Delete everything in model that is not in current list
      size_t i=1;
      while (i<=satellites->GetRows()) {
        bool found=false;

        for (size_t j=0; j<data.GetSatellites().size(); j++) {
          if (satellites->GetEntry(i).id==data.GetSatellites()[j].id) {
            found=true;
            break;
          }
        }

        if (found) {
          i++;
          continue;
        }
        else {
          satellites->Delete(i);
        }
      }

      // Now add everything to model that is not already in
      for (size_t i=0; i<data.GetSatellites().size(); i++) {
        bool found=false;

        for (size_t j=1; j<=satellites->GetRows(); j++) {
          if (satellites->GetEntry(j).id==data.GetSatellites()[i].id) {
            found=true;

            if (satellites->GetEntry(j).snr!=data.GetSatellites()[i].snr ||
                satellites->GetEntry(j).used!=data.GetSatellites()[i].used) {
              satellites->GetEntry(j)=data.GetSatellites()[i];
              satellites->RedrawRow(j);
            }
            break;
          }
        }

        if (!found) {
          satellites->Append(data.GetSatellites()[i]);
        }
      }
    }
    else {
      satellites->Clear();
    }
  }

  void UpdateStreams(const GPSData& data)
  {
    if (!data.HasFix() || !data.GetFix() ||
        !data.HasSpeed()) {
      speedStream->SetNull();
    }
    else {
      speedStream->Set(0,data.GetSpeed());
    }

    if (!data.HasFix() || !data.GetFix() ||
        !data.HasAltitude()) {
      altitudeStream->SetNull();
    }
    else {
      altitudeStream->Set(0,data.GetAltitude());
    }
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      InitializeGPS();
      Lum::OS::display->AddTimer(1,0,refreshTimer);
      Lum::OS::display->AddTimer(reconnectTime,0,reconnectCheckTimer);
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      DeinitializeGPS();
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==dataChangedAction && dataChangedAction->IsFinished()) {
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      ShowChange(dataCopy);
    }
    else if (model==refreshTimer && refreshTimer->IsFinished()) {
      GPSData dataCopy;

      GetGPSDataCopy(dataCopy);
      ShowChange(dataCopy);
      UpdateStreams(dataCopy);

      Lum::OS::display->AddTimer(1,0,refreshTimer);
    }
    else if (model==reconnectCheckTimer && reconnectCheckTimer->IsFinished()) {
      ReconnectIfDisconnected();

      Lum::OS::display->AddTimer(reconnectTime,0,reconnectCheckTimer);
    }

    Dialog::Resync(model,msg);
  }

  void HandleChange()
  {
    Lum::OS::display->QueueActionForAsyncNotification(dataChangedAction);
  }
};
#endif

#ifdef SIMPLE_CODE
static void gps_location_changed (LocationGPSDevice *device, gpointer userdata);
static void gps_location_connected (LocationGPSDevice *new_device, gpointer userdata);
static void gps_location_disconnected (LocationGPSDevice *new_device, gpointer userdata);
static void gps_location_started (LocationGPSDControl *control, gpointer userdata);
static void gps_location_stopped (LocationGPSDControl *control, gpointer userdata);
static void gps_location_error (LocationGPSDControl *control, gpointer userdata);

static void gps_location_changed(LocationGPSDevice *device, gpointer userdata)
{
  std::cout << "gps_location_changed" << std::endl;

    std::cout << "-----------------" << std::endl;
    std::cout << "Status: " << (device->online? "true" : "false") << std::endl;
    std::cout << "Detailed status: ";
    switch (device->status) {
    case LOCATION_GPS_DEVICE_STATUS_NO_FIX:
      std::cout << "No fix";
      break;
      case LOCATION_GPS_DEVICE_STATUS_FIX:
      std::cout << "Fix";
      break;
    case LOCATION_GPS_DEVICE_STATUS_DGPS_FIX:
      std::cout << "DGPS fix";
      break;
    }

    std::cout << std::endl;

    std::cout << device->satellites_in_view << " satellites in view, " << device->satellites_in_use << " used!" << std::endl;

    /* Fix information */
    LocationGPSDeviceFix *fix;
    fix = device->fix;

    std::cout << "Fix mode: ";
    switch(fix->mode) {
      case LOCATION_GPS_DEVICE_MODE_NOT_SEEN:
      std::cout << "No satelite";
      break;
    case LOCATION_GPS_DEVICE_MODE_NO_FIX:
      std::cout << "No fix";
      break;
    case LOCATION_GPS_DEVICE_MODE_2D:
      std::cout << "2D information";
      break;
    case LOCATION_GPS_DEVICE_MODE_3D:
      std::cout << "3D information";
      break;
    }
    std::cout << std::endl;

    if (fix->fields & LOCATION_GPS_DEVICE_TIME_SET) {
      std::cout << "Time: " << fix->time << " uncertainty: " << fix->ept << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET) {
      std::cout << "Latitude: " << fix->latitude << " longitude: " << fix->longitude << " uncertainty: " <<  fix->eph << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET) {
      std::cout << "Altitude: " << fix->altitude << " uncertainty: " << fix->epv << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_SPEED_SET) {
      std::cout << "Speed: " << fix->speed << " uncertainty: " << fix->eps << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_TRACK_SET) {
      std::cout << "Track: " << fix->track << " uncertainty: " << fix->epd << std::endl;
    }
    if (fix->fields & LOCATION_GPS_DEVICE_CLIMB_SET) {
      std::cout << "Climb: " << fix->climb << " uncertainty: " << fix->epc << std::endl;
    }

    fix = NULL;

    if (device->satellites!=NULL) {
      for (int i =0; i<device->satellites_in_view; i++) {
        LocationGPSDeviceSatellite* satellite=(LocationGPSDeviceSatellite*)g_ptr_array_index(device->satellites,i);

        std::cout << "Satellite id:" << satellite->prn << " S/N: " << satellite->signal_strength << " elevation: " << satellite->elevation << " azimuth: " << satellite->azimuth << " " << (satellite->in_use ? "used" : "unused") << std::endl;
      }
    }
}

static void gps_location_connected(LocationGPSDevice *new_device, gpointer userdata) {
  std::cout << "Connected..." << std::endl;
}

static void gps_location_disconnected(LocationGPSDevice *new_device, gpointer userdata){
  std::cout << "disconnected..." << std::endl;
}

static void gps_location_started(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD started!" << std::endl;
}

static void gps_location_stopped(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD stoped!" << std::endl;
}

static void gps_location_error(LocationGPSDControl *control, gpointer userdata) {
  std::cout << "GPSD error!" << std::endl;
}

static void gps_location_quit(int signal) {
  printf("[Info] SIGINT caught, exiting\n");
  g_main_loop_quit(loop);
}
#endif

#if defined(HAVE_LIB_GPS)
static void cb(struct gps_data_t* data, char * buf, size_t len, int level)
{
  assert(window!=NULL);

  Lum::OS::Guard<Lum::OS::Mutex> guard(window->dataMutex);

  GPSData                        &gpsData=window->data;

  gpsData.SetConnected(true);

  if (data->set & ONLINE_SET) {
    gpsData.SetOnline(data->online);
  }

  if (data->set & STATUS_SET) {
    gpsData.SetFix(data->status!=STATUS_NO_FIX);
  }

  if ((data->set & LATLON_SET)  && !isnan(data->fix.latitude)) {
    gpsData.SetLatitude(data->fix.latitude);
  }

  if ((data->set & LATLON_SET)  && !isnan(data->fix.longitude)) {
    gpsData.SetLongitude(data->fix.longitude);
  }

  if ((data->set & ALTITUDE_SET) && !isnan(data->fix.altitude)) {
    gpsData.SetAltitude(data->fix.altitude);
  }

  if ((data->set & SPEED_SET) && !isnan(data->fix.speed)) {
    gpsData.SetSpeed(data->fix.speed);
  }

  if ((data->set & TRACK_SET) && !isnan(data->fix.track)) {
    gpsData.SetTrack(data->fix.track);
  }

  if ((data->set & CLIMB_SET) && !isnan(data->fix.climb)) {
    gpsData.SetClimb(data->fix.climb);
  }

  if (data->set & TIME_SET) {
    Lum::Base::SystemTime time(lround(data->fix.time));
    gpsData.SetTime(time);
  }

  if (data->set & SATELLITE_SET) {
    std::vector<GPSData::Satellite> satellites;

    for (int i=0; i<data->satellites; i++) {
      GPSData::Satellite s;

      s.id=data->PRN[i];
      s.used=data->used[i];
      s.snr=data->ss[i];

      satellites.push_back(s);
    }

    gpsData.SetSatellites(satellites);
  }

  window->HandleChange();
}
#endif

#ifndef SIMPLE_CODE
class Main : public Lum::OS::MainDialog<MyWindow>
{
public:
  Main()
  {
    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"Where the #*~@# am I?");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2009, Tim Teulings");
    info.SetLicense(L"GNU Public License");
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

#else

int main(int argc, char *argv[]) {

	/* Inicialitzacio de la llibreria glib */
	g_type_init();
	g_thread_init(NULL);
	printf("[Info] Glib started\n");
	
	
	/* Location Control setup*/
	LocationGPSDControl *control = NULL;
	control = location_gpsd_control_get_default();
	if(control->can_control)
		location_gpsd_control_start (control);
	
	/* Recupera informacio actual */
	LocationGPSDevice *device = NULL;
	device = (LocationGPSDevice*)g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);
	location_gps_device_reset_last_known (device);
	printf("[Debug] Device pointer adress: %x\n", (unsigned int)device);
	printf("[Info] Device object obtained\n");
	
	/* Install signal handlers */
	signal(SIGINT, gps_location_quit);
	guint idd_changed = g_signal_connect (device, "changed", G_CALLBACK (gps_location_changed), NULL);
	guint idd_con = g_signal_connect (device, "connected", G_CALLBACK (gps_location_connected), NULL);
	guint idd_disc = g_signal_connect (device, "disconnected", G_CALLBACK (gps_location_disconnected), NULL);
	
	guint idc_run = g_signal_connect (control, "gpsd_running", G_CALLBACK (gps_location_started), NULL);
	guint idc_stop = g_signal_connect (control, "gpsd_stopped", G_CALLBACK (gps_location_stopped), NULL);
	guint idc_error = g_signal_connect (control, "error", G_CALLBACK (gps_location_error), NULL);
	
	printf("[Info] Entering the main loop event\n");
        context=g_main_context_new();
  	loop = g_main_loop_new(context, FALSE);
	/* g_timeout_add(1000, (GSourceFunc)callback, NULL); */
 	g_main_loop_run(loop);
 	
 	/* Exiting */
 	g_main_loop_unref(loop);
 	g_main_context_unref(context);
	
	/* Disconnect signal */
	if(control->can_control)
		location_gpsd_control_stop(control);
		
	g_signal_handler_disconnect(device, idd_changed);
	g_signal_handler_disconnect(device, idd_con);
	g_signal_handler_disconnect(device, idd_disc);
	
	g_signal_handler_disconnect(control, idc_run);
	g_signal_handler_disconnect(control, idc_stop);
	g_signal_handler_disconnect(control, idc_error);

	/*	Device control cleanup */
	g_object_unref(control);	
	g_object_unref(device);

	return 0;
}
#endif
