/*
  This file is part of "GiveMeFive" - A simple "five in a row" game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "GameArea.h"

#include <cstdlib>

GameArea::Point::Point()
: x(0),y(0),value(GameArea::empty)
{
  // no code
}

GameArea::Point::Point(size_t x, size_t y, Value value)
 : x(x),y(y),value(value)
{
}

GameArea::Rule::~Rule()
{
  // no code
}

GameArea::Rule1D::Rule1D(const std::wstring& pattern)
 : pattern(pattern)
{
  // no code
}

bool GameArea::Rule1D::Check(GameArea& area, const Linearisation& linearisation, Point& move)
{
  Value player=area.GetCurrentPlayer();
  Value other=area.GetOtherPlayer();

  for (size_t a=0; a<linearisation.size(); a++) {
    std::vector<Point> line;

    line=linearisation[a];

    if (line.size()>=pattern.length()) {
      // for every point in the line
      for (size_t b=0; b<line.size()-pattern.length(); b++) {
        // Match pattern
        size_t j=0;
        size_t c=b;
        bool   match=true;

        while (c<b+pattern.length() && match) {
          Point current=line[c];

          switch (pattern[j]) {
          case L' ':
            match=current.value==GameArea::empty;
            break;
          case L'.':
            match=current.value==GameArea::empty;
            if (match) {
              move=current;
            }
            break;
          case L'x':
            match=current.value==player;
            break;
          case L'#':
            match=current.value==other;
            break;
          }

          c++;
          j++;
        }

        if (match) {
          return true;
        }
      }

      // for every point in the line - but match the reverse pattern
      for (size_t b=0; b<line.size()-pattern.length(); b++) {
        // Match pattern
        size_t  j=pattern.length()-1;
        size_t  c=b;
        bool    match=true;

        while (c<b+pattern.length() && match) {
          Point current=line[c];

          switch (pattern[j]) {
          case L' ':
            match=current.value==GameArea::empty;
            break;
          case L'.':
            match=current.value==GameArea::empty;
            if (match) {
              move=current;
            }
            break;
          case L'x':
            match=current.value==player;
            break;
          case L'#':
            match=current.value==other;
            break;
          }

          c++;
          j--;
        }

        if (match) {
          return true;
        }
      }
    }
  }

  return false;
}

GameArea::GameArea()
 : state(stateHuman)
{
  srand(time(NULL));

  //
  // 4er
  //

  // my (if I have the chance to win, win :-))
  //rules.push_back(new Rule1D(L"xxxx. "));
  rules.push_back(new Rule1D(L"xxxx."));
  rules.push_back(new Rule1D(L"xxx.x"));
  rules.push_back(new Rule1D(L"xx.xx"));


  // other (if the other has the chance to win, try to avoid it)
  rules.push_back(new Rule1D(L"####."));
  rules.push_back(new Rule1D(L"###.#"));
  rules.push_back(new Rule1D(L"##.##"));

  //
  // 3er
  //

  // my
  rules.push_back(new Rule1D(L" x.xx "));
  rules.push_back(new Rule1D(L".xxx "));
  rules.push_back(new Rule1D(L"xxx. "));
  rules.push_back(new Rule1D(L"xx. x"));
  rules.push_back(new Rule1D(L"xx.x "));
  rules.push_back(new Rule1D(L"x xx "));

  // other
  rules.push_back(new Rule1D(L" ##.## "));
  rules.push_back(new Rule1D(L".### "));
  rules.push_back(new Rule1D(L"###. "));
  rules.push_back(new Rule1D(L"##. #"));
  rules.push_back(new Rule1D(L"##.# "));
  rules.push_back(new Rule1D(L"# ## "));

  //
  // 2er
  //

  // me
  rules.push_back(new Rule1D(L".xx  "));
  rules.push_back(new Rule1D(L" x.x "));
  rules.push_back(new Rule1D(L"xx.  "));
  rules.push_back(new Rule1D(L"x.x  "));
  rules.push_back(new Rule1D(L"x. x "));
  rules.push_back(new Rule1D(L"x.  x"));

  // other
  rules.push_back(new Rule1D(L".##  "));
  rules.push_back(new Rule1D(L" #.# "));
  rules.push_back(new Rule1D(L"##.  "));
  rules.push_back(new Rule1D(L"#.#  "));
  rules.push_back(new Rule1D(L"#. # "));
  rules.push_back(new Rule1D(L"#.  #"));

  //
  // 1er
  //

  // me
  rules.push_back(new Rule1D(L".x   "));
  rules.push_back(new Rule1D(L" .x  "));
  rules.push_back(new Rule1D(L"x.   "));

  // other
  rules.push_back(new Rule1D(L".#   "));
  rules.push_back(new Rule1D(L" .#  "));
  rules.push_back(new Rule1D(L"#.   "));

  RestartGame();
}

GameArea::~GameArea()
{
  for (std::list<Rule*>::iterator iter=rules.begin(); iter!=rules.end(); ++iter) {
    delete *iter;
  }

  rules.clear();
}

void GameArea::RestartGame()
{
  for (size_t i=0; i<dimension*dimension; i++) {
    area[i]=empty;
    marked[i]=0;
  }

  state=stateHuman;

  Notify();
}

bool GameArea::IsSet(size_t x, size_t y) const
{
  assert(x>=0 && x<dimension && y>=0 && y<dimension);

  return area[y*dimension+x]!=empty;
}

void GameArea::Set(size_t x, size_t y, Value value)
{
  assert(x>=0 && x<dimension && y>=0 && y<dimension);

  area[y*dimension+x]=value;

  Notify();
}

GameArea::Value GameArea::Get(size_t x, size_t y) const
{
  assert(x>=0 && x<dimension && y>=0 && y<dimension);

  return area[y*dimension+x];
}

GameArea::State GameArea::GetState() const
{
  return state;
}

bool GameArea::IsMarked(size_t x, size_t y) const
{
  assert(x>=0 && x<dimension && y>=0 && y<dimension);

  return marked[y*dimension+x];
}

void GameArea::SetMark(size_t x, size_t y)
{
  assert(x>=0 && x<dimension && y>=0 && y<dimension);

  marked[y*dimension+x]=true;
}

GameArea::Value GameArea::GetCurrentPlayer() const
{
  switch (state) {
  case stateHuman:
    return player1;
  case stateComputer:
    return player2;
  case stateFinished:
    return empty;
  }

  return empty;
}

GameArea::Value GameArea::GetOtherPlayer() const
{
  switch (state) {
  case stateHuman:
    return player2;
  case stateComputer:
    return player1;
  case stateFinished:
    return empty;
  }

  return empty;
}

void GameArea::PlayerFinished()
{
  Linearisation linearisation;

  GetLinearisation(linearisation);

  if (CheckFinished(linearisation)) {
    state=stateFinished;
  }

  if (state==stateHuman) {
    state=stateComputer;
  }
  if (state==stateComputer) {
    CalculateNextMove(linearisation);
    GetLinearisation(linearisation);

    if (CheckFinished(linearisation)) {
      state=stateFinished;
    }
    else {
      state=stateHuman;
    }
  }
}

void GameArea::GetLinearisation(Linearisation& linearisation) const
{
  linearisation.clear();

  // horiz
  for (size_t y=0; y<dimension; y++) {
    std::vector<Point> line(dimension);

    line.reserve(dimension);
    for (size_t x=0; x<dimension; x++) {
      line[x]=Point(x,y,Get(x,y));
    }
    linearisation.push_back(line);
  }

  // vert
  for (size_t x=0; x<dimension; x++) {
    std::vector<Point> line(dimension);

    for (size_t y=0; y<dimension; y++) {
      line[y]=Point(x,y,Get(x,y));
    }
    linearisation.push_back(line);
  }

  // top left to bottom right
  for (int i=(int)-dimension+1; i<(int)dimension; i++) {
    size_t             x,y;
    std::vector<Point> line;

    if (i<0) {
      x=0;
      y=-i;
    }
    else {
      x=i;
      y=0;
    }

    while (x<dimension && y<dimension) {
      line.push_back(Point(x,y,Get(x,y)));

      x++;
      y++;
    }

    linearisation.push_back(line);
  }

  // top right to bottom left
  for (size_t i=0; i<2*dimension; i++) {
    int                x,y;
    std::vector<Point> line;

    if (i<dimension) {
      x=i;
      y=0;
    }
    else {
      x=dimension-1;
      y=i-dimension+1;
    }

    while (x>=0 && y<(int)dimension) {
      line.push_back(Point(x,y,Get(x,y)));

      x-=1;
      y+=1;
    }

    linearisation.push_back(line);
  }
}

bool GameArea::CheckFinished(const Linearisation& linearisation)
{
  for (size_t a=0; a<linearisation.size(); a++) {
    std::vector<Point> line;
    Value              player=empty;
    size_t             count=0;
    size_t             start=0;

    line=linearisation[a];

    for (size_t i=0; i<line.size(); i++) {
      Point current=line[i];
      Value value=current.value;

      if (value==empty) {
        player=empty;
        count=0;
      }
      else if (value==player) {
        count++;
      }
      else {
        player=value;
        count=1;
        start=i;
      }

      if (count>=5) {
        for (size_t b=start; b<=i; b++) {
          current=line[b];
          SetMark(current.x,current.y);
        }
        Notify();
        return true;
      }
    }
  }

  return false;
}

void GameArea::CalculateNextMove(const Linearisation& linearisation)
{
  //
  // Evaluate internal rules
  //

  for (std::list<Rule*>::iterator iter=rules.begin(); iter!=rules.end(); ++iter) {
    Point move;

    if ((*iter)->Check(*this,linearisation,move)) {
      Set(move.x,move.y,GetCurrentPlayer());
      return;
    }
  }

  //
  // If no rule matches, set some stone by random
  //

  std::vector<Point> frees;

  // collect all free fields
  for (size_t y=0; y<dimension; y++) {
    for (size_t x=0; x<dimension; x++) {
      if (!IsSet(x,y)) {
        frees.push_back(Point(x,y,empty));
      }
    }
  }

  size_t pos=(size_t)(double(frees.size())*rand()/RAND_MAX);

  Set(frees[pos].x,frees[pos].y,GetCurrentPlayer());
}

