/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "ui.h"
#include "uiexpimp.h"
#include "csvfunc.h"


/* Import window */
#define DATEFORMATENTRYMAX 10

/* Macros to prevent passing a NULL pointer to functions */
#define NONNULLORZERO(s) ((s) == NULL) ? "0" : (s->data)
#define NONNULLOREMPTY(s) ((s) == NULL) ? "" : (s->data)

/* Import window */
csvlist plist;

extern char *datefmtstr;
extern unit_t curunit;

static gboolean delete_importwin( GtkWidget *widget,
                                  GdkEvent  *event,
                                  gpointer   data )
{
  gtk_widget_destroy(widget);
  return FALSE;
}

GtkTreeModel *fill_csvpreview_model(csvlist *plist, gint rows, gint cols)
{
  GtkListStore  *store;
  GtkTreeIter    iter;
  GType *coltypes;
  gint i, j;
  csvlist *prow;
  csvlistnode *pl;

  coltypes=(GType *)malloc(sizeof(GType)*rows);
  for (i=0;i<cols;i++)
    coltypes[i]=G_TYPE_STRING;

  store=gtk_list_store_newv(cols, coltypes);

  for (i=1;i<=rows;i++) {
    prow = csvgetrow(plist,i);
    gtk_list_store_append (store, &iter);
    for (j=1;j<=prow->nodes;j++) {
      pl=csvgetrowcol(plist, i, j);
      if (pl != NULL) {// should be, but won't hurt
	gtk_list_store_set (store, &iter,j-1,(char *)pl->data,-1);
      }
    }
  }
  free(coltypes);

  return GTK_TREE_MODEL (store);
}

static GtkWidget *
create_csvpreview_view (csvlist *plist, gint rows, gint cols)
{
  GtkTreeModel      *model;
  GtkWidget         *csvview;
  GtkCellRenderer   *renderer;
  GtkTreeViewColumn *col;
  gint i;
  gchar *collabel;

  model = fill_csvpreview_model(plist, rows, cols);
  csvview = gtk_tree_view_new_with_model (model);

  for (i=0;i<cols;i++) {
    col = gtk_tree_view_column_new();
    collabel=g_strdup_printf(_("Column %d"),i+1);
    gtk_tree_view_column_set_title(col, collabel);
    g_free(collabel);
    gtk_tree_view_append_column(GTK_TREE_VIEW(csvview), col);
    
    renderer = gtk_cell_renderer_text_new();
    
    /* pack cell renderer into tree view column */
    gtk_tree_view_column_pack_start(col, renderer, TRUE);

    gtk_tree_view_column_add_attribute(col, renderer, "text", i);
    gtk_tree_view_column_set_resizable(col, TRUE);
  }
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(csvview), TRUE);
  g_object_unref (model);

  return csvview;
}

void callback_startimport(GtkDialog *widget, gint arg1, AppUIData *pui)
{
  gint i;
  gint fl;
  gboolean clocale;
  gint coldes[IMPORTCOLS];
  char *date;
  gdouble km;
  gdouble trip;
  gdouble fill;
  gdouble price;
  gdouble consum;
  gdouble service;
  gdouble oil;
  gdouble tires;

  if (arg1==GTK_RESPONSE_ACCEPT) {
    ui_show_banner(_("Storing to database, please wait"));

    /* Update the UI */
    while (gtk_events_pending())
      gtk_main_iteration();
#if PLAINGTK == 1
    fl=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(pui->flspinner));
#else
    fl=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(pui->flspinner));
#endif
    for (i=0;i<IMPORTCOLS;i++) {
#if PLAINGTK == 1
      coldes[i]=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(pui->colspin[i]));
#else
      coldes[i]=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(pui->colspin[i]));
#endif
    }

    clocale=(gtk_combo_box_get_active(GTK_COMBO_BOX(pui->localecombo)) == 1);
    if (clocale)
      setlocale(LC_NUMERIC, "C");

    for (i=fl;i<=plist.nodes;i++) {

      date2sqlite(&date, gtk_entry_get_text(GTK_ENTRY(pui->dateformatentry)), (char *)(NONNULLOREMPTY((char *)(csvgetrowcol(&plist, i, coldes[0])))));
      km = user2SIlength(atof(NONNULLORZERO((char *)(csvgetrowcol(&plist, i, coldes[1])))));
      trip = user2SIlength(atof(NONNULLORZERO((char *)(csvgetrowcol(&plist, i, coldes[2])))));
      fill = user2SIvolume(atof(NONNULLORZERO((char *)(csvgetrowcol(&plist, i, coldes[3])))));
      price = atof(NONNULLORZERO((char *)(csvgetrowcol(&plist, i, coldes[4]))));
      service = atof(NONNULLORZERO((char *)(csvgetrowcol(&plist, i, coldes[5]))));
      oil = atof(NONNULLORZERO((char *)(csvgetrowcol(&plist, i, coldes[6]))));
      tires = atof(NONNULLORZERO((char *)(csvgetrowcol(&plist, i, coldes[7]))));

      consum = fill/trip*100;

      if (db_open()) {
	if (SQLITE_OK != sqlite3_bind_int(ppStmtAddRecord, 1, currentcar) /* carid */
	    || SQLITE_OK != sqlite3_bind_int(ppStmtAddRecord, 2, currentdriver) /* driverid */
	    || SQLITE_OK != sqlite3_bind_text(ppStmtAddRecord, 3, 
					      date,
					      -1, SQLITE_STATIC)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 4,
						km)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 5,
						trip)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 6,
						fill)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 7,
						consum)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 8,
						price)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 9,
						price/fill)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 10,
						service)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 11,
						oil)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 12,
						tires)
	    || SQLITE_OK != sqlite3_bind_text(ppStmtAddRecord, 13, 
					      NONNULLOREMPTY((char *)(csvgetrowcol(&plist, i, coldes[8]))),
					      -1, SQLITE_STATIC)
	    || SQLITE_DONE != sqlite3_step(ppStmtAddRecord)) {
	  /*     ui_show_banner("Problem saving result"); */
	  PDEBUG("Problem saving imported data to database");
	}
	else { /* record succesfully inserted */
	}
	sqlite3_reset(ppStmtAddRecord);
      }
      else {
	PDEBUG("Can't save new record - no database open");
      }

      free(date);

    }
    update_record_model(pui);
    if (clocale)
      setlocale(LC_NUMERIC, "");

    csvfree(&plist);
  }
  gtk_widget_destroy(GTK_WIDGET(widget));
}

GtkWidget *create_column_spinner(gint initval, gint maxval)
{
  GtkWidget *spinner;
#if PLAINGTK == 1
  GtkAdjustment *adj;
#endif

#if PLAINGTK == 1
  adj = (GtkAdjustment *) gtk_adjustment_new (initval, 0, maxval,
					      1, 1, 0.0);
  spinner = gtk_spin_button_new (adj, 1.0, 0);
  gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), FALSE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spinner),
				     GTK_UPDATE_ALWAYS);
#else
  spinner = hildon_number_editor_new(0,maxval);
  hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(spinner),initval);
#endif
  return spinner;
}

void callback_import( GtkAction * action, AppUIData *pui )
{
  gchar *filename = NULL;
  FILE *f;
  gint emptyrows;
  gint cols, rows;
  gint i;
  csvlistnode *pl;
#ifdef _GNU_SOURCE
  char *line;
  size_t llen=0;
#else
  char line[LINELEN];
#endif
  gchar *convline;
#if PLAINGTK == 1
  GtkWidget *importwindow;
#else
/*   HildonWindow *importwindow; */
  GtkWidget *importwindow;
#endif
  GtkWidget *vbox;
  GtkWidget *notebook;
  GtkWidget *table;
  GtkWidget *colscrollwin;
  GtkWidget *carscrollwin;
  GtkWidget *scrollwin;
  GtkWidget *csvview;
  GtkWidget *label;
  GtkWidget *combo;
    
  filename = interface_file_chooser ("Open file",
				     GTK_FILE_CHOOSER_ACTION_OPEN,
				     "csv",
				     HELP_ID_DATABASE);

  if (filename == NULL) {
    filename = "NULL";
  }
  else {
    ui_show_banner(_("Importing the file, please wait"));
    csvinit(&plist);
    cols=0;
    rows=0;
    emptyrows=0;
    f=fopen (filename, "r");
#ifdef _GNU_SOURCE
    while (getline(&line,&llen,f)!=-1 && emptyrows < 10)
#else
    while (fgets(line,LINELEN,f) && emptyrows < 10)
#endif
      {
	/* Update the UI */
	while (gtk_events_pending())
	  gtk_main_iteration();
	if (!g_utf8_validate(line, -1, NULL))
	  convline=g_convert(line, -1, "UTF-8", "ISO-8859-15", NULL, NULL, NULL);
	else
	  convline=g_strndup(line, strlen(line));
	csvparse(convline, &plist, &rows, &cols);
	g_free(convline);
	if (strlen(line)-1<=((csvlist *)plist.tail->data)->nodes)
	  emptyrows++;
	else
	  emptyrows=0;
      }
#ifdef _GNU_SOURCE
    free(line);
    llen=0;
#endif

    fclose(f);

    for (i=emptyrows;i>0;i--) {
      pl = plist.tail;
      csvremoverow(&plist,pl);
    }
    rows=plist.nodes;

    importwindow = gtk_dialog_new_with_buttons(_("Import"),
					       GTK_WINDOW(pui->app->mainwindow),
					       GTK_DIALOG_MODAL,
					       GTK_STOCK_OK, 
					       GTK_RESPONSE_ACCEPT,
					       GTK_STOCK_CANCEL,
					       GTK_RESPONSE_REJECT,
					       NULL);
    gtk_widget_set_size_request(GTK_WIDGET(importwindow),800,400);
  
    g_signal_connect(G_OBJECT(importwindow), "delete_event",
		     G_CALLBACK(delete_importwin), NULL);

    vbox = GTK_WIDGET(GTK_DIALOG(importwindow)->vbox);

    notebook = gtk_notebook_new ();
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
    gtk_box_pack_start (GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
    gtk_widget_show (notebook);

    scrollwin = gtk_scrolled_window_new(NULL, NULL);

    csvview = create_csvpreview_view (&plist, rows, cols);
    gtk_container_add (GTK_CONTAINER (scrollwin), csvview);

    label = gtk_label_new (_("Preview"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), scrollwin, label);

    colscrollwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(colscrollwin), 
				   GTK_POLICY_NEVER, 
				   GTK_POLICY_AUTOMATIC);

    table = gtk_table_new(11, 2, FALSE);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(colscrollwin),table);

    /* Header, left column */
    label = gtk_label_new(_("Column:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
    gtk_widget_show (label);

    /* Header, right column */
    label = gtk_label_new("#");
    gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);
    gtk_widget_show (label);

    /* 1st row, left column */
    label = gtk_label_new(_("Date:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
    gtk_widget_show (label);

    /* 2nd row, left column */
    label = gtk_label_new(_("Km:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
    gtk_widget_show (label);

    /* 3rd row, left column */
    label = gtk_label_new(_("Trip:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);
    gtk_widget_show (label);

    /* 4th row, left column */
    label = gtk_label_new(_("Fill:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);
    gtk_widget_show (label);

    /* 5th row, left column */
    label = gtk_label_new(_("Price:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 5, 6);
    gtk_widget_show (label);

/*     /\* Header, 3rd column *\/ */
/*     label = gtk_label_new(_("Column:")); */
/*     gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 0, 1); */
/*     gtk_widget_show (label); */

/*     /\* Header, 3rd column *\/ */
/*     label = gtk_label_new("#"); */
/*     gtk_table_attach_defaults(GTK_TABLE(table), label, 3, 4, 0, 1); */
/*     gtk_widget_show (label); */

    /* 1st row, 3rd column */
    label = gtk_label_new(_("Service:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 6, 7);
    gtk_widget_show (label);

    /* 2nd row, 3rd column */
    label = gtk_label_new(_("Oil:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 7, 8);
    gtk_widget_show (label);

    /* 3rd row, 3rd column */
    label = gtk_label_new(_("Tires:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 8, 9);
    gtk_widget_show (label);

    /* 4th row, left column */
    label = gtk_label_new(_("Notes:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 9, 10);
    gtk_widget_show (label);

    label = gtk_label_new(_("Date format:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 10, 11);
    gtk_widget_show (label);

    /* Right column */
    for (i=0;i<9;i++) {
      pui->colspin[i] = create_column_spinner(i+1,cols);
      gtk_table_attach_defaults(GTK_TABLE(table), pui->colspin[i], 1, 2, 1+i, 2+i);
      gtk_widget_show (pui->colspin[i]);
    }

    pui->dateformatentry = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->dateformatentry), DATEFORMATENTRYMAX);
    gtk_entry_set_text(GTK_ENTRY(pui->dateformatentry),"%Y-%m-%d");
    gtk_table_attach_defaults(GTK_TABLE(table), pui->dateformatentry, 1, 2, 10, 11);
    gtk_widget_show (pui->dateformatentry);

/*     /\* Rightmost column *\/ */
/*     for (i=0;i<4;i++) { */
/*       pui->colspin[i+5] = create_column_spinner(i+5+1,cols); */
/*       gtk_table_attach_defaults(GTK_TABLE(table), pui->colspin[i+5], 3, 4, 1+i, 2+i); */
/*       gtk_widget_show (pui->colspin[i+5]); */
/*     } */

    gtk_widget_show (table);
    gtk_widget_show(colscrollwin);
/*     gtk_widget_show(frame); */

    label = gtk_label_new (_("Column designation"));
/*     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label); */
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), colscrollwin, label);

  /*----------------------------------------------------------------*/

    carscrollwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(carscrollwin), 
				   GTK_POLICY_NEVER, 
				   GTK_POLICY_AUTOMATIC);

    table = gtk_table_new(5, 2, FALSE);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(carscrollwin),table);

    label = gtk_label_new(_("Driver:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
    gtk_widget_show (label);

    combo = create_driver_combo();
    gtk_table_attach_defaults(GTK_TABLE(table), combo, 1, 2, 0, 1);
    gtk_widget_show (combo);

    label = gtk_label_new(_("Car:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
    gtk_widget_show (label);

    combo = create_car_combo();
    gtk_table_attach_defaults(GTK_TABLE(table), combo, 1, 2, 1, 2);
    gtk_widget_show (combo);

    label = gtk_label_new(_("Unit:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
    gtk_widget_show (label);

    combo = create_unit_combo(UNITCOMBO, pui);
    gtk_table_attach_defaults(GTK_TABLE(table), combo, 1, 2, 2, 3);
    gtk_widget_show (combo);

    label = gtk_label_new(_("Start from line:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);
    gtk_widget_show (label);

    pui->flspinner = create_column_spinner(1,100);
    gtk_table_attach_defaults(GTK_TABLE(table), pui->flspinner, 1, 2, 3, 4);
    gtk_widget_show (pui->flspinner);

    label = gtk_label_new(_("Import file locale:"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);
    gtk_widget_show (label);

    pui->localecombo = gtk_combo_box_new_text();
    gtk_combo_box_append_text(GTK_COMBO_BOX (pui->localecombo),_("current"));
    gtk_combo_box_append_text(GTK_COMBO_BOX (pui->localecombo),"C");
    gtk_combo_box_set_active(GTK_COMBO_BOX (pui->localecombo),0);
    gtk_table_attach_defaults(GTK_TABLE(table), pui->localecombo, 1, 2, 4, 5);
    gtk_widget_show (pui->localecombo);

#if PLAINGTK == 0
    help_dialog_help_enable(GTK_DIALOG(importwindow),
				   HELP_ID_DATABASE,
				   pui->app->osso);
#endif


/*     g_signal_connect_swapped (importwindow, "response", */
/* 			      G_CALLBACK (callback_startimport), */
/* 			      importwindow); */
    g_signal_connect (importwindow, "response",
		      G_CALLBACK (callback_startimport),
		      pui);

    gtk_widget_show(table);
    gtk_widget_show(carscrollwin);
/*     gtk_widget_show(frame); */

    label = gtk_label_new (_("Driver/car/units"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), carscrollwin, label);

    gtk_widget_show(vbox);

    gtk_widget_show_all(GTK_WIDGET(importwindow));
  }

}

void callback_export( GtkAction * action, gpointer data )
{
  double fill;
  double trip;
  char *nickname;
  gchar *filename = NULL;
  filename = interface_file_chooser ("Save file",
				     GTK_FILE_CHOOSER_ACTION_SAVE,
				     "csv",
				     HELP_ID_DATABASE);

  if (filename == NULL) {
    filename = "NULL";
  }
  else {
    if (db_open()) {
      if (SQLITE_OK == sqlite3_bind_int(ppStmtExportCar, 1, currentcar)) {
	FILE *f = fopen (filename, "w");

	if (SQLITE_ROW != sqlite3_step(ppStmtExportCar)) {
	  /*     ui_show_banner("Problem saving result"); */
	  PDEBUG("Problem exporting");
	}
	else {
	  fprintf(f,"\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\"\n",
		  sqlite3_column_text(ppStmtExportCar,0),
		  sqlite3_column_text(ppStmtExportCar,1),
		  sqlite3_column_text(ppStmtExportCar,2),
		  sqlite3_column_text(ppStmtExportCar,3),
		  sqlite3_column_text(ppStmtExportCar,4),
		  "",
		  "",
		  "",
		  "",
		  "",
		  "",
		  "",
		  ""
		  );
	  sqlite3_reset(ppStmtExportCar);

	  fprintf(f,"\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\"\n",
		  _("Date"),
		  ( (curunit.lengthunit==SI)? _("Km") : _("Miles") ),
		  ( (curunit.lengthunit==SI)? _("Trip [km]") : _("Trip [miles]") ),
		  ( (curunit.volumeunit==SI)? _("Fill [litre]") : _("Fill [gal.]")),
		  ( (curunit.consumeunit==SI)? _("[litre/100 km]") : _("[Miles/gallon]") ),
		  _("Price"),
		  ( (curunit.volumeunit==SI)? _("Price/litre") : _("Price/gal.") ),
		  _("Service"),
		  _("Oil"),
		  _("Tires"),
		  ( (curunit.massunit==SI)? 
		        (curunit.lengthunit==SI)? _("CO2 Emissions [g/km]") : _("CO2 Emissions [g/miles]") 
		    : (curunit.lengthunit==SI)? _("CO2 Emissions [lb/100 km]") : _("CO2 Emissions [lb/100 miles]")),
		  _("Notes"),
		  _("Driver")
		  );
	  if (SQLITE_OK == sqlite3_bind_int(ppStmtExport, 1, currentcar)) {
	    while(SQLITE_ROW == sqlite3_step(ppStmtExport)) {
	    
	      db_get_driver(&nickname, sqlite3_column_int(ppStmtExport,11));

	      trip=sqlite3_column_double(ppStmtExport,2);
	      fill=sqlite3_column_double(ppStmtExport,3);

	      fprintf(f,"\"%s\",\"%.2f\",\"%.2f\",\"%.2f\",\"%.2f\",\"%.2f\",\"%.3f\",\"%.2f\",\"%.2f\",\"%.2f\",\"%.0f\",\"%s\",\"%s\"\n",
		      sqlite3_column_text(ppStmtExport,0),
		      SIlength2user(sqlite3_column_double(ppStmtExport,1)),
		      SIlength2user(trip),
		      SIvolume2user(fill),
		      SIconsumption2user(sqlite3_column_double(ppStmtExport,4)),
		      sqlite3_column_double(ppStmtExport,5),
		      SIppl2user(sqlite3_column_double(ppStmtExport,6)),
		      sqlite3_column_double(ppStmtExport,7),
		      sqlite3_column_double(ppStmtExport,8),
		      sqlite3_column_double(ppStmtExport,9),
		      SIemission2user(calc_co2_emission(fill,trip,currentcar)),
		      sqlite3_column_text(ppStmtExport,10),
		      nickname
		      );
	      if (nickname != NULL)
		free(nickname);
	    }
	  }
	}
	fclose (f);
	ui_show_banner(_("Data exported."));
      }
      else
	ui_show_banner(_("Problem exporting data."));


      sqlite3_reset(ppStmtExport);
    }
  }

}
