/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/**
 *
 * \file
 * \brief  Utility functions to fuelpad
 * \author Julius Luukko
 *
 *
 */

#define _XOPEN_SOURCE
#include <time.h>
#include <stdlib.h>
#include <string.h>

#define TIMELENMAX 12
#define DATELENMAX 20


/**
 * \fn void convdate(char *s, size_t max, const char *format, const char *date)
 * \brief Converts date with given format to %Y-%m-%d format
 * \param *s pointer to buffer where converted date will be stored
 * \param max maximum length of the converted date
 * \param *format input date format in the format undestood by strptime
 * \param *date input date
 *
 * This function is used to convert a date in given format to a format
 * that is understood as a date by sqlite3.
 *
 */
void convdate(char *s, size_t max, const char *outformat, 
	      const char *informat, const char *date)
{
  time_t t1;
  struct tm *tm;

  time(&t1);
  tm=localtime(&t1);
  strptime(date, informat, tm);
  strftime(s, max, outformat, tm);
}

void date2sqlite(char *s, size_t max, const char *format, const char *date)
{
  convdate(s, max, "%Y-%m-%d", format, date);
}

void gettimefmt(char *aika, const char *format)
{
  time_t t1;
  struct tm *lctime;

  time(&t1);
  lctime=localtime(&t1);
  strftime(aika, TIMELENMAX, format, lctime);
}

void gettime(char *aika)
{
  gettimefmt(aika, "%F");
}

int getyear(void)
{
  char aika[5];
  int ret;

  gettimefmt(aika, "%Y");
  if (strlen(aika)>0)
    ret=atof(aika);
  else
    ret=2007; /* fallback to something reasonable */
  return ret;
}

void getyearmonthday(char *datestr, int *year, int *month, int *day)
{
  char date[DATELENMAX];

  convdate(date, DATELENMAX, "%Y", "%Y-%m-%d", datestr);
  *year=atoi(date);
  convdate(date, DATELENMAX, "%m", "%Y-%m-%d", datestr);
  *month=atoi(date);
  convdate(date, DATELENMAX, "%d", "%Y-%m-%d", datestr);
  *day=atoi(date);
}
