/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/**
 *
 * \file
 * \brief  GTK+/Hildon based user interface to fuelpad
 * \author Julius Luukko
 *
 *
 */

#include "ui.h"
#include "uireport.h"
#include "uistat.h"
#include "uiexpimp.h"
#include "uialarm.h"
#include "uiicons.h"
#include <assert.h>
#include <math.h>


#define GCONF_KEY_PREFIX "/apps/fuelpad"
#define GCONF_KEY_CURRENT_DRIVER GCONF_KEY_PREFIX"/current_driver"
#define GCONF_KEY_CURRENT_CAR GCONF_KEY_PREFIX"/current_car"
#define GCONF_KEY_CURRENT_UNIT GCONF_KEY_PREFIX"/current_unit"
#define GCONF_KEY_INDIVIDUAL_UNIT GCONF_KEY_PREFIX"/individual_unit"
#define GCONF_KEY_CURRENT_LENGTHUNIT GCONF_KEY_PREFIX"/current_lengthunit"
#define GCONF_KEY_CURRENT_VOLUMEUNIT GCONF_KEY_PREFIX"/current_volumeunit"
#define GCONF_KEY_CURRENT_CONSUMEUNIT GCONF_KEY_PREFIX"/current_consumeunit"
#define GCONF_KEY_DATE_FORMAT GCONF_KEY_PREFIX"/date_format"
#define GCONF_KEY_CURRENCY GCONF_KEY_PREFIX"/currency"
#define GCONF_KEY_DATABASE GCONF_KEY_PREFIX"/database"
#define GCONF_KEY_WARNING_KM GCONF_KEY_PREFIX"/warningkm"
#define GCONF_KEY_WARNING_DAYS GCONF_KEY_PREFIX"/warningdays"
#define GCONF_KEY_REPORT_PAPER GCONF_KEY_PREFIX"/reportpaper"
#define GCONF_KEY_SELECT_COLUMNS GCONF_KEY_PREFIX"/mainviewcolumns"
#define GCONF_KEY_MAINVIEW_FONTSIZE GCONF_KEY_PREFIX"/mainviewfontsize"
#define GCONF_KEY_MAINTOOLBAR_NOTVISIBLE GCONF_KEY_PREFIX"/maintoolbar_notvisible"
#define GCONF_KEY_SECTOOLBAR_NOTVISIBLE GCONF_KEY_PREFIX"/secondarytoolbar_notvisible"

#define RESPONSE_ADDITIONAL_FIELDS 1

/* Which columns are displayed by default (a bitfield) */
#define DISPCOLDEFAULT (1<<COL_NOTES | 0 << COL_TIRES | 0 << COL_OIL | 0 << COL_SERVICE | 1 << COL_PRICEPERLITRE | 1 << COL_PRICE | 1 << COL_CONSUM | 1 << COL_FILL | 1 << COL_TRIP | 1 << COL_KM | 1 << COL_DAY)

typedef struct {
  double km;
  double trip;
  double fill;
  double price;
  double service;
  double oil;
  double tires;
} entry_t;

typedef struct {
  GString *mindate;
  GString *maxdate;
  GString *notes;
  gdouble minval[NUM_COLS];
  gdouble maxval[NUM_COLS];
  gboolean active[NUM_COLS];
} recordfilter_t;

recordfilter_t recfilter;

AppData app;
AppUIData ui;

/* a global to pass the data entered in additional fields dialog */
entry_t new_entry;


unit_t curunit = {
  .mainunit = SI,
  .lengthunit=SI,
  .volumeunit=SI,
  .consumeunit=SI,
  .individualunit = FALSE
};

/**********************************************************************
 *
 * Variables stored in gconf
 *
 */
double warnkm=1000.0;
double warndays=7.0;

/* Paper used in report and statistics */
enum papertype reportpaper=A4PAPER;

/* Which columns are shown */
gint dispcol=DISPCOLDEFAULT;

/* Main table font size */
enum fontscale fontsize=MEDIUM;

/* Toolbar visibility */
gboolean main_toolbar_visible;
gboolean secondary_toolbar_visible;

/**********************************************************************/

/* Font scale factors corresponding to enum fontscale */
double fontscalefactors[]={PANGO_SCALE_MEDIUM,
			   PANGO_SCALE_X_SMALL,
			   PANGO_SCALE_SMALL,
			   PANGO_SCALE_MEDIUM,
			   PANGO_SCALE_LARGE};

/* Possible date format strings (see strptime(3) for format descri ption) */
char *datefmtstr[]={"%Y-%m-%d", "%d.%m.%Y", "%d/%m/%Y", "%d/%m/%y", 
		    "%d-%m-%Y", "%m/%d/%Y", "%m/%d/%y"};

/* Possible values of warning times in days */
float wtimes[]={1.0, 2.0, 7.0, 14.0, 30.0};


gchar *column_headers[] = {gettext_noop("Date"),
			   gettext_noop("Km"),
			   gettext_noop("Trip\n[km]"),
			   gettext_noop("Fill\n[litre]"),
			   gettext_noop("[litre/\n100 km]"),
			   gettext_noop("Price"),
			   gettext_noop("Price/\nlitre"),
			   gettext_noop("Service"),
			   gettext_noop("Oil"),
			   gettext_noop("Tires"),
			   gettext_noop("Notes"),
};

gboolean carchanged=FALSE;

/* Prototypes of some functions */
void newrecord_additionalfields(AppUIData *pui, GtkDialog *parent);
void callback_about( GtkAction * action, gpointer data );
static GtkTreeModel *create_and_fill_model (void);
void update_car_combo(gpointer combo);
void update_driver_combo(gpointer combo);
void update_warn_label(GtkWidget *label);
void update_warn_entry(GtkWidget *entry, double value);
double get_entrywarnkm(AppUIData *pui);
void enable_mainmenu_items(AppUIData *pui);
void create_mainwin_widgets(AppUIData *pui);
void callback_recordactivated (GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn  *col, AppUIData *pui);
void filter_update(recordfilter_t *filt);
void filter_init(recordfilter_t *filt);
void filter_clear(recordfilter_t *filt);
static gboolean visible_func (GtkTreeModel *model, GtkTreeIter *iter, gpointer data);

/*************************************************************
 * Unit conversion functions 
 ************************************************************/

/* lcf = length conversion factor */
/* vcf = volume conversion factor */
double lcf[]={1.0, 1.609344, 1.609344};
double vcf[]={1.0, 3.785411784, 4.54609};

/**
 * \fn double SIlength2user(double length)
 * \brief Converts length from SI units to user units
 * \param length Length in SI units (meters)
 * \return Length in user units
 */
double SIlength2user(double length)
{
  return length/lcf[curunit.lengthunit];
}

/**
 * \fn double user2SIlength(double length)
 * \brief Converts length from user units to SI units
 * \param length Length in user units
 * \return Length in SI units
 */
double user2SIlength(double length)
{
  return length*lcf[curunit.lengthunit];
}

/**
 * \fn double SIvolume2user(double length)
 * \brief Converts volume from SI units to user units
 * \param length Volume in "SI" units (litres)
 * \return Volume in user units
 */
double SIvolume2user(double length)
{
  return length/vcf[curunit.volumeunit];
}

/**
 * \fn double user2SIvolume(double length)
 * \brief Converts volume from user units to SI units
 * \param length Volume in user units
 * \return Volume in SI units
 */
double user2SIvolume(double length)
{
  return length*vcf[curunit.volumeunit];
}

/**
 * \fn double SIconsumption2user(double consum)
 * \brief Converts consumption from SI units to user units
 * \param consum Consumption in litres/100 km
 * \return Consumption in user units
 *
 * If the global consumeunit is other than SI, the consumption
 * is converted from litres/100 km to miles per gallon (MPG).
 *
 */
double SIconsumption2user(double consum)
{
  if (curunit.consumeunit==SI)
    return consum;
  else
    return vcf[curunit.consumeunit]/lcf[curunit.consumeunit]*100.0/consum;
}

/**
 * \fn double user2SIconsumption(double consum)
 * \brief Converts consumption from  user units to SI units
 * \param consum Consumption in user units
 * \return Consumption in litres/100 km
 */
double user2SIconsumption(double consum)
{
  return SIconsumption2user(consum);
}

/**
 * \fn double SIppl2user(double ppl)
 * \brief Converts price per volume from SI units to user units
 * \param ppl Price/litre
 * \return Price/user unit
 */
double SIppl2user(double ppl)
{
  return user2SIvolume(ppl);
}

/**
 * \fn void config_save(void)
 * \brief Save all configuration data to gconf.
 *
 * Saves the current car, current driver and current units to gconf.
 *
 */
static
void config_save(void)
{
  GConfClient *gconf_client = gconf_client_get_default();

  if(!gconf_client)
    {
      g_print(_("Failed to initialize GConf.  Settings were not saved."));
      return;
    }
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_CAR,
		       currentcar, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_DRIVER, 
		       currentdriver, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_UNIT,
		       curunit.mainunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_LENGTHUNIT,
		       curunit.lengthunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_VOLUMEUNIT,
		       curunit.volumeunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_CONSUMEUNIT,
		       curunit.consumeunit, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_INDIVIDUAL_UNIT,
			curunit.individualunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_DATE_FORMAT,
		       curunit.dateformat, NULL);
  gconf_client_set_string(gconf_client,GCONF_KEY_CURRENCY,
			   curunit.currency->str, NULL);
  gconf_client_set_string(gconf_client,GCONF_KEY_DATABASE,
			  db_getfilename(), NULL);
  gconf_client_set_float(gconf_client,GCONF_KEY_WARNING_KM,
		       warnkm, NULL);
  gconf_client_set_float(gconf_client,GCONF_KEY_WARNING_DAYS,
		       warndays, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_REPORT_PAPER,
		       reportpaper, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_SELECT_COLUMNS,
		       dispcol, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_MAINVIEW_FONTSIZE,
		       (gint)fontsize, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_MAINTOOLBAR_NOTVISIBLE,
			!main_toolbar_visible, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_SECTOOLBAR_NOTVISIBLE,
			!secondary_toolbar_visible, NULL);

  gconf_client_clear_cache(gconf_client);
  g_object_unref(gconf_client);
}

/**
 * \fn void config_init(void)
 * \brief Initialize all configuration from gconf.
 */
static
void config_init(void)
{
  GConfClient *gconf_client = gconf_client_get_default();
  struct lconv *locdata;
  gchar *tmpcurrency;
  gchar *tmpdatabase;

  if(!gconf_client)
    {
      g_print(_("Failed to initialize GConf.  Settings not read."));
      return;
    }
  currentdriver = gconf_client_get_int(gconf_client, GCONF_KEY_CURRENT_DRIVER,
				       NULL);
  currentcar = gconf_client_get_int(gconf_client, GCONF_KEY_CURRENT_CAR,
				    NULL);
  curunit.mainunit = gconf_client_get_int(gconf_client, GCONF_KEY_CURRENT_UNIT,
				     NULL);
  curunit.lengthunit = gconf_client_get_int(gconf_client,
					   GCONF_KEY_CURRENT_LENGTHUNIT, NULL);
  curunit.volumeunit = gconf_client_get_int(gconf_client,
					   GCONF_KEY_CURRENT_VOLUMEUNIT, NULL);
  curunit.consumeunit = gconf_client_get_int(gconf_client,
					    GCONF_KEY_CURRENT_CONSUMEUNIT, NULL);
  curunit.individualunit = gconf_client_get_bool(gconf_client,
					 GCONF_KEY_INDIVIDUAL_UNIT, NULL);

  curunit.dateformat = gconf_client_get_int(gconf_client,
					    GCONF_KEY_DATE_FORMAT, NULL);
  tmpcurrency = gconf_client_get_string(gconf_client,
				     GCONF_KEY_CURRENCY, NULL);
  tmpdatabase = gconf_client_get_string(gconf_client,
					GCONF_KEY_DATABASE, NULL);
  warnkm = gconf_client_get_float(gconf_client,
				  GCONF_KEY_WARNING_KM, NULL);
  warndays = gconf_client_get_float(gconf_client,
				    GCONF_KEY_WARNING_DAYS, NULL);
  reportpaper=gconf_client_get_int(gconf_client,
				   GCONF_KEY_REPORT_PAPER, NULL);
  dispcol=gconf_client_get_int(gconf_client,
				   GCONF_KEY_SELECT_COLUMNS, NULL);
  fontsize=gconf_client_get_int(gconf_client,
				   GCONF_KEY_MAINVIEW_FONTSIZE, NULL);
  main_toolbar_visible=!gconf_client_get_bool(gconf_client,
					      GCONF_KEY_MAINTOOLBAR_NOTVISIBLE,
					      NULL);
  secondary_toolbar_visible=!gconf_client_get_bool(gconf_client,
						   GCONF_KEY_SECTOOLBAR_NOTVISIBLE,
						   NULL);

  gconf_client_clear_cache(gconf_client);
  g_object_unref(gconf_client);  

  if (tmpcurrency == NULL) {
    locdata=localeconv();
    curunit.currency=g_string_new(locdata->int_curr_symbol);
  }
  else
    curunit.currency=g_string_new(tmpcurrency);

  db_setfilename(tmpdatabase);

  /* Current car and driver are zero if this is the first time 
   * this program is run
   */
  if (currentcar==0) currentcar=1;
  if (currentdriver==0) currentdriver=1;
  if (curunit.mainunit>IMPERIAL || curunit.mainunit < SI) 
    curunit.mainunit=SI;
  if (curunit.lengthunit>IMPERIAL || curunit.lengthunit < SI) 
    curunit.lengthunit=SI;
  if (curunit.volumeunit>IMPERIAL || curunit.volumeunit < SI) 
    curunit.volumeunit=SI;
  if (curunit.consumeunit>IMPERIAL || curunit.consumeunit < SI) 
    curunit.consumeunit=SI;
  if (curunit.dateformat<0 || curunit.dateformat>=sizeof(datefmtstr)/sizeof(char *))
    curunit.dateformat=0;
  if (warnkm==0.0)
    warnkm=1000;
  if (warndays==0.0)
    warndays=7.0;

  /* Limit reportpaper between 0 and 1 */ 
  if (reportpaper<0) reportpaper=0;
  else if (reportpaper>1) reportpaper=1;

  /* Reset shown columns to default value */
  if (dispcol == 0)
    dispcol=DISPCOLDEFAULT;

  /* Reset fontsize to MEDIUM*/
  if (fontsize<XSMALL || fontsize>LARGE)
    fontsize=MEDIUM;
}


/**
 * \fn void config_init(void)
 * \param argc argc from main()
 * \param argv argv from main()
 * \brief The first function to be called before any other ui functions
 *
 * This function initializes localization, gtk and reads gconf data.
 *
 */
void ui_init(int   argc,
	     char *argv[])
{
  /* Initialize localization. */
  setlocale(LC_ALL, "");
  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);

  gtk_init (&argc, &argv);
  config_init();
  filter_init(&recfilter);
}

/**
 * \fn void ui_main_loop(void)
 * \brief User interface main loop
 *
 * This functions calls gtk_main() and when gtk_main() exists, saves
 * the gconf data.
 *
 */
void ui_main_loop(void)
{
  gtk_main ();
  config_save();
  filter_clear(&recfilter);
}

#if PLAINGTK == 0
gint dbus_callback (const gchar *interface, const gchar *method,
		    GArray *arguments, gpointer data,
		    osso_rpc_t *retval)
{
  PDEBUG ("fuelpad dbus: %s, %s\n", interface, method);

  if (!strcmp (method, "top_application"))
      gtk_window_present (GTK_WINDOW (data));

  retval->type = DBUS_TYPE_INVALID;
  return OSSO_OK;
}

/**
 * \fn void osso_hw_state_callback(osso_hw_state_t *state, gpointer data)
 * \brief Osso hardware state callback
 *
 * This functions is a callback for hardware state changes. It
 * implements only the shutdown_ind state, which calls fuelpad_exit()
 * and exits with exit(1).
 *
 */
static void osso_hw_state_callback(osso_hw_state_t *state, gpointer data)
{
  if(state->shutdown_ind) {
    fuelpad_exit();
    exit(1);
  }
}

#endif

/**
 * \fn gboolean delete_event( GtkWidget *widget, GdkEvent  *event, gpointer   data )
 * \brief Main window delete_event callback
 * \return FALSE
 */
static gboolean delete_event( GtkWidget *widget,
                              GdkEvent  *event,
                              gpointer   data )
{
  gtk_main_quit ();
#if PLAINGTK == 0
  osso_deinitialize(app.osso);
#endif
  return FALSE;
}

/**
 * \fn void main_window_fullscreen(gpointer window)
 * \brief Puts the window to full screen mode and scales if necessary
 *
 * Calls gtk_window_fullscreen() or gtk_window_unfullscreen(). If
 * statistics window is present, scale the canvas.
 *
 */
void main_window_fullscreen(gpointer data)
{
  AppUIData *a;

  a=(AppUIData *)data;
  a->mainfullscreen = !a->mainfullscreen;
  if (a->mainfullscreen) {
    gtk_window_fullscreen(GTK_WINDOW(a->app->mainwindow));
  } else {
    gtk_window_unfullscreen(GTK_WINDOW(a->app->mainwindow));
  }
}

void cb_main_fullscreen( GtkAction * action, gpointer data )
{
  main_window_fullscreen(data);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(((AppUIData *)(data))->main_menu_item_fullscreen),
				 ((AppUIData *)(data))->mainfullscreen);
}

/* Callback for hardware keys */
/* See maemo tutorial */
#if PLAINGTK == 0
gboolean key_press_cb(GtkWidget * widget, GdkEventKey * event,
                      AppUIData *pui)
{
  switch (event->keyval) {

    case GDK_F6:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->main_menu_item_fullscreen),
				     !pui->mainfullscreen);
      return TRUE;
  }

  return FALSE;
}
#endif

void main_toolbar_show_hide(AppUIData *pui)
{
  if (main_toolbar_visible)
    gtk_widget_show(GTK_WIDGET(pui->main_toolbar));
  else
    gtk_widget_hide(GTK_WIDGET(pui->main_toolbar));
}

void secondary_toolbar_show_hide(AppUIData *pui)
{
  if (secondary_toolbar_visible)
    gtk_widget_show(GTK_WIDGET(pui->secondary_toolbar));
  else
    gtk_widget_hide(GTK_WIDGET(pui->secondary_toolbar));
}

void cb_main_toolbar( GtkAction * action, AppUIData *pui)
{
  main_toolbar_visible = !main_toolbar_visible;
  main_toolbar_show_hide(pui);
}

void cb_secondary_toolbar( GtkAction * action, AppUIData *pui)
{
  secondary_toolbar_visible = !secondary_toolbar_visible;
  secondary_toolbar_show_hide(pui);
}

void update_font_scale(AppUIData *pui)
{
  gint i;
  GtkTreeViewColumn   *col;
  GtkWidget           *label;
  PangoAttrList       *attrs;
  PangoAttribute      *attr;
  GList               *rendererlist;

  for (i=COL_DAY; i<COL_ID; i++) {
    col=gtk_tree_view_get_column(GTK_TREE_VIEW(pui->view), i);
    label = gtk_tree_view_column_get_widget(col); 
    attrs=gtk_label_get_attributes(GTK_LABEL(label));
    attr = pango_attr_scale_new(fontscalefactors[fontsize]);
    attr->start_index = 0;
    attr->end_index = -1;
    pango_attr_list_change(attrs, attr);

    rendererlist=gtk_tree_view_column_get_cell_renderers(col);
    g_object_set(rendererlist->data, "scale", fontscalefactors[fontsize], NULL);
    g_list_free(rendererlist);
  }
}

void cb_fontsize_x_small( GtkAction * action, AppUIData *pui)
{
  fontsize=XSMALL;
  update_font_scale(pui);
  update_record_model(pui);
}

void cb_fontsize_small( GtkAction * action, AppUIData *pui)
{
  fontsize=SMALL;
  update_font_scale(pui);
}

void cb_fontsize_medium( GtkAction * action, AppUIData *pui)
{
  fontsize=MEDIUM;
  update_font_scale(pui);
}

void cb_fontsize_large( GtkAction * action, AppUIData *pui)
{
  fontsize=LARGE;
  update_font_scale(pui);
}

void ui_show_animation(GtkWidget **banner, const char *s)
{
#if PLAINGTK == 0
  *banner = hildon_banner_show_animation(GTK_WIDGET(app.mainwindow), NULL, s);
/*   PDEBUG(s); */
#else
  fputs(s,stderr);
#endif
}

void ui_show_banner(const char *s)
{
#if PLAINGTK == 0
  hildon_banner_show_information(GTK_WIDGET(app.mainwindow), NULL, s);
  PDEBUG(s);
#else
  fputs(s,stderr);
#endif
}

void ui_remove_banner(GtkWidget *banner)
{
#if PLAINGTK == 0
  gtk_widget_destroy(GTK_WIDGET(banner));
#endif
}

/**
 * \fn void edit_driver_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback for edit car dialog
 *
 * This callback changes the entered driver data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void edit_car_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      if (SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 1,
					 gtk_entry_get_text(GTK_ENTRY(pui->entrymark)),
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 2,
					    gtk_entry_get_text(GTK_ENTRY(pui->entrymodel)),
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 3,
					   atoi(gtk_entry_get_text(GTK_ENTRY(pui->entryyear))))
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 4,
					    gtk_entry_get_text(GTK_ENTRY(pui->entryregister)),
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 5,
					    gtk_entry_get_text(GTK_ENTRY(pui->entrycarnotes)),
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 6,
					   currentcar)
	  || SQLITE_DONE != sqlite3_step(ppStmtUpdateCar)) {
	PDEBUG("Problem udpating car properties");
      }
      else {
	g_print("Car data updated\n");
      }
      sqlite3_reset(ppStmtUpdateCar);
      update_car_combo(pui->carcombo);
    }
    else {
      PDEBUG("Can't update driver properties - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void edit_driver_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback editing driver data
 *
 * This callback updates the driver data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void edit_driver_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      if (SQLITE_OK != sqlite3_bind_text(ppStmtUpdateDriver, 1, 
					 gtk_entry_get_text(GTK_ENTRY(pui->entryname)), 
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateDriver, 2, 
					    gtk_entry_get_text(GTK_ENTRY(pui->entrynick)), 
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateDriver, 3,
					   currentdriver)
	  || SQLITE_DONE != sqlite3_step(ppStmtUpdateDriver)) {
	PDEBUG("Problem updatig the driver data");
      }
      else { /* New driver added */
/* 	gtk_combo_box_append_text(GTK_COMBO_BOX (pui->drivercombo),gtk_entry_get_text(GTK_ENTRY(pui->entrynick))); */
	g_print("Driver data updated\n");
      }
      sqlite3_reset(ppStmtUpdateDriver);
      update_driver_combo(pui->drivercombo);
    }
    else {
      PDEBUG("Can't update driver data - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void add_driver_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback for add car dialog
 *
 * This callback stores the entered new driver data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * add_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void add_driver_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      if (SQLITE_OK != sqlite3_bind_text(ppStmtAddDriver, 1, 
					 gtk_entry_get_text(GTK_ENTRY(pui->entryname)), 
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddDriver, 2, 
					    gtk_entry_get_text(GTK_ENTRY(pui->entrynick)), 
					    -1, SQLITE_STATIC)
	  || SQLITE_DONE != sqlite3_step(ppStmtAddDriver)) {
	PDEBUG("Problem saving the new driver");
      }
      else { /* New driver added */
	gtk_combo_box_append_text(GTK_COMBO_BOX (pui->drivercombo),gtk_entry_get_text(GTK_ENTRY(pui->entrynick)));
      }
      sqlite3_reset(ppStmtAddDriver);
    }
    else {
      PDEBUG("Can't save new driver - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void create_driver_editwin(AppUIData *pui, GtkWidget *dialog)
 * \brief Creates a driver editing window
 *
 * This function is used for creating an edit window both for adding a
 * new driver and editing an existing one.
 *
 */
void create_driver_editwin(AppUIData *pui, GtkWidget *dialog)
{
  GtkWidget *table;
  GtkWidget *label;

  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, FALSE, FALSE, 5);

  label=gtk_label_new(_("Full name:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryname = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryname), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryname, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryname);

  label=gtk_label_new(_("Short name:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrynick = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrynick), NICKMAXLEN);
  gtk_table_attach(GTK_TABLE(table), pui->entrynick, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrynick);

  gtk_widget_show(table);

}

/**
 * \fn void add_driver_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for adding a new driver to the database
 *
 * This callback opens a dialog which allows to add a new driver to
 * the database.
 *
 */
void edit_driver_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Edit driver properties"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_driver_editwin(pui, dialog);

  if (SQLITE_OK == sqlite3_bind_int(ppStmtOneDriver, 1, currentdriver)) {
    while (SQLITE_ROW == sqlite3_step(ppStmtOneDriver)) {
      gtk_entry_set_text(GTK_ENTRY(pui->entryname),sqlite3_column_text(ppStmtOneDriver,0));
      gtk_entry_set_text(GTK_ENTRY(pui->entrynick),sqlite3_column_text(ppStmtOneDriver,1));
    }
    sqlite3_reset(ppStmtOneDriver);
  }

#if PLAINGTK == 0
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDDRIVER,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (edit_driver_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void add_driver_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for adding a new driver to the database
 *
 * This callback opens a dialog which allows to add a new driver to
 * the database.
 *
 */
void add_driver_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Add a new driver"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_driver_editwin(pui, dialog);

#if PLAINGTK == 0
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDDRIVER,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (add_driver_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void delete_driver_cb( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a driver from the database
 *
 * This callback deletes a driver from the database. All the records
 * and events, which have this driver as a driver are also deleted. The
 * user is asked for a confirmation before deleting.
 *
 */
void delete_driver_cb( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  int numrec;
  int numevents;
  GString *message;
  gint confirm;
  char nickname[NICKMAXLEN];

  if (db_open()) { /* delete */
    if (db_get_numof_drivers() <= 1) {
      message=g_string_new(_("The driver can't be deleted. There is only one driver left and the database must contain at least one driver."));
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_information(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_INFO,
					  GTK_BUTTONS_OK,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      g_string_free(message, TRUE);
    }
    else {
      numrec=db_get_numof_records(currentdriver);
      numevents=db_get_numof_events(currentdriver);
      db_get_current_driver(nickname, NICKMAXLEN);
      message=g_string_new(_("Are you sure you want to delete driver"));
      g_string_append(message, " ");
      g_string_append(message, nickname);
      g_string_append(message, ". ");
      if (numrec+numevents==0)
	g_string_append(message,_("There are no records or events for this driver."));
      else {
	if (numrec>0 && numevents == 0)
	  g_string_append_printf(message,_("There are %d records for this driver, which will also be deleted."), numrec);
	else if (numrec>0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d records and %d events for this driver, which will also be deleted."), numrec, numevents);
	else if (numrec == 0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d events for this driver, which will also be deleted."), numevents);
      }
      
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      
      if(confirm == GTK_RESPONSE_OK) {
	if (!db_delete_driver(currentdriver)) {
	  PDEBUG("Problem deleting the driver");
	}
	else { /* remove the record if succesfully removed from db */
	  update_driver_combo(pui->drivercombo);
	  update_record_model(pui);
	  update_totalkm(pui);
	}
      } /* if (confirm == GTK_RESPONSE_OK) */
      
      g_string_free(message, TRUE);
    }
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

/**
 * \fn void delete_car_cb( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a car from the database
 *
 * This callback deletes a car from the database. All the records and
 * events for this car are also deleted. The user is asked for a
 * confirmation before deleting.
 *
 */
void delete_car_cb( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  int numrec;
  int numevents;
  GString *message;
  gint confirm;
  char car[CARMAXLEN];

  if (db_open()) { /* delete */
    if (db_get_numof_cars() <= 1) {
      message=g_string_new(_("The car can't be deleted. There is only one car left and the database must contain at least one car."));
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_information(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_INFO,
					  GTK_BUTTONS_OK,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      g_string_free(message, TRUE);
    }
    else {
      numrec=db_get_numof_carrecords(currentcar);
      numevents=db_get_numof_carevents(currentcar);
      db_get_current_car(car, CARMAXLEN);
      message=g_string_new(_("Are you sure you want to delete car"));
      g_string_append(message, " ");
      g_string_append(message, car);
      g_string_append(message, ". ");
      if (numrec+numevents==0)
	g_string_append(message,_("There are no records or events for this car."));
      else {
	if (numrec>0 && numevents == 0)
	  g_string_append_printf(message,_("There are %d records for this car, which will also be deleted."), numrec);
	else if (numrec>0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d records and %d events for this car, which will also be deleted."), numrec, numevents);
	else if (numrec == 0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d events for this car, which will also be deleted."), numevents);
      }
      
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      
      if(confirm == GTK_RESPONSE_OK) {
	if (!db_delete_car(currentcar)) {
	  PDEBUG("Problem deleting the car");
	}
	else { /* remove the record if succesfully removed from db */
	  update_car_combo(pui->carcombo);
	  update_record_model(pui);
	  update_totalkm(pui);
	}
      } /* if (confirm == GTK_RESPONSE_OK) */
      
      g_string_free(message, TRUE);
    }
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

/**
 * \fn void add_car_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback for add car dialog
 *
 * This callback stores the entered new car data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function add_car which 
 * is called with the data (i.e. sqlite would not be used here)
 *
 */
void add_car_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  char listitemtext[LISTITEMMAX]="";

  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      if (SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 1, 
					 gtk_entry_get_text(GTK_ENTRY(pui->entrymark)),
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 2,
					    gtk_entry_get_text(GTK_ENTRY(pui->entrymodel)),
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtAddCar, 3,
					   atoi(gtk_entry_get_text(GTK_ENTRY(pui->entryyear))))
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 4,
					    gtk_entry_get_text(GTK_ENTRY(pui->entryregister)),
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 5,
					    gtk_entry_get_text(GTK_ENTRY(pui->entrycarnotes)),
					    -1, SQLITE_STATIC)
	  || SQLITE_DONE != sqlite3_step(ppStmtAddCar)) {
	PDEBUG("Problem saving the new car");
      }
      else { /* New car added */
	g_snprintf(listitemtext, LISTITEMMAX, "%s %s",gtk_entry_get_text(GTK_ENTRY(pui->entrymark)),gtk_entry_get_text(GTK_ENTRY(pui->entryregister)));

	gtk_combo_box_append_text(GTK_COMBO_BOX (pui->carcombo),listitemtext);
      }
      sqlite3_reset(ppStmtAddCar);
    }
    else {
      PDEBUG("Can't save new driver - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void create_car_editwin(AppUIData *pui, GtkWidget *dialog)
 * \brief Creates a car editing window
 *
 * This function is used for creating an edit window both for adding a
 * new car and editing an existing one.
 *
 */
void create_car_editwin(AppUIData *pui, GtkWidget *dialog)
{
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *scrollwin;

  scrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), scrollwin, TRUE, TRUE, 0);

  /* Mark */
  table = gtk_table_new(5, 2, FALSE);
/*   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, FALSE, FALSE, 5); */
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin),table);

  label=gtk_label_new(_("Mark:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrymark = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrymark), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entrymark, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrymark);

  /* Model */
  label=gtk_label_new(_("Model:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrymodel = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrymodel), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entrymodel, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrymodel);

  /* Year */
  label=gtk_label_new(_("Year:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryyear = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryyear), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryyear, 1, 2, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryyear);

  /* Register */
  label=gtk_label_new(_("Register:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryregister = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryregister), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryregister, 1, 2, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryregister);

  /* Notes */
  label=gtk_label_new(_("Notes:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrycarnotes = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrycarnotes), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entrycarnotes, 1, 2, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrycarnotes);

  gtk_widget_show(table);
  gtk_widget_show(scrollwin);
}

/**
 * \fn void add_car_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for adding a new car to the database
 *
 * This callback opens a dialog which allows to add a new car to the
 * database.
 *
 */
void add_car_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Add a new car"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_car_editwin(pui, dialog);

#if PLAINGTK == 0
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDCAR,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (add_car_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void edit_car_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for edit car properties
 *
 * This callback opens a dialog which allows to edit car properties.
 *
 */
void edit_car_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Edit car properties"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_car_editwin(pui, dialog);

  if (SQLITE_OK == sqlite3_bind_int(ppStmtOneCar, 1, currentcar)) {
    while (SQLITE_ROW == sqlite3_step(ppStmtOneCar)) {
      gtk_entry_set_text(GTK_ENTRY(pui->entrymark),sqlite3_column_text(ppStmtOneCar,0));
      gtk_entry_set_text(GTK_ENTRY(pui->entrymodel),sqlite3_column_text(ppStmtOneCar,1));
      gtk_entry_set_text(GTK_ENTRY(pui->entryyear),sqlite3_column_text(ppStmtOneCar,2));
      gtk_entry_set_text(GTK_ENTRY(pui->entryregister),sqlite3_column_text(ppStmtOneCar,3));
      gtk_entry_set_text(GTK_ENTRY(pui->entrycarnotes),sqlite3_column_text(ppStmtOneCar,4));
    }
    sqlite3_reset(ppStmtOneCar);
  }

#if PLAINGTK == 0
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDCAR,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (edit_car_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void dateformat_changed_cb (GtkWidget *widget, gpointer
 *                                 callback_data ) 
 * \brief Callback for date format selection combo box
 *
 * This callback changes the dateformat in the curunit structure
 * according to the selection made in the date format combo box.
 *
 */
void dateformat_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  curunit.dateformat = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
}

/**
 * \fn void driver_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for driver selection combo box
 *
 */
void driver_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;
  unsigned int i;

  index = gtk_combo_box_get_active(GTK_COMBO_BOX(widget))+1;
  i=1;
  while (SQLITE_ROW == sqlite3_step(ppStmtDriver) && i<index) {
    i++;
  }
  currentdriver=sqlite3_column_int(ppStmtDriver,2);
  sqlite3_reset(ppStmtDriver);
}

/**
 * \fn void car_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for car selection combo box
 *
 * This callback creates the contents of the record table after car
 * has been changed.
 *
 */
void car_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;
  unsigned int i;

  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget))+1;
  i=1;
  while (SQLITE_ROW == sqlite3_step(ppStmtCar) && i<index) {
    i++;
  }
  currentcar=sqlite3_column_int(ppStmtCar,2);
  sqlite3_reset(ppStmtCar);

  carchanged=TRUE;
}

/**
 * \fn void warntime_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for alarm warning time combo changed
 *
 */
void warntime_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;

  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
  warndays=wtimes[index];
}

/**
 * \fn void paper_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for report paper combo changed
 *
 */
void paper_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;

  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
  reportpaper=(enum papertype)index;
}

/**
 * \fn void update_labels(void)
 * \brief Updates the headers to correspond to the current unit
 *
 */
void update_column_headers(GtkWidget *view)
{
  GtkTreeViewColumn   *col;
  GtkWidget           *label;

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_DAY);
  label = gtk_tree_view_column_get_widget(col); 
  gtk_label_set_text(GTK_LABEL(label), gettext(column_headers[COL_DAY]));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_KM);
  label = gtk_tree_view_column_get_widget(col); 
  if (curunit.lengthunit==SI)
    gtk_label_set_text(GTK_LABEL(label), _("Km"));
  else
    gtk_label_set_text(GTK_LABEL(label), _("Miles"));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_TRIP);
  label = gtk_tree_view_column_get_widget(col); 
  if (curunit.lengthunit==SI)
    gtk_label_set_text(GTK_LABEL(label), _("Trip\n[km]"));
  else
    gtk_label_set_text(GTK_LABEL(label), _("Trip\n[miles]"));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_FILL);
  label = gtk_tree_view_column_get_widget(col); 
  if (curunit.volumeunit==SI)
    gtk_label_set_text(GTK_LABEL(label), _("Fill\n[litre]"));
  else
    gtk_label_set_text(GTK_LABEL(label), _("Fill\n[gal.]"));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_CONSUM);
  label = gtk_tree_view_column_get_widget(col); 
  if (curunit.consumeunit==SI)
    gtk_label_set_text(GTK_LABEL(label), _("[litre/\n100 km]"));
  else
    gtk_label_set_text(GTK_LABEL(label), _("[Miles/\ngallon]"));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_PRICE);
  label = gtk_tree_view_column_get_widget(col); 
  gtk_label_set_text(GTK_LABEL(label), gettext(column_headers[COL_PRICE]));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_PRICEPERLITRE);
  label = gtk_tree_view_column_get_widget(col); 
  if (curunit.volumeunit==SI)
    gtk_label_set_text(GTK_LABEL(label), _("Price/\nlitre"));
  else
    gtk_label_set_text(GTK_LABEL(label), _("Price/\ngal."));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_SERVICE);
  label = gtk_tree_view_column_get_widget(col); 
  gtk_label_set_text(GTK_LABEL(label), gettext(column_headers[COL_SERVICE]));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_OIL);
  label = gtk_tree_view_column_get_widget(col); 
  gtk_label_set_text(GTK_LABEL(label), gettext(column_headers[COL_OIL]));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_TIRES);
  label = gtk_tree_view_column_get_widget(col); 
  gtk_label_set_text(GTK_LABEL(label), gettext(column_headers[COL_TIRES]));

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), COL_NOTES);
  label = gtk_tree_view_column_get_widget(col); 
  gtk_label_set_text(GTK_LABEL(label), gettext(column_headers[COL_NOTES]));

}

enum unit get_unit(GtkWidget *widget)
{
  enum unit tmpunit;

  tmpunit = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
  if (tmpunit>IMPERIAL || tmpunit < SI) tmpunit=SI;

  return tmpunit;
}

/**
 * \fn void unit_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for unit selection combo boxes
 * \param callback_data selects between different unit combo boxes
 *
 * This callback sets the corresponding unit variable (mainunit,
 * lengthunit, volumeunit, consumeunit) to the value
 * read from the combo box.
 *
 */
void unit_changed_cb (GtkWidget *widget, AppUIData *pui )
{
  enum unit tmpunit;
  double oldwarnkm;

  if (GTK_IS_WIDGET(pui->entrywarnkm))
    oldwarnkm=get_entrywarnkm(pui);
  tmpunit=get_unit(widget);
  curunit.mainunit = tmpunit;
  if (!curunit.individualunit) {
    curunit.lengthunit = tmpunit;
    curunit.volumeunit = tmpunit;
    curunit.consumeunit = tmpunit;
    if (pui->lengthcombo != NULL) {
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->lengthcombo),tmpunit);
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->volumecombo),tmpunit);
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->consumecombo),tmpunit);
    }
  }
  if (GTK_IS_WIDGET(pui->entrywarnkm)) {
    update_warn_label(pui->labelentrywarn);
    update_warn_entry(pui->entrywarnkm, oldwarnkm);
  }
}

void l_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  double oldwarnkm;

  if (GTK_IS_WIDGET(pui->entrywarnkm))
    oldwarnkm=get_entrywarnkm(pui);
  curunit.lengthunit = get_unit(widget);
  if (GTK_IS_WIDGET(pui->entrywarnkm)) {
    update_warn_label(pui->labelentrywarn);
    update_warn_entry(pui->entrywarnkm, oldwarnkm);
  }
}

void v_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  curunit.volumeunit = get_unit(widget);
}

void c_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  curunit.consumeunit = get_unit(widget);
}

/**
 * \fn void unit_check_cb (GtkWidget *widget, AppUIData *pui )
 * \brief Callback for check button for selecting the unit individually
 *
 * This function either enables or disables the unit selection
 * comboboxes and sets the global variable individualunit according to
 * the state of the check button.
 *
 */
void unit_check_cb (GtkWidget *widget, AppUIData *pui )
{
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget))) 
  {
    /* If control reaches here, the toggle button is down */
    curunit.individualunit = TRUE;
    gtk_widget_set_sensitive(pui->unitcombo, FALSE);
    gtk_widget_set_sensitive(pui->lengthcombo, TRUE);
    gtk_widget_set_sensitive(pui->volumecombo, TRUE);
    gtk_widget_set_sensitive(pui->consumecombo, TRUE);
  } else {  
  /* If control reaches here, the toggle button is up */
    curunit.individualunit = FALSE;
    gtk_widget_set_sensitive(pui->unitcombo, TRUE);
    gtk_widget_set_sensitive(pui->lengthcombo, FALSE);
    gtk_widget_set_sensitive(pui->volumecombo, FALSE);
    gtk_widget_set_sensitive(pui->consumecombo, FALSE);
 }
}

/**
 * \fn void update_car_changed(AppUIData *pui)
 * \brief Creates a combo box for selecting a unit
 * \param *pui Pointer to the UI data structure
 *
 * This function can be used after the current driver has been changed
 * to save the driver to gconf and update the labels in the main window.
 *
 */
void update_car_changed(AppUIData *pui)
{
  config_save();
  update_record_model(pui);
}

double get_entrywarnkm(AppUIData *pui)
{
  return user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrywarnkm))));
}

void settings_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  gboolean warn;

  g_string_assign(curunit.currency, gtk_entry_get_text(GTK_ENTRY(pui->entrycurrency)));
  warnkm=get_entrywarnkm(pui);
  update_car_changed(pui);
  gtk_widget_destroy(GTK_WIDGET(dialog));

  /* Update the next event estimates */
  warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
  update_reminder_toolbutton (pui, warn);
}

void car_apply_cb (GtkWidget *widget, AppUIData *pui )
{
  gboolean warn;

  update_car_changed(pui);

  /* Update the next event estimates */
  warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
  update_reminder_toolbutton (pui, warn);
}


/**
 * \fn GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui)
 * \brief Creates a combo box for selecting a unit
 * \param whichunit UNITCOMBO, LENGTHCOMBO, VOLUMECOMBO or CONSUMECOMBO
 * \param pui pointer to user interface data
 * \return Pointer to the created combo box
 *
 * This function is used to create a unit selection combo box. The
 * unit selection combo box containts selections SI, US and
 * Imperial. The default value is read from mainunit,
 * lengthunit, volumeunit or consumeunit
 * depending on the value of the parameter whichunit.
 *
 */
GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui)
{
  GtkWidget *combo;
  enum unit tmpunit=curunit.mainunit;

  combo = gtk_combo_box_new_text();
  
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"SI");
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"US");
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"Imperial");
  
  switch (whichunit) {
  case UNITCOMBO: 
    tmpunit = curunit.mainunit;
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(unit_changed_cb), pui);
    break;
  case LENGTHCOMBO:
    tmpunit = curunit.lengthunit;
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(l_unit_changed_cb), pui);
    break;
  case VOLUMECOMBO:
    tmpunit = curunit.volumeunit;
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(v_unit_changed_cb), pui);
    break;
  case CONSUMECOMBO:
    tmpunit = curunit.consumeunit;
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(c_unit_changed_cb), pui);
    break;
  }
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),tmpunit);

  return combo;
}

void fill_driver_combo(GtkWidget *combo)
{
  gint i;
  gint active;
  gint64 id;

  active=0;
  i=0;
  if (db_open()) {
    while(SQLITE_ROW == sqlite3_step(ppStmtDriver)) {
      gtk_combo_box_append_text(GTK_COMBO_BOX (combo),sqlite3_column_text(ppStmtDriver,0));
      id=sqlite3_column_int64(ppStmtDriver,2);
      if (id==currentdriver)
	active=i;
      i++;
    }
    sqlite3_reset(ppStmtDriver);
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),active);
  }
}

/**
 * \fn void update_driver_combo(gpointer combo)
 * \brief Update the contents of driver combo
 * \param combo Pointer to the already existing combo vox
 *
 * This function can be used to update the driver selection combo box
 * after adding a new driver or deleting one.
 *
 */
void update_driver_combo(gpointer combo)
{
  GtkTreeModel *model;
  gint i, count;

  model = gtk_combo_box_get_model(GTK_COMBO_BOX(combo));
  count = gtk_tree_model_iter_n_children(model, NULL);
  for (i=0; i < count; i++)
    gtk_combo_box_remove_text(GTK_COMBO_BOX(combo), 0);
  fill_driver_combo(GTK_WIDGET(combo));
}


GtkWidget *create_driver_combo(void)
{
  GtkWidget *combo;

  combo = gtk_combo_box_new_text();
  fill_driver_combo(combo);
  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(driver_changed_cb), NULL);

  return combo;
}

/**
 * \fn void fill_car_combo(GtkWidget *combo)
 * \brief Fills the car combo with all the cars in the database
 *
 * This callback fills the car combo with all the cars found in the
 * database.
 *
 * \todo This should be changed so that fuelpad.c has a function which
 * is called to obtain the data (i.e. sqlite would not be used here)
 *
 */
void fill_car_combo(GtkWidget *combo)
{
  gint i;
  gint active;
  gint64 id;
  char listitemtext[LISTITEMMAX]="";

  active=0;
  i=0;
  if (db_open()) {
    while(SQLITE_ROW == sqlite3_step(ppStmtCar)) {
      
      g_snprintf(listitemtext, LISTITEMMAX, "%s %s",sqlite3_column_text(ppStmtCar,0),sqlite3_column_text(ppStmtCar,1));

      gtk_combo_box_append_text(GTK_COMBO_BOX (combo),listitemtext);
      id=sqlite3_column_int64(ppStmtCar,2);
      if (id==currentcar)
	active=i;
      i++;
    }
    sqlite3_reset(ppStmtCar);
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),active);
  }
}

/**
 * \fn void update_car_combo(gpointer combo)
 * \brief Update the contents of car combo
 * \param combo Pointer to the already existing combo vox
 *
 * This function can be used to update the car selection combo box
 * after adding a new car or deleting one.
 *
 */
void update_car_combo(gpointer combo)
{
  GtkTreeModel *model;
  gint i, count;

  model = gtk_combo_box_get_model(GTK_COMBO_BOX(combo));
  count = gtk_tree_model_iter_n_children(model, NULL);
  for (i=0; i < count; i++)
    gtk_combo_box_remove_text(GTK_COMBO_BOX(combo), 0);
  fill_car_combo(GTK_WIDGET(combo));
}

GtkWidget *create_car_combo(void)
{
  GtkWidget *combo;

  combo = gtk_combo_box_new_text();
  fill_car_combo(combo);
  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(car_changed_cb), NULL);
  return combo;
}

#define MAXWARNLEN 10
void update_warn_label(GtkWidget *label)
{
  if (curunit.lengthunit==SI)
    gtk_label_set_text(GTK_LABEL(label), _("km before"));
  else
    gtk_label_set_text(GTK_LABEL(label), _("miles before"));
}

void update_warn_entry(GtkWidget *entry, double value)
{
  char       s[MAXWARNLEN];

  g_snprintf(s,MAXWARNLEN,"%.0f",SIlength2user(value));
  gtk_entry_set_text(GTK_ENTRY(entry),s);
}

void callback_settings( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *notebook;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *unitcheck;
  GtkWidget *unitscrollwin;
  gint       i;
  gchar *warntexts[] = {gettext_noop("1 day"),
			gettext_noop("2 days"),
			gettext_noop("1 week"),
			gettext_noop("2 weeks"),
			gettext_noop("1 month")};

  dialog = gtk_dialog_new_with_buttons(_("Settings"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);
  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
  gtk_box_pack_start (GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
  gtk_widget_show (notebook);

  /** Driver and car notebook page **/

  table = gtk_table_new(4, 5, FALSE);
/*     gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 5); */

  label=gtk_label_new(_("Database:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  label=gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(label), db_getfilename());
  gtk_editable_set_editable(GTK_EDITABLE(label), FALSE);
  gtk_table_attach(GTK_TABLE(table), label, 1, 5, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  /* First row */
  label=gtk_label_new(_("Driver:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->drivercombo = create_driver_combo();
  gtk_table_attach(GTK_TABLE(table), pui->drivercombo, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->drivercombo);

  button = gtk_button_new_with_label(_("Edit"));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, 1, 2,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_driver_cb), &ui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(_("Add new"));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 3, 4, 1, 2,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_driver_cb), &ui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(_("Delete"));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 4, 5, 1, 2,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_driver_cb), &ui);
  gtk_widget_show(button);

  /* Second row */
  label=gtk_label_new(_("Car:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->carcombo = create_car_combo();
  gtk_table_attach(GTK_TABLE(table), pui->carcombo, 1, 2, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);

  gtk_widget_show(pui->carcombo);

  button = gtk_button_new_with_label(_("Edit"));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_car_cb), pui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(_("Add new"));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 3, 4, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_car_cb), pui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(_("Delete"));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 4, 5, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_car_cb), pui);
  gtk_widget_show(button);

  /* Third row */
  label=gtk_label_new(_("Date format:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->datecombo = gtk_combo_box_new_text();
  g_signal_connect (GTK_OBJECT (pui->datecombo), "changed",
		    G_CALLBACK(dateformat_changed_cb), NULL);
  for (i=0;i<sizeof(datefmtstr)/sizeof(char *);i++)
    gtk_combo_box_append_text(GTK_COMBO_BOX (pui->datecombo),datefmtstr[i]);
  gtk_combo_box_set_active(GTK_COMBO_BOX (pui->datecombo),curunit.dateformat);
  gtk_table_attach(GTK_TABLE(table), pui->datecombo, 1, 2, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->datecombo);

  gtk_widget_show(table);

  label = gtk_label_new (_("Driver and car"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);


  /** Reminders notebook page **/

  table = gtk_table_new(2, 3, FALSE);

  /* First row */
  label=gtk_label_new(_("Warn"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrywarnkm=gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrywarnkm), MAXWARNLEN);
  update_warn_entry(pui->entrywarnkm, warnkm);
  gtk_table_attach(GTK_TABLE(table), pui->entrywarnkm, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrywarnkm);

  pui->labelentrywarn=gtk_label_new(NULL);
  update_warn_label(pui->labelentrywarn);
  gtk_table_attach(GTK_TABLE(table), pui->labelentrywarn, 2, 3, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->labelentrywarn);

  /* Second row */
  label=gtk_label_new(_("Warn"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->warntimecombo=gtk_combo_box_new_text();
  for (i=0;i<sizeof(warntexts)/sizeof(char *);i++)
    gtk_combo_box_append_text(GTK_COMBO_BOX (pui->warntimecombo),gettext(warntexts[i]));
  i=0;
  while(i<sizeof(wtimes)/sizeof(float) && warndays!=wtimes[i])
    i++;
  gtk_combo_box_set_active(GTK_COMBO_BOX (pui->warntimecombo),i);
  g_signal_connect (GTK_OBJECT (pui->warntimecombo), "changed",
		    G_CALLBACK(warntime_changed_cb), NULL);
  gtk_table_attach(GTK_TABLE(table), pui->warntimecombo, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->warntimecombo);

  label=gtk_label_new(_("before"));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

    
  gtk_widget_show(table);

  label = gtk_label_new (_("Reminders"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  /** Units notebook page **/

  unitscrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(unitscrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);
  table = gtk_table_new(6, 3, FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(unitscrollwin),table);

  /* First row */
  label=gtk_label_new(_("Currency"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrycurrency = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrycurrency), 30);
  gtk_entry_set_text(GTK_ENTRY(pui->entrycurrency),curunit.currency->str);
  gtk_table_attach(GTK_TABLE(table), pui->entrycurrency, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrycurrency);

  label=gtk_label_new(_("Unitsystem:"));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  label=gtk_label_new(_("Length unit:"));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->lengthcombo = create_unit_combo(LENGTHCOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->lengthcombo, 1, 2, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->lengthcombo, FALSE);
  gtk_widget_show(pui->lengthcombo);

  label=gtk_label_new(_("Volume unit:"));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->volumecombo = create_unit_combo(VOLUMECOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->volumecombo, 1, 2, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->volumecombo, FALSE);
  gtk_widget_show(pui->volumecombo);

  label=gtk_label_new(_("Fuel consumption unit:"));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->consumecombo = create_unit_combo(CONSUMECOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->consumecombo, 1, 2, 5, 6, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->consumecombo, FALSE);
  gtk_widget_show(pui->consumecombo);

  /* Must be the last combo since the toggle callback uses the other combos */
  pui->unitcombo = create_unit_combo(UNITCOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->unitcombo, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->unitcombo);

  /* Must be the last since the toggle callback uses the combos */
  unitcheck = gtk_check_button_new_with_label(_("Select individually"));
  g_signal_connect (GTK_OBJECT (unitcheck), "toggled",
		    G_CALLBACK(unit_check_cb), pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), unitcheck, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), unitcheck, 0, 1, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(unitcheck);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (unitcheck), curunit.individualunit);

  gtk_widget_show(table);
  gtk_widget_show(unitscrollwin);

  label = gtk_label_new (_("Units"));
  /*     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label); */
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), unitscrollwin, label);

  /** Reports notebook page **/

  table = gtk_table_new(1, 2, FALSE);
  /* First row */
  label=gtk_label_new(_("Paper"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->papercombo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX (pui->papercombo),"A4");
  gtk_combo_box_append_text(GTK_COMBO_BOX (pui->papercombo),"Letter");
  gtk_combo_box_set_active(GTK_COMBO_BOX (pui->papercombo),0);
  g_signal_connect (GTK_OBJECT (pui->papercombo), "changed",
		    G_CALLBACK(paper_changed_cb), NULL);
  switch (reportpaper) {
  case A4PAPER:
    gtk_combo_box_set_active(GTK_COMBO_BOX (pui->papercombo),0);
    break;
  case LETTERPAPER:
    gtk_combo_box_set_active(GTK_COMBO_BOX (pui->papercombo),1);
  }
  gtk_table_attach(GTK_TABLE(table), pui->papercombo, 1, 2, 0, 1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->papercombo);

  gtk_widget_show(table);

  label = gtk_label_new (_("Report"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  /* Notebooks done */

#if PLAINGTK == 0
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_SETTINGS,
				 pui->app->osso);
#endif

/*   g_signal_connect_swapped (dialog, "response", */
/* 			    G_CALLBACK (settings_response), */
/* 			    dialog); */
  g_signal_connect (dialog, "response",
		    G_CALLBACK (settings_response),
		    pui);

  gtk_widget_show(dialog);
}

static void
cell_data_func (GtkTreeViewColumn *col,
		      GtkCellRenderer   *renderer,
		      GtkTreeModel      *model,
		      GtkTreeIter       *iter,
		      gpointer           user_data)
{
/*   gfloat  speed; */
  double  speed;
  gchar   buf[20];

  gtk_tree_model_get(model, iter, ((data_func_t *)user_data)->col, &speed, -1);
  g_snprintf(buf, sizeof(buf), ((data_func_t *)user_data)->format, speed);

  g_object_set(renderer, "text", buf, NULL);
}

gint
sort_iter_compare_func (GtkTreeModel *model,
                        GtkTreeIter  *a,
                        GtkTreeIter  *b,
                        gpointer      userdata)
{
  gint sortcol = GPOINTER_TO_INT(userdata); /* not used yet */
  gint ret = 0;

  switch (sortcol)
    {
    case COL_KM:
    case COL_TRIP:
    case COL_FILL:
    case COL_CONSUM:
    case COL_PRICE:
    case COL_PRICEPERLITRE:
    case COL_SERVICE:
    case COL_OIL:
    case COL_TIRES: {
      
      double result1, result2;

      gtk_tree_model_get(model, a, sortcol, &result1, -1);
      gtk_tree_model_get(model, b, sortcol, &result2, -1);
    
      if (result1 != result2)
	{
	  if (!isnan(result1) && !isnan(result2))
	    ret = (result1 > result2) ? 1 : -1;
	  else {
	    if (isnan(result1) && !isnan(result2)) {
		  ret = -1;
	    }
	    else if (!isnan(result1) && isnan(result2)) {
	      ret = 1;
	    }
	  }
	}
      /* else both equal => ret = 0 */
    }
      break;
    case COL_DAY: {
      gchar *date1, *date2;

      gtk_tree_model_get(model, a, sortcol, &date1, -1);
      gtk_tree_model_get(model, b, sortcol, &date2, -1);

      date2sqlite(date1, ENTRYDATEMAX, datefmtstr[curunit.dateformat], date1);
      date2sqlite(date2, ENTRYDATEMAX, datefmtstr[curunit.dateformat], date2);

      ret=strcmp(date1,date2);

      g_free(date1);
      g_free(date2);

    }
      break;
    case COL_NOTES:{

        gchar *name1, *name2;
	
        gtk_tree_model_get(model, a, sortcol, &name1, -1);
        gtk_tree_model_get(model, b, sortcol, &name2, -1);
	
        if (name1 == NULL || name2 == NULL)
	  {
	    if (name1 == NULL && name2 == NULL)
	      break; /* both equal => ret = 0 */
	    
	    ret = (name1 == NULL) ? -1 : 1;
	  }
        else
	  {
	    ret = g_utf8_collate(name1,name2);
	  }
	
        g_free(name1);
        g_free(name2);
      }
      break;
    }

  return ret;
}

void update_record_model(AppUIData *pui)
{
  GtkTreeModel        *model;

  ui_show_banner(_("Updating, please wait"));
  /* Update the UI */
  while (gtk_events_pending())
    gtk_main_iteration();

  carchanged=FALSE;

  model = create_and_fill_model();
  gtk_tree_view_set_model(GTK_TREE_VIEW(pui->view),model);

  update_column_headers(pui->view);
  update_totalkm(pui);
}

GtkTreeModel *get_filter(GtkWidget *view)
{
  GtkTreeModel *sortable;
  GtkTreeModel *filter;

  sortable=gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  if (sortable != NULL)
    filter=gtk_tree_model_sort_get_model(GTK_TREE_MODEL_SORT(sortable));
  else
    filter=NULL;

  return filter;
}

/**
 * \fn void get_store_and_iter(GtkTreeModel *model,
			GtkWidget *view,
			GtkTreeIter *iter,
			GtkListStore **store,
			GtkTreeIter *storeiter)
 * \brief Obtains the list store and iter of a view
 * \param model if model available or NULL
 * \param view the outermost tree view
 * \param iter if iter available or NULL
 * \param store pointer to a list store pointer
 * \param storeiter pointer to a list iter pointer (or NULL)
 *
 * This functions obtains the list store embedded inside a
 * GtkTreeModelSort and a GtkTreeModelFilter. It also converts the
 * iter of the outermost model to the iter of the innermost iter
 * corresponding to the list store.
 *
 */
void get_store_and_iter(GtkTreeModel *model,
			GtkWidget *view,
			GtkTreeIter *iter,
			GtkListStore **store,
			GtkTreeIter *storeiter)
{
  GtkTreeModel *sortable;
  GtkTreeModel *filter;
  GtkTreeIter filteriter;

  sortable=gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  filter=gtk_tree_model_sort_get_model(GTK_TREE_MODEL_SORT(sortable));
  *store = GTK_LIST_STORE(gtk_tree_model_filter_get_model(GTK_TREE_MODEL_FILTER(filter)));
  if (model==NULL || iter==NULL) 
    storeiter=NULL;
  else
  {
    gtk_tree_model_sort_convert_iter_to_child_iter(GTK_TREE_MODEL_SORT(model), &filteriter, iter);
    gtk_tree_model_filter_convert_iter_to_child_iter(GTK_TREE_MODEL_FILTER(filter), storeiter, &filteriter);
  }
}

static GtkTreeModel *
create_and_fill_model (void)
{
  GtkListStore  *store;
  GtkTreeIter    iter;
  GtkTreeModel        *filter;
  GtkTreeSortable     *sortable;
  char date[ENTRYDATEMAX];
  char notes[ENTRYNOTESMAX];
  gchar *convnotes;

  if (db_open()) {
    store = gtk_list_store_new (NUM_COLS, G_TYPE_STRING, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_STRING, G_TYPE_INT64, G_TYPE_BOOLEAN);

    if (SQLITE_OK == sqlite3_bind_int(ppStmtRecords, 1, currentcar)) {
      while(SQLITE_ROW == sqlite3_step(ppStmtRecords)) {

	convdate(date, ENTRYDATEMAX, datefmtstr[curunit.dateformat], "%Y-%m-%d", sqlite3_column_text(ppStmtRecords,0));

	/* Convert notes to utf8 (earlier versions might have imported wrongly encoded data */
	g_snprintf(notes, ENTRYNOTESMAX, "%s", sqlite3_column_text(ppStmtRecords,10));
	if (!g_utf8_validate(notes, -1, NULL))
	  convnotes=g_convert(notes, -1, "UTF-8", "ISO-8859-15", NULL, NULL, NULL);
	else
	  convnotes=g_strndup(notes, strlen(notes));
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    /* 			  COL_DAY, sqlite3_column_text(ppStmtRecords,0), */
			    COL_DAY, date,
			    COL_KM, SIlength2user(sqlite3_column_double(ppStmtRecords,1)),
			    COL_TRIP, SIlength2user(sqlite3_column_double(ppStmtRecords,2)),
			    COL_FILL, SIvolume2user(sqlite3_column_double(ppStmtRecords,3)),
			    COL_CONSUM, SIconsumption2user(sqlite3_column_double(ppStmtRecords,4)),
			    COL_PRICE, sqlite3_column_double(ppStmtRecords,5),
			    COL_PRICEPERLITRE, SIppl2user(sqlite3_column_double(ppStmtRecords,6)),
			    COL_SERVICE, sqlite3_column_double(ppStmtRecords,7),
			    COL_OIL, sqlite3_column_double(ppStmtRecords,8),
			    COL_TIRES, sqlite3_column_double(ppStmtRecords,9),
			    /* 			COL_INSURANCE, sqlite3_column_double(ppStmtRecords,5), */
			    /* 			COL_OTHER, sqlite3_column_double(ppStmtRecords,5), */
/* 			    COL_NOTES, sqlite3_column_text(ppStmtRecords,10), */
			    COL_NOTES, convnotes,
			    COL_ID, sqlite3_column_int64(ppStmtRecords,11),
			    COL_VISIBLE, TRUE,
			    -1);
	g_free(convnotes);
    
      }
    }

    sqlite3_reset(ppStmtRecords);

    filter = gtk_tree_model_filter_new(GTK_TREE_MODEL(store), NULL);
/*     gtk_tree_model_filter_set_visible_column(GTK_TREE_MODEL_FILTER(filter), */
/* 					     COL_VISIBLE); */
    gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filter),
					   visible_func,
                                           (gpointer)&recfilter,
					   NULL);
    sortable = GTK_TREE_SORTABLE(gtk_tree_model_sort_new_with_model(filter));

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_DAY, 
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_DAY), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_TRIP,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_TRIP), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_CONSUM,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_CONSUM), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_PRICEPERLITRE,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_PRICEPERLITRE), NULL);

    /* set initial sort order */
    gtk_tree_sortable_set_sort_column_id(sortable, COL_DAY, GTK_SORT_ASCENDING);

    return GTK_TREE_MODEL (sortable);
  }
  else
    return NULL;
}

#define MAXSTRLEN 20
static void
list_selection_changed_cb (GtkTreeSelection *selection, AppUIData *pui)
{
  GtkTreeIter iter;
  GtkTreeModel *model;
/*         gchar *colstring; */
  gint64 id;
  double value;
  char s[MAXSTRLEN];

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      gtk_tree_model_get (model, &iter, COL_ID, &id, -1);
      
/*       gtk_tree_model_get (model, &iter, COL_SERVICE, &value,-1); */
/*       g_snprintf(s,MAXSTRLEN,"%.2f",value); */
/*       gtk_label_set_text(GTK_LABEL(pui->servicelabel),s); */

/*       gtk_tree_model_get (model, &iter, COL_OIL, &value,-1); */
/*       g_snprintf(s,MAXSTRLEN,"%.2f",value); */
/*       gtk_label_set_text(GTK_LABEL(pui->oillabel),s); */

/*       gtk_tree_model_get (model, &iter, COL_TIRES, &value,-1); */
/*       g_snprintf(s,MAXSTRLEN,"%.2f",value); */
/*       gtk_label_set_text(GTK_LABEL(pui->tireslabel),s); */
    }
}

static gboolean
visible_func (GtkTreeModel *model,
              GtkTreeIter  *iter,
              gpointer      data)
{
  gint i;
  gchar *str;
  gdouble colval;
  gboolean visible = FALSE;
  recordfilter_t *filt;

  filt=(recordfilter_t *)data;

  gtk_tree_model_get (model, iter, COL_DAY, &str, -1);

  if (str) {
    visible = TRUE;
    if (filt->active[COL_DAY]) {
      if (str) { /* If not still empty */
	date2sqlite(str, ENTRYDATEMAX, datefmtstr[curunit.dateformat], str);
	visible = strcmp(filt->mindate->str, str) <= 0 && strcmp(filt->maxdate->str, str) >= 0;
	g_free(str);
      }
    }

    /** \todo does not take COL_NOTES into account yet */
    for (i=COL_KM; i<COL_NOTES;i++) {
      if (filt->active[i]) {
	gtk_tree_model_get (model, iter, i, &colval, -1);
	visible = colval >= filt->minval[i] && colval <= filt->maxval[i];
      }
    }
  }

  return visible;
}

static GtkWidget *
create_view_and_model ()
{
  GtkWidget *view;
/*   static data_func_t dfdata1; */
  static data_func_t dfdata2;
  static data_func_t dfdata3;
  static data_func_t dfdata4;
  static data_func_t dfdata5;
  static data_func_t dfdata6;
  static data_func_t dfdata7;
  static data_func_t dfdata8;
  static data_func_t dfdata9;
  static data_func_t dfdata10;

  GtkTreeViewColumn   *col;
  GtkCellRenderer     *renderer;
  GtkTreeModel        *model;
  GtkTreeSelection    *select;

  GtkWidget           *label;
  PangoAttrList       *attrs;
  PangoAttribute      *attr;

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(fontscalefactors[fontsize]);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  model = create_and_fill_model ();
  view = gtk_tree_view_new_with_model (model);

  /* --- Column #1 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_DAY]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_DAY);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

/*   dfdata1.col = COL_DAY; */
/*   strncpy(dfdata1.format,"%.0f",DFFORMATLEN); */
/*   gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata1, NULL); */

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_DAY);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_DAY), dispcol & (1<<COL_DAY));

  /* --- Column #2 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_KM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_KM);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata2.col = COL_KM;
  strncpy(dfdata2.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata2, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_KM);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_KM), dispcol & (1<<COL_KM));

  /* --- Column #3 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_TRIP]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_TRIP);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata3.col = COL_TRIP;
  strncpy(dfdata3.format,"%.1f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata3, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_TRIP);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_TRIP), dispcol & (1<<COL_TRIP));

  /* --- Column #4 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_FILL]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_FILL);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_FILL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_FILL), dispcol & (1<<COL_FILL));

  dfdata4.col = COL_FILL;
  strncpy(dfdata4.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata4, NULL);

  /* --- Column #5 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_CONSUM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_CONSUM);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_CONSUM);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_CONSUM), dispcol & (1<<COL_CONSUM));

  dfdata5.col = COL_CONSUM;
  strncpy(dfdata5.format,"%.1f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata5, NULL);


  /* --- Column #6 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_PRICE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_PRICE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_PRICE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_PRICE), dispcol & (1<<COL_PRICE));

  dfdata6.col = COL_PRICE;
  strncpy(dfdata6.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata6, NULL);

  /* --- Column #7 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_PRICEPERLITRE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_PRICEPERLITRE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_PRICEPERLITRE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_PRICEPERLITRE), dispcol & (1<<COL_PRICEPERLITRE));

  dfdata7.col = COL_PRICEPERLITRE;
  strncpy(dfdata7.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata7, NULL);

  /* --- Column #8 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_SERVICE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_SERVICE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_SERVICE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_SERVICE), dispcol & (1<<COL_SERVICE));

  dfdata8.col = COL_SERVICE;
  strncpy(dfdata8.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata8, NULL);

  /* --- Column #9 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_OIL]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_OIL);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_OIL);

  dfdata9.col = COL_OIL;
  strncpy(dfdata9.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata9, NULL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_OIL), dispcol & (1<<COL_OIL));

  /* --- Column #10 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_TIRES]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_TIRES);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_TIRES);

  dfdata10.col = COL_TIRES;
  strncpy(dfdata10.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata10, NULL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_TIRES), dispcol & (1<<COL_TIRES));

  /* --- Column #11 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_NOTES]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_NOTES);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_NOTES);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_NOTES), dispcol & (1<<COL_NOTES));

  /* Invisible column to distinguish the results */
  /* --- Column #last --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Id");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_ID);

/*   renderer = gtk_cell_renderer_text_new (); */
/*   gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view), */
/*                                                -1,       */
/*                                                "Id",   */
/*                                                renderer, */
/*                                                "text", COL_ID, */
/*                                                NULL); */
  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_ID), FALSE);

  /* Invisible column to distinguish the results */
  /* --- Column #last+1 --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Visible");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_VISIBLE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_VISIBLE), FALSE);

/*   model = create_and_fill_model (); */

  /* Change the unit dependent headers */
  update_column_headers(view);

/*   gtk_tree_view_set_model (GTK_TREE_VIEW (view), model); */
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);

  /* The tree view has acquired its own reference to the
   *  model, so we can drop ours. That way the model will
   *  be freed automatically when the tree view is destroyed */

  g_object_unref (model);

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (select), "changed",
		    G_CALLBACK (list_selection_changed_cb),
		    &ui);
  g_signal_connect(view, "row-activated",
		   G_CALLBACK (callback_recordactivated),
		   &ui);

  return view;
}

void select_columns_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  gint i;
  gboolean checked;

  for (i=0;i<(NUM_COLS-2);i++) {
    checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pui->columncheck[i]));
    if (checked)
      dispcol |= (1<<i);
    else
      dispcol &= ~(1<<i);
  }
  gtk_widget_destroy(GTK_WIDGET(pui->mainvbox));
  create_mainwin_widgets(pui);

  gtk_widget_destroy(GTK_WIDGET(dialog));
}


#define SELCOLLINES 4.0
void callback_select_columns( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *table;
  gint i;
  gint j;
  gint col;
  gfloat numcols;

  dialog = gtk_dialog_new_with_buttons(_("Select columns"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);

  numcols=ceilf(COL_ID/SELCOLLINES);

  table = gtk_table_new(numcols, COL_ID/numcols, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 5);

  for (col=0,j=0,i=0;i<COL_ID;i++,j++) {
    if (j==(int)SELCOLLINES) {
      j=0;
      col++;
    }
    pui->columncheck[i]=gtk_check_button_new_with_label (gettext(column_headers[i]));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (pui->columncheck[i]), dispcol & (1<<i));
    gtk_table_attach(GTK_TABLE(table), pui->columncheck[i], col, col+1, j, j+1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (pui->columncheck[i]);
  }



  gtk_widget_show(table);

#if PLAINGTK == 0
    hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_MAINWIN,
				   pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (select_columns_response),
		    pui);
  gtk_widget_show(dialog);
  
}

void filter_update(recordfilter_t *filt)
{
  char date[ENTRYDATEMAX];

  db_get_min_date(date,ENTRYDATEMAX);
  g_string_append_printf(filt->mindate,"%s",date);
  db_get_max_date(date,ENTRYDATEMAX);
  g_string_append_printf(filt->maxdate,"%s",date);
}

void filter_init(recordfilter_t *filt)
{
  gint i;

  filt->mindate=g_string_new("");
  filt->maxdate=g_string_new("");
  filt->notes=g_string_new("");

  filter_update(filt);

  /* Initialize all filter nonactive */
  for (i=0; i<=COL_NOTES;i++)
    filt->active[i] = FALSE;

}

void filter_clear(recordfilter_t *filt)
{
  g_string_free(filt->mindate, TRUE);
  g_string_free(filt->maxdate, TRUE);
  g_string_free(filt->notes, TRUE);
}

void filter_records_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  gint i;
  char mindate[ENTRYDATEMAX];
  char maxdate[ENTRYDATEMAX];
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif

  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      for (i=0;i<COL_ID;i++) {
	recfilter.active[i] = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pui->filtercheck[i]));
      }

      /* Get the minimum date */
#if PLAINGTK == 0
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->filterminentry[0]), &year, &month, &day);
      g_snprintf(mindate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite((char *)mindate, ENTRYDATEMAX, datefmtstr[curunit.dateformat], 
		      gtk_entry_get_text(GTK_ENTRY(pui->filterminentry[0])));
#endif
      g_string_assign(recfilter.mindate, mindate);

      /* Get the maximum date */
#if PLAINGTK == 0
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->filtermaxentry[0]), &year, &month, &day);
      g_snprintf(maxdate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite((char *)maxdate, ENTRYDATEMAX, datefmtstr[curunit.dateformat], 
		      gtk_entry_get_text(GTK_ENTRY(pui->filtermaxentry[0])));
#endif
      g_string_assign(recfilter.maxdate, maxdate);
      
      for (i=COL_KM;i<COL_NOTES;i++) {
	recfilter.minval[i]=atof(gtk_entry_get_text(GTK_ENTRY(pui->filterminentry[i])));
	recfilter.maxval[i]=atof(gtk_entry_get_text(GTK_ENTRY(pui->filtermaxentry[i])));
      }

    }
    else {
      PDEBUG("Can't filter records - no database open");
    }
  }

  gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(get_filter(pui->view)));

  gtk_widget_destroy(GTK_WIDGET(dialog));
}

void callback_filter_records( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *scrollwin;
  gint i;
  GString *s;
  char date[ENTRYDATEMAX];
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif

  dialog = gtk_dialog_new_with_buttons(_("Filter records"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);

  scrollwin=gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);

  table = gtk_table_new(3, 5, FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin),table);

  for (i=0;i<COL_ID;i++) {

    pui->filtercheck[i]=gtk_check_button_new_with_label (gettext(column_headers[i]));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (pui->filtercheck[i]), recfilter.active[i]);
    gtk_table_attach(GTK_TABLE(table), pui->filtercheck[i], 0, 1, i, i+1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (pui->filtercheck[i]);

    s=g_string_new("");
    if (i==COL_DAY) {
      db_get_min_date(date,ENTRYDATEMAX);
/*       g_string_append_printf(s,"%s",date); */
#if PLAINGTK == 1
	  convdate(date, ENTRYDATEMAX, datefmtstr[curunit.dateformat], "%Y-%m-%d", date);

	  pui->filterminentry[i]=gtk_entry_new();
	  gtk_entry_set_text(GTK_ENTRY(pui->filterminentry[i]),date);
#else
	  pui->filterminentry[i] = hildon_date_editor_new();
	  getyearmonthday(date, 
			  &year,
			  &month,
			  &day);
	  hildon_date_editor_set_date(HILDON_DATE_EDITOR(pui->filterminentry[i]), year, month, day);
#endif
    }
    else if (i!=COL_NOTES) {
      g_string_append_printf(s,"%.2f",db_get_min(i));
      pui->filterminentry[i]=gtk_entry_new();
      gtk_entry_set_text(GTK_ENTRY(pui->filterminentry[i]),s->str);
    }
    else {
      pui->filterminentry[i]=gtk_entry_new();
    }
    gtk_table_attach(GTK_TABLE(table), pui->filterminentry[i], 1, 2, i, i+1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->filterminentry[i]);
    g_string_free(s, TRUE);

    if (i!=COL_NOTES) {
      s=g_string_new("");
      if (i==COL_DAY) {
	db_get_max_date(date,ENTRYDATEMAX);
/* 	g_string_append_printf(s,"%s",date); */
#if PLAINGTK == 1
	  convdate(date, ENTRYDATEMAX, datefmtstr[curunit.dateformat], "%Y-%m-%d", date);

	  pui->filtermaxentry[i]=gtk_entry_new();
	  gtk_entry_set_text(GTK_ENTRY(pui->filtermaxentry[i]),date);
#else
	  pui->filtermaxentry[i] = hildon_date_editor_new();
	  getyearmonthday(date, 
			  &year,
			  &month,
			  &day);
	  hildon_date_editor_set_date(HILDON_DATE_EDITOR(pui->filtermaxentry[i]), year, month, day);
#endif
      }
      else {
	g_string_append_printf(s,"%.2f",db_get_max(i));
	pui->filtermaxentry[i]=gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(pui->filtermaxentry[i]),s->str);
      }
      gtk_table_attach(GTK_TABLE(table), pui->filtermaxentry[i], 2, 3, i, i+1, 
		       GTK_EXPAND|GTK_FILL,
		       0, 0, 5);
      gtk_widget_show(pui->filtermaxentry[i]);
      g_string_free(s, TRUE);
    }
  }



  gtk_widget_show(table);
  gtk_widget_show(scrollwin);

#if PLAINGTK == 0
    hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_FILTER,
				   pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (filter_records_response),
		    pui);
  gtk_widget_show(dialog);
  
}


void update_totalkm(AppUIData *pui)
{
  GString *str;
  float totalkm;
  float totalfill;

  str=g_string_new(NULL);

  totalkm=db_get_totalkm();
  totalfill=db_get_totalfill();
  g_string_printf(str, "%.0f ", SIlength2user(totalkm));
  if (curunit.lengthunit==SI)
    g_string_append_printf(str, _("km"));
  else
    g_string_append_printf(str, _("miles"));
  gtk_label_set_text(GTK_LABEL(pui->stb_totalkmlabel),str->str);

  if (totalkm != 0.0)
    g_string_printf(str, "%.1f ",SIconsumption2user(totalfill/totalkm*100));
  else
    g_string_printf(str, "%.1f ",0.0);
  if (curunit.consumeunit==SI)
    g_string_append_printf(str, _("l/100 km"));
  else
    g_string_append_printf(str, _("MPG"));
  gtk_label_set_text(GTK_LABEL(pui->stb_avgconsumlabel),str->str);

  g_string_printf(str, "%.1f", db_get_totalcost());
  g_string_append_printf(str, " %s", curunit.currency->str);
  gtk_label_set_text(GTK_LABEL(pui->stb_totcostlabel),str->str);

  g_string_free(str, TRUE);

}

#if PLAINGTK == 0
static HildonFileSystemModel* get_file_system_model(GtkWidget *ref_widget)
{
 return HILDON_FILE_SYSTEM_MODEL(g_object_new(HILDON_TYPE_FILE_SYSTEM_MODEL,
					      "ref_widget", ref_widget, NULL));
}
#endif

gchar * interface_file_chooser (gchar *header, 
				GtkFileChooserAction action, 
				const gchar *ext,
				gchar *help_id)
{
    GtkWidget *dialog;
    gchar *filename = NULL;
#if PLAINGTK == 0
    HildonFileSystemModel *model = NULL;
#endif

#if PLAINGTK == 1
    dialog = gtk_file_chooser_dialog_new (header,
					  GTK_WINDOW(app.mainwindow),
					  action,
					  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					  action == GTK_FILE_CHOOSER_ACTION_SAVE ? GTK_STOCK_SAVE : GTK_STOCK_OPEN, 
					  GTK_RESPONSE_OK,
					  NULL);
#else
    if( (model = get_file_system_model(GTK_WIDGET(app.mainwindow)) ) == NULL)
    {
       g_print("could not get file system model\n\n");
       return NULL;
    }
    dialog = hildon_file_chooser_dialog_new_with_properties
                    (GTK_WINDOW(app.mainwindow),
		             "file_system_model", model,
		             "action", action,
	         	     NULL);
    hildon_file_chooser_dialog_set_extension(HILDON_FILE_CHOOSER_DIALOG(dialog), (gchar *)ext);
#endif
#if PLAINGTK == 0
    hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				   help_id,
				   app.osso);
#endif
    gtk_widget_show_all (GTK_WIDGET (dialog));

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    }
    gtk_widget_destroy(dialog);
    return filename;
}

void internal_close_db(AppUIData *pui)
{
  GtkListStore *store;

  db_close();
  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
  gtk_list_store_clear(store);
}

void callback_close_db( GtkAction * action, AppUIData *pui )
{
  internal_close_db(pui);
  enable_mainmenu_items(pui);
}

void callback_open_db( GtkAction * action, AppUIData *pui )
{
  gchar *filename = NULL;

  filename = interface_file_chooser ("Open database",
				     GTK_FILE_CHOOSER_ACTION_OPEN,
				     "db",
				     HELP_ID_DATABASE);

  if (filename == NULL) {
    filename = "NULL";
  }
  else {
    internal_close_db(pui);
    db_setfilename(filename);
    db_connect();
/*     update_record_model(); */
    update_car_changed(pui);
    enable_mainmenu_items(pui);
  }
}

/**
 * \fn void add_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
 * \brief Response callback for add record dialog
 *
 * This callback adds the data entered in the add record dialog to the
 * database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void add_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkListStore  *store;
  GtkTreeIter    iter;
  GtkAdjustment *vertical;
  GtkTreeIter    storeiter;

  char date[ENTRYDATEMAX];
  gdouble km;
  gdouble trip;
  gdouble fill;
  gdouble price;
  gdouble consum;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif

  if (!db_open()) {
       gtk_widget_destroy(GTK_WIDGET(dialog));
  }
  else {
    switch (arg1){
    case GTK_RESPONSE_ACCEPT: 
      if (carchanged)
	update_car_changed(pui);

#if PLAINGTK == 0
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->entrydate), &year, &month, &day);
      g_snprintf(date, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite((char *)date, ENTRYDATEMAX, datefmtstr[curunit.dateformat], 
		  gtk_entry_get_text(GTK_ENTRY(pui->entrydate)));
#endif

      km = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm))));
      trip = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip))));
      fill = user2SIvolume(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryfill))));
      price = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryprice)));
      consum = fill/trip*100;
      if (db_open()) {
	if (SQLITE_OK != sqlite3_bind_int(ppStmtAddRecord, 1, currentcar) /* carid */
	    || SQLITE_OK != sqlite3_bind_int(ppStmtAddRecord, 2, currentdriver) /* driverid */
	    || SQLITE_OK != sqlite3_bind_text(ppStmtAddRecord, 3, 
					      date,
					      -1, SQLITE_STATIC)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 4,
						km)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 5,
						trip)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 6,
						fill)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 7,
						consum)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 8,
						price)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 9,
						price/fill)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 10,
						new_entry.service)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 11,
						new_entry.oil)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 12,
						new_entry.tires)
	    || SQLITE_OK != sqlite3_bind_text(ppStmtAddRecord, 13, 
					      gtk_entry_get_text(GTK_ENTRY(pui->entrynotes)),
					      -1, SQLITE_STATIC)
	    || SQLITE_DONE != sqlite3_step(ppStmtAddRecord)) {
	  /*     ui_show_banner("Problem saving result"); */
	  PDEBUG("Problem saving result");
	}
	else { /* record succesfully inserted */
	  convdate(date, ENTRYDATEMAX, datefmtstr[curunit.dateformat], "%Y-%m-%d", date);
	  get_store_and_iter(NULL, pui->view, NULL, &store, NULL);
	  gtk_list_store_append (store, &storeiter);
	  gtk_list_store_set (store, &storeiter,
			      COL_DAY, date,
			      COL_KM, SIlength2user(km),
			      COL_TRIP, SIlength2user(trip),
			      COL_FILL, SIvolume2user(fill),
			      COL_CONSUM, SIconsumption2user(consum),
			      COL_PRICE, price,
			      COL_PRICEPERLITRE, price/SIvolume2user(fill),
			      COL_SERVICE, new_entry.service,
			      COL_OIL, new_entry.oil,
			      COL_TIRES, new_entry.tires,
			      /* 			COL_INSURANCE, sqlite3_column_double(ppStmtRecords,5), */
			      /* 			COL_OTHER, sqlite3_column_double(ppStmtRecords,5), */
			      COL_NOTES, gtk_entry_get_text(GTK_ENTRY(pui->entrynotes)),
			      COL_ID, db_get_last_insert_rowid(),
			      COL_VISIBLE, TRUE,
			      -1);
	  
	  update_totalkm(pui);

	  /* Scroll to the end of list */
	  vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->datascrollwin);
	  vertical->value = vertical->upper;
	  gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->datascrollwin,vertical);
	  gtk_adjustment_value_changed(vertical);
	  
	}
	sqlite3_reset(ppStmtAddRecord);
      }
      else {
	PDEBUG("Can't save new record - no database open");
      }
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case RESPONSE_ADDITIONAL_FIELDS:
      newrecord_additionalfields(pui, dialog);
      break;
    }
  }
}

/**
 * \fn void edit_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
 * \brief Response callback for edit record dialog
 *
 * This callback alters the data edited in the edit record dialog to
 * the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void edit_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  char buf[BUFMAX];

  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkListStore  *store;
  GtkTreeIter    iter;
  GtkTreeIter    storeiter;

  gint64 id;
  char date[ENTRYDATEMAX];
  gdouble km;
  gdouble trip;
  gdouble fill;
  gdouble price;
  gdouble consum;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif

  if (!db_open()) {
       gtk_widget_destroy(GTK_WIDGET(dialog));
  }
  else {
    switch (arg1) {
    case GTK_RESPONSE_ACCEPT:
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	gtk_tree_model_get (model, &iter, COL_ID, &id, -1);

#if PLAINGTK == 0
	hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->entrydate), &year, &month, &day);
	g_snprintf(date, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
	date2sqlite((char *)date, ENTRYDATEMAX, datefmtstr[curunit.dateformat], 
		    gtk_entry_get_text(GTK_ENTRY(pui->entrydate)));
#endif

	km = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm))));
	trip = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip))));
	fill = user2SIvolume(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryfill))));
	price = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryprice)));
	consum = fill/trip*100;
	if (db_open()) {
	  if (SQLITE_OK != sqlite3_bind_text(ppStmtUpdateRecord, 1,
					     date,
					     -1, SQLITE_STATIC)
	      || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 2,
						  km)
	      || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 3,
						  trip)
	      || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 4,
						  fill)
	      || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 5,
						  consum)
	      || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 6,
						price)
	      || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 7,
						  price/fill)
	      || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 8,
						  new_entry.service)
	      || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 9,
						  new_entry.oil)
	      || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 10,
						  new_entry.tires)
	      || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateRecord, 11, 
						gtk_entry_get_text(GTK_ENTRY(pui->entrynotes)),
						-1, SQLITE_STATIC)
	      || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateRecord, 12,
					       id)
	      || SQLITE_DONE != sqlite3_step(ppStmtUpdateRecord)) {
	    /*     ui_show_banner("Problem saving result"); */
	    PDEBUG("Problem saving result");
	  }
	  else {
	    convdate(date, ENTRYDATEMAX, datefmtstr[curunit.dateformat], "%Y-%m-%d", date);
	    get_store_and_iter(model, pui->view, &iter, &store, &storeiter);
	    gtk_list_store_set (store, &storeiter,
				COL_DAY, date,
				COL_KM, SIlength2user(km),
				COL_TRIP, SIlength2user(trip),
				COL_FILL, SIvolume2user(fill),
				COL_CONSUM, SIconsumption2user(consum),
				COL_PRICE, price,
				COL_PRICEPERLITRE, price/SIvolume2user(fill),
				COL_SERVICE, new_entry.service,
				COL_OIL, new_entry.oil,
				COL_TIRES, new_entry.tires,
				/* 			COL_INSURANCE, sqlite3_column_double(ppStmtRecords,5), */
				/* 			COL_OTHER, sqlite3_column_double(ppStmtRecords,5), */
				COL_NOTES, gtk_entry_get_text(GTK_ENTRY(pui->entrynotes)),
				COL_ID, id,
				COL_VISIBLE, TRUE,
				-1);
	    
	    update_totalkm(pui);
	
	  }
	  
	  sqlite3_reset(ppStmtUpdateRecord);
	}
	else {
	  PDEBUG("Can't save new driver - no database open");
	}
      }
      else {
	PDEBUG("editrecord: Something wrong with finding the selection\n");
      }
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case RESPONSE_ADDITIONAL_FIELDS:
      newrecord_additionalfields(pui, dialog);

      g_snprintf(buf,BUFMAX,"%.2f",new_entry.service);
      gtk_entry_set_text(GTK_ENTRY(pui->entryservice),buf);

      g_snprintf(buf,BUFMAX,"%.2f",new_entry.oil);
      gtk_entry_set_text(GTK_ENTRY(pui->entryoil),buf);

      g_snprintf(buf,BUFMAX,"%.2f",new_entry.tires);
      gtk_entry_set_text(GTK_ENTRY(pui->entrytires),buf);

      break;
    }
  }
}

gboolean callback_kmadded (GtkWidget     *widget,
                           GdkEventFocus *event,
                           AppUIData     *pui)
{
  float trip;
  float newkm;
  float lastkm;
  char buf[ENTRYTRIPMAX];

  trip=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip)));
  if (trip<0.1) {
    newkm=user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm))));
    lastkm=db_get_last_refill();
    snprintf(buf, ENTRYTRIPMAX, "%.1f",SIlength2user(newkm-lastkm));
    gtk_entry_set_text(GTK_ENTRY(pui->entrytrip),buf);
  }
  return FALSE;
}

/**
 * \fn void create_record_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
 * \brief Creates the contents of a record adding or editing dialog
 * \param *pui Pointer to the UI data structure
 * \param *dialog Pointer to the already created dialog
 * \param add Set this to 1 if an add dialog is needed
 *
 * This function is used for creating an edit window both for adding a
 * new record and editing an existing one.
 *
 */
void create_record_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
{
  GtkWidget *table;
  GtkWidget *label;
#if PLAINGTK == 1  
  char date[ENTRYDATEMAX];
  char aika[TIMELENMAX];
#endif
  GtkListStore *store;
  GtkEntryCompletion *completion;

  if (add)
    table = gtk_table_new(4, 4, FALSE);
  else
    table = gtk_table_new(4, 3, FALSE);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, FALSE, FALSE, 5);

  /* First row, first entry */
  label=gtk_label_new(_("Date:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if PLAINGTK == 1  
  pui->entrydate = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrydate), ENTRYDATEMAX);
  gettimefmt(aika, datefmtstr[curunit.dateformat]);
  gtk_entry_set_text(GTK_ENTRY(pui->entrydate),aika);
#else
  pui->entrydate = hildon_date_editor_new();
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entrydate, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrydate);
  
  /* First row, second entry */
  if (curunit.lengthunit==SI)
    label=gtk_label_new(_("Km:"));
  else
    label=gtk_label_new(_("Miles:"));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(label);

  pui->entrykm = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrykm), ENTRYKMMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrykm),
#if HILDON == 1
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
		 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrykm), "autocap", FALSE, NULL);
#endif
#endif
  if (add) {
    g_signal_connect(pui->entrykm, "focus-out-event", 
		     G_CALLBACK(callback_kmadded), &ui);
  }

  gtk_table_attach(GTK_TABLE(table), pui->entrykm, 3, 4, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrykm);

  /* Second row, first entry */
  label=gtk_label_new(_("Trip:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
	    GTK_EXPAND|GTK_FILL,
		    0, 0, 5);
  gtk_widget_show(label);

  pui->entrytrip = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrytrip), ENTRYTRIPMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrytrip),
#if HILDON == 1
              "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrytrip), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entrytrip, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrytrip);

  /* Second row, second entry */
  label=gtk_label_new(_("Fill:"));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryfill = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryfill), ENTRYFILLMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryfill),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryfill), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryfill, 3, 4, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryfill);

  /* Third row, first entry */
  label=gtk_label_new(_("Price:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryprice = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryprice), ENTRYPRICEMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryprice),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryprice), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryprice, 1, 2, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryprice);

  /* Third row, second entry */
  label=gtk_label_new(_("Notes:"));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrynotes = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrynotes), ENTRYNOTESMAX);
  gtk_table_attach(GTK_TABLE(table), pui->entrynotes, 3, 4, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrynotes);

  completion = gtk_entry_completion_new();
  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
  gtk_entry_completion_set_model(completion, GTK_TREE_MODEL(store));
  gtk_entry_completion_set_text_column(completion, COL_NOTES);
  gtk_entry_set_completion(GTK_ENTRY(pui->entrynotes), completion);

  if (add) {
    /* Forth row, first entry */
    label=gtk_label_new(_("Car:"));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->carcombo = create_car_combo();
    gtk_table_attach(GTK_TABLE(table), pui->carcombo, 1, 2, 3, 4, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->carcombo);

    /* Forth row, second entry */
    label=gtk_label_new(_("Driver:"));
    gtk_table_attach(GTK_TABLE(table), label, 2, 3, 3, 4, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);
    
    pui->drivercombo = create_driver_combo();
    gtk_table_attach(GTK_TABLE(table), pui->drivercombo, 3, 4, 3, 4, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->drivercombo);
  }

  /* Table ready - show it */
  gtk_widget_show (table);

}


void callback_newrecord( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *label;

  if (db_open()) {
    dialog = gtk_dialog_new_with_buttons(_("Add a new record"),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 _("Additional fields..."),
					 RESPONSE_ADDITIONAL_FIELDS,
					 NULL);
    new_entry.service=0.0;
    new_entry.oil=0.0;
    new_entry.tires=0.0;
    create_record_editwin(pui, dialog, 1);

#if PLAINGTK == 0
    hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_ADDRECORD,
				   pui->app->osso);
#endif

/*     g_signal_connect_swapped (dialog, "response", */
/* 			      G_CALLBACK (add_record_response), */
/* 			      dialog); */
    g_signal_connect (dialog, "response",
		      G_CALLBACK (add_record_response),
		      &ui);
  }
  else {
    dialog = gtk_dialog_new_with_buttons(_("Add a new record"),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, 
					 GTK_RESPONSE_REJECT,
					 NULL);

    label=gtk_label_new("Can't access database - adding records not possible");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

    g_signal_connect_swapped (dialog, "response",
			      G_CALLBACK (gtk_widget_destroy),
			      dialog);
  }


  gtk_widget_show(dialog);
}

void additional_fields_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  if (arg1==GTK_RESPONSE_ACCEPT) {
    new_entry.service = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryservice)));
    new_entry.oil=atof(gtk_entry_get_text(GTK_ENTRY(pui->entryoil)));
    new_entry.tires=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytires)));
  }
}


void create_additionalfieldswin(AppUIData *pui, GtkWidget *dialog)
{
  GtkWidget *table;
  GtkWidget *label;

    table = gtk_table_new(1, 3, FALSE);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, FALSE, FALSE, 5);

    /* First row, first entry */
    label=gtk_label_new(_("Service:"));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->entryservice = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->entryservice), ENTRYSERVICEMAX);
#if PLAINGTK == 0
    g_object_set(G_OBJECT(pui->entryservice),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
    g_object_set(G_OBJECT(pui->entryservice), "autocap", FALSE, NULL);
#endif
#endif
    gtk_table_attach(GTK_TABLE(table), pui->entryservice, 1, 2, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->entryservice);

    /* Second row, first entry */
    label=gtk_label_new(_("Oil:"));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->entryoil = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->entryoil), ENTRYOILMAX);
#if PLAINGTK == 0
    g_object_set(G_OBJECT(pui->entryoil),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
    g_object_set(G_OBJECT(pui->entryoil), "autocap", FALSE, NULL);
#endif
#endif
    gtk_table_attach(GTK_TABLE(table), pui->entryoil, 1, 2, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->entryoil);

    /* Third row, first entry */
    label=gtk_label_new(_("Tires:"));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->entrytires = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->entrytires), ENTRYTIRESMAX);
#if PLAINGTK == 0
    g_object_set(G_OBJECT(pui->entrytires),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
    g_object_set(G_OBJECT(pui->entrytires), "autocap", FALSE, NULL);
#endif
#endif
    gtk_table_attach(GTK_TABLE(table), pui->entrytires, 1, 2, 2, 3, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->entrytires);

    /* Table ready - show it */
    gtk_widget_show (table);

}

void newrecord_additionalfields(AppUIData *pui, GtkDialog *parent)
{
  GtkWidget *dialog;
  GtkWidget *label;

  dialog = gtk_dialog_new_with_buttons(_("Additional fields"),
				       GTK_WINDOW(parent),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  if (db_open()) {
    create_additionalfieldswin(pui, dialog);

#if PLAINGTK == 0
    hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_ADDRECORD,
				   pui->app->osso);
#endif

/*     g_signal_connect_swapped (dialog, "response", */
/* 			      G_CALLBACK (additional_fields_response), */
/* 			      dialog); */
    g_signal_connect (dialog, "response",
		      G_CALLBACK (additional_fields_response),
		      &ui);
  }
  else {
    label=gtk_label_new("Can't access database - adding records not possible");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

  }
  /* Destroy the dialog when the user responds to it (e.g. clicks a button) */
  g_signal_connect_swapped (dialog, "response",
			    G_CALLBACK (gtk_widget_destroy),
			    dialog);

  gtk_widget_show(dialog);
}

void callback_prevrecord( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter    iter;
  GtkTreePath   *path;
  GtkAdjustment *vertical;

  if (db_open()) {
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->datascrollwin);
	vertical->value -= vertical->step_increment;
	if (vertical->value<vertical->lower)
	  vertical->value = vertical->lower;
	/* 	  gtk_tree_model_iter_prev(model,&iter); */
	path = gtk_tree_model_get_path(model, &iter);
	if (gtk_tree_path_prev(path)) {
	  gtk_tree_model_get_iter(model, &iter, path);
	  gtk_tree_selection_select_iter(selection, &iter);
	}
	gtk_tree_path_free(path);
	gtk_tree_selection_select_iter(selection, &iter);
	gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->datascrollwin,vertical);
	gtk_adjustment_value_changed(vertical);
      }
  }
}

void callback_nextrecord( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter    iter;
  GtkAdjustment *vertical;

  if (db_open()) {
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->datascrollwin);
	vertical->value += vertical->step_increment;
	if (vertical->value>vertical->upper)
	  vertical->value = vertical->upper;
	gtk_tree_model_iter_next(model,&iter);
	gtk_tree_selection_select_iter(selection, &iter);
	gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->datascrollwin,vertical);
	gtk_adjustment_value_changed(vertical);
      }
  }
}

/**
 * \fn void callback_editrecord( GtkAction * action, AppUIData *pui )
 * \brief Callback for editing a record
 *
 * This callback creates the record editing dialog
 *
 * \todo This should be changed so that fuelpad.c has a function for
 * obtaining the record data (i.e. sqlite would not be used here)
 *
 */
void callback_editrecord( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *label;

  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  gint64 id;
  char date[ENTRYDATEMAX];
  char buf[BUFMAX];
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  char notes[ENTRYNOTESMAX];
  gchar *convnotes;

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      dialog = gtk_dialog_new_with_buttons(_("Edit a record"),
					   GTK_WINDOW(pui->app->mainwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL,
					   GTK_RESPONSE_REJECT,
					   _("Additional fields..."),
					   RESPONSE_ADDITIONAL_FIELDS,
					   NULL);
      create_record_editwin(pui, dialog, 0);

      gtk_tree_model_get (model, &iter, COL_ID, &id, -1);
      /** \todo Data should be read from the store, not from database */
      if (SQLITE_OK == sqlite3_bind_int(ppStmtOneRecord, 1, id)) {
	while (SQLITE_ROW == sqlite3_step(ppStmtOneRecord)) {
#if PLAINGTK == 1
	  convdate(date, ENTRYDATEMAX, datefmtstr[curunit.dateformat], "%Y-%m-%d", sqlite3_column_text(ppStmtOneRecord,0));

	  gtk_entry_set_text(GTK_ENTRY(pui->entrydate),date);
#else
	  getyearmonthday(sqlite3_column_text(ppStmtOneRecord,0), 
			  &year,
			  &month,
			  &day);
	  hildon_date_editor_set_date(HILDON_DATE_EDITOR(pui->entrydate), year, month, day);
#endif

	  g_snprintf(buf,BUFMAX,"%.1f",SIlength2user(sqlite3_column_double(ppStmtOneRecord,1)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrykm),buf);

	  g_snprintf(buf,BUFMAX,"%.1f",SIlength2user(sqlite3_column_double(ppStmtOneRecord,2)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrytrip),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",SIvolume2user(sqlite3_column_double(ppStmtOneRecord,3)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryfill),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,4));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryprice),buf);

	  new_entry.service = sqlite3_column_double(ppStmtOneRecord,5);
	  new_entry.oil = sqlite3_column_double(ppStmtOneRecord,6);
	  new_entry.tires = sqlite3_column_double(ppStmtOneRecord,7);

	/* Convert notes to utf8 (earlier versions might have imported wrongly encoded data */
	  g_snprintf(notes, ENTRYNOTESMAX, "%s", sqlite3_column_text(ppStmtOneRecord,8));
	  if (!g_utf8_validate(notes, -1, NULL))
	    convnotes=g_convert(notes, -1, "UTF-8", "ISO-8859-15", NULL, NULL, NULL);
	  else
	    convnotes=g_strndup(notes, strlen(notes));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrynotes),convnotes);
	  g_free(convnotes);

#if PLAINGTK == 0
	  hildon_help_dialog_help_enable(GTK_DIALOG(dialog),
					 HELP_ID_ADDRECORD,
					 pui->app->osso);
#endif

/* 	  g_signal_connect_swapped (dialog, "response", */
/* 				    G_CALLBACK (edit_record_response), */
/* 				    dialog); */
	  g_signal_connect (dialog, "response",
			    G_CALLBACK (edit_record_response),
			    &ui);
	}
      }
      else {
	PDEBUG("Problem obtaining record\n");
      }
    }
    else {
      dialog = gtk_dialog_new_with_buttons(_("Edit a record"),
					   GTK_WINDOW(pui->app->mainwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   NULL);

      label=gtk_label_new(_("Select a record first"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
      gtk_widget_show(label);

      g_signal_connect_swapped (dialog, "response",
				G_CALLBACK (gtk_widget_destroy),
				dialog);
    }
    sqlite3_reset(ppStmtOneRecord);

  }
  else {
    dialog = gtk_dialog_new_with_buttons(_("Edit a record"),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 NULL);

    label=gtk_label_new("Can't access database - editing records not possible");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

    /* Destroy the dialog when the user responds to it (e.g. clicks a button) */
    g_signal_connect_swapped (dialog, "response",
			      G_CALLBACK (gtk_widget_destroy),
			      dialog);
  }

  gtk_widget_show(dialog);
}

void callback_recordactivated (GtkTreeView        *treeview,
                               GtkTreePath        *path,
                               GtkTreeViewColumn  *col,
                               AppUIData *pui)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  model = gtk_tree_view_get_model(treeview);

  if (gtk_tree_model_get_iter(model, &iter, path)) {
    callback_editrecord(NULL, pui);
  }
}


/**
 * \fn void callback_deleterecord( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a record
 *
 * This callback deletes the currently selected record. The user is
 * asked for a confirmation. If there is also a reminder event
 * associated with this record, it may be also optionally deleted.
 *
 * \todo This should be changed so that fuelpad.c has a function for
 * obtaining the needed data (i.e. sqlite would not be used here)
 *
 */
void callback_deleterecord( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  GtkWidget *check;
  GtkWidget *separator;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeIter storeiter;
  GtkListStore *store;
  GtkTreeModel *model;
  gint64 id;
  gint confirm;
  gint numevents;
  gboolean delevent;

  if (db_open()) { /* delete */

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      gtk_tree_model_get (model, &iter, COL_ID, &id, -1);

#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->mainwindow), 
						_("Are you sure you want to delete the current record"));
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  _("Are you sure you want to delete the current record"));
      gtk_widget_show(confirmdlg);
#endif
      numevents=db_get_numof_events_with_record(id);
      if (numevents>0) {
	separator=gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(confirmdlg)->vbox), separator, TRUE, TRUE, 0);
	gtk_widget_show(separator);

	check=gtk_check_button_new_with_label(_("Delete the corresponding reminder event"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (check), TRUE);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(confirmdlg)->vbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);
      }

      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      if (numevents>0)
	delevent = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));
      else
	delevent = FALSE;
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
    
      if(confirm == GTK_RESPONSE_OK) {
	if (SQLITE_OK != sqlite3_bind_int(ppStmtDeleteRecord, 1, id)
	    ||SQLITE_DONE != sqlite3_step(ppStmtDeleteRecord)) {
	  PDEBUG("Problem deleting the record");
	  sqlite3_reset(ppStmtDeleteRecord);
	}
	else { /* remove the record if succesfully removed from db */
	  sqlite3_reset(ppStmtDeleteRecord);
	  get_store_and_iter(model, pui->view, &iter, &store, &storeiter);

	  gtk_list_store_remove(store, &storeiter);
	  
	  update_totalkm(pui);

	  /* Delete the corresponding alarmevent */
	  if (numevents > 0 && delevent == TRUE) {
	    if (SQLITE_OK != sqlite3_bind_int(ppStmtDeleteEventwithRecordid, 1, id)
		||SQLITE_DONE != sqlite3_step(ppStmtDeleteEventwithRecordid)) {
	      PDEBUG("Problem deleting the event");
	    }
	    else { /* Alarm event deleted, we should update
		    the next event estimates if event window open*/
	      g_print("Alarm event deleted\n");
	    }
	    sqlite3_reset(ppStmtDeleteEventwithRecordid);
	  }

	}
      } /* if (confirm == GTK_RESPONSE_OK) */
    }
    else
      PDEBUG("deleterecord: Something wrong with finding the selection\n");
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

#if PLAINGTK == 0
void help_activated(GtkWidget *win, gchar *help_id)
{
   osso_return_t retval;

   if (!help_id)
      return;

#if HILDON == 1
   retval = hildon_help_show(
      app.osso, help_id, 0);
#else
   retval = ossohelp_show(
      app.osso, help_id, 0);
#endif
}
#endif

#if PLAINGTK == 0
void callback_help( GtkAction * action, gpointer data )
{
#if HILDON == 1
/*   hildon_help_show(app.osso, HELP_ID_INTRO, HILDON_HELP_SHOW_DIALOG); */
  hildon_help_show(app.osso, HELP_ID_INTRO, 0);
#else
  ossohelp_show(app.osso, HELP_ID_INTRO, 0);
#endif
}
#endif

void callback_about( GtkAction * action, gpointer data )
{
  GtkWidget *dialog;

  dialog = gtk_message_dialog_new (GTK_WINDOW(app.mainwindow),
				   GTK_DIALOG_MODAL,
				   GTK_MESSAGE_INFO,
				   GTK_BUTTONS_OK,
				   "FuelPad v. %s",
				   VERSION);

  gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog),
					   "Copyright (C) 2007-2009 Julius Luukko\nLicensed under GNU GPL v. 2\nhttp://fuelpad.garage.maemo.org/");

  /* Destroy the dialog when the user responds to it (e.g. clicks a button) */
  g_signal_connect_swapped (dialog, "response",
			    G_CALLBACK (gtk_widget_destroy),
			    dialog);

  gtk_widget_show(dialog);
}

/**
 * \fn void update_reminder_toolbutton(AppUIData *pui, gboolean warn)
 * \brief Display the reminder icon
 * \param warn If TRUE, display "no active reminders" icon, otherwise
 * "active reminders" icon
 *
 * This callback displays a different icon depending on the parameter warn.
 *
 */
void update_reminder_toolbutton(AppUIData *pui, gboolean warn)
{
  static GdkPixbuf *pixbufred=NULL;
  static GdkPixbuf *pixbufgreen=NULL;
  static GtkWidget *imagered=NULL;
  static GtkWidget *imagegreen=NULL;
  GtkWidget *image;

  if (pixbufred == NULL) {
    pixbufred = gdk_pixbuf_new_from_xpm_data((const char **)&exclamation_red_xpm);
  }
  if (pixbufgreen == NULL) {
    pixbufgreen = gdk_pixbuf_new_from_xpm_data((const char **)&exclamation_green_xpm);
  }
  if (!GTK_IS_WIDGET(imagered)) {
    imagered = gtk_image_new_from_pixbuf(pixbufred);
  }
  if (!GTK_IS_WIDGET(imagegreen)) {
    imagegreen = gtk_image_new_from_pixbuf(pixbufgreen);
  }

  if (warn)
    image = imagered;
  else
    image = imagegreen;

  gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(pui->mtb_reminder),image);
  gtk_widget_show_all(GTK_WIDGET(pui->app->mainwindow));

}

#if PLAINGTK == 1
static void create_mainwin_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui)
#else
  static void create_mainwin_toolbar(HildonWindow * main_window, AppUIData *pui)
#endif
{
  /* Create toolbar */
  pui->main_toolbar = gtk_toolbar_new();

  /* Create toolbar button items */
  pui->mtb_add = gtk_tool_button_new_from_stock(GTK_STOCK_ADD);
  pui->mtb_edit = gtk_tool_button_new_from_stock(GTK_STOCK_EDIT);
  pui->mtb_up = gtk_tool_button_new_from_stock(GTK_STOCK_GO_UP);
  pui->mtb_down = gtk_tool_button_new_from_stock(GTK_STOCK_GO_DOWN);
  pui->mtb_delete = gtk_tool_button_new_from_stock(GTK_STOCK_DELETE);
  pui->mtb_reminder = gtk_tool_button_new(NULL, NULL);
  pui->mtb_close = gtk_tool_button_new_from_stock(GTK_STOCK_QUIT);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_add),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_edit),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_up),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_down),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_delete),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_reminder),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_close),TRUE);
   
  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_add, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_edit, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_up, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_down, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_delete, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_reminder, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_close, -1);

  g_signal_connect(G_OBJECT(pui->mtb_add), "clicked",
		   G_CALLBACK(callback_newrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_edit), "clicked",
		   G_CALLBACK(callback_editrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_up), "clicked",
		   G_CALLBACK(callback_prevrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_down), "clicked",
		   G_CALLBACK(callback_nextrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_delete), "clicked",
		   G_CALLBACK(callback_deleterecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_reminder), "clicked",
		   G_CALLBACK(callback_alarm), pui);
  g_signal_connect(G_OBJECT(pui->mtb_close), "clicked",
		   G_CALLBACK(delete_event), NULL);

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(pui->main_toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(main_window, GTK_TOOLBAR(pui->main_toolbar));
#endif
}

#if PLAINGTK == 1
static void create_secondary_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui)
#else
  static void create_secondary_toolbar(HildonWindow * main_window, AppUIData *pui)
#endif
{
  GString *str;
  float totalkm;
  float totalfill;

  PangoAttrList *attrs;
  PangoAttribute *attr;


  /* Create toolbar */
  pui->secondary_toolbar = gtk_toolbar_new();

  /* Create toolbar items */

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(PANGO_SCALE_SMALL);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  /* Car combo */
  pui->stb_car = gtk_tool_item_new();
  pui->stb_carcombo = create_car_combo();
  gtk_container_add(GTK_CONTAINER(pui->stb_car), GTK_WIDGET(pui->stb_carcombo));

  /* Apply car combo selection */
  pui->stb_carapply = gtk_tool_button_new_from_stock(GTK_STOCK_APPLY);

  str=g_string_new(NULL);

  /* Total distance */
  totalkm=db_get_totalkm();
  g_string_printf(str, "%.0f ", SIlength2user(totalkm));
  if (curunit.lengthunit==SI)
    g_string_append_printf(str, _("km"));
  else
    g_string_append_printf(str, _("miles"));
  pui->stb_totalkm = gtk_tool_item_new();
  pui->stb_totalkmlabel=gtk_label_new(str->str);
  gtk_label_set_selectable(GTK_LABEL(pui->stb_totalkmlabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_totalkmlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_totalkm), GTK_WIDGET(pui->stb_totalkmlabel));

  /* Average consumption */
  totalfill=db_get_totalfill();
  if (totalkm != 0.0)
    g_string_printf(str, "%.1f ",SIconsumption2user(totalfill/totalkm*100));
  else
    g_string_printf(str, "%.1f ",0.0);
  if (curunit.consumeunit==SI)
    g_string_append_printf(str, _("l/100 km"));
  else
    g_string_append_printf(str, _("MPG"));
  pui->stb_avgconsum = gtk_tool_item_new();
  pui->stb_avgconsumlabel=gtk_label_new(str->str);
  gtk_label_set_selectable(GTK_LABEL(pui->stb_avgconsumlabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_avgconsumlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_avgconsum), GTK_WIDGET(pui->stb_avgconsumlabel));

  /* Total cost */
  g_string_printf(str, "%.1f %s", db_get_totalcost(), curunit.currency->str);
  pui->stb_totcost = gtk_tool_item_new();
  pui->stb_totcostlabel=gtk_label_new(str->str);
  gtk_label_set_selectable(GTK_LABEL(pui->stb_totcostlabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_totcostlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_totcost), GTK_WIDGET(pui->stb_totcostlabel));

  g_string_free(str, TRUE);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_car),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_carapply),FALSE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_totalkm),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_avgconsum),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_totcost),TRUE);

  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_car, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_carapply, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_totalkm, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_avgconsum, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_totcost, -1);

  g_signal_connect(G_OBJECT(pui->stb_carapply), "clicked",
		   G_CALLBACK(car_apply_cb), pui);

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(pui->secondary_toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(main_window, GTK_TOOLBAR(pui->secondary_toolbar));
#endif
}

void enable_mainmenu_items(AppUIData *pui)
{
  gboolean dbopen;

  dbopen=db_open();
  
  gtk_widget_set_sensitive(pui->mm_item_new, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_edit, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_delete, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_monthly, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_report, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_close, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_import, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_export, dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_add), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_edit), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_up), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_down), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_delete), dbopen);
}

void create_mainwin_menu(GtkWidget *root, AppUIData *pui)
{ 
  GSList *radio_menu_group = NULL;

  pui->main_menu = gtk_menu_new();
  pui->mm_menu_db = gtk_menu_new();
  pui->mm_menu_record = gtk_menu_new();
  pui->mm_menu_stat = gtk_menu_new();
  pui->mm_menu_view = gtk_menu_new();
  pui->mm_menu_toolbar = gtk_menu_new();
  pui->mm_menu_fontsize = gtk_menu_new();

  /* Create main menu items */
  pui->mm_item_db = gtk_menu_item_new_with_label(_("Database"));
  pui->mm_item_record = gtk_menu_item_new_with_label(_("Record"));
  pui->mm_item_stat = gtk_menu_item_new_with_label(_("Statistics"));
  pui->mm_item_alarm = gtk_menu_item_new_with_label(_("Reminders..."));
  pui->mm_item_view = gtk_menu_item_new_with_label(_("View"));
  pui->mm_item_settings = gtk_menu_item_new_with_label(_("Settings..."));
#if PLAINGTK == 0
  pui->mm_item_help = gtk_menu_item_new_with_label(_("Help"));
#endif
  pui->mm_item_about = gtk_menu_item_new_with_label(_("About"));
  pui->mm_item_exit = gtk_menu_item_new_with_label(_("Exit"));

  /* Create database menu items */
  pui->mm_item_open = gtk_menu_item_new_with_label(_("Open..."));
  pui->mm_item_close = gtk_menu_item_new_with_label(_("Close"));
  pui->mm_item_import = gtk_menu_item_new_with_label(_("Import..."));
  pui->mm_item_export = gtk_menu_item_new_with_label(_("Export..."));

  /* Create record menu items */
  pui->mm_item_new = gtk_menu_item_new_with_label(_("New"));
  pui->mm_item_edit = gtk_menu_item_new_with_label(_("Edit"));
  pui->mm_item_delete = gtk_menu_item_new_with_label(_("Delete"));

  /* Create statistics menu items */
  pui->mm_item_monthly = gtk_menu_item_new_with_label(_("Monthly data"));
  pui->mm_item_report = gtk_menu_item_new_with_label(_("Yearly report"));

  /* Create view menu items */
  pui->main_menu_item_fullscreen = gtk_check_menu_item_new_with_label(_("Full screen"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->main_menu_item_fullscreen),
				 pui->mainfullscreen);
  pui->mm_item_toolbar = gtk_menu_item_new_with_label(_("Toolbars"));
  pui->mm_item_fontsize = gtk_menu_item_new_with_label(_("Table font size"));
  pui->mm_item_columnselect = gtk_menu_item_new_with_label(_("Select columns..."));
  pui->mm_item_filter = gtk_menu_item_new_with_label(_("Filter records..."));

  /* Create toolbar menu items */
  pui->mm_item_toolbar_main = gtk_check_menu_item_new_with_label(_("Buttons"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_toolbar_main),
				 main_toolbar_visible);
  pui->mm_item_toolbar_secondary = gtk_check_menu_item_new_with_label(_("Information"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_toolbar_secondary),
				 secondary_toolbar_visible);

  /* Create fontsize menu items */
  pui->mm_item_fontsize_x_small = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Extra small"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->mm_item_fontsize_x_small));
  if (fontsize == XSMALL)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_x_small), TRUE);
  pui->mm_item_fontsize_small = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Small"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->mm_item_fontsize_small));
  if (fontsize == SMALL)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_small), TRUE);
  pui->mm_item_fontsize_medium = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Medium"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->mm_item_fontsize_medium));
  if (fontsize == MEDIUM)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_medium), TRUE);
  pui->mm_item_fontsize_large = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Large"));
  if (fontsize == LARGE)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_large), TRUE);

  /* Add menu items to right menus */
  /* Main menu */
  gtk_menu_append(pui->main_menu, pui->mm_item_db);
  gtk_menu_append(pui->main_menu, pui->mm_item_record);
  gtk_menu_append(pui->main_menu, pui->mm_item_stat);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(pui->main_menu, pui->mm_item_alarm);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(pui->main_menu, pui->mm_item_view);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(pui->main_menu, pui->mm_item_settings);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
#if PLAINGTK == 0
  gtk_menu_append(pui->main_menu, pui->mm_item_help);
#endif
  gtk_menu_append(pui->main_menu, pui->mm_item_about);
  gtk_menu_append(pui->main_menu, pui->mm_item_exit);

  /* Database menu */
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_open);
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_close);
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_import);
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_export);

  /* Record menu */
  gtk_menu_append(pui->mm_menu_record, pui->mm_item_new);
  gtk_menu_append(pui->mm_menu_record, pui->mm_item_edit);
  gtk_menu_append(pui->mm_menu_record, pui->mm_item_delete);

  /* Statistics menu */
  gtk_menu_append(pui->mm_menu_stat, pui->mm_item_monthly);
  gtk_menu_append(pui->mm_menu_stat, pui->mm_item_report);

  /* View menu */
  gtk_menu_append(pui->mm_menu_view, pui->main_menu_item_fullscreen);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_toolbar);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_fontsize);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_columnselect);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_filter);

  /* Toolbars menu */
  gtk_menu_append(pui->mm_menu_toolbar, pui->mm_item_toolbar_main);
  gtk_menu_append(pui->mm_menu_toolbar, pui->mm_item_toolbar_secondary);

  /* Font size menu */
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_x_small);
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_small);
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_medium);
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_large);

#if PLAINGTK == 1
  pui->mm_menubar = gtk_menu_bar_new();
  gtk_box_pack_start (GTK_BOX (root), pui->mm_menubar, FALSE, FALSE, 2);
  gtk_widget_show (pui->mm_menubar);
  pui->mm_item_file = gtk_menu_item_new_with_label ("fuelpad");
  gtk_widget_show (pui->mm_item_file);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (pui->mm_item_file), pui->main_menu);
  gtk_menu_bar_append (GTK_MENU_BAR (pui->mm_menubar), pui->mm_item_file);
#else
  hildon_window_set_menu(HILDON_WINDOW(pui->app->mainwindow), GTK_MENU(pui->main_menu));
#endif
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_db), pui->mm_menu_db);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_record), pui->mm_menu_record);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_stat), pui->mm_menu_stat);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_view), pui->mm_menu_view);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_toolbar), pui->mm_menu_toolbar);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_fontsize), pui->mm_menu_fontsize);

  /* Attach the callback functions to the activate signal */
  g_signal_connect(G_OBJECT(pui->mm_item_alarm), "activate",
		   G_CALLBACK(callback_alarm), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_settings), "activate",
		   G_CALLBACK(callback_settings), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_monthly), "activate",
		   G_CALLBACK(callback_stat), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_report), "activate",
		   G_CALLBACK(callback_report), pui);
#if PLAINGTK == 0
  g_signal_connect(G_OBJECT(pui->mm_item_help), "activate",
		   G_CALLBACK(callback_help), NULL);
#endif
  g_signal_connect(G_OBJECT(pui->mm_item_about), "activate",
		   G_CALLBACK(callback_about), NULL);
  g_signal_connect(G_OBJECT(pui->mm_item_exit), "activate",
		   G_CALLBACK(delete_event), NULL);

  g_signal_connect(G_OBJECT(pui->mm_item_open), "activate",
		   G_CALLBACK(callback_open_db), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_close), "activate",
		   G_CALLBACK(callback_close_db), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_import), "activate",
		   G_CALLBACK(callback_import), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_export), "activate",
		   G_CALLBACK(callback_export), NULL);

  g_signal_connect(G_OBJECT(pui->mm_item_new), "activate",
		   G_CALLBACK(callback_newrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_edit), "activate",
		   G_CALLBACK(callback_editrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_delete), "activate",
		   G_CALLBACK(callback_deleterecord), pui);

  g_signal_connect(G_OBJECT(pui->main_menu_item_fullscreen), "toggled",
		   G_CALLBACK(cb_main_fullscreen), (gpointer)pui);
  g_signal_connect(G_OBJECT(pui->mm_item_columnselect), "activate",
		   G_CALLBACK(callback_select_columns), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_filter), "activate",
		   G_CALLBACK(callback_filter_records), pui);

  g_signal_connect(G_OBJECT(pui->mm_item_toolbar_main), "toggled",
		   G_CALLBACK(cb_main_toolbar), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_toolbar_secondary), "toggled",
		   G_CALLBACK(cb_secondary_toolbar), pui);

  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_x_small), "toggled",
			   G_CALLBACK(cb_fontsize_x_small), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_small), "toggled",
			   G_CALLBACK(cb_fontsize_small), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_medium), "toggled",
			   G_CALLBACK(cb_fontsize_medium), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_large), "toggled",
			   G_CALLBACK(cb_fontsize_large), pui);

  /* Make all menu widgets visible */
  gtk_widget_show_all(GTK_WIDGET(pui->main_menu));
}

void create_mainwin_widgets(AppUIData *pui)
{
  GtkWidget *vbox;

  pui->mainvbox = gtk_vbox_new (FALSE, 0);
  vbox = pui->mainvbox;
  gtk_container_add (GTK_CONTAINER (app.mainwindow), vbox);

  create_mainwin_menu(vbox, pui);

  pui->datascrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start (GTK_BOX(vbox), pui->datascrollwin, TRUE, TRUE, 0);

  pui->view = create_view_and_model ();
  gtk_container_add (GTK_CONTAINER (pui->datascrollwin), pui->view);

#if PLAINGTK == 1
  create_mainwin_toolbar(app.mainwindow, vbox, pui);
#else
  create_mainwin_toolbar(app.mainwindow, pui);
#endif

#if PLAINGTK == 1
  create_secondary_toolbar(app.mainwindow, vbox, pui);
#else
  create_secondary_toolbar(app.mainwindow, pui);
#endif

  enable_mainmenu_items(pui);

  gtk_widget_show_all(GTK_WIDGET(vbox));

  /*----------------------------------------------------------------*/
}

void ui_create_mainwin(void)
{
#if PLAINGTK == 0
  osso_return_t ret;
#endif

  gboolean warn;

#if PLAINGTK == 0
  app.osso = osso_initialize ("org.luukko.fuelpad", VERSION, TRUE, NULL);
  if (app.osso == NULL) {
    fprintf (stderr, "osso_initialize failed.\n");
    exit (1);
  }

  ret = osso_rpc_set_default_cb_f (app.osso, dbus_callback, NULL);
  if (ret != OSSO_OK)
    {
      fprintf (stderr, "osso_rpc_set_default_cb_f failed: %d.\n", ret);
      exit (1);
    }

    osso_hw_set_event_cb(app.osso, NULL, osso_hw_state_callback, NULL);

#endif

#if PLAINGTK == 0
  /* Create the hildon program and setup the title */
  app.program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("Fuelpad");

  /* Create HildonWindow and set it to HildonProgram */
  app.mainwindow = HILDON_WINDOW(hildon_window_new());

  hildon_program_add_window(app.program, app.mainwindow);

  /* Link app data to ui data */
  ui.app = &app;

  /* Callback for hardware keys */
  g_signal_connect(G_OBJECT(app.mainwindow), "key_press_event",
		   G_CALLBACK(key_press_cb), &ui);

#else
  /* Create the main window */
  app.mainwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (app.mainwindow), "fuelpad");

  /* Link app data to ui data */
  ui.app = &app;
#endif

  g_signal_connect (G_OBJECT (app.mainwindow), "delete_event",
		    G_CALLBACK (delete_event), NULL);

  create_mainwin_widgets(&ui);

  /* View the reminder system toolbar item */
  ui.alarmview = create_alarmview_and_model (&ui);
  warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(ui.alarmview)));
  update_reminder_toolbutton (&ui, warn);

  /* Begin the main application */
  gtk_widget_show_all(GTK_WIDGET(app.mainwindow));

  /* Show or hide toolbars */
  main_toolbar_show_hide(&ui);
  secondary_toolbar_show_hide(&ui);

}
