/*
  This source is part of the FindMine program.
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Control.h"

#include <cstdlib>

#include <Lum/Base/String.h>

#include <Lum/OS/Driver.h>

static Lum::Object::Prefs *prefs=new Sweeper::Prefs();

Sweeper::Sweeper()
: xOff(0),yOff(0),sWidth(0),sHeight(0),bombs(0),oneButtonMode(false),game(NULL),themeIndex(0)
{
  SetPrefs(::prefs);

  sMaxWidth=100;
  sMaxHeight=100;

  status=new Lum::Model::ULong();
  status->Set(waiting);

  font=Lum::OS::display->GetFont(Lum::OS::Display::fontTypeFixed,
                                 Lum::OS::Display::fontScaleFootnote);

  cellFrame=Lum::OS::display->GetFrame(Lum::OS::Display::buttonFrameIndex);
  cellBackground=Lum::OS::display->GetFill(Lum::OS::Display::buttonBackgroundFillIndex);
  leftArrow=Lum::OS::display->GetImage(Lum::OS::Display::arrowLeftImageIndex);
  rightArrow=Lum::OS::display->GetImage(Lum::OS::Display::arrowRightImageIndex);
  upArrow=Lum::OS::display->GetImage(Lum::OS::Display::arrowUpImageIndex);
  downArrow=Lum::OS::display->GetImage(Lum::OS::Display::arrowDownImageIndex);
}

Sweeper::~Sweeper()
{
  delete game;
}

void Sweeper::EvaluateThemeName()
{
  for (size_t i=0; i<themes.size(); i++) {
    if (themes[i].name==themeName) {
      themeIndex=i;

      xOff=0;
      yOff=0;

      if (sWidth>0 && sHeight>0) {
        delete game;
        game=Lum::OS::driver->CreateBitmap(GetAreaWidth()*GetBoxSize(),
                                                     GetAreaHeight()*GetBoxSize());
        RedrawArea();

        Redraw();
      }
      break;
    }
  }
}

void Sweeper::CalcSize()
{
  width=16*24;
  height=16*24;

  minWidth=10*24;
  minHeight=10*24;

  Lum::Object::CalcSize();
}

void Sweeper::DrawCell(size_t row, size_t column)
{
  size_t            x,y,frameA,frameB,frameSize;
  Lum::OS::DrawInfo *draw=game->GetDrawInfo();

  x=row*GetBoxSize();
  y=column*GetBoxSize();

  frameA=std::max(cellFrame->leftBorder,cellFrame->topBorder);
  frameB=std::max(cellFrame->rightBorder,cellFrame->bottomBorder);
  frameSize=frameA+frameB;

  Lum::OS::FillRef fill=Lum::OS::display->GetFill(Lum::OS::Display::backgroundFillIndex);

  fill->Draw(draw,x,y,GetBoxSize(),GetBoxSize(),x,y,GetBoxSize(),GetBoxSize());

  // Check if we have a filled gaming area
  if (area[row][column].active) {
    // if the given cell is already unrevealed...
    if (area[row][column].open) {
      // if there is a bomb...
      if (area[row][column].bomb) {
        if (GetImage(bombImage)!=NULL) {
          GetImage(bombImage)->Draw(draw,x,y);
        }
        else {
          draw->PushForeground(Lum::OS::display->textColor);
          draw->FillArc(x+frameA,
                        y+frameA,
                        GetBoxSize()-frameSize,
                        GetBoxSize()-frameSize,
                        0,360*64);
          draw->PopForeground();
        }
      }
      // no bomb...
      else {
        // paint the number of hidden neighbours...
        if (area[row][column].count!=0) {
          std::wstring string=Lum::Base::NumberToWString((unsigned long)area[row][column].count);
          draw->PushFont(font,Lum::OS::Font::normal);
          draw->PushForeground(Lum::OS::display->textColor);
          draw->DrawString(x+(GetBoxSize()-font->StringWidth(string,Lum::OS::Font::normal)) / 2,
                           y+(GetBoxSize()-font->height) / 2+font->ascent,
                           string);
          draw->PopForeground();
          draw->PopFont();
        }
      }
    }
    // Cell is still hidden
    else {
      // if we are finished playing, draw hidden bomb cells as unhidden and having a bomb...
      if ((status->Get()!=playing) && (area[row][column].bomb)) {
        if (GetImage(bombImage)!=NULL) {
          GetImage(bombImage)->Draw(draw,x,y);
        }
        else {
          draw->PushForeground(Lum::OS::display->textColor);
          draw->FillArc(x+frameA,
                        y+frameA,
                        GetBoxSize()-frameSize,
                        GetBoxSize()-frameSize,
                        0,360*64);
          draw->PopForeground();
        }
      }
      // if the cell is marked as having a bomb, do special drawing...
      else if (area[row][column].marked) {
        if (GetImage(markedImage)!=NULL) {
          GetImage(markedImage)->Draw(draw,x,y);
        }
        else {
          cellBackground->Draw(draw,
                               x+cellFrame->leftBorder,
                               y+cellFrame->topBorder,
                               GetBoxSize()-cellFrame->minWidth,
                               GetBoxSize()-cellFrame->minHeight,
                               x+cellFrame->leftBorder,
                               y+cellFrame->topBorder,
                               GetBoxSize()-cellFrame->minWidth,
                               GetBoxSize()-cellFrame->minHeight);
          draw->PushForeground(Lum::OS::display->fillColor);
          draw->FillRectangle(x+frameA,
                              y+frameA,
                              GetBoxSize()-frameSize,
                              GetBoxSize()-frameSize);
          draw->PopForeground();
          cellFrame->Draw(draw,x,y,GetBoxSize(),GetBoxSize());
        }
      }
      // draw the cell as normal hidden cell...
      else {
        if (GetImage(hiddenImage)!=NULL) {
          GetImage(hiddenImage)->Draw(draw,x,y);
        }
        else {
          cellBackground->Draw(draw,
                               x+cellFrame->leftBorder,
                               y+cellFrame->topBorder,
                               GetBoxSize()-cellFrame->minWidth,
                               GetBoxSize()-cellFrame->minHeight,
                               x+cellFrame->leftBorder,
                               y+cellFrame->topBorder,
                               GetBoxSize()-cellFrame->minWidth,
                               GetBoxSize()-cellFrame->minHeight);
          cellFrame->Draw(draw,x,y,GetBoxSize(),GetBoxSize());
        }
      }
    }
  }
}

bool Sweeper::GetNeighbour(size_t n,
                           size_t column, size_t row,
                           size_t& x, size_t& y)
{
  /*
    012
    345
    678
   */

  switch (n) {
  case 0:
    x=column-1;
    y=row-1;
    break;
  case 1:
    x=column;
    y=row-1;
    break;
  case 2:
    x=column+1;
    y=row-1;
    break;
  case 3:
    x=column-1;
    y=row;
    break;
  case 4:
    x=column;
    y=row;
    break;
  case 5:
    x=column+1;
    y=row;
    break;
  case 6:
    x=column-1;
    y=row+1;
    break;
  case 7:
    x=column;
    y=row+1;
    break;
  case 8:
    x=column+1;
    y=row+1;
    break;
  }

  return x<sWidth && y<sHeight;
}

void Sweeper::RecalcArea()
{
  bool changed,notFound;

  size_t x,y;

  /* automatically open cells which have 0 neighbours */

  do {
    changed=false;
    notFound=false;

    /* calculate neighbourhood */
    for (size_t column=0; column<sWidth; column++) {
      for (size_t row=0; row<sHeight; row++) {
        size_t oldBombs,bombs;

        oldBombs=area[column][row].count;
        bombs=0;

        for (size_t n=0; n<9; n++) {
          if (GetNeighbour(n,column,row,x,y)) {
            if (area[x][y].bomb) {
              bombs++;
            }
          }
        }

        if (bombs!=oldBombs) {
          area[column][row].count=bombs;

          if (area[column][row].open) {
            DrawCell(column,row);
          }
        }

        // check if you have opened a bomb
        if (area[column][row].open && area[column][row].bomb && (status->Get()!=lost)) {
          timer.Pause();
          status->Set(lost);
          RedrawArea(); // We redraw the complete area to show all bombs
          Redraw();
        }

        // still undetected bombs
        if (!area[column][row].bomb && !area[column][row].open) {
          notFound=true;
        }

        // still unmarked bombs
        if (area[column][row].bomb && !area[column][row].marked) {
          notFound=true;
        }
      }
    }

    /* open automatically when neighbour has count 0 and is opened */
    for (size_t column=0; column<sWidth; column++) {
      for (size_t row=0; row<sHeight; row++) {

        if (area[column][row].open && (area[column][row].count==0)) {
          for (size_t n=0; n<9; n++) {
            if (GetNeighbour(n,column,row,x,y)) {
              if (!area[x][y].open) {
                area[x][y].open=true;
                DrawCell(x,y);
                changed=true;
              }
            }
          }
        }
      }
    }

    if ((status->Get()==playing) && !notFound) {
      timer.Pause();
      status->Set(won);
    }
  } while (changed);
}

size_t Sweeper::GetBoxSize() const
{
  return themes[themeIndex].boxSize;
}

Lum::Images::Image* Sweeper::GetImage(Image image) const
{
  return themes[themeIndex].images[image];
}

bool Sweeper::IsAreaCompletelyVisible() const
{
  return GetAreaWidth()<=width/GetBoxSize() && GetAreaHeight()<=height/GetBoxSize();
}

bool Sweeper::CanScrollLeft() const
{
  return xOff>0;
}

bool Sweeper::CanScrollUp() const
{
  return yOff>0;
}

bool Sweeper::CanScrollRight() const
{
  return xOff+(width/GetBoxSize()-2)<GetAreaWidth();
}

bool Sweeper::CanScrollDown() const
{
  return yOff+(height/GetBoxSize()-2)<GetAreaHeight();
}

void Sweeper::ScrollLeft()
{
  assert(xOff>0);

  xOff--;
  Redraw();
}

void Sweeper::ScrollUp()
{
  assert(yOff>0);

  yOff--;
  Redraw();
}

void Sweeper::ScrollRight()
{
  //assert();

  xOff++;
  Redraw();
}

void Sweeper::ScrollDown()
{
  //assert();

  yOff++;
  Redraw();
}

void Sweeper::RedrawArea()
{
  for (size_t x=0; x<sWidth; x++) {
    for (size_t y=0; y<sHeight; y++) {
      DrawCell(x,y);
    }
  }
}

void Sweeper::OpenField(size_t x, size_t y)
{
  if (area[x][y].active) {
    if (!area[x][y].open) {
      if (status->Get()==playing && (timer.IsStoped() || timer.IsPausing())) {
        timer.Start();
      }

      area[x][y].open=true;
      DrawCell(x,y);

      RecalcArea();

      Redraw();
    }
  }
}

void Sweeper::ToggleMark(size_t x, size_t y)
{
  if (area[x][y].active) {
    if (!area[x][y].open) {
      if (status->Get()==playing && (timer.IsStoped() || timer.IsPausing())) {
        timer.Start();
      }

      area[x][y].marked=!area[x][y].marked;
      DrawCell(x,y);

      RecalcArea();

      Redraw();
    }
  }
}

void Sweeper::Draw(int x, int y, size_t w, size_t h)
{
  Object::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  DrawBackground(x,y,w,h);

  if (sWidth==0 || sHeight==0) {
    return;
  }

  Lum::OS::DrawInfo *draw=GetDrawInfo();

  if (IsAreaCompletelyVisible()) {
    draw->CopyFromBitmap(game,
                         0,0,GetAreaWidth()*GetBoxSize(),GetAreaHeight()*GetBoxSize(),
                         this->x,this->y);
  }
  else {
    draw->CopyFromBitmap(game,
                         xOff*GetBoxSize(),yOff*GetBoxSize(),
                         width-2*GetBoxSize(),
                         height-2*GetBoxSize(),
                         this->x+GetBoxSize(),this->y+GetBoxSize());

    if (CanScrollUp()) {
      if (GetImage(upImage)!=NULL) {
        GetImage(upImage)->Draw(draw,
                                this->x+(width-GetBoxSize())/2,
                                this->y);
      }
      else {
        upArrow->Draw(draw,
                      this->x+(width-GetBoxSize())/2,this->y,
                      GetBoxSize(),GetBoxSize());
      }
    }

    // Left
    if (CanScrollLeft()) {
      if (GetImage(leftImage)!=NULL) {
        GetImage(leftImage)->Draw(draw,
                                  this->x,
                                  this->y+(height-GetBoxSize())/2);
      }
      else {
         leftArrow->Draw(draw,
                         this->x,this->y+(height-GetBoxSize())/2,
                         GetBoxSize(),GetBoxSize());
      }
    }

    // Down
    if (CanScrollDown()) {
      if (GetImage(downImage)!=NULL) {
        GetImage(downImage)->Draw(draw,
                                  this->x+(width-GetBoxSize())/2,
                                  this->y+height-GetBoxSize());
      }
      else {
         downArrow->Draw(draw,
                         this->x+(width-GetBoxSize())/2,this->y+height-GetBoxSize(),
                         GetBoxSize(),GetBoxSize());
      }
    }

    // Right
    if (CanScrollRight()) {
      if (GetImage(rightImage)!=NULL) {
        GetImage(rightImage)->Draw(draw,
                                   this->x+width-GetBoxSize(),
                                   this->y+(height-GetBoxSize())/2);
      }
      else {
         rightArrow->Draw(draw,
                          this->x+width-GetBoxSize(),this->y+(height-GetBoxSize())/2,
                          GetBoxSize(),GetBoxSize());
      }
    }
  }
}

bool Sweeper::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (!visible || status->Get()!=playing) {
    return false;
  }

  if (event.type==Lum::OS::MouseEvent::down && PointIsIn(event)) {
    mbdx=event.x;
    mbdy=event.y;

    return true;
  }
  else if (event.type==Lum::OS::MouseEvent::up && PointIsIn(event) &&
           event.button==Lum::OS::MouseEvent::button1 && event.qualifier==Lum::OS::qualifierButton1) {

    size_t x,y;

    if (!IsAreaCompletelyVisible()) {
      if ((size_t)event.x<this->x+GetBoxSize() && CanScrollLeft()) {
        ScrollLeft();
        return true;
      }
      if ((size_t)event.x>=this->x+width-GetBoxSize() && CanScrollRight()) {
        ScrollRight();
        return true;
      }
      if ((size_t)event.y<this->y+GetBoxSize() && CanScrollUp()) {
        ScrollUp();
        return true;
      }
      if ((size_t)event.y>=this->y+height-GetBoxSize() && CanScrollDown()) {
        ScrollDown();
        return true;
      }

      x=(event.x-this->x)/GetBoxSize()-1+xOff;
      y=(event.y-this->y)/GetBoxSize()-1+yOff;
    }
    else {
      x=(event.x-this->x)/GetBoxSize();
      y=(event.y-this->y)/GetBoxSize();
    }

    if (oneButtonMode) {
      if ((size_t)std::abs(mbdx-event.x)>=GetBoxSize()/3 || (size_t)std::abs(mbdy-event.y)>=GetBoxSize()/3) {
        OpenField(x,y);
      }
      else {
        ToggleMark(x,y);
      }
    }
    else {
      OpenField(x,y);
    }

    return true;
  }
  else if (event.type==Lum::OS::MouseEvent::up && PointIsIn(event) &&
           event.button==Lum::OS::MouseEvent::button3 &&
           event.qualifier==Lum::OS::qualifierButton3) {

    if (oneButtonMode) {
      return true;
    }

    size_t x,y;

    if (!IsAreaCompletelyVisible()) {
      if ((size_t)event.x<this->x+GetBoxSize() && CanScrollLeft()) {
        return true;
      }
      if ((size_t)event.x>=this->x+width-GetBoxSize() && CanScrollRight()) {
        return true;
      }
      if ((size_t)event.y<this->y+GetBoxSize() && CanScrollUp()) {
        return true;
      }
      if ((size_t)event.y>=this->y+height-GetBoxSize() && CanScrollDown()) {
        return true;
      }

      x=(event.x-this->x)/GetBoxSize()-1+xOff;
      y=(event.y-this->y)/GetBoxSize()-1+yOff;
    }
    else {
      x=(event.x-this->x)/GetBoxSize();
      y=(event.y-this->y)/GetBoxSize();
    }

    ToggleMark(x,y);

    return true;
  }

  return false;
}

bool Sweeper::SetSize(size_t width, size_t height, size_t bombs)
{
  if (width==sWidth && height==sHeight && bombs==this->bombs) {
    // Don't erase game zone, if nothing has changed.
    return true;
  }

  if (width>sMaxWidth || height>sMaxHeight || bombs>width*height) {
    return false;
  }

  sWidth=width;
  sHeight=height;
  this->bombs=bombs;

  delete game;
  game=Lum::OS::driver->CreateBitmap(GetAreaWidth()*GetBoxSize(),
                                     GetAreaHeight()*GetBoxSize());

  Run();

  RedrawArea();

  Redraw();

  return true;
}

void Sweeper::SetOneButtonMode(bool set)
{
  oneButtonMode=set;
}

size_t Sweeper::GetAreaWidth() const
{
  return sWidth;
}

size_t Sweeper::GetAreaHeight() const
{
  return sHeight;
}

size_t Sweeper::GetMines() const
{
  return bombs;
}

Lum::Model::ULong* Sweeper::GetStatusModel() const
{
  return status.Get();
}

void Sweeper::Run()
{
  size_t cells,bombs;

  /* Set a more random seed */
  srand(time(NULL));

  /* initialize */
  for (size_t x=0; x<sMaxWidth; x++) {
    for (size_t y=0; y<sMaxHeight; y++) {
      area[x][y].bomb=false;
      area[x][y].open=false;
      area[x][y].marked=false;
      area[x][y].count=0;
      area[x][y].active=false;
    }
  }

  cells=sWidth*sHeight;
  bombs=this->bombs;

  for (size_t x=0; x<sWidth; x++) {
    for (size_t y=0; y<sHeight; y++) {
      area[x][y].active=true;

      area[x][y].bomb=((float)cells*rand()/RAND_MAX)<bombs;

      if (area[x][y].bomb) {
        bombs--;
      }

      cells--;
    }
  }

  status->Set(playing);

  RedrawArea();

  xOff=0;
  yOff=0;

  Redraw();

  timer.Reset();
}

Sweeper::Status Sweeper::GetStatus() const
{
  return (Status)status->Get();
}

time_t Sweeper::GetElapsedTime() const
{
  return timer.GetTime();
}

void Sweeper::SetStatusToWaiting()
{
  status->Set(waiting);
  if (visible) {
    Redraw();
  }
}
