/*
 * This file is part of maptile-loader
 * Copyright (C) 2007  Pekka Rönkkö (pronkko@gmail.com)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <signal.h>
#include <glib.h>
#include <glib-object.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libosso.h>
#include <string.h>
#include <stdlib.h>

/* Which connection framework we should use?
   Note that to use Maemo 4.x connection framework 
   you should also change the configure.ac to include
   correct connectivity pkg-config libs and cflags */
#define MT_USE_MAEMO_4x_IAP
#undef MT_USE_MAEMO_3x_IAP

#if defined (MT_USE_MAEMO_4x_IAP)
#include <conic.h>
#elif defined (MT_USE_MAEMO_3x_IAP)
#include <osso-ic-dbus.h>
#include <osso-ic.h>
#endif

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus-glib.h>

#include "config.h"
#include "maptile-types.h"
#include "libmaptileloader/maptile-repo.h"
#include "maptile-loader.h"
#include "maptile-module-manager.h"

#define MAPTILE_REPO_CONF "~/.maptile-loader/maptile-loader.conf"

/* DBUS definitions and RPC methods */
const gchar * MAPTILE_LOADER_DBUS_SERVICE = "org.cityguide.maptileloader";
const gchar * MAPTILE_LOADER_DBUS_OBJECT_PATH = "/org/cityguide/maptileloader";
const gchar * MAPTILE_LOADER_DBUS_INTERFACE = "org.cityguide.maptileloader";

const gchar * MAPTILE_LOADER_FUNCTION_GET_SUPPORTED_MAPS = "MaptileLoaderGetSupportedMaps";
const gchar * MAPTILE_LOADER_FUNCTION_GET_MAPTILE = "MaptileLoaderGetMaptile";

/* Module manager handles plugins */
static MaptileModuleManager *module_manager = NULL;

/* Data structures for handling repo types */
static GHashTable *repo_types_hash_table = NULL;
static GList *repo_types_hash_table_keys = NULL;

/* Data structures for handling actual map repos */
static GHashTable *repo_hash_table = NULL;

/* IAP connection */
#if defined (MT_USE_MAEMO_4x_IAP)
static ConIcConnection *ic_conn = NULL;
#endif

/* Flag for IAP connection (connected = TRUE) */
static gboolean ic_connected = FALSE;
#if defined(MT_USE_MAEMO_3x_IAP)
static gboolean ic_connecting = FALSE;
/* FIXME: Proxy usage is not yet implemented */
static gchar *iap_http_proxy_host = NULL;
static gint iap_http_proxy_port = 0;
#endif

/* Module (plugin) init function */
static gboolean maptile_loader_init_modules();

/* Maptile repo init function */
static gboolean maptile_loader_init_repos();

/* Program cleanup function, called at the end of execution */
static void maptile_loader_deinit();

/* Connectivity callbacks */
static void maptile_loader_check_con_ic();
#if defined (MT_USE_MAEMO_4x_IAP)
static void on_connection_event (ConIcConnection *conn, 
                                 ConIcConnectionEvent *event,
                                 gpointer user_data);
#elif defined(MT_USE_MAEMO_3x_IAP)
static DBusHandlerResult get_connection_status_signal_cb(DBusConnection *connection,
                                                         DBusMessage *message,
                                                         void *user_data);
static void iap_callback(struct iap_event_t *event,
                         void *arg);
#endif
/* Repo hash table destroyers */
static void repo_keys_destroy(gpointer data);
static void  repo_values_destroy(gpointer data);

/* DBUS callback */
static DBusHandlerResult dbus_cb_default(DBusConnection *connection, DBusMessage *message, void *user_data);

/* DBUS RPC handler helper functions */
static void maptile_loader_get_supported_maps(gchar ***supported_maps, guint *n_supported_maps);
static gchar *maptile_loader_get_tile(gchar *maptype, guint tilex, guint tiley, guint zoom);

/* Signal handler */
static void signalmanager(int sig)
{
    g_debug("Signal %d. Quitting.", sig);
    maptile_loader_deinit();
    exit(0);
}

/* Main function */
int main( void )
{


    GMainLoop *loop;
    dbus_bool_t result; 

    osso_context_t *_osso = NULL;

    _osso = osso_initialize ( MAPTILE_LOADER_DBUS_SERVICE, "0.1", TRUE, NULL );
    g_assert ( _osso );

    DBusConnection * dbconn = osso_get_dbus_connection( _osso );    
    static DBusObjectPathVTable dmap_req_vtable = {
        .message_function    = dbus_cb_default,
        .unregister_function = NULL
    };

    result = dbus_connection_register_object_path(dbconn,
                                                "/org/cityguide/maptileloader",
                                                &dmap_req_vtable,
                                                NULL);
    if (!result)
    {
        g_warning("dbus connection failed\n");
        return -1;
    }

    /* GType system needs to be initialized for plugin system */
    g_type_init();

    loop = g_main_loop_new(NULL, FALSE);

/* Initialize connectivity */
#if defined (MT_USE_MAEMO_4x_IAP)

    ic_conn = con_ic_connection_new();
    if ( !ic_conn )
    {
        g_warning("IC connection object creation failed!\n");
        return -1;
    }

    g_signal_connect (ic_conn, "connection-event",
                      G_CALLBACK(on_connection_event), NULL);

    if ( !con_ic_connection_connect(ic_conn, CON_IC_CONNECT_FLAG_AUTOMATICALLY_TRIGGERED))
        g_warning("IC connection request failed due to DBUS error!\n");

#elif defined(MT_USE_MAEMO_3x_IAP)

    DBusConnection *dbus_conn_system = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    gchar *filter_string = g_strdup_printf(
              "interface=%s", ICD_DBUS_INTERFACE);
    /* add match */
    dbus_bus_add_match(dbus_conn_system, filter_string, NULL);

    g_free (filter_string);
    /* add the callback */
    dbus_connection_add_filter(dbus_conn_system,
                               get_connection_status_signal_cb,
                               NULL, NULL);
    osso_iap_cb(iap_callback);

#endif


    /* Gnome VFS needs to be initialized for the directory handling */
    if ( !gnome_vfs_initialized() )
        gnome_vfs_init();

    /* Initialize plugins */
    if (!maptile_loader_init_modules())
        exit(-1);

    /* Initialize maptile repositories */
    if (!maptile_loader_init_repos())
        exit(-1);

    struct sigaction sa;
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = signalmanager;
    sigaction(SIGTERM, &sa, NULL);
    sigaction(SIGINT, &sa, NULL);
    sigaction(SIGBUS, &sa, NULL);

    /* Begin the main loop */
    g_main_loop_run(loop);

    return 0;
}

/* Maptile hash key memory freeing */
static void repo_keys_destroy(gpointer data)
{
    g_free((gchar *)data);
}

/* Maptile repo hash data freeing */
static void repo_values_destroy(gpointer data)
{
    MaptileRepo *repo = MAPTILE_REPO(data);
    g_object_unref(repo);
}

/* Initialize plugin system, load plugins into memory */
static gboolean maptile_loader_init_modules()
{
    GType *repos;
    guint n_repos;
    guint i;

    module_manager = g_object_new(MAPTILE_TYPE_MODULE_MANAGER,
                                    "module-path", MODULEDIR, NULL);


    if ( repo_types_hash_table == NULL )
        repo_types_hash_table = g_hash_table_new(g_str_hash, g_str_equal);

    if ( repo_hash_table == NULL )
        repo_hash_table = g_hash_table_new_full(g_str_hash, g_str_equal, repo_keys_destroy, repo_values_destroy);

    repos = g_type_children (MAPTILE_TYPE_REPO, &n_repos);

    if ( !n_repos )
    {
        return FALSE;
    }

    for (i = 0; i < n_repos; i++)
    {
        MaptileRepoClass *klass;

        klass = g_type_class_ref (repos[i]);
        /* Insert types to hash table so we can create the instances of maptile repositories of
           correct type */
        g_debug("Inserting plugin type %s", klass->name);
        g_hash_table_insert(repo_types_hash_table, g_strdup(klass->name), (gpointer)repos[i]);
        /* Insert plugin names into list to ease getting the types thorough DBUS */
        repo_types_hash_table_keys = g_list_append(repo_types_hash_table_keys, g_strdup(klass->name));

        g_type_class_unref (klass);
    }

    g_free(repos);

    return TRUE;
}

/* Free reserved memory at the end of execution */
static void maptile_loader_deinit()
{
    if ( module_manager )
        g_object_unref(module_manager);

    if ( repo_types_hash_table )
        g_hash_table_destroy(repo_types_hash_table);

    if ( repo_hash_table )
        g_hash_table_destroy(repo_hash_table);

    if ( repo_types_hash_table_keys )
    {
        GList *tmp = repo_types_hash_table_keys;
        while ( tmp != NULL )
        {
            g_free(tmp->data);
            tmp = g_list_next(tmp);
        }
        g_list_free(repo_types_hash_table_keys);
    }
}

/* Read repos from the config file and save them to hash table */
static gboolean maptile_loader_init_repos()
{
    GError *error = NULL;
    gchar *filename = gnome_vfs_expand_initial_tilde(MAPTILE_REPO_CONF);
    GKeyFile *repo_file = g_key_file_new();
    if (!g_key_file_load_from_file(repo_file, filename,
                                   G_KEY_FILE_NONE, &error)) {
        g_warning("Error loading repositories from %s: %s", filename,
                  error->message);
        g_error_free(error);
        return FALSE;
    }

    gsize num_groups = 0;
    gchar **groups = g_key_file_get_groups(repo_file, &num_groups);
    if (groups == NULL) {
        g_warning("No repositories found in %s", filename);
        g_key_file_free(repo_file);
        return FALSE;
    }

    int i = 0;
    MaptileRepo *repo;
    GType repo_type;
    gchar *type = NULL;
    gchar *url = NULL;
    gchar *cache = NULL;
    gchar *use_ic = NULL;
    gboolean use_ic_flag = FALSE;
    gboolean verified = TRUE;

    if ( g_hash_table_size(repo_hash_table) > 0 )
    {
        g_hash_table_destroy(repo_hash_table);
        repo_hash_table = g_hash_table_new_full(g_str_hash, g_str_equal, repo_keys_destroy, repo_values_destroy);
    }

    for (i = 0; i < num_groups; i++) {

        g_debug("Creating repo for maptile %s", groups[i]);

        cache = g_key_file_get_string(repo_file, groups[i], "cache", NULL);

        /* Check for existence of the cache URI */
        GnomeVFSURI *map_dir_uri = gnome_vfs_uri_new(cache);
        if(!gnome_vfs_uri_exists(map_dir_uri))
        {
            GnomeVFSURI *parent, *curr_uri;
            GList *list = NULL;

            list = g_list_prepend(list, curr_uri = map_dir_uri);
            while(GNOME_VFS_ERROR_NOT_FOUND
                    == gnome_vfs_make_directory_for_uri(
                        parent = gnome_vfs_uri_get_parent(curr_uri), 0755))
                list = g_list_prepend(list, curr_uri = parent);

            while(list != NULL)
            {
                if(verified)
                {
                    verified = (GNOME_VFS_OK
                            == gnome_vfs_make_directory_for_uri(
                                (GnomeVFSURI*)list->data, 0755));
                }
                gnome_vfs_uri_unref((GnomeVFSURI*)list->data);
                list = g_list_remove(list, list->data);
            }
            /* verified now equals result of last make-dir attempt. */
        }

        if(!verified)
        {
            /* Failed to create Map Cache directory. */
            g_warning("Unable to create map cache directory:%s, skipping maptile repo  %s", cache, groups[i]);
            continue;
        }

        type = g_key_file_get_string(repo_file, groups[i], "type", NULL);
        url = g_key_file_get_string(repo_file, groups[i], "url", NULL);

        use_ic = g_key_file_get_string(repo_file, groups[i], "use_ic", NULL);
        if (!strcmp(use_ic, "yes"))
            use_ic_flag = TRUE;
        else
            use_ic_flag = FALSE;

        repo_type = (GType) g_hash_table_lookup(repo_types_hash_table, type);

        repo = g_object_new(repo_type, "map-name", groups[i],
                                           "map-data-uri", url,
                                           "map-cache-uri", cache,
                                           "con-ic-needed", use_ic_flag,
                                           NULL);
        if ( repo )
            g_hash_table_insert(repo_hash_table, g_strdup(groups[i]), repo);
        else
            g_warning("Repo creation for %s failed", groups[i]);

    }

    g_key_file_free(repo_file);
    g_free(filename);
    return TRUE;
}

/* Put plugin names into array of strings */
static void maptile_loader_get_supported_maps(gchar ***supported_maps, guint *n_supported_maps)
{
    guint i = 0;
    *n_supported_maps = 0;

    GList *tmp = repo_types_hash_table_keys;

    if ( tmp )
        *supported_maps = g_new0(gchar *, g_list_length(repo_types_hash_table_keys));
    while ( tmp != NULL )
    {
        (*supported_maps)[i] = g_strdup((gchar *)tmp->data);
        tmp = g_list_next(tmp);
        i++;
        *n_supported_maps = i;
   }
}

/* Get plugin from the hahs table */
static gchar *maptile_loader_get_tile(gchar *maprepo, guint tilex, guint tiley, guint zoom)
{
    MaptileRepo    *repo;
    gchar          *tile;

    repo = (MaptileRepo *) g_hash_table_lookup(repo_hash_table, (gconstpointer)maprepo);

    if (!repo)
        g_warning("Could not find repo type for %s", maprepo);

    maptile_repo_get_tile (repo, tilex, tiley, zoom, &tile,
		    &maptile_loader_check_con_ic);

    return tile;
}

/*
 * DBUS method handler:
 * call: MAPTILE_LOADER_FUNCTION_GET_SUPPORTED_MAPS = "MaptileLoaderGetSupportedMaps"
 *       - parameters: none
 *       - returns: array of strings ("GoogleMaps""MicrosoftVE")
 * call: MAPTILE_LOADER_FUNCTION_GET_MAPTILE = "MaptileLoaderGetMaptile";
 *       - parameters: maptype, string (eg. Google Street)
 *                     orientation, float (eg. 0.0)
 *                     tilex, int (X of tile)
 *                     tiley, int (Y of tile)
 *                     zoom, float (zoom level, eg. 2.0)
 *       - returns: URI to local cache where the tile is saved
 */
static DBusHandlerResult
dbus_cb_default(DBusConnection *connection, DBusMessage *message, void *user_data)
{
    DBusHandlerResult res = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    DBusMessage * reply = NULL;

    if (dbus_message_is_method_call(message, MAPTILE_LOADER_DBUS_INTERFACE, MAPTILE_LOADER_FUNCTION_GET_MAPTILE) )
    {
        gchar *maptype;
        guint tilex;
        guint tiley;
        guint zoom;

        if ( dbus_message_get_args( message, NULL,
                                    DBUS_TYPE_STRING, &maptype,
                                    DBUS_TYPE_UINT32, &tilex,
                                    DBUS_TYPE_UINT32, &tiley,
                                   /* float? */ DBUS_TYPE_UINT32, &zoom,
                                    DBUS_TYPE_INVALID) != FALSE )
        {
            gchar *maptile = maptile_loader_get_tile(maptype, tilex, tiley, zoom);
            /* Create reply */
            reply = dbus_message_new_method_return(message);
            if (reply)
            {
                dbus_message_append_args(reply,
                    DBUS_TYPE_STRING, &maptile,
                    DBUS_TYPE_INVALID);
            }
            res = DBUS_HANDLER_RESULT_HANDLED;
            g_free(maptile);
        }
        else
        {
            g_warning("Received request for maptile but the parameters are wrong!\n");
            res = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }

    }
    else if (dbus_message_is_method_call(message, MAPTILE_LOADER_DBUS_INTERFACE, MAPTILE_LOADER_FUNCTION_GET_SUPPORTED_MAPS) )
    {
        gchar **supported_maps;
        guint n_supported_maps;

        maptile_loader_get_supported_maps(&supported_maps, &n_supported_maps);

        reply = dbus_message_new_method_return(message);

        if (reply)
        {
            dbus_message_append_args(reply,
                DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &supported_maps, n_supported_maps,
                DBUS_TYPE_INVALID);
        }
        res = DBUS_HANDLER_RESULT_HANDLED;
    }
    else
    {
        res = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    /* Send the reply */
    if (reply)
    {
        if ( dbus_connection_send( connection, reply, NULL) == FALSE )
            g_warning("DBUS handler error sending reply!\n");
        else
            res = DBUS_HANDLER_RESULT_HANDLED;
    }

    return res;
}

/* Connectivity functions and callbacks */
static void maptile_loader_check_con_ic()
{
#if defined (MT_USE_MAEMO_4x_IAP)
    if ( !ic_connected )
        if ( !con_ic_connection_connect(ic_conn, CON_IC_CONNECT_FLAG_NONE) )
            g_warning("IAP connection failed due to DBUS error!\n");
#elif defined(MT_USE_MAEMO_3x_IAP)
    if(!ic_connected && !ic_connecting)
    {
        ic_connecting = TRUE;
        osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, NULL);
    }
#endif
}

#if defined (MT_USE_MAEMO_4x_IAP)
static void on_connection_event (ConIcConnection *conn, ConIcConnectionEvent *event, gpointer user_data)
{
    gboolean is_connected;

    g_return_if_fail (CON_IC_IS_CONNECTION(conn));

    switch (con_ic_connection_event_get_error(event)) {
    case CON_IC_CONNECTION_ERROR_NONE:
        break;
    case CON_IC_CONNECTION_ERROR_INVALID_IAP:
        g_warning ("conic: IAP is invalid");
        break;
    case CON_IC_CONNECTION_ERROR_CONNECTION_FAILED:
        g_warning ("conic: connection failed");
        break;
    case CON_IC_CONNECTION_ERROR_USER_CANCELED:
        g_warning ("conic: user cancelled");
        break;
    default:
        g_return_if_reached ();
    }

    switch (con_ic_connection_event_get_status(event)) {
    case CON_IC_STATUS_CONNECTED:
        is_connected = TRUE;
        break;
    case CON_IC_STATUS_DISCONNECTED:
        is_connected = FALSE;
        break;
    case CON_IC_STATUS_DISCONNECTING:
        is_connected = FALSE;
        break;
    default:
        g_return_if_reached ();
    }

    if (is_connected != ic_connected) {
        ic_connected = is_connected;
    }
}
#elif defined (MT_USE_MAEMO_3x_IAP)
static DBusHandlerResult
get_connection_status_signal_cb(DBusConnection *connection,
        DBusMessage *message, void *user_data)
{
    gchar *iap_name = NULL, *iap_nw_type = NULL, *iap_state = NULL;

    /* check signal */
    if(!dbus_message_is_signal(message,
                ICD_DBUS_INTERFACE,
                ICD_STATUS_CHANGED_SIG))
    {
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if(!dbus_message_get_args(message, NULL,
                DBUS_TYPE_STRING, &iap_name,
                DBUS_TYPE_STRING, &iap_nw_type,
                DBUS_TYPE_STRING, &iap_state,
                DBUS_TYPE_INVALID))
    {
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if(!strcmp(iap_state, "CONNECTED"))
    {
        if(!ic_connected)
            ic_connected = TRUE;
    }
    else if(ic_connected)
    {
        ic_connected = FALSE;
    }

    return DBUS_HANDLER_RESULT_HANDLED;
}

static void
iap_callback(struct iap_event_t *event, void *arg)
{
    ic_connecting = FALSE;
    if(event->type == OSSO_IAP_CONNECTED && !ic_connected)
    {
        ic_connected = TRUE;
    }
}
#endif

/* end of discovery-map-fw.c */
