#
# _radio.py
# Easy API - radio module.
#
# Copyright (C) 2007-2008 UFCG - Federal University of Campina Grande
# Embedded Systems and Pervasive Computing Lab.
#
# Contact: Mario Hozano Lucas de Souza <hozano@embedded.ufcg.edu.br>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

"""This module encapsulates a set of functions to reproduce broadcasted FM
radios. These functions include tuning, auto-scan for radios or get
available stations.
"""
import pygst
pygst.require("0.10")
import gst
import time
import math

__all__ = ["start", "stop", "tune", "get_frequency", "next_station", 
           "previous_station", "scan", "get_volume", "set_volume"]

class Radio:
    """This module encapsulates a set of functions to reproduce broadcasted FM
    radios. These functions include tuning, auto-scan for radios or get
    available stations.
    """

    _MIN_FREQ = 87500
    _MAX_FREQ = 106900
    _FREQ_STEP = 200
    _OK_STRENGHT = 10000

    def __init__(self):
        """Constructor for Radio class.
        """
        self.radio = gst.element_factory_make("radiobin", "radio")
        # We are using this local property instead of the "volume" property of 
        # radiobin because the method 'self.radio.get_property("volume")' that 
        # return the "volume" property of radiobin is not working properly.
        self._volume = 3

    def start(self):
        """Start reproducing radio.
        """
        self.radio.set_state(gst.STATE_PLAYING)
        self.set_volume(self._volume)

    def stop(self):
        """Stop reproducing radio.
        """
        self.radio.set_state(gst.STATE_NULL)

    def tune(self, frequency):
        """Tune radio to the specified frequency.

        @type  frequency: integer
        @param frequency: frequency value specified in kHz
        """
        self.radio.set_property('frequency', frequency)

    def get_frequency(self):
        """Get radio's actual frequency.

        @rtype:  integer
        @return: frequency value specified in kHz
        """
        return self.radio.get_property('frequency')

    def next_station(self):
        """Search and tune next station founded.
        """
        return self.scan(forward_direction=True)

    def previous_station(self):
        """Search and tune previous station founded.
        """
        return self.scan(forward_direction=False)

    def scan(self, forward_direction=True):
        """Search and auto tune radio to founded station. The search could be
        performed in a forward or in a backward way.

        @type  forward_direction: boolean
        @param forward_direction: True to tune the next station or False to
                                  tune the previous station
        """
        increment = forward_direction and self._FREQ_STEP or -self._FREQ_STEP
        old_frequency = frequency = self.get_frequency()

        frequency -= frequency % 100
        if not (frequency / 100) % 2:
            frequency += increment / 2

        while True:
            frequency += increment

            if frequency < self._MIN_FREQ:
                frequency = self._MAX_FREQ

            if frequency > self._MAX_FREQ:
                frequency = self._MIN_FREQ

            self.tune(frequency)
            # This time.sleep is necessary to radiobin correct sintonize and
            # get the correct value of the signal's strenght
            time.sleep(0.05)

            if self.radio.get_property('signal-strength') > self._OK_STRENGHT:
                return frequency

            elif (old_frequency % frequency) < (self._FREQ_STEP / 2):
                self.tune(old_frequency)
                return old_frequency

    def get_volume(self):
        """Return the volume level of the radio.

        @rtype:  unsigned integer
        @return: radio's volume level
        """
        # This method is returning this local property because the method 
        # 'self.radio.get_property("volume")' that must return the "volume" 
        # property of radiobin is not working properly.
        return self._volume

    def set_volume(self, volume=None):
        """Set the volume level of the radio.The values range is from 0 to 10.

        @type  volume: unsigned integer
        @param volume: number that set volume level, ranged from 0 to 10
        """
        # Radiobin's uses v4l2 and it's volume property is specified in (%).
        # Due to standardize with other Easy's modules, the parameter 'volume'
        # specified in (0-10) scale is multiplied by 10, to be in (%).
        if volume <= 10.0 and volume >= 0.0:
            self.radio.set_property("volume", volume * 10)
            self._volume = volume

_radio = Radio()

start = _radio.start
stop = _radio.stop
tune = _radio.tune
get_frequency = _radio.get_frequency
next_station = _radio.next_station
previous_station = _radio.previous_station
scan = _radio.scan
get_volume = _radio.get_volume
set_volume = _radio.set_volume