/*
  DiskUsage - shows the disk usage under a number of operating systems.
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <math.h>

#include <iomanip>
#include <iostream>
#include <list>
#include <locale>
#include <set>
#include <sstream>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Object.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/ActionDialog.h>
#include <Lum/Dlg/Progress.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/DataStream.h>
#include <Lum/Model/Integer.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Grid.h>
#include <Lum/Label.h>
#include <Lum/Menu.h>
#include <Lum/PercentBar.h>
#include <Lum/PercentPie.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>

#include "config.h"

#if defined(linux)
#include <string.h>
#include <sys/vfs.h>
#include <mntent.h>
#elif defined(sun)
#include <stdio.h>
#include <string.h>
#include <sys/mnttab.h>
#include <sys/types.h>
#include <sys/statvfs.h>
#elif defined(__APPLE__)
#include <sys/param.h>
#include <sys/ucred.h>
#include <sys/mount.h>
#elif defined(__WIN32__) || defined(WIN32)
#include <windows.h>
#include <winnt.h>
#undef GetObject
#endif

static Lum::AppInfo info;

struct DiskInfo
{
  std::wstring  name;
  std::wstring  dir;
  double        blocks;
  double        free;
  double        avail;
  double        unit; // Multiplicator to get the size in kilobytes
};

bool GetDisks(std::list<DiskInfo>& list)
{
#if defined(linux)
  FILE          *file;
  struct mntent *inf;
  struct statfs stats;
  DiskInfo      info;

  file=setmntent("/etc/mtab","r");

  if (file==NULL) {
    return false;
  }

  while (true) {
    memset(&stats,0,sizeof(stats));
    inf=getmntent(file);
    while ((inf!=NULL) && ((statfs(inf->mnt_dir,&stats)!=0) || (stats.f_blocks==0))) {
      inf=getmntent(file);
    }
    if ((inf!=NULL) && (statfs(inf->mnt_dir,&stats)==0)) {
      info.name=Lum::Base::StringToWString(inf->mnt_fsname);
      info.dir=Lum::Base::StringToWString(inf->mnt_dir);
      info.blocks=stats.f_blocks;
      info.free=stats.f_bfree;
      info.avail=stats.f_bavail;
      info.unit=stats.f_bsize/1024;

      list.push_back(info);
    }
    else {
      break;
    }
  }

  endmntent(file);

  return true;
#elif defined(sun)
  FILE           *file;
  struct mnttab  tab;
  struct statvfs stats;
  DiskInfo       info;

  file=fopen("/etc/mnttab","r");

  if (file==NULL) {
    return false;
  }

  while (getmntent(((Private*)handle)->handle,&tab)==0) {
    memset(&stats,0,sizeof(stats));
    if ((statvfs(tab.mnt_mountp,&stats)==0)) {
      info.name=Lum::Base::StringToWString(tab.mnt_special);
      info.dir=Lum::Base::StringToWString(list[x].f_mntonname);
      info.blocks=stats.f_blocks;
      info.free=stats.f_bfree;
      info.avail=stats.f_bavail;
      info.unit=1; // TODO

      list.push_back(info);
    }
    else {
      break;
    }
  }

  fclose(file);

  return true;
#elif defined(__APPLE__)
  struct statfs* devlist;
  size_t         size;
  DiskInfo       info;

  size=getmntinfo(&devlist,MNT_NOWAIT);

  if (size==0) {
    return false;
  }
  for (size_t x=0; x<size; x++) {
    info.name=Lum::Base::StringToWString(devlist[x].f_fstypename);
    info.dir=Lum::Base::StringToWString(devlist[x].f_mntonname);
    info.blocks=devlist[x].f_blocks;
    info.free=devlist[x].f_bfree;
    info.avail=devlist[x].f_bavail;
    info.unit=1; // TODO

    list.push_back(info);
  }

  return true;
#elif defined(__WIN32__) || defined(WIN32)
  UINT    old;
  wchar_t buffer[10240];
  wchar_t buffer2[MAX_PATH+1];
  size_t  offset=0;
  DWORD   compLength,flags;
  //UINT    type;

  old=SetErrorMode(SEM_FAILCRITICALERRORS);
  GetLogicalDriveStringsW(1024,buffer);

  while (buffer[offset]!='\0') {
    DiskInfo       info;
    ULARGE_INTEGER available,bytes,free;

    size_t x=0;
    while (buffer[offset]!='\0') {
      info.name.append(1,buffer[offset]);
      info.dir.append(1,buffer[offset]);
      ++x;
      ++offset;
    }
    ++offset;

    /*
    type=GetDriveTypeW(info.name.c_str());
    if (type==DRIVE_UNKNOWN || type==DRIVE_NO_ROOT_DIR || type==DRIVE_REMOVABLE) {
      continue;
    }*/

    if (GetVolumeInformationW(info.name.c_str(),buffer2,MAX_PATH+1,NULL,&compLength,&flags,NULL,0)!=0) {
      info.dir=buffer2;

      if (info.dir==L"OS") {
	      info.dir=info.name;
      }
    }

    if (GetDiskFreeSpaceExW(info.name.c_str(),&available,&bytes,&free)) {
      info.blocks=bytes.QuadPart/1024;
      info.free=free.QuadPart/1024;
      info.avail=available.QuadPart/1024;
      info.unit=1;
      list.push_back(info);
    }
  }

  SetErrorMode(old);

  return true;
#else
  return false;
#endif
}

std::wstring DoubleToWString(double value, double unit)
{
  std::stringstream buffer;

  buffer.imbue(std::locale(""));
  buffer.setf(std::ios::fixed);
  buffer<< std::setprecision(1);

  if (ceil(value*unit)>=1024*1024*1024) {
    buffer << (value*unit)/(1024*1024*1024) << " TB";
  }
  else if (ceil(value*unit)>=1024*1024) {
    buffer << (value*unit)/(1024*1024) << " GB";
  }
  else if (ceil(value*unit)>=1024) {
    buffer<< std::setprecision(1);
    buffer << (value*unit)/1024 << " MB";
  }
  else {
    buffer << value*unit << " KB";
  }

  return Lum::Base::StringToWString(buffer.str());
}

Lum::TextValue* GetTextValueLeft(const Lum::Model::StringRef& value)
{
  Lum::TextValue   *text;
  Lum::OS::FontRef font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption1));

  text=Lum::TextValue::Create(value,Lum::TextValue::left,true,false);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,25);
  // TODO: font

  return text;
}

Lum::TextValue* GetTextValueRight(const Lum::Model::StringRef& value)
{
  Lum::TextValue   *text;
  Lum::OS::FontRef font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption1));

  text=Lum::TextValue::Create(value,Lum::TextValue::right,true,false);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,25);
  // TODO: font

  return text;
}

class DirEntry
{
public:
  Lum::Base::Path                  path;
  double                           size;
  std::map<std::wstring,DirEntry*> subDirs;

public:
  DirEntry(const Lum::Base::Path& path)
   : path(path),
     size(0)
  {
    // no code
  }

  ~DirEntry()
  {
    for (std::map<std::wstring, DirEntry*>::const_iterator iter=subDirs.begin();
         iter!=subDirs.end();
         ++iter) {
      delete iter->second;
    }
  }

  void AccumulateSize()
  {
    for (std::map<std::wstring, DirEntry*>::const_iterator iter=subDirs.begin();
         iter!=subDirs.end();
         ++iter) {
      iter->second->AccumulateSize();
      size+=iter->second->size;
    }
  }
};

class Map : public Lum::Object
{
private:
  class Rect
  {
  public:
    int          x;
    int          y;
    size_t       width;
    size_t       height;
    std::wstring label;
  };

private:
  DirEntry              *top;
  double                size;

  std::list<Rect>       labels;

  Lum::Model::StringRef label;

public:
  Map(DirEntry *top, double size)
   : top(top),
     size(size),
     label(new Lum::Model::String(L""))
  {
    // no code
  }

  ~Map()
  {
    delete top;
  }

  Lum::Model::String* GetLabel() const
  {
    return label.Get();
  }

  void CalcSize()
  {
    minWidth=100;
    minHeight=100;

    width=minWidth;
    height=minHeight;

    Object::CalcSize();
  }

  void DrawEntry(Lum::OS::DrawInfo *draw,
                 DirEntry* entry,
                 int x, int y, size_t width, size_t height,
                 int depth)
  {
    size_t full=lround(entry->size);
    size_t area=width*height;
    int    offset;

    // TODO: Instead dividing hardcode by 1024 to avoid overlow,
    // Calculate factor based on the full value!

    if (area==0 || full==0) {
      return;
    }

    if (width>=height) {
      offset=x-1;
    }
    else {
      offset=y-1;
    }

    for (std::map<std::wstring, DirEntry*>::const_iterator iter=entry->subDirs.begin();
         iter!=entry->subDirs.end();
         ++iter) {

      if (iter->second->size==0) {
        continue;
      }

      Rect rect;

      rect.label=iter->second->path.GetPath();

      size_t subArea=lround((area*iter->second->size)/full);

      if (subArea==0) {
        continue;
      }

      if (width>=height) {
        size_t w;

        w=subArea/height;

        if (w<=1) {
          continue;
        }

        rect.x=offset+1;
        rect.y=y;
        rect.width=w-1;
        rect.height=height;

        labels.push_front(rect);

        draw->PushForeground(Lum::OS::Display::fillColor);
        draw->FillRectangle(offset+1,y,w-1,height);
        draw->PopForeground();

        draw->PushForeground(Lum::OS::Display::blackColor);
        draw->DrawLine(offset+w,y,offset+w,y+height-1);
        draw->PopForeground();

        DrawEntry(draw,iter->second,offset+1,y,w-1,height,depth+1);

        offset+=w;
      }
      else {
        size_t h;

        h=subArea/width;

        if (h<=1) {
          continue;
        }

        rect.x=x;
        rect.y=offset+1;
        rect.width=width;
        rect.height=h-1;

        labels.push_front(rect);

        draw->PushForeground(Lum::OS::Display::fillColor);
        draw->FillRectangle(x,offset+1,width,h-1);
        draw->PopForeground();

        draw->PushForeground(Lum::OS::Display::blackColor);
        draw->DrawLine(x,offset+h,x+width-1,offset+h);
        draw->PopForeground();

        DrawEntry(draw,iter->second,x,offset+1,width,h-1,depth+1);

        offset+=h;
      }
    }
  }

  void Draw(int x, int y, size_t w, size_t h)
  {
    Object::Draw(x,y,w,h); /* We must call Draw of our superclass */

    if (!OIntersect(x,y,w,h)) {
      return;
    }

    Lum::OS::DrawInfo *draw=GetDrawInfo();
    size_t            fillFactor;

    draw->PushForeground(Lum::OS::Display::whiteColor);
    draw->FillRectangle(this->x,this->y,width,height);
    draw->PopForeground();

    draw->PushForeground(Lum::OS::Display::blackColor);
    draw->DrawRectangle(this->x,this->y,width,height);
    draw->PopForeground();

    fillFactor=lround((100*top->size)/size);

    labels.clear();
    DrawEntry(draw,top,this->x+1,this->y+1,((width-2)*fillFactor)/100,height-2,1);
  }

  void ShowLabel(int x, int y)
  {
    for (std::list<Rect>::const_iterator iter=labels.begin();
         iter!=labels.end();
         ++iter) {
      if (x>=iter->x && y>=iter->y &&
          x<iter->x+(int)iter->width && y<iter->y+(int)iter->height) {
        label->Set(iter->label);
        return;
      }
    }

    label->Set(L"");
  }

  bool HandleMouseEvent(const Lum::OS::MouseEvent& event)
  {
    if (!visible) {
      return false;
    }

    if (event.type==Lum::OS::MouseEvent::down && PointIsIn(event) &&
        event.button==Lum::OS::MouseEvent::button1) {
      ShowLabel(event.x,event.y);
      return true;
    }
    else if (event.type==Lum::OS::MouseEvent::move && event.IsGrabed()) {
      ShowLabel(event.x,event.y);
    }
    else if (event.IsGrabEnd()) {
      ShowLabel(event.x,event.y);
    }

    return false;
  }
};

class MapWindow : public Lum::Dlg::ActionDialog
{
private:
  DirEntry *top;
  double   size;

public:
  MapWindow(DirEntry *top, double size)
   : top(top),
     size(size)
  {
    Observe(GetClosedAction());
  }

  Lum::Object* GetContent()
  {
    Lum::Panel     *vert;
    Map            *map;

    vert=Lum::VPanel::Create(true,true);
    vert->SetWidth(Lum::Base::Size::workHRel,80);
    vert->SetHeight(Lum::Base::Size::workVRel,60);

    map=new Map(top,size);
    map->SetFlex(true,true);

    vert->Add(map);
    vert->AddSpace();
    vert->Add(Lum::TextValue::Create(map->GetLabel(),true,false));

    return vert;
  }

  void GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
  {
    Lum::Dlg::ActionDialog::CreateActionInfosClose(actions,GetClosedAction());
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }

    Lum::Dlg::ActionDialog::Resync(model,msg);
  }
};

class ScanTask : public Lum::Dlg::ProgressTask
{
private:
  Lum::Base::Path dir;

public:
  DirEntry        *top;
  double          size;

public:
  ScanTask()
   : top(NULL),
     size(0)
  {
    // no code
  }

  void SetSize(double size)
  {
    this->size=size;
  }

  void SetDir(const Lum::Base::Path& dir)
  {
    this->dir=dir;
  }

  void Run()
  {
    unsigned long              scannedDirs=0;
    std::list<Lum::Base::Path> todos;
    std::set<unsigned long>    hardLinks;

    delete top;

    todos.push_back(dir);

    while (todos.size()>0 && !IsAborted()) {
      Lum::Base::Path              todo;
      Lum::Base::DirScanner::Entry entry;
      Lum::Base::Status            status;
      Lum::Base::DirScannerRef     scanner;
      DirEntry                     *e;

      todo=todos.front();
      todos.pop_front();
      e=new DirEntry(todo);

      scannedDirs++;
      SetAction(L"Scaned "+
                Lum::Base::NumberToWString(scannedDirs)+
                L" from "+
                Lum::Base::NumberToWString(scannedDirs+todos.size())+
                L" directories...");
      SetProgress((100*scannedDirs)/(scannedDirs+todos.size()));

      //
      // Scan the directory
      //

      scanner=Lum::Base::DirScanner::Create(todo);

      while (scanner->GetNext(entry,status)) {
        if (entry.name!=L"." && entry.name!=L"..") {
          if (entry.isDir && !entry.isSoftLink &&
              !(entry.isHardLink && hardLinks.find(entry.linkId)!=hardLinks.end())) {
            Lum::Base::Path subDir(todo);

            subDir.AppendDir(entry.name);

            todos.push_back(subDir);
          }
          else if (!(entry.isHardLink && hardLinks.find(entry.linkId)!=hardLinks.end())) {
            e->size+=entry.size;
            //e->size+=entry.blockSize;
          }
        }

        if (entry.isHardLink) {
          hardLinks.insert(entry.linkId);
        }
      }

      e->size/=1024;

      //
      // Link into the directory tree
      //

      if (top==NULL) {
        top=e;
      }
      else {
        DirEntry *current=top;

        for (size_t i=dir.GetPartCount(); i<todo.GetPartCount()-1; i++) {
          assert(current->subDirs.find(todo.GetPart(i))!=current->subDirs.end());

          current=current->subDirs[todo.GetPart(i)];
        }

        assert(current!=NULL);

        current->subDirs[todo.GetPart(todo.GetPartCount()-1)]=e;
      }
    }

    if (IsAborted()) {
      delete top;
      top=NULL;
      todos.clear();

      return;
    }

    top->AccumulateSize();
  }
};

typedef Lum::Model::StdTable<DiskInfo>      DiskInfoModel;
typedef Lum::Base::Reference<DiskInfoModel> DiskInfoModelRef;

class DiskInfoDataProvider : public DiskInfoModel::DataProvider
{
private:
  Lum::PercentBar                 *percent;
  Lum::Model::DoubleDataStreamRef data;

public:
  DiskInfoDataProvider()
  : percent(new Lum::PercentBar()),
    data(new Lum::Model::DoubleDataStream())
  {
    percent->SetFlex(true,true);
    percent->SetMinWidth(Lum::Base::Size::stdCharWidth,15);
    percent->SetModel(data);

    data->SetNotificationMode(Lum::Model::DoubleDataStream::notifyExplicit);
    data->SetChannels(3);
  }

  virtual ~DiskInfoDataProvider()
  {
    delete percent;
  }

  std::wstring GetString(const DiskInfoModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      return iter->name;
    case 2:
      return iter->dir;
    default:
      return L"";
    }
  }

  Lum::Object* GetObject(const DiskInfoModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 3:
      data->Set(0,(iter->blocks-iter->free)/iter->blocks);
      data->Set(1,iter->avail/iter->blocks);
      data->Set(2,(iter->free-iter->avail)/iter->blocks);

      return percent;
    default:
      return NULL;
    }
  }
};

class MainWindow : public Lum::Dialog
{
private:
  Lum::Model::ActionRef              about;

  Lum::Model::ActionRef              showMap;

  Lum::Model::IntRef                 refreshTime;
  Lum::Model::ActionRef              timer;

  Lum::Table                         *table;
  Lum::Model::SingleLineSelectionRef selection;
  DiskInfoModelRef                   tableModel;
  Lum::Model::StringRef              device;
  Lum::Model::StringRef              dir;
  Lum::Model::StringRef              used;
  Lum::Model::StringRef              free;
  Lum::Model::StringRef              total;
  Lum::Model::DoubleDataStreamRef    data;

  std::list<DiskInfo>                list;

  ScanTask                           scanTask;

public:
  MainWindow()
  : about(new Lum::Model::Action()),
    showMap(new Lum::Model::Action()),
    refreshTime(new Lum::Model::Int(10)),
    timer(new Lum::Model::Action()),
    selection(new Lum::Model::SingleLineSelection),
    tableModel(new DiskInfoModel(new DiskInfoDataProvider())),
    device(new Lum::Model::String()),
    dir(new Lum::Model::String()),
    used(new Lum::Model::String()),
    free(new Lum::Model::String()),
    total(new Lum::Model::String()),
    data(new Lum::Model::DoubleDataStream())
  {
    data->SetNotificationMode(Lum::Model::DoubleDataStream::notifyExplicit);
    data->SetChannels(3);

    scanTask.SetParent(this);
    scanTask.SetCaption(L"Scanning directory...");

    Observe(about);
    Observe(showMap);
    Observe(refreshTime);
    Observe(timer);
    Observe(selection);
    Observe(GetOpenedAction());

    Observe(scanTask.GetAbortedAction());
    Observe(scanTask.GetFinishedAction());
  }

  void PreInit()
  {
    Lum::Panel            *panel,*panel2;
    Lum::PercentPie       *percent;
    Lum::Model::HeaderRef headerModel;

    panel=Lum::VPanel::Create(true,true);
    panel->SetWidth(Lum::Base::Size::workHRel,20);
    panel->SetHeight(Lum::Base::Size::workVRel,40);

    panel2=Lum::HPanel::Create(true,false);
    panel2->Add(Lum::Label::Create()
                ->AddLabel(L"Device:",GetTextValueLeft(device))
                ->AddLabel(L"Directory:",GetTextValueLeft(dir))
                ->AddLabel(L"Size:",GetTextValueRight(total))
                ->AddLabel(L"Used:",GetTextValueRight(used))
                ->AddLabel(L"Free:",GetTextValueRight(free)));
    panel2->AddSpace(true);

    percent=new Lum::PercentPie();
    percent->SetModel(data);
    panel2->Add(percent);

    panel->Add(panel2);
    panel->AddSpace();

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Dir",Lum::Base::Size::stdCharWidth,10);
    headerModel->AddColumn(L"Dev",Lum::Base::Size::stdCharWidth,10);
    headerModel->AddColumn(L"Size",Lum::Base::Size::stdCharWidth,15,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetModel(tableModel);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetDoubleClickAction(showMap);
    table->SetShowHeader(true);
    table->GetTableView()->SetAutoFitColumns(true);
    panel->Add(table);

    SetMain(panel);

    Lum::MenuDesc *menu=new Lum::MenuDesc();

    menu
      ->AddMenuItemSub(_ld(menuProject))
        ->AddMenuItemAction(_ld(menuProjectQuit),Lum::OS::qualifierControl,L"q",GetClosedAction());
    menu
      ->AddMenuItemSub(_ld(menuHelp))
        ->AddMenuItemAction(_ld(menuHelpHelp),NULL)
        ->AddMenuItemAction(_ld(menuHelpAbout),about);

    SetMenu(menu);

    Dialog::PreInit();
  }

  void UpdateDisplay()
  {
    list.clear();
    if (GetDisks(list)) {
      std::list<DiskInfo>::const_iterator iter;

      iter=list.begin();

      // First try to update existing entries in the current table
      while (iter!=list.end()) {
        size_t i;

        for (i=1; i<=tableModel->GetRows(); i++) {
          if (iter->dir==tableModel->GetEntry(i).dir) {
            break;
          }
        }

        if (i<=tableModel->GetRows()) {
          tableModel->GetEntry(i)=*iter;
          tableModel->RedrawRow(i);
        }
        else {
          tableModel->Append(*iter);
        }

        ++iter;
      }

      // Now see, if entries in the table have been removed and must
      // be deleted
      size_t i=1;

      while(i<=tableModel->GetRows()) {
        std::wstring                        directory=tableModel->GetEntry(i).dir;
        std::list<DiskInfo>::const_iterator iter;

        iter=list.begin();
        while (iter!=list.end()) {
          if (iter->dir==directory) {
            break;
          }

          ++iter;
        }

        if (iter==list.end()) {
          tableModel->Delete(i);
        }
        else {
          i++;
        }
      }

    }
  }

  void ShowDetail()
  {
    std::list<DiskInfo>::const_iterator iter;

    if (selection->HasSelection()) {
      iter=list.begin();
      while (iter!=list.end() && iter->dir!=tableModel->GetEntry(selection->GetLine()).dir) {
        ++iter;
      }
    }
    else {
      iter=list.end();
    }

    if (iter!=list.end()) {
      device->Set(iter->name);
      dir->Set(iter->dir);

      total->Set(DoubleToWString(iter->blocks,iter->unit));
      used->Set(DoubleToWString(iter->blocks-iter->free,iter->unit));
      free->Set(DoubleToWString(iter->free,iter->unit)+L" ("+DoubleToWString(iter->avail,iter->unit)+L")");

      data->Enable();
      data->Set(0,(iter->blocks-iter->free)/iter->blocks); // used
      data->Set(1,iter->avail/iter->blocks); // free without reserved
      data->Set(2,(iter->free-iter->avail)/iter->blocks); // reserved
      data->Notify();
    }
    else {
      device->Set(L"");

      used->Set(L"");
      free->Set(L"");
      total->Set(L"");

      data->Disable();
    }
  }

  void InitiateScan()
  {
    if (!selection->HasSelection()) {
      return;
    }

    std::list<DiskInfo>::const_iterator iter=list.begin();

    while (iter!=list.end() && iter->dir!=tableModel->GetEntry(selection->GetLine()).dir) {
      ++iter;
    }

    if (iter==list.end()) {
      return;
    }

    Lum::Base::Path path;

    path.SetNativeDir(iter->dir);

    scanTask.SetSize(iter->blocks*iter->unit);
    scanTask.SetDir(path);
    scanTask.Start();
  }

  void ShowMap()
  {
    DirEntry  *top;
    MapWindow *window;

    top=scanTask.top;
    scanTask.top=NULL;

    window=new MapWindow(top,top->size/*scanTask.size*/);
    window->SetParent(this);

    if (window->Open()) {
      window->EventLoop();
      window->Close();
    }

    delete window;
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      UpdateDisplay();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
    }
    else if (model==timer && timer->IsFinished()) {
      UpdateDisplay();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
    }
    else if (model==refreshTime && IsOpen()) {
      Lum::OS::display->RemoveTimer(timer);
      if (refreshTime->Get()>0) {
        Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
      }
    }
    else if (model==selection) {
      ShowDetail();
    }
    else if (model==showMap && showMap->IsFinished()) {
      InitiateScan();
    }
    else if (model==scanTask.GetFinishedAction() &&
             scanTask.GetFinishedAction()->IsFinished()) {
      ShowMap();
    }
    else if (model==about && about->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }

};

class Main : public Lum::OS::MainDialog<MainWindow>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Show usage of all available disks"));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2004, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::OS::MainDialog<MainWindow>::Prepare();
  }
};

LUM_MAIN(Main,L"DiskUsage")
