/*
  DiceJinni - A dice simulator
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "DiceEdit.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/Value.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/String.h>
#include <Lum/Table.h>

class RangeDialog : public Lum::Dialog
{
private:
  Lum::Model::StringRef startModel;
  Lum::Model::StringRef endModel;

  Lum::Model::ActionRef okAction;

  bool                  hasResult;
  size_t                start;
  size_t                end;

public:
  RangeDialog()
   : startModel(new Lum::Model::String(L"1")),
     endModel(new Lum::Model::String(L"6")),
     okAction(new Lum::Model::Action()),
     hasResult(false)
  {
    Observe(startModel);
    Observe(endModel);
    Observe(okAction);
    Observe(GetClosedAction());
  }

  void PreInit()
  {
    Lum::String *s,*e;

    s=Lum::String::Create(startModel,5,true,false);
    s->SetAlignment(Lum::String::right);
    s->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typePosDec));
    e=Lum::String::Create(endModel,5,true,false);
    e->SetAlignment(Lum::String::right);
    e->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typePosDec));

    SetMain(Lum::VPanel::Create(false,false)
            ->Add(Lum::Label::Create(true,false)
                  ->AddLabel(L"Number range:",
                             Lum::HPanel::Create(true,false)
                             ->Add(s)
                             ->AddSpace()
                             ->Add(e)))
            ->AddSpace()
            ->Add(Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction(),true,false)));
  }

  bool HasResult() {
    return hasResult;
  }

  void GetResult(size_t &start, size_t& end)
  {
    assert(hasResult);

    start=this->start;
    end=this->end;
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      hasResult=true;
      Exit();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }
    else if (model==startModel || model==endModel) {
      if (Lum::Base::WStringToNumber(*startModel,start) &&
          Lum::Base::WStringToNumber(*endModel,end) &&
          end>=start) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }
    }

    Lum::Dialog::Resync(model,msg);
  }

  static bool GetRange(Lum::OS::Window* parent, size_t& start, size_t& end)
  {
    RangeDialog *dlg;
    bool        res;

    dlg=new RangeDialog();
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    if (res) {
      dlg->GetResult(start,end);
    }

    delete dlg;

    return res;
  }
};

DiceEdit::DiceEdit(Dice& dice)
 : dice(dice),
   valuesModel(new ValuesModel(dice.values)),
   selection(new Lum::Model::SingleLineSelection()),
   okAction(new Lum::Model::Action()),
   addAction(new Lum::Model::Action()),
   addNumbersAction(new Lum::Model::Action()),
   editAction(new Lum::Model::Action()),
   removeAction(new Lum::Model::Action()),
   success(false)
{
  EvaluateRights();

  Observe(selection);

  Observe(okAction);
  Observe(addAction);
  Observe(addNumbersAction);
  Observe(editAction);
  Observe(removeAction);

  Observe(GetClosedAction());
}

void DiceEdit::EvaluateRights()
{
  if (valuesModel->GetRows()>0) {
    okAction->Enable();
  }
  else {
    okAction->Disable();
  }

  if (selection.Valid() && selection->HasSelection()) {
    editAction->Enable();
    removeAction->Enable();
  }
  else {
    editAction->Disable();
    removeAction->Disable();
  }
}

void DiceEdit::PreInit()
{
  Lum::Panel            *horiz;
  Lum::Table            *table;
  Lum::Model::HeaderRef headerModel;

  horiz=Lum::HPanel::Create(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Value",Lum::Base::Size::stdCharWidth,30,true);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetHeight(Lum::Base::Size::stdCharHeight,8);
  table->SetShowHeader(true);
  table->GetTableView()->SetAutoHSize(true);
  table->GetTableView()->SetAutoFitColumns(true);
  table->SetModel(valuesModel);
  table->SetHeaderModel(headerModel);
  table->SetSelection(selection);
  table->SetDoubleClickAction(editAction);
  horiz->Add(table);

  horiz->AddSpace();

  horiz->Add(Lum::VPanel::Create(false,true)
             ->Add(Lum::Button::Create(_(L"BTN_ADD",L"_Add"),addAction,true,false))
             ->AddSpace()
             ->Add(Lum::Button::Create(_(L"BTN_ADD_NUMBERS",L"Add _numbers"),addNumbersAction,true,false))
             ->AddSpace()
             ->Add(Lum::Button::Create(_(L"BTN_UPDATE",L"_Edit"),editAction,true,false))
             ->AddSpace()
             ->Add(Lum::Button::Create(_(L"BTN_REMOVE",L"_Remove"),removeAction,true,false)));

  SetMain(Lum::VPanel::Create(true,true)
          ->Add(horiz)
          ->AddSpace()
          ->Add(Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction(),true,false)));

  Dialog::PreInit();
}

void DiceEdit::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==okAction && okAction->IsFinished()) {
    success=true;
    Exit();
  }
  else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==selection && dynamic_cast<const Lum::Model::Selection::Selected*>(&msg)!=NULL) {
    EvaluateRights();
  }
  else if (model==addAction && addAction->IsFinished()) {
    bool                  result;
    Lum::Model::StringRef value(new Lum::Model::String(L""));

    result=Lum::Dlg::TextInput::GetText(GetWindow(),
                                        L"Enter dice value",
                                        L"Please enter the value for this\n"
                                        L"side of the current dice:",
                                        value);

    if (result && value->Length()>0) {
      Value v;

      v.value=*value;
      valuesModel->Append(v);

      configurationChanged=true;
      EvaluateRights();
    }
  }
  else if (model==addNumbersAction && addNumbersAction->IsFinished()) {
    bool   result=false;
    size_t start=0;
    size_t end=0;

    result=RangeDialog::GetRange(GetWindow(),start,end);

    if (result) {
      for (size_t i=start; i<=end; i++) {
        Value v;

        v.value=Lum::Base::NumberToWString(i);
        valuesModel->Append(v);
      }
      configurationChanged=true;
      EvaluateRights();
    }
  }
  else if (model==editAction && editAction->IsFinished()  && editAction->IsEnabled()) {
    assert(selection->HasSelection());

    bool                  result;
    Lum::Model::StringRef value(new Lum::Model::String(valuesModel->GetEntry(selection->GetLine()).value));

    result=Lum::Dlg::TextInput::GetText(GetWindow(),
                                        L"Enter dice value",
                                        L"Please enter the value for this\n"
                                        L"side of the current dice:",
                                        value);

    if (result) {
      valuesModel->GetEntry(selection->GetLine()).value=*value;
      valuesModel->RedrawRow(selection->GetLine());

      configurationChanged=true;
      EvaluateRights();
    }
  }
  else if (model==removeAction && removeAction->IsFinished() && removeAction->IsEnabled()) {
    assert(selection->HasSelection());

    valuesModel->Delete(selection->GetLine());
    // Could be the currently displayed one
    setsModel->Notify();

    configurationChanged=true;
    EvaluateRights();
  }

  Dialog::Resync(model,msg);
}

bool DiceEdit::GetSuccess() const
{
  return success;
}

