/*
 * CoverFinder - Find and download cover images from Amazon 
 * Copyright (C) 2007 - Sven Salzwedel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <gtk/gtk.h>
#include <glib/gstdio.h>

#include "config.h"
#include "cb.h"
#include "ui.h"
#include "search.h"
#include "amazon.h"

static GtkActionEntry action_entries[] = {
	{"FileMenu", NULL, "File"},
	{"EditMenu", NULL, "Edit"},
	{"HelpMenu", NULL, "Help"},
	{"Save", GTK_STOCK_SAVE_AS, "_Save as...", "<control>S", "Save as...",
	 G_CALLBACK (cb_action_save)},
	{"ClearCache", GTK_STOCK_CLEAR, "Clear Cache", NULL, "Clear cover image cache",
	 G_CALLBACK (cb_action_clear_cache)},
	{"Quit", GTK_STOCK_QUIT, "_Quit", "<control>Q", "Quit", G_CALLBACK (cb_action_quit)},
	{"Info", GTK_STOCK_ABOUT, "_Info", NULL, "Info", G_CALLBACK (cb_action_info)}
};

static void init_search_widgets (CFGui * gui, GtkBox * parent, gpointer cb_data)
{
	GtkWidget *hbox, *widget;
	GtkTreeIter iter;
	GtkCellRenderer *renderer;
	GtkListStore *providers;
	GtkTooltips *tooltips;
	gint i;

	widget = gtk_alignment_new (0.5, 0.5, 0, 0);
	gtk_widget_show (widget);
	hbox = gtk_hbox_new (FALSE, 6);
	gtk_widget_show (hbox);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
	gtk_container_add (GTK_CONTAINER (widget), hbox);
	gtk_box_pack_start (parent, widget, FALSE, TRUE, 0);

	widget = gtk_label_new ("Search:");
	gtk_widget_show (widget);
	gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, TRUE, 0);

	widget = gtk_entry_new ();
	gtk_widget_show (widget);
	g_signal_connect (G_OBJECT (widget), "activate", G_CALLBACK (cb_search), cb_data);
	gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, TRUE, 0);
	gui->search_entry = widget;
	
	tooltips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tooltips, gui->search_entry, 
						  "Use \"title:foo\" to search for an album with the title \"foo\". "
						  "Use \"artist:bar\" to search fo albums by the artist \"bar\". "
						  "You can also enter both special search keywords, separated by semicolon, "
						  "i.e. title:foo;artist:bar will search for album \"foo\" by artist "
						  "\"bar\".", NULL);


	providers =
		gtk_list_store_new (PROVIDER_NUM_COLUMNS, G_TYPE_STRING, G_TYPE_INT, G_TYPE_POINTER);
	for (i = 0; i < amazon_num_providers; i++) {
		SProvider provider = amazon_providers[i];

		gtk_list_store_append (providers, &iter);
		gtk_list_store_set (providers, &iter, PROVIDER_COLUMN_DESCRIPTION, provider.description,
							PROVIDER_COLUMN_TYPE, provider.type, PROVIDER_COLUMN_DATA,
							provider.data, -1);
	}

	widget = gtk_combo_box_new_with_model (GTK_TREE_MODEL (providers));
	gtk_widget_show (widget);
	gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, TRUE, 0);

	renderer = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (widget), renderer, TRUE);
	gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (widget), renderer, "text",
									PROVIDER_COLUMN_DESCRIPTION, NULL);
	gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 0);
	gui->providers_combo = widget;
}

static void init_search_result_widgets (CFGui * gui, GtkBox * parent, gpointer cb_data)
{
	GtkWidget *swindow, *tview;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;
	GtkListStore *results;

	swindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (swindow);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swindow), GTK_POLICY_AUTOMATIC,
									GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (swindow), GTK_SHADOW_IN);
	gtk_container_set_border_width (GTK_CONTAINER (swindow), 6);
	gtk_box_pack_start (GTK_BOX (parent), swindow, TRUE, TRUE, 0);

	results =
		gtk_list_store_new (RESULT_NUM_COLUMNS, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_POINTER);
	tview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (results));
	gtk_widget_show (tview);
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (tview), FALSE);
	g_signal_connect (G_OBJECT (tview), "button-press-event", G_CALLBACK (cb_result_button_pressed),
					  cb_data);
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tview));
	g_signal_connect (G_OBJECT (selection), "changed", G_CALLBACK (cb_result_selected), cb_data);
	gtk_container_add (GTK_CONTAINER (swindow), tview);

	renderer = gtk_cell_renderer_pixbuf_new ();
	col =
		gtk_tree_view_column_new_with_attributes ("Thumbnail", renderer, "pixbuf",
												  RESULT_COLUMN_THUMBNAIL, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tview), col);

	renderer = gtk_cell_renderer_text_new ();
	col =
		gtk_tree_view_column_new_with_attributes ("Title", renderer, "markup",
												  RESULT_COLUMN_DESCRIPTION, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tview), col);

	gui->results_view = tview;
}

static void init_uimanager (CFGui * gui, GtkBox * parent, gpointer cb_data)
{
	GtkActionGroup *agrp;
	GError *err = NULL;
	GtkWidget *menu;

	gui->uimgr = gtk_ui_manager_new ();
	agrp = gtk_action_group_new ("default");
	gtk_action_group_add_actions (agrp, action_entries, G_N_ELEMENTS (action_entries), cb_data);
	if (gtk_ui_manager_add_ui_from_file (gui->uimgr, DATADIR "/" PACKAGE "/ui.xml", &err) == 0) {
		ui_error_dialog (gui, err);
		return;
	}
	gtk_ui_manager_insert_action_group (gui->uimgr, agrp, 0);
	gtk_window_add_accel_group (GTK_WINDOW (gui->window),
								gtk_ui_manager_get_accel_group (gui->uimgr));

	menu = gtk_ui_manager_get_widget (gui->uimgr, "/MainMenu");
	gtk_widget_show (menu);
	gtk_box_pack_start (parent, menu, FALSE, TRUE, 0);
}

static void init_statusbar (CFGui * gui, GtkBox * parent, gpointer cb_data)
{
	GtkWidget *statusbar, *progressbar;

	statusbar = gtk_statusbar_new ();
	gtk_widget_show (statusbar);
	gtk_statusbar_set_has_resize_grip (GTK_STATUSBAR (statusbar), TRUE);
	gtk_box_pack_start (parent, statusbar, FALSE, TRUE, 0);
	gui->statusbar = statusbar;

	progressbar = gtk_progress_bar_new ();
	gtk_widget_show (progressbar);
	gtk_box_pack_start (GTK_BOX (statusbar), progressbar, FALSE, FALSE, 0);
	gui->progressbar = progressbar;
}

CFGui *ui_init (gpointer cb_data)
{
	GtkWidget *vbox, *window;
	CFGui *gui;

	gui = g_new (CFGui, 1);
	gui->results_list = NULL;
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_default_size (GTK_WINDOW (window), 500, 375);
	gtk_window_set_title (GTK_WINDOW (window), "CoverFinder");
	gtk_window_set_icon_name (GTK_WINDOW (window), "coverfinder");
	gtk_widget_show (window);
	g_signal_connect (G_OBJECT (window), "delete-event", G_CALLBACK (cb_window_delete), NULL);
	gui->window = window;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox);

	init_uimanager (gui, GTK_BOX (vbox), cb_data);
	init_search_widgets (gui, GTK_BOX (vbox), cb_data);
	init_search_result_widgets (gui, GTK_BOX (vbox), cb_data);
	init_statusbar (gui, GTK_BOX (vbox), cb_data);
	gtk_container_add (GTK_CONTAINER (window), vbox);
	ui_set_save_enabled (gui, FALSE);
	ui_select_search_entry (gui);

	return gui;
}

void ui_error_dialog (CFGui * gui, GError * err)
{
	GtkWidget *dialog;

	dialog =
		gtk_message_dialog_new (GTK_WINDOW (gui->window), GTK_DIALOG_DESTROY_WITH_PARENT,
								GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, err->message);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

void ui_set_save_enabled (CFGui * gui, gboolean state)
{
	GtkAction *action;

	action = gtk_ui_manager_get_action (gui->uimgr, "/MainMenu/FileMenu/Save");
	gtk_action_set_sensitive (action, state);
}


void ui_push_status (CFGui * gui, const gchar * ctx, const gchar * msg)
{
	guint ctxid;

	ctxid = gtk_statusbar_get_context_id (GTK_STATUSBAR (gui->statusbar), ctx);
	gtk_statusbar_push (GTK_STATUSBAR (gui->statusbar), ctxid, msg);
}

void ui_pop_status (CFGui * gui, const gchar * ctx)
{
	guint ctxid;

	ctxid = gtk_statusbar_get_context_id (GTK_STATUSBAR (gui->statusbar), ctx);
	gtk_statusbar_pop (GTK_STATUSBAR (gui->statusbar), ctxid);
}


void ui_set_busy_begin (CFGui * gui)
{
	gui->window_cursor = gdk_cursor_new (GDK_WATCH);
	gdk_window_set_cursor (gui->window->window, gui->window_cursor);
	gtk_widget_set_sensitive (gui->search_entry, FALSE);
}

void ui_set_busy_end (CFGui * gui)
{

	gdk_window_set_cursor (gui->window->window, NULL);
	gdk_cursor_destroy (gui->window_cursor);
	gtk_widget_set_sensitive (gui->search_entry, TRUE);
}

void ui_select_search_entry (CFGui *gui)
{
	gtk_widget_grab_focus (gui->search_entry);
}

void ui_progress_set (CFGui *gui, gdouble val)
{
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (gui->progressbar), val);
}

void ui_progress_reset (CFGui *gui)
{
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (gui->progressbar), 0.0);
}
