/*
 * CoverFinder - Find and download cover images from Amazon 
 * Copyright (C) 2007 - Sven Salzwedel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>

#include "config.h"
#include "search.h"
#include "amazon.h"
#include "net.h"

SProvider amazon_providers[] = {
	{"Amazon (us)", SPROVIDER_AMAZON, "com"}
	,
	{"Amazon (uk)", SPROVIDER_AMAZON, "co.uk"}
	,
	{"Amazon (de)", SPROVIDER_AMAZON, "de"}
	,
	{"Amazon (jp)", SPROVIDER_AMAZON, "jp"}
	,
	{"Amazon (fr)", SPROVIDER_AMAZON, "fr"}
	,
	{"Amazon (ca)", SPROVIDER_AMAZON, "ca"}
};

int amazon_num_providers = G_N_ELEMENTS (amazon_providers);

static gchar *prepare_token (const gchar * token, GError ** err)
{
	gchar *enc_token, *esc_token;
	gsize c_written;

	if ((enc_token = g_convert (token, -1, "ISO-8859-1", "UTF-8", NULL, &c_written, err)) == NULL) {
		return NULL;
	}
	if ((esc_token = net_escape (enc_token, err)) == NULL) {
		g_free (enc_token);
		return NULL;
	}

	g_free (enc_token);
	return (esc_token);
}

static void url_append (GString * url, gchar * str, const gchar * param)
{
	if (str == NULL)
		return;
	g_string_append (url, param);
	g_string_append (url, str);
	g_free (str);
}

static gchar *get_url (const gchar * locale, const gchar * search_string, guint64 page, GError ** err)
{
	gchar **tokens, **stokens, *ret;
	gchar *title = NULL, *artist = NULL, *keywords = NULL;
	gint i = 0;
	GString *url;

	tokens = g_strsplit (search_string, ";", -1);
	while (tokens[i] != NULL) {
		stokens = g_strsplit (tokens[i++], ":", 2);
		if (stokens[0] == NULL) {
			continue;
		}
		if (stokens[1] != NULL) {
			if (!strcmp (stokens[0], "title")) {
				title = prepare_token (stokens[1], err);
			} else if (!strcmp (stokens[0], "artist")) {
				artist = prepare_token (stokens[1], err);
			}
		} else {
			if (keywords != NULL)
				g_free (keywords);
			keywords = prepare_token (stokens[0], err);
		}
		g_strfreev (stokens);

		if (*err != NULL) {
			g_strfreev (tokens);
			return NULL;
		}
	}
	g_strfreev (tokens);

	url = g_string_new ("");
	g_string_append_printf (url, AWS_URL, locale);
	url_append (url, title, AWS_SEARCH_TITLE);
	url_append (url, artist, AWS_SEARCH_ARTIST);
	url_append (url, keywords, AWS_SEARCH_KEYWORDS);
	if (page > 0) {
		g_string_append_printf (url, AWS_PAGE, page);
	}

	ret = url->str;
	g_string_free (url, FALSE);
	return ret;
}

static xmlDoc *get_resultdoc (const gchar * locale, const gchar * search_string, guint64 page, GError ** err)
{
	gchar *tmp_file_n, *url;
	gint fd;
	gboolean net_result;
	xmlDoc *ret;

	if ((url = get_url (locale, search_string, page, err)) == NULL) {
		return NULL;
	}

	if ((fd = g_file_open_tmp (PACKAGE "-XXXXXX", &tmp_file_n, err)) == -1) {
		g_free (url);
		g_free (tmp_file_n);
		return NULL;
	}

	net_result = net_getfile (url, tmp_file_n, err);
	g_free (url);
	close (fd);
	if (net_result == FALSE) {
		return NULL;
	}
	ret = xmlParseFile (tmp_file_n);
	g_unlink (tmp_file_n);
	g_free (tmp_file_n);
	if (ret == NULL) {
		g_set_error (err, 0, 0, "Invalid XML response from server.");
		return NULL;
	}

	return ret;
}

static gchar *get_childvalue (xmlNodePtr parent, xmlChar * childname)
{
	xmlNodePtr element;

	element = parent->children;
	while (element != NULL) {
		if (!xmlStrcmp (element->name, childname) && element->children != NULL) {
			return (gchar *) xmlStrdup (element->children->content);
		}
		element = element->next;
	}
	return NULL;
}

static gchar *get_childvalue_markup (xmlNodePtr parent, xmlChar * childname)
{
	gchar *val, *ret = NULL;

	val = get_childvalue (parent, childname);
	if (val != NULL) {
		ret = g_markup_escape_text (val, -1);
		g_free (val);
	}
	return ret;
}

static const gchar *get_binding_type (xmlNodePtr parent, xmlChar * childname)
{
	xmlChar *binding;
	const gchar *ret;

	binding = (xmlChar *) get_childvalue (parent, childname);
	if (!xmlStrcmp (binding, AWS_XML_BINDING_AUDIO_CD) || !xmlStrcmp (binding, AWS_XML_BINDING_CD)
		|| !xmlStrcmp (binding, AWS_XML_BINDING_CD_AUDIO)) {
		ret = SRESULT_TYPE_CDDA;
	} else if (!xmlStrcmp (binding, AWS_XML_BINDING_DVD)) {
		ret = SRESULT_TYPE_DVD;
	} else {
		ret = SRESULT_TYPE_OTHER;
	}
	g_free (binding);

	return ret;
}

static xmlNodePtr check_document (xmlDoc * result_doc)
{
	xmlNodePtr node;

	node = xmlDocGetRootElement (result_doc);
	if (node == NULL || xmlStrcmp (node->name, AWS_XML_ROOT)) {
		return NULL;
	}

	node = node->children;
	if (node == NULL || xmlStrcmp (node->name, AWS_XML_OREQ)) {
		return NULL;
	}

	node = node->next;
	if (node == NULL || xmlStrcmp (node->name, AWS_XML_ITEMS)) {
		return NULL;
	}

	return node->children;
}

static SResult *get_result (xmlNodePtr element)
{
	SResult *result;
	gint has_img = 0;

	result = search_result_new ();
	while (element != NULL) {
		if (!xmlStrcmp (element->name, AWS_XML_ITEM_ATTRIB)) {
			result->title = get_childvalue_markup (element, AWS_XML_ITEM_ATTRIB_TITLE);
			result->artist = get_childvalue_markup (element, AWS_XML_ITEM_ATTRIB_ARTIST);
			result->type = get_binding_type (element, AWS_XML_ITEM_ATTRIB_BINDING);
		} else if (!xmlStrcmp (element->name, AWS_XML_ITEM_SMALLIMG) && has_img < 1) {
			result->image_url = get_childvalue (element, AWS_XML_ITEM_IMG_URL);
			has_img = 1;
		} else if (!xmlStrcmp (element->name, AWS_XML_ITEM_MEDIUMIMG) && has_img < 2) {
			result->image_url = get_childvalue (element, AWS_XML_ITEM_IMG_URL);
			has_img = 2;
		} else if (!xmlStrcmp (element->name, AWS_XML_ITEM_LARGEIMG)) {
			result->image_url = get_childvalue (element, AWS_XML_ITEM_IMG_URL);
			has_img = 3;
		}
		element = element->next;
	}
	if (has_img == 0) {
		search_result_free (result);
		result = NULL;
	} else {
		result->markup = search_result_get_markup (result);
	}

	return result;
}

static void get_results (xmlNodePtr node, GSList ** result_list)
{
	SResult *result;
	xmlNodePtr element;

	while (node != NULL) {
		if (!xmlStrcmp (node->name, AWS_XML_ITEM)) {
			element = node->children;
			if ((result = get_result (element)) != NULL) {
				*result_list = g_slist_prepend (*result_list, result);
			}
		}

		node = node->next;
	}
}

/*static guint64 get_pages (xmlNodePtr node)
{
	while (node != NULL) {
		if (!xmlStrcmp (node->name, AWS_XML_TOTAL_PAGES)) {
			return g_ascii_strtoull ((gchar *) node->children->content, NULL, 0);
		}
		node = node->next;
	}

	return 0;
}*/

GSList *amazon_search (const gchar * locale, const gchar * token, GError ** err)
{
	xmlDoc *result_doc;
	xmlNodePtr node;
	GSList *result_list = NULL;

	if ((result_doc = get_resultdoc (locale, token, 0, err)) == NULL
		|| (node = check_document (result_doc)) == NULL) {
		return NULL;
	}

	get_results (node, &result_list);
	xmlFreeDoc (result_doc);

	xmlCleanupParser ();
	result_list = g_slist_reverse (result_list);

	return g_slist_sort (result_list, (GCompareFunc) search_results_cmp);
}
