// (c) 2008, Andrew V. Sichevoi
// http://thekondor.net/conler

#include <stdlib.h>
#include <glib.h>

#ifdef MAEMO
  #include <gconf/gconf.h>
  #include <gconf/gconf-client.h>
#endif

#include "common.h"
#include "const.h"
#include "types.h"

static gboolean simulate = FALSE;   /* Simulate mode */
static gchar* config_path = NULL;   /* Alternative config */
static gchar* ap_alt_id = NULL;     /* Alternative AP id */

static GOptionEntry entries[] =
{
  {"config", 'c', 0, G_OPTION_ARG_STRING, &config_path,
    "Path to an alternative configuration file, default: " CONFIG_PATH, "cfg"},
  {"simulate", 's', 0, G_OPTION_ARG_NONE, &simulate,
    "Don't execute commands, only print", NULL},
  {"cid", 'i', 0, G_OPTION_ARG_STRING, &ap_alt_id,
    "Run commands for AP with id = P", "P"},
  { NULL }
};

static gchar* get_connection_name(const gchar* icd_conn_id)
{
    if (!icd_conn_id)
        return NULL;
   
    if (DIABLO != get_maemo_version())
        return g_strdup(icd_conn_id); 

#ifdef MAEMO
    GConfClient* gconf_client;
    GError* error = NULL;
    gchar* key;
    gchar* conn_name;

    g_type_init();
    gconf_client = gconf_client_get_default();
    if (!gconf_client) {
        g_warning("No GConf client is available!\n");
        return NULL;
    }

    key = g_strdup_printf("/system/osso/connectivity/IAP/%s/name", icd_conn_id);
    conn_name = gconf_client_get_string(gconf_client, key, &error);
    if (error) {
        g_warning("AP name obtaining error: %s\n", error->message);
        g_error_free(error);
    }

    g_free(key);
    g_object_unref(gconf_client);

    return conn_name;
#else
    return g_strdup(icd_conn_id);
#endif
}

static AccessPoint_s* find_access_point(GSList* access_points,
                                        const gchar* const name)
{
    if (!access_points || !name)
        return NULL;

    for (GSList* iter = access_points; iter; iter = g_slist_next(iter))
        if (!g_ascii_strcasecmp(AP_LIST_CAST(iter)->name, name))
            return AP_LIST_CAST(iter);

    return NULL;
}

static void execute_command(const gchar* cmd, gpointer simulation_mode)
{
  if (*((gboolean *)simulation_mode))
    g_print("%s\n", cmd);
  else
  {
    GError* error = NULL;
    g_spawn_command_line_async(cmd, &error);
    if (error) {
        g_warning("Command execution error: %s", error->message);
        g_error_free(error);
    }
  }
}

/* Check functions */
static void check_connection_type()
{
  const gchar* icd_type = g_getenv("ICD_CONNECTION_TYPE");

  if (icd_type && !g_ascii_strcasecmp("WLAN_INFRA", icd_type))
    return;

  g_print("Error: no WiFi connection is available\n");
  exit(1);
}

static void check_config(const gchar* config)
{
  if (g_file_test(config, G_FILE_TEST_EXISTS))
    return;

  g_print("Error: Configuration file '%s' isn't available\n", config);
  exit(1);
}

static void process_config(const gchar* config_path, ConlerCfg_s* cfg)
{
  if (read_cfg(config_path, cfg))
    return;

  g_print("Error: Configuration file can't be parsed\n");
  g_print("Please check '%s' for validness\n", config_path);
  exit(1);
}


/* Here we go */
int main(int argc, char** argv)
{
  ConlerCfg_s cfg; 
  GError* error = NULL;
  GOptionContext* context;

  g_set_application_name(PROGRAM_FULL_ID);
  g_print("%s v%s, %s\n\n", PROGRAM_ID, PROGRAM_VERSION, COPYRIGHT);

  context = g_option_context_new("- run AP-specific commands");
  g_option_context_add_main_entries(context, entries, NULL);

  if (!g_option_context_parse(context, &argc, &argv, &error)) {
    g_print("Error: option parsing failed: %s\n", error->message);
    g_print("Use '--help' to see the list of available keys\n");
    g_option_context_free(context);
    exit(1);
  }

 if (!simulate)
    check_connection_type();

  /* Obtain AP */
  gchar* ap_id = ap_alt_id;
  if (!ap_id)  
    ap_id = get_connection_name(g_getenv("ICD_CONNECTION_ID"));
  
  if (!ap_id) {
    g_print("Error: Can't obtain AP name\n");
    exit(1);
  }

  /* Obtain path to the configuration file */
  gchar* config = config_path;
  if (!config_path)
    config = CONFIG_PATH;
  check_config(config);

  process_config(config, &cfg);

  /* Execute AP-specific commands */
  AccessPoint_s* ap;
  ap = find_access_point(cfg.access_points, ap_id);
  if (!ap) {
    g_print("Info: Nothing to do for '%s'\n", ap_id);
    exit(1);
  }

  if (!ap->disabled)
    g_slist_foreach(ap->commands, (GFunc)execute_command, (gpointer)&simulate);
  
  /* Saying thanks to the used memory and releasing it */
  if (config_path)
    g_free(config_path);

  g_free(ap_id);
//  if (ap_alt_id)
//    g_free(ap_alt_id);

//  if (!ap_alt_id && ap_id)
//    g_free(ap_id);
//  else if (ap_alt_id)
//    g_free(ap_alt_id);


  g_option_context_free(context);
  free_cfg(&cfg);

  return 0;
}

