#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{GraphController}.

@todo: Document this module.
"""

import edje

from models.tripsmodel import TripsModel, ALTITUDE, SPEED, RPM
from trips.graphview import GraphView
from trips.eventview import EventView
from trips.zoomview import ZoomView
from common.carlog import DEBUG

ZOOM = 30*60

class GraphController(object):
    """
    @todo: Document this class.
    """
    zoom = [ZOOM, ZOOM * 2, ZOOM * 4, ZOOM * 8, ZOOM * 16]
    max = [100, 4000, 100]

    def __init__(self, canvas, parent):
        self.parent = parent
        self.model = TripsModel()
        graph_image = canvas.Image()
        graph_image.alpha_set(True)
        self.model.init(graph_image)
        self.view = GraphView(canvas, self, graph_image)
        self.event = EventView(canvas, self)
        self.zoom_view = ZoomView(canvas, self)
        self.selected = None
        self.total_time = 0
        self.current_point = 0
        self.realtime = False
        self.zoom_index = 0
        self.field = SPEED
        self.scale = (0, self.max[self.field])
        self.model.add_data_update_cb(self._data_update_cb)

    def set_view(self, theme, view):
        view.part_swallow("trip_view", self.view)
        view.part_swallow("trip_event", self.event)
        view.part_swallow("zoom_view", self.zoom_view)

        self._scale_update(self.scale, True)
        color = edje.file_data_get(theme, "trips:line-color")
        if color:
            r, g, b = [(int(x)) for x in color.split(":")]
            color = (r, g, b)
        else:
            color = (255, 255, 255)

        line_width = edje.file_data_get(theme, "trips:line-width")

        line_width = int(line_width) if line_width else 3

        self.zoom_view.set_theme(theme)
        self.model.set_theme(color, line_width, self.view.get_height())
        self.view.show()
        self.update()

    def set_zoom_down(self):
        if self.zoom_index < (len(self.zoom) -1):
            self.zoom_index += 1
            if self.current_point + self.zoom[self.zoom_index] > self.total_time:
                self.current_point = self.total_time - self.zoom[self.zoom_index]
            self.update()

    def set_zoom_up(self):
        if self.zoom_index > 0:
            self.zoom_index -= 1
            self.update()

    def get_zoom_state(self):
        return (self.zoom_index > 0), (self.zoom_index < len(self.zoom) -1)

    def get_field(self):
        return self.field

    def split_screen(self, splited):
        self.zoom_view.split_screen(splited)

    def update(self):
        if self.current_point < 0:
            self.current_point = 0
        self.model.set_graph(self.field, self.view.get_width(),
                                self.scale, self.zoom_index)
        self.model.update_graph()
        self.update_position()

    def update_position(self):
        zoom = self.zoom[self.zoom_index]
        self.view.move_graph(self.current_point, zoom)
        self.zoom_view.update_times(self.current_point, zoom)

    def move(self, x):
        zoom = self.zoom[self.zoom_index]
        new_point = int(self.current_point - (float(zoom) / self.view.size[0]) * x)
        self.realtime = True if new_point + zoom >= self.total_time else False
        if x > 0 and new_point <= 0:
            return

        if x < 0 and new_point + zoom > self.total_time:
            return
        self.view.move_graph(new_point)
        self.zoom_view.update_times(new_point, zoom)
        self.current_point = new_point

    def end_move(self):
        self.selected_point()

    def selected_point(self, frac=None):
        if frac:
            self.selected = frac
        if self.selected:
            zoom = self.zoom[self.zoom_index]
            point = int(zoom * self.selected) + self.current_point
            if point <= self.total_time:
                latlon = self.model.get_location(point)
                if latlon:
                    self.parent.set_map_position(latlon[0], latlon[1])

    def _scale_update(self, scale, reset=False):
        max = self.max[self.field]
        if reset:
            self.scale = (0, max)
        value = None
        if scale:
            if scale[0] < self.scale[0]:
                value = scale[0] - (scale[0] % max)
                self.scale = int(round(value)), self.scale[1]
            if scale[1] > self.scale[1]:
                value = (self.max[self.field] - (scale[1] % max)) % max
                value += scale[1]
                self.scale = self.scale[0], int(round(value))
        if reset or value is not None:
            self.parent.scale_set(self.scale)
            return True


    def load_file_trip(self, file):
        if self.model.load_file(file):
            self.__reset_trip()
            return True

    def load_current_trip(self):
        if self.model.load_current_trip():
            self.__reset_trip()
            return True

    def __reset_trip(self):
        self.total_time = self.model.get_times()[2]
        self.current_point = 0
        stats = self.model.get_speed_stats()
        self._scale_update(stats, True)
        self.update()

    def set_data_rpm(self):
        self.field = RPM
        stats = self.model.get_rpm_stats()
        self._scale_update(stats, True)
        self.update()

    def set_data_speed(self):
        self.field = SPEED
        stats = self.model.get_speed_stats()
        self._scale_update(stats, True)
        self.update()

    def set_data_altitude(self):
        self.field = ALTITUDE
        stats = self.model.get_altitude_stats()
        self._scale_update(stats, True)
        self.update()

    def _data_update_cb(self, speed, rpm, gps_data, trip_data):
        self.total_time = self.model.get_times()[2]
        if self.field == SPEED:
            value = speed
        elif self.field == RPM:
            value = rpm
        else:
            value = gps_data[2]

        if self._scale_update((0, value)):
            self.update()

        if self.realtime and self.current_point < self.total_time - \
                    self.zoom[self.zoom_index]:
            self.current_point = self.total_time - \
                int(self.zoom[self.zoom_index] * 0.9)
            self.update_position()
