#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License as
#  published by the Free Software Foundation; either version 3 of the
#  License, or (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program. If not, see <http://www.gnu.org/licenses/>.
#


from gauges.gaugeview import GaugeView

class SpeedBarView(GaugeView):

    def __init__(self, canvas, image=None, pieces=7):
        """
        Speed Bar viewer constructor
        @param canvas: Evas canvas object
        @param image: bar image key
        @param pieces: number of the bars
        """
        GaugeView.__init__(self, canvas)
        self.dx = self.dy = self.px = 0
        self.pieces = pieces >= 1 and pieces or 1
        if image is None:
            raise ValueError("Image not specified")
        self.image = self.Image(file=image)
        self.sx, sy = self.image.image_size_get()
        self.clipper.resize(self.sx, sy)
        self.bar_width = self.sx / pieces
        self.image.size_set(self.sx, sy)
        self.image.fill_set(0, 0, self.sx, sy)
        self.image.clip_set(self.clipper)
        self.image.show()
        self.set_value(0, 0)

    def move(self, x, y):
        """ Move objects """
        if self.dx != x or self.dy != y:
            self.dx = x
            self.dy = y
            self.image.move(x, y)
            self.clipper.move(x + self.px, y)

    def set_value(self, value_abs, value_rel):
        """
        Change current value
        @param value_abs: absolute value (not used)
        @param value_rel: relative value from 0.0 to 1.0
        """
        if value_rel <= 0:
            self.px = self.sx
        elif value_rel >= 1:
            self.px = 0
        else:
            self.px = int(self.sx - round(value_rel * self.pieces) * \
                self.bar_width)
        self.clipper.move(self.dx - self.px, self.dy)

def get_param_list():
    return (("image", "img"), ("pieces", "int"))

def get_view(canvas, **params):
    return SpeedBarView(canvas, **params)
