#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{TripsModel}.

@var ALTITUDE:  Trip altitude handle.
@var RPM:       Trip RPM handle.
@var SPEED:     Trip Speed handle.
"""

import os, dbus, time, ecore

from main import trip
from common.singleton import Singleton
from common.carlog import DEBUG, ERROR
from ConfigParser import ConfigParser, DEFAULTSECT
from models.obdmodel import OBDModel
from models.gpsmodel import GPSModel
from models.dbusmodel import CarmandDbusModel, DBUS_BUS_NAME, \
    DBUS_OBJECT_PATH,  DBUS_IFACE_TRIP

ALTITUDE, RPM, SPEED = range(3)

class TripsModel(Singleton, dbus.Interface):
    """
    Implements trip-related features and data.
    """

    def __init__(self):
        Singleton.__init__(self)

        dbus.Interface.__init__(self, CarmandDbusModel().get_bus_object(),
            DBUS_IFACE_TRIP)
        self.initalized = False
        self.update = False
        self.trip_data = None
        self.obdmodel = OBDModel()
        self.gpsmodel = GPSModel()
        self._data_update_cbs = []
        self._trip_reseted_cbs = []

        bus = CarmandDbusModel().get_bus()
        bus.add_signal_receiver(self._trip_reseted,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_TRIP, signal_name = 'TripReseted')
        bus.add_signal_receiver(self._data_flushed,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_TRIP, signal_name = 'DataFlushed')
        bus.add_signal_receiver(self._trip_data,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_TRIP, signal_name = 'DataAvailable')

    def load_file(self, file):
        """
        Loads a trip file.

        @type   file:   file
        @param  file:   Trip file.
        @raise  Exception:  Error loading trip file.
        @rtype: boolean
        @return:    C{True} if trip is initialized, C{False} otherwise.
        """
        if self.initalized:
            self.initalized = False
        try:
            trip.load_file(file)
        except Exception, err:
            ERROR("Error loading trip file: %s" % err)
            return self.initialized

        self.update = False
        self.initalized = True
        return self.initalized

    def load_current_trip(self):
        """
        Loads current trip.

        @raise DBusException: Error getting trip folders.
        @raise DBusException: Error getting time precision.
        @rtype: boolean
        @return:    C{True} if current trip is loaded, C{False} otherwise.
        """
        try:
            folder = self.ActualTripFolder()
        except dbus.exceptions.DBusException, err:
            ERROR("Error in get trip folders: %s" % err)
            return False

        file = os.path.join(folder, "trip.dat")
        self.FlushData()
        up = self.update
        if self.load_file(file):
            if not up:
                try:
                    prec = CarmandDbusModel().GetTimePrecision()
                except dbus.exceptions.DBusException, err:
                    ERROR("Error in get time precision: %s" % err)
                    self.update = False
                    return False

                ecore.timer_add(prec, self.__data_update_cb)
            self.update = True
            return True

    def get_times(self):
        """
        Gets trip times.

        @raise  Exception:  Error getting trip times.
        @rtype: tuple
        @return:    Trip times C{(start time, finish time, total time)}.
        """
        if self.initalized:
            try:
                return trip.get_period()
            except Exception, err:
                DEBUG("Error in get trip times: %s" % err)
        return (0, 0, 0)

    def init(self, image):
        """
        Initializes a trip on a new image.

        @raise Exception: Error initializing graph.
        """
        try:
            trip.init(image)
        except Exception, err:
            DEBUG("Error in init graph: %s" % err)


    def set_graph(self, field, width, scale, zoom):
        """
        Sets trip graph properties.

        @type   field:  number
        @param  field:  Field value.
        @type   width:  number
        @param  width:  Width value.
        @type   scale:  tuple
        @param  scale:  Scale values C{(min, max)}.
        @type   zoom:   number
        @param  zoom:   Zoom value.
        @raise Exception: Error setting trip graph.
        """
        try:
            trip.set_graph(field, width, scale, zoom)
        except Exception, err:
            DEBUG("Error in set graph: %s" % err)

    def set_theme(self, color, line_width, height):
        """
        Sets Carman theme properties on trip.

        @type   color:  tuple
        @param  color:  C{(red, green, blue)} color values.
        @type   line_width: number
        @param  line_width: Line width value.
        @type   height: number
        @param  height: Height value.
        @raise Exception: Error setting theme on trip.
        """
        try:
            trip.set_theme(color, line_width, height)
        except Exception, err:
            DEBUG("Error in set theme: %s" % err)

    def update_graph(self):
        """
        Updates trip graph with latest values.

        @raise  Exception: Error updating trip graph.
        """
        try:
            trip.update_graph()
        except Exception, err:
            DEBUG("Error in update graph: %s" % err)

    def get_speed_stats(self):
        """
        Returns trip speed statistics.

        @raise  Exception:  Error getting trip speed statistics.
        @rtype: tuple
        @return:    Trip speed statistics C{(min_speed, max_speed, avg_speed)}.
        """
        if self.initalized:
            try:
                return trip.get_speed_stats()
            except Exception, err:
                DEBUG("Error in get trip speed avg: %s" % err)

    def get_rpm_stats(self):
        """
        Returns trip RPM statistics.

        @raise  Exception:  Error getting trip RPM statistics.
        @rtype: tuple
        @return:    Trip RPM statistics c{(min_rpm, max_rpm, avg_rpm)}.
        """
        if self.initalized:
            try:
                return trip.get_rpm_stats()
            except Exception, err:
                DEBUG("Error in get trip RPM avg: %s" % err)

    def get_altitude_stats(self):
        """
        Returns trip altitude statistics.

        @raise  Exception:  Error getting trip altitude statistics.
        @rtype: tuple
        @return:    Trip altitude statistics c{(min_alt, max_alt, avg_alt)}.
        """
        if self.initalized:
            try:
                return trip.get_altitude_stats()
            except Exception, err:
                DEBUG("Error in get trip RPM avg: %s" % err)

    def get_max_speed(self):
        """
        Returns trip maximum speed value.

        @rtype: number
        @return:    Trip maximum speed value, if any.
        """
        stat = self.get_speed_stats()
        if stat:
            return stat[1]

    def get_max_rpm(self):
        """
        Returns trip maximum RPM value.

        @rtype: number
        @return:    Trip maximum RPM value, if any.
        """
        stat = self.get_rpm_stats()
        if stat:
            return stat[1]

    def get_max_altitude(self):
        """
        Returns trip maximum altitude value.

        @rtype: number
        @return:    Trip maximum altitude value, if any.
        """
        stat = self.get_altitude_stats()
        if stat:
            return stat[1]

    def get_location(self, time):
        """
        Returns trip location from a given time.

        @type   time:   number
        @param  time:   Time (in seconds) since trip start.
        @raise Exception: Error getting trip location from given time.
        @rtype: tuple
        @return: Trip location C{(lat, lon)}.
        """
        if self.initalized:
            try:
                return trip.get_latlon(time)
            except Exception, err:
                DEBUG("Error in get trip location: %s" % err)

    def get_all_locations(self):
        """
        Returns all trip locations.

        @raise Exception: Error getting trip locations.
        @rtype: tuple
        @return: Tuple of trip locations C{(lat, lon)}.
        """
        if self.initalized:
            try:
                return trip.get_track()
            except Exception, err:
                DEBUG("Error in get trip track: %s" % err)

    def get_trip_distance(self):
        """
        Returns trip distance.

        @raise Exception: Error getting trip distance.
        @rtype: number
        @return: Trip distance.
        """
        if self.initalized:
            try:
                return trip.get_trip_distance()
            except Exception, err:
                DEBUG("Error in get trip distance: %s" % err)

    def is_initalized(self):
        """
        Verifies if trip is initialized.

        @rtype: boolean
        @return: C{True} if trip is initialized, C{False} otherwise.
        """
        return self.initalized

    def is_current(self):
        """
        Verifies if trip is updated.

        @rtype: boolean
        @return: C{True} if trip is updated, C{False} otherwise.
        """
        return self.initalized and self.update

    def add_data_update_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when new trip
        data is available.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self._data_update_cbs:
            self._data_update_cbs.append(cb)

    def del_data_update_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when new
        trip data is available.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self._data_update_cbs:
            self._data_update_cbs.remove(cb)

    def add_trip_reseted_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when trip is
        resetted.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self._trip_reseted_cbs:
            self._trip_reseted_cbs.append(cb)

    def del_trip_reseted_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when trip
        is resetted.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """

        if cb in self._trip_reseted_cbs:
            self._trip_reseted_cbs.remove(cb)

    def _data_flushed(self):
        """
        Display a debug message every time a new trip data is flushed.
        """
        DEBUG("Data flushed received")

    def _trip_reseted(self):
        """
        Calls registered callbacks that wants to be called when trip is
        resetted.
        """
        DEBUG("Trip Reseted received")
        for cb in self._trip_reseted_cbs:
            cb()

    def _trip_data(self, *param):
        """
        Updates trip data with given data.

        @type   param:  tuple
        @param  param:  Trip data.
        """
        self.trip_data = param

    def __data_update_cb(self):
        """
        Callback which updates trip data using a time precision.

        @raise  Exception:  Error adding trip data.
        @rtype: boolean
        @return:    C{True} if update is successful, C{False} otherwise.
        """
        if not self.update:
            return False
        gps_data = self.gpsmodel.get_last_data()
        speed = self.obdmodel.get_last_data("0D")
        rpm = self.obdmodel.get_last_data("0C")
        trip_data = self.trip_data

        if speed:
            speed = int(speed[0])
        elif gps_data:
            speed = int(gps_data[3])

        if rpm:
            rpm = int(rpm[0])

        if not gps_data:
            gps_data = (None, None, None)

        if not trip_data:
            trip_data = [None]

        try:
            trip.add_data(gps_data[0], gps_data[1], gps_data[2], rpm, speed,
                trip_data[0])
        except Exception, err:
            DEBUG("Error in add trip data %s" % err)
            return False

        for cb in self._data_update_cbs:
            cb(speed, rpm, gps_data, self.trip_data)

        return True

    def change_trip_name(self, path, name):
        """
        Changes a trip name.

        @type   path:   string
        @param  path:   Trip path.
        @type   name:   string
        @param  name:   New trip name.
        """
        info = os.path.join(path, "info.txt")
        if os.path.exists(info):
            config = ConfigParser()
            config.read(info)
            config.set(DEFAULTSECT, 'tag', name)
            file = open(info, "w")
            config.write(file)
            file.close()

    def get_trip_info(self, path):
        """
        Returns trip info.

        @type   path:   string
        @param  path:   Trip path.
        @rtype: list
        @return:    Trip info C{[tag, date_time, path]}.
        """
        info = os.path.join(path, "info.txt")
        if os.path.exists(info):
            config = ConfigParser()
            config.read(info)
            epoch = int(config.get(DEFAULTSECT, 'start'))
            date_time = time.strftime("%d %b, %Y <hour>%H:%M",
                    time.localtime(epoch))

            tag = ""
            if config.has_option(DEFAULTSECT, 'tag'):
                tag = config.get(DEFAULTSECT, 'tag').strip()

            return [tag, date_time, path]
