# connmodel - Starts the connection
#
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{ConnectionModel}.

@note: This module tries to import C{conic} module (C{ImportError} raises when
not found). C{conic} module is only available on Maemo-like distributions.
@var    DBUS_ADDRESS: DBus system bus address.
@var    DBUS_BUS_NAME: Dbus internet connection daemon bus name.
@var    DBUS_OBJECT_PATH: Dbus internet connection daemon object path.
@var    DBUS_IFACE: Dbus internet connection daemon interface.
"""

import dbus
import common.ossohelper as osso
from common.singleton import Singleton
from common.carlog import DEBUG, WARNING
try:
    import conic
except ImportError:
    conic = None
    WARNING("conic module not found - are you running on a desktop?")

DBUS_ADDRESS     = "unix:path=/var/run/dbus/system_bus_socket"
DBUS_BUS_NAME    = "com.nokia.icd"
DBUS_OBJECT_PATH = "/com/nokia/icd"
DBUS_IFACE       = "com.nokia.icd"

class ConnectionModel(Singleton):
    """
    Represents and handles all Carman connections which use maemo libconic.

    @cvar CONNECTING: connecting status for libconic connections.
    @cvar CONNECTED: connected status for libconic connections.
    @cvar DISCONNECTING: disconnecting status for libconic connections.
    @cvar IDLE: idle status for libconic status.
    """

    CONNECTING    = "CONNECTING"
    CONNECTED     = "CONNECTED"
    DISCONNECTING = "DISCONNECTING"
    IDLE          = "IDLE"

    def __init__(self):
        Singleton.__init__(self)

        self._conn = None
        self._status_changed_cbs = []
        self._connection_lost_cbs = []
        self._connection_enabled_cbs = []

        if osso.get_device_state() != osso.DEVICE_STATE_NORMAL:
            self._last_status = self.DISCONNECTING
        else:
            self._last_status = self.CONNECTED

        if conic:
            self._start()

    def _event_cb(self, *args):
        """
        This method is registered as a libconic callback. It is used just for
        debugging purposes.

        @type args: object
        @param args: set of arguments from libconic.
        """
        DEBUG("Event conic callback %s" % args)

    def _changed_status(self, name, infra, status, aux):
        """
        Method called when libconic reports a status changed.

        @type name: string
        @param name: connection name.
        @type infra: string
        @param infra: infraestructure name.
        @type status: string
        @param status: new status.
        @type aux: object
        @param aux: not used.
        """
        DEBUG("Changed status %s" % status)
        self._last_status = status
        for cb in self._status_changed_cbs:
            cb(status)
        if status == self.DISCONNECTING:
            for cb in self._connection_lost_cbs:
                cb()
        elif status == self.CONNECTED:
            for cb in self._connection_enabled_cbs:
                cb()
                self.del_connection_enabled_cb(cb)

    def _start(self):
        """
        Starts libconic connection and connect signals.
        """
        self._bus = dbus.bus.BusConnection(DBUS_ADDRESS)
        self._conn = conic.Connection()
        self._bus.add_signal_receiver(self._changed_status, "status_changed",
            DBUS_IFACE, DBUS_BUS_NAME, DBUS_OBJECT_PATH)
        self._conn.connect("connection-event", self._event_cb)
        self._conn.set_property("automatic-connection-events", True)
        self._conn.request_connection(conic.CONNECT_FLAG_NONE)

    def add_status_changed_cb(self, cb):
        """
        Adds a callback to status changed callback list.

        @type cb: callback
        @param cb: callback to be added to the status changed callbacks list.
        """
        if callable(cb) and cb not in self._status_changed_cbs:
            self._status_changed_cbs.append(cb)

    def del_status_changed_cb(self, cb):
        """
        Removes a callback from status changed callbacks list.

        @type cb: callback
        @param cb: callback to be removed from status changed callbacks list.
        """
        if cb in self._status_changed_cbs:
            self._status_changed_cbs.remove(cb)

    def add_connection_lost_cb(self, cb):
        """
        Adds a callback to connection lost callbacks list.

        @type cb: callback
        @param cb: callback to be added to connection lost callbacks list.
        """
        if callable(cb) and cb not in self._connection_lost_cbs:
            self._connection_lost_cbs.append(cb)

    def del_connection_lost_cb(self, cb):
        """
        Removes a callback from connection lost callbacks list.

        @type cb: callback
        @param cb: callback to be removed from connection lost callbacks list.
        """
        if cb in self._connection_lost_cbs:
            self._connection_lost_cbs.remove(cb)

    def add_connection_enabled_cb(self, cb):
        """
        Adds a callback to connection enabled callbacks list.

        @type cb: callback
        @param cb: callback to be added to connection enabled callbacks list.
        """
        if callable(cb) and cb not in self._connection_enabled_cbs:
            self._connection_enabled_cbs.append(cb)

    def del_connection_enabled_cb(self, cb):
        """
        Removes a callback from connection enabled callbacks list.

        @type cb: callback
        @param cb: callback to be removed from connection enabled callbacks
        list.
        """
        if cb in self._connection_enabled_cbs:
            self._connection_enabled_cbs.remove(cb)

    def try_connect(self):
        """
        Request connection from libconic.
        """
        if self._conn:
            self._conn.request_connection(conic.CONNECT_FLAG_NONE)

    def Status(self):
        """
        Returns connection last status.

        @rtype: string
        @return: connect last status.
        """
        return self._last_status
