#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{TrackController} and L{TrackAuxController}
"""

import edje
from maps.trackview import TrackView
from models.mapmodel import MapModel
from models.gpsmodel import GPSModel
from models.trackmodel import TrackModel

class TrackController(object):
    """
    Controls features related to route design.

    @type canvas: L{evas.Canvas}
    @param canvas: Instance of L{evas.Canvas}.
    @type gps_model: L{GPSModel}
    @param gps_model: Instance of L{GPSModel}.
    @type trip: boolean
    @param trip: C{True} if track view from trip data, C{False} otherwise.
    @ivar   __move_cb: When set, this callback moves L{TrackView}'s pointer.
    """
    def __init__(self, canvas, gps_model=None, trip=False):
        self.zoom = None
        self.last_fix = None
        self.last_gps_xy = None
        self.last_trip_fix = None
        self.__move_cb = None
        self.last_angle = 0
        self.gps_ready = False

        self.gps_model = gps_model
        self.map_model = MapModel()

        self.track_model = TrackModel()

        gps = gps_model is not None

        self.track_view = TrackView(canvas, self, gps=gps, trip=trip)

        if gps_model:
            gps_model.add_status_changed_cb(self.__status_changed_cb)
            gps_model.add_data_available_cb(self.__data_available_cb)

    def __status_changed_cb(self, status):
        """
        Called when the GPS status has changed.

        @type status: string
        @param status: GPS status.
        """
        if status != GPSModel.FIXED:
            self.gps_ready = False

    def __data_available_cb(self, *data):
        """
        Called when there is data available from the GPS.

        @type data: object
        @param data: GPS data.
        """
        if not self.gps_ready:
            self.gps_ready = True
            self.track_view.new_track()
            self.track_model.new_track()
        angle = data[4] == data[4] and data[4] or self.last_angle
        if self.last_angle and self.last_fix and \
          abs(self.last_fix[0] - data[0]) <= 0.000004 and \
          abs(self.last_fix[1] - data[1]) <= 0.000004:
            angle = self.last_angle
        else:
            self.last_angle = angle
        if data[0] == data[0] and data[1] == data[1]:
            self.last_fix = data
            x, y = self.map_model.latlon_to_xy(data[0], data[1])
            self.track_model.add_point((x, y))

            if self.zoom is not None:
                self.last_gps_xy = x >> self.zoom, y >> self.zoom
                self.track_view.set_gps_position(self.last_gps_xy[0],
                    self.last_gps_xy[1], angle)
                if self.__move_cb:
                    self.__move_cb(*self.last_gps_xy)

    def set_move_cb(self, cb):
        """
        Called when the user navigates through the map screen.

        @type cb: callback
        @param cb: If callable, it is called with the last datas tracking.
        """
        if callable(cb):
            self.__move_cb = cb
            if self.last_gps_xy:
                cb(self.last_gps_xy[0], self.last_gps_xy[1], False)
        else:
            self.__move_cb = None

    def get_lock(self):
        """
        Verifies if L{__move_cb} is callable.

        @rtype: boolean
        @return: C{True} if L{__move_cb} is callable, C{False} otherwise.
        """
        return callable(self.__move_cb)

    def set_view(self, theme, view):
        """
        Sets the theme screen.

        @type theme: file
        @param theme: theme.edj.
        @type view: TripsView
        @param view: Trips to be drawn on the map.
        """
        track_color = edje.file_data_get(theme, "tracks:main-line-color")
        if track_color:
            r, g, b = [(int(x)) for x in track_color.split(":")]
            self.track_view.set_theme(theme, (r, g, b))
        view.part_swallow("track_view", self.track_view)

    def set_position(self, x, y, zoom):
        """
        Sets the position according to the given coordinates.

        @type x: int
        @param x: coordinate x.
        @type y: int
        @param y: coordinate y.
        @type zoom: int
        @param zoom: Zoom of the map.
        """
        self.track_view.set_position(x, y, zoom)
        if zoom != self.zoom:
            self.zoom = zoom
            if self.last_fix:
                x, y = self.map_model.latlon_to_xy(self.last_fix[0],
                    self.last_fix[1])
                self.last_gps_xy = x >> zoom, y >> zoom
                self.track_view.set_gps_position(self.last_gps_xy[0],
                    self.last_gps_xy[1], self.last_angle)
            if self.last_trip_fix:
                self.set_trip_position(*self.last_trip_fix)

    def set_trip_position(self, lat, lon):
        """
        Sets trip position according to the given latitude and longitude.

        @type lat: int
        @param lat: Latitude.
        @type lon: int
        @param lon: Longitude.
        """
        x, y = self.map_model.latlon_to_xy(lat, lon)
        self.track_view.set_trip_position(x >> self.zoom, y >> self.zoom)
        self.last_trip_fix = lat, lon

    def get_gps_position(self):
        """
        Returns the GPS position.

        @rtype: tuple
        @return: GPS position (M{x}, M{y}).
        """
        return self.track_view.get_gps_position()

    def finalize(self):
        """
        Finishes the track model.
        """
        self.track_model.finalize()

    def get_track(self, left, top, right, bottom):
        """
        Returns the latest track points.

        @type left: int
        @param left: Left side reference.
        @type top: int
        @param top: Top side reference.
        @type right: int
        @param right: Right side reference.
        @type bottom: int
        @param bottom: Bottom side reference.
        """
        track = self.track_model.get_track_area((left << self.zoom,
            top << self.zoom, right << self.zoom, bottom << self.zoom))
        return [[(x >> self.zoom, y >> self.zoom) for x, y in points]
            for points in track]

    def add_track(self, positions):
        """
        Adds new track to L{TrackModel}.

        @type positions: list
        @param positions: List of coordinates.
        """
        self.track_model.clear()
        self.track_model.new_track()
        for position in positions:
            if position is None:
                self.track_model.new_track()
            else:
                x, y = self.map_model.latlon_to_xy(position[0], position[1])
                self.track_model.add_point((x, y))

    def add_point(self, latlong):
        """
        Adds a new point to be drawn on the screen.

        @type latlong: list
        @param latlong: List containing latitude and longitude data.
        """
        if latlong is None:
            self.track_model.new_track()
        else:
            x, y = self.map_model.latlon_to_xy(latlong[0], latlong[1])
            self.track_model.add_point((x, y))

    def clear(self):
        """
        Clears all tracking history.
        """
        self.track_model.clear()
        self.track_view.clear()
        self.track_model.new_track()
        self.track_model.clear_selected_track()

class TrackAuxController(TrackController):
    """
    @type canvas: Canvas
    @param canvas: Instance of L{evas.Canvas}.
    """
    def __init__(self, canvas):
        TrackController.__init__(self, canvas)

    def set_view(self, theme, view):
        """
        Sets screen with theme, trips and track.

        @type theme: file
        @param theme: theme.edj.
        @type view: TripsView
        @param view: Trips to be drawn on the map.
        """
        track_color = edje.file_data_get(theme, "tracks:aux-line-color")
        if track_color:
            r, g, b = [(int(x)) for x in track_color.split(":")]
            self.track_view.set_theme(theme, (r, g, b))
        view.part_swallow("track_aux_view", self.track_view)

    def set_track_model(self, models, track):
        """
        Sets the models and the current selected track.

        @deprecated: This method is not being used.
        @type models: list
        @param models: List of models.
        @type track: string
        @param track: Track filename path.
        """
        self.track_model.set_models(models, track)

    def get_selected_track(self):
        """
        Returns the current selected track.

        @deprecated: This method is not used.
        @rtype: string
        @return: Track filename path.
        """
        return self.track_model.get_selected_track()
