# sensors.py - sensors definitions
# -*- coding: utf-8 -*-
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
Implements L{Sensor}.
"""

from common.carlog import INFO
from common.carmanconfig import CarmanConfig

class Sensor(object):
    """
    Represents a car sensor.

    @type pid: number
    @param pid: the sensor pid (e.g. 0C)
    @type description: string
    @param description: a string describing the sensor
    @param short_description: a string with a short description of
        the sensor
    @type unit: tuple
    @param unit: a tuple with the unit of the sensor data
        (metric and imperial)
    @type min: tuple
    @param min: a tuple with the minimum value for the sensor data
        (metric and imperial)
    @type max: tuple
    @param max: a tuple with the maximum value for the sensor data
    @type calc_func: callback
    @param calc_func: a callable object that convertes the byte array
        received from the scantool to a meanful data
    @type convert_func: callback
    @param convert_func: a callable object that convests a value from the
        metric unit to the imperial
    """
    def __init__(self, pid, description, short_description, unit, min, max,
                 calc_func, convert_func):
        self._pid = pid
        self._description = description
        self._short_description = short_description
        self._unit = unit
        self._min = min
        self._max = max
        self._calc = calc_func
        self._convert = convert_func

        self._process_pid()
        self._process_desc()
        self._process_lists()
        self._process_functions()

    # __init__

    def _process_pid(self):
        """
        Validates the pid format.
        """
        if not isinstance(self._pid, basestring) or\
            len(self._pid) != 2 and len(self._pid) != 4:
            raise ValueError('invalid format of pid: %s' % self._pid)
    # _process_pid

    def _process_desc(self):
        """
        Validates the short description and description format.
        """
        if not isinstance(self._description, basestring):
            raise ValueError('invalid format of description: %s'\
                             % self._description)
        if not isinstance(self._short_description, basestring):
            raise ValueError('invalid format of short description: %s'\
                             % self._short_description)
    # _process_desc

    def _process_lists(self):
        """
        Validates the unit, minimum and maximum tuples.
        """
        if not isinstance(self._unit, list) or not isinstance(self._min, list)\
            or not isinstance(self._max, list):
            raise ValueError('metric unit, min and max should be a list')

        num_units = len(CarmanConfig.get_all_units())
        if not len(self._unit) == num_units or \
            not len(self._min) == num_units or \
            not len(self._max) == num_units:
            raise ValueError('metric unit, min and max should be a list'\
                             ' with %s elements' % num_units)

        if None in (self._unit[0], self._min[0], self._max[0]):
            raise ValueError('metric unit, min and max could not be None')

        imperial = (self._unit[1], self._min[1], self._max[1])
        if None in imperial and not imperial == (None, None, None):
            INFO('pid %s - imperial unit, min or max is missing,' \
                    ' ignoring all of them' % self._pid)
            self._unit[1] = None
            self._min[1] = None
            self._max[1] = None
            self._convert = None

        try:
            self._min[0] = int(self._min[0])
        except ValueError:
            raise ValueError('invalid format of min: %s' % self._min[0])
        if self._min[1] is not None:
            try:
                self._min[1] = int(self._min[1])
            except ValueError:
                raise ValueError('invalid format of min: %s' % self._min[1])

        try:
            self._max[0] = int(self._max[0])
        except ValueError:
            raise ValueError('invalid format of min: %s' % self._max[0])
        if self._max[1] is not None:
            try:
                self._max[1] = int(self._max[1])
            except ValueError:
                raise ValueError('invalid format of min: %s' % self._max[1])
    # _process_tuples

    def bla(self, data): return data

    def _process_functions(self):
        """
        Validates the calc and conversion callable object.
        """
        if self._calc is None:
            if self._pid in __calc_func__:
                self._calc = self.bla
 #                self._calc = __calc_func__[self._pid]
            else:
                raise ValueError('sensor value calculation function not'
                                 ' available for pid %s.'\
                                 % self._pid)
        else:
            if not callable(self._calc):
                raise ValueError('sensor value calculation function not'
                                 ' available for pid %s.'\
                                 % self._pid)
            else:
                try:
                    self._calc(['00', '00', '00', '00', '00', '00'])
                except Exception:
                    raise ValueError('sensor value calculation function could'
                                 ' not be used for pid %s.'\
                                 % self._pid)

        if self._convert is None and self._unit[1] is not None:
            if self._pid in __convert_func__:
                self._convert = __convert_func__[self._pid]
            else:
                raise ValueError('sensor value conversion function not'
                                 ' available for pid %s.'\
                                 % self._pid)
        elif self._unit[1] is not None:
            if not callable(self._convert):
                raise ValueError('sensor value conversion function not'
                                 ' available for pid %s.'\
                                 % self._pid)
            else:
                try:
                    self._convert(0)
                except Exception:
                    raise ValueError('sensor value conversion function could'
                                 ' not be used for pid %s.'\
                                 % self._pid)
    # _process_functions

    def get_pid(self):
        """
        Returns the PID of this sensor.

        @rtype: string
        @return: Sensor PID.
        """
        return self._pid
    # get_pid

    def get_description(self):
        """
        Returns the desciption of this sensor.

        @rtype: string
        @return: Sensor description.
        """
        return self._description
    # get_description

    def get_short_description(self):
        """
        Returns the short description of this sensor.

        @rtype: string
        @return: Sensor short description.
        """
        return self._short_description
    # get_short_description

    def get_unit(self, is_metric=True):
        """
        Returns the unit of this sensor in the metric or imperial system.

        @type is_metric: boolean
        @param is_metric: Metric or Imperial.
        @rtype: string
        @return: Sensor unit.
        """
        if not is_metric and self._unit[1] is not None:
            return self._unit[1]
        return self._unit[0]
    # get_unit

    def get_min(self, is_metric=True):
        """
        Returns the minimum value of this sensor in the metric
        or imperial system.

        @type is_metric: boolean
        @param is_metric: Metric or Imperial.
        @rtype: number
        @return: Sensor's min value.
        """
        if not is_metric and self._min[1] is not None:
            return self._min[1]
        return self._min[0]
    # get_min

    def get_max(self, is_metric=True):
        """
        Returns the minimum value of this sensor in the metric
        or imperial system.

        @type is_metric: boolean
        @param is_metric: Metric or Imperial.
        @rtype: number
        @return: Sensor's max value.
        """
        if not is_metric and self._max[1] is not None:
            return self._max[1]
        return self._max[0]
    # get_max

    def calculate_value(self, data, is_metric=True):
        """
        Calculates the real value of this sensor given an OBD-II data.

        @type data: number
        @param data: OBD-II data
        @type is_metric: boolean
        @param is_metric: Metric or Imperial.
        @rtype: number
        @return: Real value to the given OBD-II data.
        """
        ret = self._calc(data)
        if not is_metric and self._convert is not None:
            ret = self._convert(ret)
        return ret
    # calculate_value

    def convert_to_imperial(self, data):
        """
        Converts the value to imperial metric.

        @type data: number
        @param data: The value to be converted to imperial.
        @rtype: number
        @return: The imperial value.
        """
        if self._convert is not None:
            return self._convert(data)
        return data
    # convert_to_imperial

    def has_imperial(self):
        """
        Returns if this sensor has imperial unit.

        @rtype: boolean
        @return: C{True} if sensor has imperial unit, C{False} otherwise.
        """
        return self._unit[1] is not None
    # has_imperial
# Sensor

#==============================================================================
#  Block containing default functions for OBD-II data conversion. These
#  functions should not be called directly. For further information, 
#  check http://en.wikipedia.org/wiki/OBD-II_PIDs
#==============================================================================
def _extract_data(data, index) :
    """
    Extracts an element of the data list.

    @type data: list
    @param data: A list to extract data from
    @type index: list
    @param index: A list containing the positions to extract data
    @rtype: list
    @return: List of extracted elements.
    """
    ret = []
    for idx in index:
        try:
            ret.append(int(data[idx], 16))
        except IndexError:
            ret.append(0)
    return ret
# _extract_value()

def _value_to_int(data):
    """
    Returns the first byte of the OBD-II data as an integer.

    @type data: number
    @param data: The OBD-II data to convert
    @rtype: number
    @return: First byte of the OBD-II data as an integer.
    """
    return _extract_data(data, [0])[0]
# _value_to_int()

def _calc_percentage(data):
    """
    Returns a percentage of the first byte of the OBD-II data.

    @type data: number
    @param data: The OBD-II data to convert
    @rtype: number
    @return: Percentage of the first byte of the OBD-II data.
    """
    value = _extract_data(data, [0])[0]
    return value * 100 / 255.0
# _calc_load_engine()

def _calc_temperature(data):
    """
    Returns a temperature from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted
    @rtype: number
    @return: temperature from the OBD-II data.
    """
    value = _extract_data(data, [0])[0]
    return value - 40
# _calc_temperature()

def _calc_fuel_perc_trim(data):
    """
    Returns the fuel trim percentage from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The fuel trim percentage from the OBD-II data.
    """
    value = _extract_data(data, [0])[0]
    return 0.7812 * (value - 128)
# _calc_fuel_perc_trim()

def _calc_fuel_perc_trim2(data):
    """
    Returns the fuel trim of the second byte of the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The fuel trim of the second byte of the OBD-II data.
    """
    value = _extract_data(data, [1])[0]
    return 0.7812 * (value - 128)
# _calc_fuel_perc_trim()

def _calc_timing_advance(data):
    """
    Returns the timing advance from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The timing advance from the OBD-II data.
    """
    value = _extract_data(data, [0])[0]
    return (value / 2.0) - 64
# _calc_timing_advance()

def _calc_fuel_pressure(data):
    """
    Returns the fuel pressure from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The fuel pressure from the OBD-II data.
    """
    value = _extract_data(data, [0])[0]
    return value * 3
# _calc_fuel_pressure()

def _calc_engine_rpm(data):
    """
    Returns the engine RPM from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The engine RPM from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return ((value1 * 256) + value2) / 4.0
# _calc_engine_rpm()

def _calc_maf(data):
    """
    Returns the MAF value from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The MAF value from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return ((value1 * 256) + value2) / 100.0
# _calc_maf()

def _calc_seconds(data):
    """
    Returns the seconds from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The seconds from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return (value1 * 256) + value2
# _calc_seconds()

def _calc_distance(data):
    """
    Returns the distance from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The distance from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return (value1 * 256) + value2
# _calc_distance()

def _calc_fuel_pressure2(data):
    """
    Returns the fuel pressure from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The fuel pressure from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return ((value1 * 256) + value2) * 0.079
# _calc_fuel_pressure()

def _calc_fuel_pressure_diesel(data):
    """
    Returns the fuel (diesel) pressure from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The fuel (diesel) pressure from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return ((value1 * 256) + value2) * 10
# _calc_fuel_pressure_diesel()

def _calc_o2_voltage(data):
    """
    Returns the O2 sensor voltage from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to convert
    @rtype: number
    @return: The O2 sensor voltage from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [2, 3])
    return ((value1 * 256) + value2) * 0.000122
# _calc_o2_voltage()

def _calc_egr_error(data):
    """
    Returns the EGR error from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The EGR error from the OBD-II data.
    """
    value = _extract_data(data, [0])[0]
    return (value * 0.78125) - 100
# _calc_egr_error()

def _calc_vapor_pressure(data):
    """
    Returns the vapor pressure from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The vapor pressure from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return (((value1 * 256) + value2) / 4.0) - 8192
# _calc_vapor_pressure()

def _calc_o2_amp(data):
    """
    Returns the O2 sensor amperage from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The O2 sensor amperage from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [2, 3])
    return (((value1 * 256) + value2) * 0.00391) - 128
# _calc_o2_amp()

def _calc_cat_temp(data):
    """
    Returns the catalyst temperature from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The catalyst temperature from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return (((value1 * 256) + value2) / 10.0) - 40
# _calc_cat_temp()

def _calc_ctr_voltage(data):
    """
    Returns the control module voltage

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The control module voltage.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return ((value1 * 256) + value2) / 1000.0
# _calc_ctr_voltage()

def _calc_abs_load(data):
    """
    Returns the engine absolute load value from OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The engine absolute load value from OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return (((value1 * 256) + value2) * 100) / 255.0
# _calc_abs_load()

def _calc_cmd_eq(data):
    """
    Returns the equivalence ration from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: data
    @return: The equivalence ration from the OBD-II data.
    """
    value1, value2 = _extract_data(data, [0, 1])
    return ((value1 * 256) + value2) * 0.0000305
# _calc_cmd_eq()

def _calc_oxygen_voltage(data):
    """
    Returns the oxygen sensor voltage from the OBD-II data.

    @type data: number
    @param data: The OBD-II data to be converted.
    @rtype: number
    @return: The oxygen sensor voltage from the OBD-II data.
    """
    value = _extract_data(data, [0])[0]
    return value * 0.005
# _calc_oxygen_voltage

__calc_func__ = {
    '04' : _calc_percentage,
    '05' : _calc_temperature,
    '06' : _calc_fuel_perc_trim,
    '07' : _calc_fuel_perc_trim,
    '08' : _calc_fuel_perc_trim,
    '09' : _calc_fuel_perc_trim,
    '0A' : _calc_fuel_pressure,
    '0B' : _value_to_int,
    '0C' : _calc_engine_rpm,
    '0D' : _value_to_int,
    '0E' : _calc_timing_advance,
    '0F' : _calc_temperature,
    '10' : _calc_maf,
    '11' : _calc_percentage,
    '14:1' : _calc_oxygen_voltage,
    '14:2' : _calc_fuel_perc_trim2,
    '15:1' : _calc_oxygen_voltage,
    '15:2' : _calc_fuel_perc_trim2,
    '16:1' : _calc_oxygen_voltage,
    '16:2' : _calc_fuel_perc_trim2,
    '17:1' : _calc_oxygen_voltage,
    '17:2' : _calc_fuel_perc_trim2,
    '18:1' : _calc_oxygen_voltage,
    '18:2' : _calc_fuel_perc_trim2,
    '19:1' : _calc_oxygen_voltage,
    '19:2' : _calc_fuel_perc_trim2,
    '1A:1' : _calc_oxygen_voltage,
    '1A:2' : _calc_fuel_perc_trim2,
    '1B:1' : _calc_oxygen_voltage,
    '1B:2' : _calc_fuel_perc_trim2,
    '1F' : _calc_seconds,
    '21' : _calc_distance,
    '22' : _calc_distance,
    '22' : _calc_fuel_pressure2,
    '23' : _calc_fuel_pressure_diesel,
    '24:1' : _calc_cmd_eq,
    '24:2' : _calc_o2_voltage,
    '25:1' : _calc_cmd_eq,
    '25:2' : _calc_o2_voltage,
    '26:1' : _calc_cmd_eq,
    '26:2' : _calc_o2_voltage,
    '27:1' : _calc_cmd_eq,
    '27:2' : _calc_o2_voltage,
    '28:1' : _calc_cmd_eq,
    '28:2' : _calc_o2_voltage,
    '29:1' : _calc_cmd_eq,
    '29:2' : _calc_o2_voltage,
    '2A:1' : _calc_cmd_eq,
    '2A:2' : _calc_o2_voltage,
    '2B:1' : _calc_cmd_eq,
    '2B:2' : _calc_o2_voltage,
    '2C' : _calc_percentage,
    '2D' : _calc_egr_error,
    '2E' : _calc_percentage,
    '2F' : _calc_percentage,
    '30' : _value_to_int,
    '31' : _calc_distance,
    '32' : _calc_vapor_pressure,
    '33' : _value_to_int,
    '34:1' : _calc_cmd_eq,
    '34:2' : _calc_o2_amp,
    '35:1' : _calc_cmd_eq,
    '35:2' : _calc_o2_amp,
    '36:1' : _calc_cmd_eq,
    '36:2' : _calc_o2_amp,
    '37:1' : _calc_cmd_eq,
    '37:2' : _calc_o2_amp,
    '38:1' : _calc_cmd_eq,
    '38:2' : _calc_o2_amp,
    '39:1' : _calc_cmd_eq,
    '39:2' : _calc_o2_amp,
    '3A:1' : _calc_cmd_eq,
    '3A:2' : _calc_o2_amp,
    '3B:1' : _calc_cmd_eq,
    '3B:2' : _calc_o2_amp,
    '3C' : _calc_cat_temp,
    '3D' : _calc_cat_temp,
    '3E' : _calc_cat_temp,
    '3F' : _calc_cat_temp,
    '42' : _calc_ctr_voltage,
    '43' : _calc_abs_load,
    '44' : _calc_cmd_eq,
    '45' : _calc_percentage,
    '46' : _calc_temperature,
    '47' : _calc_percentage,
    '48' : _calc_percentage,
    '49' : _calc_percentage,
    '4A' : _calc_percentage,
    '4B' : _calc_percentage,
    '4C' : _calc_percentage,
    '4C' : _calc_percentage,
    '4D' : _calc_seconds,
    '4E' : _calc_seconds
}

#===============================================================================
#  This block contain functions used in the conversion from metric to the
#  imperial unit.
#===============================================================================
def _convert_temp(value):
    """
    Converts the temperature from °C to °F.

    @type value: number
    @param value: The temperature in °C.
    @rtype: number
    @return: The °F temperature.
    """
    return ((value * 9) / 5.0) - 32
# _convert_temp()

def _convert_press(value):
    """
    Converts the pressure from kPa to inMg.

    @type value: number
    @param value: The pressure in kPa.
    @rtype: number
    @return: The pressure in inMg.
    """
    return (value * 29) / 100.0
# _convert_press()

def _convert_press2(value):
    """
    Converts the pressure from Pa to inMg.

    @type value: number
    @param value: The pressure in Pa.
    @rtype: number
    @return: The pressure in inMg.
    """
    return (value * 29) / 100000.0
# _convert_press()

def _convert_speed(value):
    """
    Converts the speed from Km/h to MPH.

    @type value: number
    @param value: The speed in Km/h.
    @rtype: number
    @return: The speed in MPH.
    """
    return (value * 62) / 100.0
# _convert_vel()

def _convert_air(value):
    """
    Converts the air flow from g/s to lb/min.

    @type value: number
    @param value: The air flow in g/s.
    @rtype: number
    @return: The air from in lb/min.
    """
    return (value * 132) / 1000.0
# _convert_air()

def _convert_distance(value):
    """
    Converts the distance from Km to Miles.

    @type value: number
    @param value: The distance in Km.
    @rtype: number
    @return: The distance in Miles.
    """
    return value * 0.6213
# _convert_distance()

__convert_func__ = {
    '05' : _convert_temp,
    '0F' : _convert_temp,
    '46' : _convert_temp,

    '0A' : _convert_press,

    '0B' : _convert_press,
    '0D' : _convert_speed,


    '21' : _convert_distance,
    '10' : _convert_air,
    '22' : _convert_press,
    '23' : _convert_press,
    '31' : _convert_distance,
    '32' : _convert_press2,
    '33' : _convert_press,
    '3C' : _convert_temp,
    '3D' : _convert_temp,
    '3E' : _convert_temp,
    '3F' : _convert_temp,
}

#===============================================================================
#  This block contain functions used in the conversion from imperial to the
#  metric unit.
#===============================================================================
def _unconvert_temp(value):
    """
    Converts the temperature from °F to °C.

    @type value: number
    @param value: The temperature in °F.
    @rtype: number
    @return: The temperature in °C.
    """
    return ((value + 32) * 5) / 9.0
    #return ((value * 9) / 5.0) - 32
# _unconvert_temp()

def _unconvert_press(value):
    """
    Converts the pressure from inMg to kPa.

    @type value: number
    @param value: The pressure in inMg.
    @rtype: number
    @return: The pressure in inMg.
    """
    return (value * 100 ) / 29.0
    #return (value * 29) / 100.0
# _unconvert_press()

def _unconvert_press2(value):
    """
    Converts the pressure from inMg to Pa

    @type value: number
    @param value: The pressure in inMg.
    @rtype: number
    @return: The pressure in Pa.
    """
    return (value * 100000) / 29.0
    #return (value * 29) / 100000.0
# _unconvert_press2()

def _unconvert_speed(value):
    """
    Converts the speed from MPH to Km/h.

    @type value: number
    @param value: The speed in MPH
    @rtype: number
    @return: The speed in Km/h.
    """
    return (value * 100) / 62.0
    #return (value * 62) / 100.0
# _unconvert_speed()

def _unconvert_air(value):
    """
    Converts the air flow from lb/min to g/s.

    @type value: number
    @param value: The air flow in lb/min.
    @rtype: number
    @return: The air flow in g/s.
    """
    return (value * 1000) / 132.0
    #return (value * 132) / 1000.0
# _unconvert_air()

def _unconvert_distance(value):
    """
    Converts the distance from Miles to Km.

    @type value: number
    @param value: The distance in Miles.
    @rtype: number
    @return: The distance in Km.
    """
    return value / 0.6213
    #return value * 0.6213
# _unconvert_distance()

__unconvert_func__ = {
    '05' : _unconvert_temp,
    '0A' : _unconvert_press,
    '0B' : _unconvert_press,
    '0D' : _unconvert_speed,
    '0F' : _unconvert_temp,
    '10' : _unconvert_air,
    '21' : _unconvert_distance,
    '22' : _unconvert_press,
    '23' : _unconvert_press,
    '31' : _unconvert_distance,
    '32' : _unconvert_press2,
    '33' : _unconvert_press,
    '3C' : _unconvert_temp,
    '3D' : _unconvert_temp,
    '3E' : _unconvert_temp,
    '3F' : _unconvert_temp,
    '46' : _unconvert_temp
}
